/*
    Melody maid
    copyright (c) 1998-2006 Kazuki IWAMOTO http://www.maid.org/ iwm@maid.org

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
*/
#include "abort.h"
#include "add.h"
#include "command.h"
#include "edit.h"
#include "export.h"
#include "file.h"
#include "general.h"
#include "genre.h"
#include "mm_dialog.h"
#include "mm_list.h"
#include "mmaid.h"
#include "process.h"
#include "reload.h"
#include "root.h"
#include "version.h"
#include "misc/fileio.h"
#include "orz/orzcharlist.h"
#include "orz/orzcharset.h"
#include "orz/orzconf.h"
#include "orz/orzhistory.h"
#include "orz/orzmdi.h"
#ifdef G_OS_WIN32
# include <gdk/gdkwin32.h>
#endif /* G_OS_WIN32 */


/******************************************************************************
*                                                                             *
* ja:メニュー関数群                                                           *
*                                                                             *
******************************************************************************/
void
command_new (GtkWidget *widget,
             gpointer   user_data)
{
  file_open_edit (NULL);
}


void
command_open (GtkWidget *widget,
              gpointer   user_data)
{
  gchar *file = NULL, *path, *title;
  GtkStockItem stock_item;
  GtkWidget *dialog;

  gtk_stock_lookup (GTK_STOCK_OPEN, &stock_item);
  title = misc_mnemonic_to_text (stock_item.label);
#if GTK_CHECK_VERSION(2,4,0)
  dialog = gtk_file_chooser_dialog_new (title,
                        GTK_WINDOW (window), GTK_FILE_CHOOSER_ACTION_OPEN,
                        GTK_STOCK_CANCEL, GTK_RESPONSE_CANCEL,
                        GTK_STOCK_OPEN, GTK_RESPONSE_ACCEPT,
                        NULL);
  if (open_path)
    gtk_file_chooser_set_current_folder (GTK_FILE_CHOOSER (dialog), open_path);
  if (gtk_dialog_run (GTK_DIALOG (dialog)) == GTK_RESPONSE_ACCEPT)
    file = gtk_file_chooser_get_filename (GTK_FILE_CHOOSER (dialog));
#else /* not GTK_CHECK_VERSION(2,4,0) */
  dialog = gtk_file_selection_new (title);
  if (open_path)
    {
      path = g_strconcat (open_path, G_SEARCHPATH_SEPARATOR_S, NULL);
      gtk_file_selection_set_filename (GTK_FILE_SELECTION (dialog), path);
      g_free (path);
    }
  gtk_file_selection_hide_fileop_buttons (GTK_FILE_SELECTION (dialog));
  if (gtk_dialog_run (GTK_DIALOG (dialog)) == GTK_RESPONSE_OK)
    file = g_strdup (gtk_file_selection_get_filename
                                                (GTK_FILE_SELECTION (dialog)));
#endif /* not GTK_CHECK_VERSION(2,4,0) */
  g_free (title);
  gtk_widget_destroy (dialog);
  if (file)
    {
      g_free (open_path);
      path = fileio_get_full_path (file);
      open_path = g_path_get_dirname (path);
      g_free (path);
      file_open_edit (file);
      g_free (file);
    }
}


void
command_close (GtkWidget *widget,
               gpointer   user_data)
{
  gint page;

  if ((page = gtk_notebook_get_current_page (GTK_NOTEBOOK (mdi))) >= 0
    && prompt_close ((MmaidWindow *)orz_mdi_get_data (ORZ_MDI (mdi), page)))
    gtk_notebook_remove_page (GTK_NOTEBOOK (mdi), page);
}


void
command_save (GtkWidget *widget,
              gpointer   user_data)
{
  gchar *root = NULL, *tmp;
  const gchar *file;
  gint i = G_MAXINT, page_num;
  MmaidWindow *mmaid;

  page_num = gtk_notebook_get_current_page (GTK_NOTEBOOK (mdi));
  mmaid = orz_mdi_get_data (ORZ_MDI (mdi), page_num);
  file = orz_mdi_get_file (ORZ_MDI (mdi), page_num);
  tmp = g_path_get_dirname (file);
  root = g_strconcat (tmp, G_DIR_SEPARATOR_S, NULL);
  g_free (tmp);
  if (!orz_mdi_get_created (ORZ_MDI (mdi), page_num))
    {
      gchar *path;
      gint leng;

#if GLIB_CHECK_VERSION(2,6,0)
      path = g_filename_display_name (root);
#else /* not GLIB_CHECK_VERSION(2,6,0) */
      path = g_filename_to_utf8 (root, -1, NULL, NULL, NULL);
#endif /* not GLIB_CHECK_VERSION(2,6,0) */
      leng = g_strlen (path);
      for (i = mm_list_length (MM_LIST (mmaid->list)) - 1; i >= 0; i--)
        {
          const ID3Tag *id3tag;

          id3tag = mm_list_get_nth (MM_LIST (mmaid->list), i);
          if (g_strfilencmp (id3tag->file, path, leng))
            break;
        }
      g_free (path);
    }
  if (i >= 0)
    {
      command_saveas (widget, user_data);
    }
  else
    {
      gboolean result = TRUE;

      if (fileio_isfile (file))
        {
          gchar *utf8str;
          GtkWidget *dialog;

#if GLIB_CHECK_VERSION(2,6,0)
          utf8str = g_filename_display_name (file);
#else /* not GLIB_CHECK_VERSION(2,6,0) */
          utf8str = g_filename_to_utf8 (file, -1, NULL, NULL, NULL);
#endif /* not GLIB_CHECK_VERSION(2,6,0) */
          dialog = gtk_message_dialog_new (GTK_WINDOW (window),
                    GTK_DIALOG_MODAL | GTK_DIALOG_DESTROY_WITH_PARENT,
                    GTK_MESSAGE_INFO, GTK_BUTTONS_YES_NO,
                    _("%s\nThis file is existed.\n\nOver write?"), utf8str);
          g_free (utf8str);
          g_signal_connect (G_OBJECT (dialog), "key-press-event",
                                    G_CALLBACK (misc_dialog_key_press), NULL);
          result = gtk_dialog_run (GTK_DIALOG (dialog)) == GTK_RESPONSE_YES;
          gtk_widget_destroy (dialog);
        }
      if (result &&  (mmaid->filekind == FILE_KIND_DB
                        ? file_save_db (file, mmaid, root, mmaid->charset)
                        : file_save_list (file, mmaid, root, mmaid->charset,
                                                            mmaid->playlist))
                && (mmaid->filekind == FILE_KIND_DB || mmaid->playlist != 2))
        orz_mdi_set_edited (ORZ_MDI (mdi), page_num, FALSE);
    }
  g_free (root);
}


/* ja:ラジオボタンが変更された */
static void
command_save_toggled_radio0 (GtkWidget *widget,
                             GtkWidget *hbox)
{
  gtk_widget_set_sensitive (hbox, FALSE);
}


/* ja:ラジオボタンが変更された */
static void
command_save_toggled_radio1 (GtkWidget *widget,
                             GtkWidget *hbox)
{
  gtk_widget_set_sensitive (hbox, TRUE);
}


/* ja:コンボボックスが変更された */
static void
command_save_changed (GtkWidget *widget,
                      GtkWidget *dialog)
{
  gchar *root, *utf8str;

#if GTK_CHECK_VERSION(2,4,0)
# if GTK_CHECK_VERSION(2,6,0)
  utf8str = gtk_combo_box_get_active_text (GTK_COMBO_BOX (widget));
# else /* not GTK_CHECK_VERSION(2,6,0) */
  GtkTreeIter iter;

  if (gtk_combo_box_get_active_iter (GTK_COMBO_BOX (widget), &iter))
        gtk_tree_model_get (gtk_combo_box_get_model (GTK_COMBO_BOX (widget)),
                                                    &iter, 0, &utf8str, -1);
# endif /* not GTK_CHECK_VERSION(2,6,0) */
  root = g_filename_from_utf8 (utf8str, -1, NULL, NULL, NULL);
  gtk_file_chooser_set_current_folder (GTK_FILE_CHOOSER (dialog), root);
  g_free (root);
  g_free (utf8str);
#else /* not GTK_CHECK_VERSION(2,4,0) */
  gchar *file, *name, *root;

  name = g_path_get_basename (gtk_file_selection_get_filename
                                                (GTK_FILE_SELECTION (dialog)));
  root = g_filename_from_utf8
            (gtk_entry_get_text (GTK_ENTRY (widget)), -1, NULL, NULL, NULL);
  file = g_strconcat (root, name, NULL);
  gtk_file_selection_set_filename (GTK_FILE_SELECTION (dialog), file);
  g_free (file);
  g_free (name);
  g_free (root);
#endif /* not GTK_CHECK_VERSION(2,4,0) */
}


/* ja:コンボボックスが変更された */
static void
command_save_edited (GtkWidget *widget,
                     GtkWidget *dialog)
{
#if GTK_CHECK_VERSION(2,4,0)
  gtk_dialog_set_response_sensitive (GTK_DIALOG (dialog), GTK_RESPONSE_ACCEPT,
#else /* not GTK_CHECK_VERSION(2,4,0) */
  gtk_dialog_set_response_sensitive (GTK_DIALOG (dialog), GTK_RESPONSE_OK,
#endif /* not GTK_CHECK_VERSION(2,4,0) */
                                orz_charset_is_valid (ORZ_CHARSET (widget)));
}


void
command_saveas (GtkWidget *widget,
                gpointer   user_data)
{
  gchar *file, *root = NULL, *charset, *name, *path, *tmp, **ary, *title;
  gint playlist, i, page_num;
  guint filekind;
  GList *glist = NULL;
  GtkStockItem stock_item;
  GtkWidget *dialog, *hbox0, *hbox1, *vbox;
  GtkWidget *combo0, *combo1, *label0, *label1;
  GtkWidget *radio0, *radio1, *radio2, *radio3, *radio4;
  MmaidWindow *mmaid;

  page_num = gtk_notebook_get_current_page (GTK_NOTEBOOK (mdi));
  mmaid = orz_mdi_get_data (ORZ_MDI (mdi), page_num);
  /* ja:共通のルートを求める */
  for (i = mm_list_length (MM_LIST (mmaid->list)) - 1; i >= 0; i--)
    {
      const ID3Tag *id3tag;

      id3tag = mm_list_get_nth (MM_LIST (mmaid->list), i);
      if (root)
        {
          gint j;

          for (j = 0; root[j] != '\0'; j++)
            if (root[j] != id3tag->file[j])
              break;
          root[j] = '\0';
        }
      else
        {
          root = g_strdup (id3tag->file);
        }
    }
  path = g_strrchr (root, G_DIR_SEPARATOR);
  if (!path)
    {
      dialog = gtk_message_dialog_new (GTK_WINDOW (window),
                            GTK_DIALOG_MODAL | GTK_DIALOG_DESTROY_WITH_PARENT,
                            GTK_MESSAGE_WARNING, GTK_BUTTONS_OK,
                            _("Root Directory Error"));
      g_signal_connect (G_OBJECT (dialog), "key-press-event",
                                    G_CALLBACK (misc_dialog_key_press), NULL);
      gtk_dialog_run (GTK_DIALOG (dialog));
      gtk_widget_destroy (dialog);
      return;
    }
  *path = '\0';
#ifdef G_OS_WIN32
  if (g_strlen (root) > 2
            && G_IS_DIR_SEPARATOR (root[0]) && G_IS_DIR_SEPARATOR (root[1]))
    {
      path = g_strdup (G_DIR_SEPARATOR_S G_DIR_SEPARATOR_S);
      ary = g_strsplit (root + 2, G_DIR_SEPARATOR_S, 0);
    }
  else
    {
      ary = g_strsplit (root, G_DIR_SEPARATOR_S, 0);
    }
#else /* not G_OS_WIN32 */
  ary = g_strsplit (root, G_DIR_SEPARATOR_S, 0);
  path = NULL;
#endif /* not G_OS_WIN32 */
  g_free (root);
  root = NULL;
  for (i = 0; ary[i]; i++)
    {
      path = path ? g_strconcat (path, ary[i], G_DIR_SEPARATOR_S, NULL)
                  : g_strconcat (      ary[i], G_DIR_SEPARATOR_S, NULL);
      glist = g_list_append (glist, path);
    }
  g_strfreev (ary);
  if (save_path)
    {
      gchar *utf8str;

      utf8str = g_filename_to_utf8 (save_path, -1, NULL, NULL, NULL);
      if (g_strfilencmp (utf8str, path, g_strlen (utf8str)) == 0)
        root = g_strconcat (save_path, G_DIR_SEPARATOR_S, NULL);
      g_free (utf8str);
    }
  if (!root)
    root = g_filename_from_utf8 (path, -1, NULL, NULL, NULL);
  /* ja:ダイアログ */
  gtk_stock_lookup (GTK_STOCK_SAVE_AS, &stock_item);
  title = misc_mnemonic_to_text (stock_item.label);
#if GTK_CHECK_VERSION(2,4,0)
  dialog = gtk_file_chooser_dialog_new (title,
                        GTK_WINDOW (window), GTK_FILE_CHOOSER_ACTION_SAVE,
                        GTK_STOCK_CANCEL, GTK_RESPONSE_CANCEL,
                        GTK_STOCK_SAVE_AS, GTK_RESPONSE_ACCEPT,
                        NULL);
#else /* not GTK_CHECK_VERSION(2,4,0) */
  dialog = gtk_file_selection_new (title);
#endif /* not GTK_CHECK_VERSION(2,4,0) */
  g_free (title);
  /* ja:ラジオボタン */
  radio0 = gtk_radio_button_new_with_mnemonic (NULL, _("_iDB File"));
  radio1 = gtk_radio_button_new_with_mnemonic_from_widget
                                (GTK_RADIO_BUTTON (radio0), _("_Play List"));
  radio2 = gtk_radio_button_new_with_mnemonic (NULL, _("Rela_tive Path"));
  radio3 = gtk_radio_button_new_with_mnemonic_from_widget
                            (GTK_RADIO_BUTTON (radio2), _("Ab_solute Path"));
  radio4 = gtk_radio_button_new_with_mnemonic_from_widget
                            (GTK_RADIO_BUTTON (radio3), _("File Name _Only"));
  hbox0 = gtk_hbox_new (FALSE, 0);
  if (mmaid->filekind == FILE_KIND_DB)
    {
      gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (radio0), TRUE);
      gtk_widget_set_sensitive (hbox0, FALSE);
    }
  else
    {
      gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (radio1), TRUE);
    }
  switch (mmaid->playlist)
    {
      case  1: gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (radio3), TRUE);
               break;
      case  2: gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (radio4), TRUE);
               break;
      default: gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (radio2), TRUE);
    }
  g_signal_connect (G_OBJECT (radio0), "toggled",
                            G_CALLBACK (command_save_toggled_radio0), hbox0);
  g_signal_connect (G_OBJECT (radio1), "toggled",
                            G_CALLBACK (command_save_toggled_radio1), hbox0);
  /* ja:コンボボックス */
#if GTK_CHECK_VERSION(2,4,0)
  combo0 = gtk_combo_box_new_text ();
  for (i = g_list_length (glist) - 1; i >= 0; i--)
    {
      gtk_combo_box_prepend_text (GTK_COMBO_BOX (combo0),
                                                g_list_nth_data (glist, i));
      if (g_strfilecmp (g_list_nth_data (glist, i), root) == 0)
        gtk_combo_box_set_active (GTK_COMBO_BOX (combo0), 0);
    }
  g_signal_connect (G_OBJECT (combo0), "changed",
                                    G_CALLBACK (command_save_changed), dialog);
#else /* not GTK_CHECK_VERSION(2,4,0) */
  combo0 = gtk_combo_new ();
  gtk_combo_set_popdown_strings (GTK_COMBO (combo0), glist);
  gtk_combo_set_value_in_list (GTK_COMBO (combo0), FALSE, FALSE);
  gtk_combo_set_case_sensitive (GTK_COMBO (combo0), TRUE);
  gtk_editable_set_editable (GTK_EDITABLE (GTK_COMBO (combo0)->entry), FALSE);
  gtk_entry_set_text (GTK_ENTRY (GTK_COMBO (combo0)->entry), root);
  g_signal_connect (G_OBJECT (GTK_COMBO (combo0)->entry), "changed",
                                    G_CALLBACK (command_save_changed), dialog);
#endif /* not GTK_CHECK_VERSION(2,4,0) */
  for (i = g_list_length (glist) - 1; i >= 0; i--)
    g_free (g_list_nth_data (glist, i));
  g_list_free (glist);
  /* ja:文字符号化方式 */
  combo1 = orz_charset_new ();
  orz_charset_add_candidate (ORZ_CHARSET (combo1), charset_list);
  orz_charset_set_charset (ORZ_CHARSET (combo1), mmaid->charset);
  g_signal_connect (G_OBJECT (combo1), "edited",
                                    G_CALLBACK (command_save_edited), dialog);
  /* ja:ラベル */
  label0 = gtk_label_new_with_mnemonic (_("_Root"));
  label1 = gtk_label_new_with_mnemonic (_("Character _Encoding"));
#if GTK_CHECK_VERSION(2,4,0)
  gtk_label_set_mnemonic_widget (GTK_LABEL(label0), combo0);
#else /* not GTK_CHECK_VERSION(2,4,0) */
  gtk_label_set_mnemonic_widget (GTK_LABEL(label0), GTK_COMBO (combo0)->entry);
#endif /* not GTK_CHECK_VERSION(2,4,0) */
  gtk_label_set_mnemonic_widget (GTK_LABEL(label1), combo1);
  /* ja:フレームとボックス */
  vbox = gtk_vbox_new (FALSE, SPACING);
  gtk_container_set_border_width (GTK_CONTAINER (vbox), SPACING);
  hbox1 = gtk_hbox_new (FALSE, 0);
  gtk_box_pack_start (GTK_BOX (hbox1), radio0, FALSE, FALSE, 0);
  gtk_box_pack_start (GTK_BOX (hbox1), radio1, FALSE, FALSE, 0);
  gtk_box_pack_start (GTK_BOX (hbox0), gtk_label_new ("("), FALSE, FALSE, 0);
  gtk_box_pack_start (GTK_BOX (hbox0), radio2, FALSE, FALSE, 0);
  gtk_box_pack_start (GTK_BOX (hbox0), radio3, FALSE, FALSE, 0);
  gtk_box_pack_start (GTK_BOX (hbox0), radio4, FALSE, FALSE, 0);
  gtk_box_pack_start (GTK_BOX (hbox0), gtk_label_new (")"), FALSE, FALSE, 0);
  gtk_box_pack_start (GTK_BOX (hbox1), hbox0, FALSE, FALSE, 0);
  gtk_box_pack_start (GTK_BOX (vbox), hbox1, TRUE, TRUE, 0);
  hbox0 = gtk_hbox_new (FALSE, 0);
  gtk_box_pack_start (GTK_BOX (hbox0), label0, FALSE, FALSE, 0);
  gtk_box_pack_start (GTK_BOX (hbox0), combo0, TRUE, TRUE, 0);
  gtk_box_pack_start (GTK_BOX (vbox), hbox0, TRUE, TRUE, 0);
  hbox0 = gtk_hbox_new (FALSE, 0);
  gtk_box_pack_start (GTK_BOX (hbox0), label1, FALSE, FALSE, 0);
  gtk_box_pack_start (GTK_BOX (hbox0), combo1, TRUE, TRUE, 0);
  gtk_box_pack_start (GTK_BOX (vbox), hbox0, TRUE, TRUE, 0);
  gtk_widget_show_all (vbox);
  /* ja:ダイアログ */
#if GTK_CHECK_VERSION(2,4,0)
  gtk_file_chooser_set_extra_widget (GTK_FILE_CHOOSER (dialog), vbox);
  if (root)
    gtk_file_chooser_set_current_folder (GTK_FILE_CHOOSER (dialog), root);
  name = g_path_get_basename (orz_mdi_get_file (ORZ_MDI (mdi), page_num));
  tmp = g_filename_to_utf8 (name, -1, NULL, NULL, NULL);
  gtk_file_chooser_set_current_name (GTK_FILE_CHOOSER (dialog), tmp);
  g_free (root);
  g_free (tmp);
  g_free (name);
  if (gtk_dialog_run (GTK_DIALOG (dialog)) == GTK_RESPONSE_ACCEPT)
    {
# if ! GTK_CHECK_VERSION(2,6,0)
      GtkTreeIter iter;
# endif /* not GTK_CHECK_VERSION(2,6,0) */

      file = gtk_file_chooser_get_filename (GTK_FILE_CHOOSER (dialog));
# if GTK_CHECK_VERSION(2,6,0)
      root = gtk_combo_box_get_active_text (GTK_COMBO_BOX (combo0));
# else /* not GTK_CHECK_VERSION(2,6,0) */
      if (gtk_combo_box_get_active_iter (GTK_COMBO_BOX (combo0), &iter))
        gtk_tree_model_get (gtk_combo_box_get_model (GTK_COMBO_BOX (combo0)),
                                                        &iter, 0, &root, -1);
# endif /* not GTK_CHECK_VERSION(2,6,0) */
      charset = orz_charset_get_charset (ORZ_CHARSET (combo1));
      filekind = gtk_toggle_button_get_active
                (GTK_TOGGLE_BUTTON (radio0)) ? FILE_KIND_DB : FILE_KIND_LIST;
      if (gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (radio3)))
        playlist = 1;
      else if (gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (radio4)))
        playlist = 2;
      else
        playlist = 0;
    }
#else /* not GTK_CHECK_VERSION(2,4,0) */
  gtk_box_pack_end (GTK_BOX (GTK_FILE_SELECTION (dialog)->main_vbox),
                                                        vbox, FALSE, FALSE, 0);
  name = g_path_get_basename (orz_mdi_get_file (ORZ_MDI (mdi), page_num));
  tmp = root ? g_strconcat (root, name, NULL)
             : g_build_filename (save_path, name, NULL);
  gtk_file_selection_set_filename (GTK_FILE_SELECTION (dialog), tmp);
  g_free (root);
  g_free (tmp);
  g_free (name);
  gtk_file_selection_hide_fileop_buttons (GTK_FILE_SELECTION (dialog));
  if (gtk_dialog_run (GTK_DIALOG (dialog)) == GTK_RESPONSE_OK)
    {
      file = g_strdup (gtk_file_selection_get_filename
                                                (GTK_FILE_SELECTION (dialog)));
      root = g_strdup (gtk_entry_get_text
                                    (GTK_ENTRY (GTK_COMBO (combo0)->entry)));
      charset = orz_charset_get_charset (ORZ_CHARSET (combo1));
      filekind = gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (radio0))
                                            ? FILE_KIND_DB : FILE_KIND_LIST;
      if (gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (radio3)))
        playlist = 1;
      else if (gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (radio4)))
        playlist = 2;
      else
        playlist = 0;
    }
#endif /* not GTK_CHECK_VERSION(2,4,0) */
  else
    {
      file = root = charset = NULL;
      filekind = FILE_KIND_DB;
      playlist = 0;
    }
  gtk_widget_destroy (dialog);
  if (file)
    {
      gboolean result = TRUE;

      if (fileio_isfile (file))
        {
          gchar *utf8str;

#if GLIB_CHECK_VERSION(2,6,0)
          utf8str = g_filename_display_name (file);
#else /* not GLIB_CHECK_VERSION(2,6,0) */
          utf8str = g_filename_to_utf8 (file, -1, NULL, NULL, NULL);
#endif /* not GLIB_CHECK_VERSION(2,6,0) */
          dialog = gtk_message_dialog_new (GTK_WINDOW (window),
                    GTK_DIALOG_MODAL | GTK_DIALOG_DESTROY_WITH_PARENT,
                    GTK_MESSAGE_INFO, GTK_BUTTONS_YES_NO,
                    _("%s\nThis file is existed.\n\nOver write?"), utf8str);
          g_free (utf8str);
          g_signal_connect (G_OBJECT (dialog), "key-press-event",
                                    G_CALLBACK (misc_dialog_key_press), NULL);
          result = gtk_dialog_run (GTK_DIALOG (dialog)) == GTK_RESPONSE_YES;
          gtk_widget_destroy (dialog);
        }
      if (result && (filekind == FILE_KIND_DB
                    ? file_save_db (file, mmaid, root, charset)
                    : file_save_list (file, mmaid, root, charset, playlist)))
        {
          gchar *path;

          g_free (save_path);
          path = fileio_get_full_path (file);
          save_path = g_path_get_dirname (path);
          g_free (path);
          /* ja:キャラクターセット */
          g_free (mmaid->charset);
          mmaid->charset = g_strdup (charset);
          if (filekind == FILE_KIND_DB || playlist != 2)
            {
              /* ja:諸設定 */
              orz_mdi_set_created (ORZ_MDI (mdi), page_num, FALSE);
              orz_mdi_set_edited (ORZ_MDI (mdi), page_num, FALSE);
              orz_mdi_set_file (ORZ_MDI (mdi), page_num, file);
            }
          mmaid->filekind = filekind;
          if (filekind == FILE_KIND_LIST)
            playlist = mmaid->playlist = playlist;
          orz_history_add_file (ORZ_HISTORY (history), file);
        }
      g_free (file);
      g_free (root);
      g_free (charset);
    }
}


void
command_reload (GtkWidget *widget,
                gpointer   user_data)
{
  gchar *text;
  const gchar *file;
  gint page_num;
  MmaidWindow *mmaid;

  page_num = gtk_notebook_get_current_page (GTK_NOTEBOOK (mdi));
  mmaid = orz_mdi_get_data (ORZ_MDI (mdi), page_num);
  file = orz_mdi_get_file (ORZ_MDI (mdi), page_num);
  text = orz_mdi_get_edited (ORZ_MDI (mdi), page_num) ? g_strdup_printf
            (_("File %s was edited.\nRestore to existed file?"), file) : NULL;
  if (reload_dialog (mmaid, text))
    {
      /* ja:削除 */
      mm_list_delete_all (MM_LIST (mmaid->list));
      /* ja:再読み込み */
      file_open_db (file, mmaid);
      orz_mdi_set_edited (ORZ_MDI (mdi), page_num, FALSE);
      charlist_renewal_all (mmaid->charset);
      set_menu_bar (mmaid);
    }
}


static void
command_dialog_abort (GtkWidget *widget,
                      GtkWidget *dialog)
{
  userbreak = FALSE;
}


void
command_export (GtkWidget *widget,
                gpointer   user_data)
{
  gint i, count;
  GtkWidget *dialog;
  ExportInfo *expinfo;
  MmaidWindow *mmaid;

  mmaid = orz_mdi_get_data (ORZ_MDI (mdi), -1);
  expinfo = export_dialog (mmaid->charset);
  if (!expinfo)
    return;
  def_hold = expinfo->hold;
  def_gnum = expinfo->gnum;
  userbreak = TRUE;
  dialog = mm_dialog_new ();
  gtk_window_set_title (GTK_WINDOW (dialog), _("Export"));
  mm_list_set_editable (MM_LIST (MM_DIALOG (dialog)->list), TRUE);
  g_signal_connect (G_OBJECT (dialog), "destroy",
                                            G_CALLBACK (gtk_main_quit), NULL);
  g_signal_connect (G_OBJECT (MM_DIALOG (dialog)->abort_button),
                        "clicked", G_CALLBACK (command_dialog_abort), dialog);
  g_signal_connect_swapped (G_OBJECT (MM_DIALOG (dialog)->ok_button),
                "clicked", G_CALLBACK (gtk_widget_destroy), G_OBJECT (dialog));
  g_object_set_data (G_OBJECT (dialog), "user_data", mmaid);
  gtk_widget_show (MM_DIALOG (dialog)->abort_button);
  gtk_widget_show (dialog);
  gtk_grab_add (dialog);
  count = mm_list_length (MM_LIST (mmaid->list));
  for (i = 0; i < count && userbreak; i++)
    {
      while (gtk_events_pending ())
        gtk_main_iteration ();
      if (mm_list_is_selected (MM_LIST (mmaid->list), i))
        {
          gchar *file, *str;
          gint j, k, n, gnum;
          ID3Tag *id3tag;

          id3tag = copy_id3tag (mm_list_get_nth (MM_LIST (mmaid->list), i));
          file = g_filename_from_utf8 (id3tag->file, -1, NULL, NULL, NULL);
          switch (file_is_type (file))
            {
              case FILE_KIND_ERROR:
                /* ja:ファイルが存在しないとき */
                str = _("File Open Error");
                break;
              case FILE_KIND_TAG:
                if (expinfo->hold)
                  {
                    /* ja:既存のタグを保持し、かつ既にタグがあるとき */
                    str = _("Have Tag");
                    break;
                  }
              default:
                gnum = -1;
                if (id3tag->genre)
                  for (j = 0; j < genres; j++)
                    for (k = 0; k < 256; k++)
                      if (genre[j][k]
                                    && !g_strcmp (genre[j][k], id3tag->genre))
                        {
                          gnum = k;
                          goto loop;
                        }
                loop:
                if (gnum == -1)
                  {
                    gnum = expinfo->gnum;
                    g_free (id3tag->genre);
                    id3tag->genre = NULL;
                    for (j = 0; j < genres; j++)
                      if (genre[j][gnum])
                        {
                          id3tag->genre = g_strdup (genre[j][gnum]);
                          break;
                        }
                  }
                if (!id3tag->charset)
                  id3tag->charset = g_strdup (expinfo->charset);
                switch (file_save_id3tag (file, id3tag, gnum))
                  {
                    case FILE_SAVE_SUCCESS:
                      str = _("Success");
                      break;
                    case FILE_SAVE_CHARSET_TITLE:
                      str = _("Invalid Character Encoding of Title");
                      break;
                    case FILE_SAVE_CHARSET_ARTIST:
                      str = _("Invalid Character Encoding of Artist");
                      break;
                    case FILE_SAVE_CHARSET_ALBUM:
                      str = _("Invalid Character Encoding of Album");
                      break;
                    case FILE_SAVE_CHARSET_YEAR:
                      str = _("Invalid Character Encoding of Year");
                      break;
                    case FILE_SAVE_CHARSET_COMMENT:
                      str = _("Invalid Character Encoding of Comment");
                      break;
                    case FILE_SAVE_LENGTH_TITLE:
                      str = _("Invalid Length of Title");
                      break;
                    case FILE_SAVE_LENGTH_ARTIST:
                      str = _("Invalid Length of Artist");
                      break;
                    case FILE_SAVE_LENGTH_ALBUM:
                      str = _("Invalid Length of Album");
                      break;
                    case FILE_SAVE_LENGTH_YEAR:
                      str = _("Invalid Length of Year");
                      break;
                    case FILE_SAVE_LENGTH_COMMENT:
                      str = _("Invalid Length of Comment");
                      break;
                    default:
                      str = _("Error");
                  }
            }
          g_free (file);
          n = mm_list_append (MM_LIST (MM_DIALOG (dialog)->list), id3tag);
          mm_list_set_status (MM_LIST (MM_DIALOG (dialog)->list), n, str);
        }
    }
  g_free (expinfo->charset);
  g_free (expinfo);
  gtk_widget_hide (MM_DIALOG (dialog)->abort_button);
  gtk_widget_show (MM_DIALOG (dialog)->ok_button);
  gtk_main ();
}


void
command_exit (GtkWidget *widget,
              gpointer   user_data)
{
  GdkEvent event;

  event.any.type = GDK_DELETE;
  event.any.window = window->window;
  event.any.send_event = FALSE;
  gdk_event_put (&event);
}


void
command_up (GtkWidget *widget,
            gpointer   user_data)
{
  gint i, count, page_num;
  MmaidWindow *mmaid;

  page_num = gtk_notebook_get_current_page (GTK_NOTEBOOK (mdi));
  mmaid = orz_mdi_get_data (ORZ_MDI (mdi), page_num);
  count = mm_list_length (MM_LIST (mmaid->list));
  for (i = 1; i < count; i++)
    if (!mm_list_is_selected (MM_LIST (mmaid->list), i - 1)
                            && mm_list_is_selected (MM_LIST (mmaid->list), i))
      mm_list_swap (MM_LIST (mmaid->list), i - 1, i);
  orz_mdi_set_edited (ORZ_MDI (mdi), page_num, TRUE);
}


void
command_down (GtkWidget *widget,
              gpointer   user_data)
{
  gint i, page_num;
  MmaidWindow *mmaid;

  page_num = gtk_notebook_get_current_page (GTK_NOTEBOOK (mdi));
  mmaid = orz_mdi_get_data (ORZ_MDI (mdi), page_num);
  for (i = mm_list_length (MM_LIST (mmaid->list)) - 2; i >= 0 ; i--)
    if (mm_list_is_selected (MM_LIST (mmaid->list), i)
                        && !mm_list_is_selected (MM_LIST (mmaid->list), i + 1))
      mm_list_swap (MM_LIST (mmaid->list), i, i + 1);
  orz_mdi_set_edited (ORZ_MDI (mdi), page_num, TRUE);
}


void
command_root (GtkWidget *widget,
              gpointer   user_data)
{
  root_dialog ((MmaidWindow *)orz_mdi_get_data (ORZ_MDI (mdi), -1));
}


static void
command_dialog_ok (GtkWidget *widget,
                   GtkWidget *dialog)
{
  gboolean edited = FALSE;
  gint i, count;
  MmaidWindow *mmaid;

  mmaid = g_object_get_data (G_OBJECT (dialog), "user_data");
  count = mm_list_length (MM_LIST (MM_DIALOG (dialog)->list));
  for (i = 0; i < count; i++)
    {
      gint n;
      const ID3Tag *id3tag;

      id3tag = mm_list_get_nth (MM_LIST (MM_DIALOG (dialog)->list), i);
      n = mm_list_search (MM_LIST (mmaid->list),
                                            MM_LIST_FILE, id3tag->file, -1);
      if (n >= 0)
        {
          if (mm_list_get_active_refresh
                                    (MM_LIST (MM_DIALOG (dialog)->list), i))
            {
              mm_list_change (MM_LIST (mmaid->list), n,
                                    MM_LIST_ID3TAG, copy_id3tag (id3tag), -1);
              edited = TRUE;
            }
          else if (mm_list_get_active_delete
                                    (MM_LIST (MM_DIALOG (dialog)->list), i))
            {
              mm_list_delete (MM_LIST (mmaid->list), n);
              edited = TRUE;
            }
        }
      else
        {
          if (mm_list_get_active_add (MM_LIST (MM_DIALOG (dialog)->list), i))
            {
              mm_list_append (MM_LIST (mmaid->list), copy_id3tag (id3tag));
              edited = TRUE;
            }
        }
    }
  if (edited)
    orz_mdi_set_edited (ORZ_MDI (mdi),
                    orz_mdi_get_page_from_data (ORZ_MDI (mdi), mmaid), TRUE);
  gtk_widget_destroy (dialog);
}


void
command_refresh (GtkWidget *widget,
                 gpointer   user_data)
{
  gboolean selected = FALSE;
  gchar *title;
  gint i, count;
  GtkStockItem stock_item;
  GtkWidget *dialog;
  MmaidWindow *mmaid;

  mmaid = orz_mdi_get_data (ORZ_MDI (mdi), -1);
  /* ja:選択の数を数える */
  count = mm_list_length (MM_LIST (mmaid->list));
  for (i = 0; i < count; i++)
    if (mm_list_is_selected (MM_LIST (mmaid->list), i))
      {
        selected = TRUE;
        break;
      }
  userbreak = TRUE;
  dialog = mm_dialog_new ();
  gtk_stock_lookup (GTK_STOCK_REFRESH, &stock_item);
  title = misc_mnemonic_to_text (stock_item.label);
  gtk_window_set_title (GTK_WINDOW (dialog), title);
  g_free (title);
  mm_list_set_editable (MM_LIST (MM_DIALOG (dialog)->list), TRUE);
  g_signal_connect (G_OBJECT (MM_DIALOG (dialog)->abort_button),
                        "clicked", G_CALLBACK (command_dialog_abort), dialog);
  g_signal_connect (G_OBJECT (MM_DIALOG (dialog)->ok_button),
                        "clicked", G_CALLBACK (command_dialog_ok), dialog);
  g_signal_connect_swapped (G_OBJECT (MM_DIALOG (dialog)->cancel_button),
                "clicked", G_CALLBACK (gtk_widget_destroy), G_OBJECT (dialog));
  g_object_set_data (G_OBJECT (dialog), "user_data", mmaid);
  gtk_widget_show (MM_DIALOG (dialog)->abort_button);
  gtk_widget_show (dialog);
  gtk_grab_add (dialog);
  for (i = 0; i < count && userbreak; i++)
    {
      while (gtk_events_pending ())
        gtk_main_iteration ();
      if (!selected || mm_list_is_selected (MM_LIST (mmaid->list), i))
        {
          gchar *file;
          gint n;
          const ID3Tag *id3tag0;
          ID3Tag *id3tag1;

          id3tag0 = mm_list_get_nth (MM_LIST (mmaid->list), i);
          file = g_filename_from_utf8 (id3tag0->file, -1, NULL, NULL, NULL);
          switch (file_is_type (file))
            {
              case FILE_KIND_UNKNOWN:
                id3tag1 = file_open_id3tag (file, id3tag0->charset
                                            ? id3tag0->charset : charset_list);
                n = mm_list_append (MM_LIST (MM_DIALOG (dialog)->list),
                                                                    id3tag1);
                if (!cmp_id3tag (id3tag0, id3tag1))
                  mm_list_show_refresh (MM_LIST (MM_DIALOG (dialog)->list), n);
                mm_list_show_delete (MM_LIST (MM_DIALOG (dialog)->list), n);
                break;
              case FILE_KIND_TAG:
                id3tag1 = file_open_id3tag (file, id3tag0->charset
                                            ? id3tag0->charset : charset_list);
                n = mm_list_append (MM_LIST (MM_DIALOG (dialog)->list),
                                                                    id3tag1);
                mm_list_show_delete (MM_LIST (MM_DIALOG (dialog)->list), n);
                if (!cmp_id3tag (id3tag0, id3tag1))
                  {
                    mm_list_set_sensitive_delete
                                (MM_LIST (MM_DIALOG (dialog)->list), n, FALSE);
                    mm_list_show_refresh (MM_LIST (MM_DIALOG (dialog)->list),
                                                                            n);
                    mm_list_set_active_refresh
                                (MM_LIST (MM_DIALOG (dialog)->list), n, TRUE);
                  }
                break;
              default:
                n = mm_list_append (MM_LIST (MM_DIALOG (dialog)->list),
                                                        copy_id3tag (id3tag0));
                mm_list_show_delete (MM_LIST (MM_DIALOG (dialog)->list), n);
                mm_list_set_active_delete (MM_LIST (MM_DIALOG (dialog)->list),
                                                                    n, TRUE);
                break;
            }
          g_free (file);
        }
    }
  if (userbreak)
    {
      gtk_widget_hide (MM_DIALOG (dialog)->abort_button);
      gtk_widget_show (MM_DIALOG (dialog)->ok_button);
      gtk_widget_show (MM_DIALOG (dialog)->cancel_button);
      g_signal_connect (G_OBJECT (dialog), "destroy",
                                            G_CALLBACK (gtk_main_quit), NULL);
      gtk_main ();
    }
  else
    {
      gtk_widget_destroy (dialog);
    }
}


void
command_add (GtkWidget *widget,
             gpointer   user_data)
{
  add_dialog ((MmaidWindow *)orz_mdi_get_data (ORZ_MDI (mdi), -1));
}


void
command_edit (GtkWidget *widget,
              gpointer   user_data)
{
  edit_dialog ((MmaidWindow *)orz_mdi_get_data (ORZ_MDI (mdi), -1));
}


static void
command_search_recursive (const gchar *path,
                          GtkWidget   *dialog)
{
  GDir *gdir;

  gdir = g_dir_open (path, 0, NULL);
  if (gdir)
    {
      const gchar *name;

      while (userbreak && (name = g_dir_read_name (gdir)))
        {
          gchar *file;

          while (gtk_events_pending ())
            gtk_main_iteration ();
#ifndef G_OS_WIN32
          if (name[0] == '.')
            continue;
#endif /* not G_OS_WIN32 */
          file = g_build_filename (path, name, NULL);
          if (g_file_test (file, G_FILE_TEST_IS_DIR))
            {
              command_search_recursive (file, dialog);
            }
          else
            {
              gchar *utf8file;
              gint n0, n1;
              const ID3Tag *id3tag0;
              ID3Tag *id3tag1;
              MmaidWindow *mmaid;

              mmaid = g_object_get_data (G_OBJECT (dialog), "user_data");
              utf8file = g_filename_to_utf8 (file, -1, NULL, NULL, NULL);
              n0 = mm_list_search (MM_LIST (mmaid->list),
                                                MM_LIST_FILE, utf8file, -1);
              g_free (utf8file);
              id3tag0 = n0 >= 0 ? mm_list_get_nth (MM_LIST (mmaid->list), n0)
                                : NULL;
              switch (file_is_type (file))
                {
                  case FILE_KIND_UNKNOWN:
                    id3tag1 = file_open_id3tag (file, charset_list);
                    n1 = mm_list_append (MM_LIST (MM_DIALOG (dialog)->list),
                                                                    id3tag1);
                    if (n0 >= 0)
                      {
                        if (!cmp_id3tag (id3tag0, id3tag1))
                          mm_list_show_refresh
                                (MM_LIST (MM_DIALOG (dialog)->list), n1);
                        mm_list_show_delete
                                (MM_LIST (MM_DIALOG (dialog)->list), n1);
                      }
                    else
                      {
                        mm_list_show_add
                                (MM_LIST (MM_DIALOG (dialog)->list), n1);
                        if (g_strlen (file) > 4 && g_ascii_strcasecmp (".mp3",
                                            file + g_strlen (file) - 4) == 0)
                          mm_list_set_active_add
                                (MM_LIST (MM_DIALOG (dialog)->list), n1, TRUE);
                      }
                    break;
                  case FILE_KIND_TAG:
                    id3tag1 = file_open_id3tag (file, charset_list);
                    n1 = mm_list_append (MM_LIST (MM_DIALOG (dialog)->list),
                                                                    id3tag1);
                    if (n0 >= 0)
                      {
                        mm_list_show_delete
                                (MM_LIST (MM_DIALOG (dialog)->list), n1);
                        if (!cmp_id3tag (id3tag0, id3tag1))
                          {
                            mm_list_set_sensitive_delete
                                (MM_LIST (MM_DIALOG (dialog)->list),
                                                                    n1, FALSE);
                            mm_list_show_refresh
                                (MM_LIST (MM_DIALOG (dialog)->list), n1);
                            mm_list_set_active_refresh
                                (MM_LIST (MM_DIALOG (dialog)->list), n1, TRUE);
                          }
                      }
                    else
                      {
                        mm_list_show_add
                                (MM_LIST (MM_DIALOG (dialog)->list), n1);
                        mm_list_set_active_add
                                (MM_LIST (MM_DIALOG (dialog)->list), n1, TRUE);
                      }
                    break;
                  default:
                    if (n0 >= 0)
                      {
                        n1 = mm_list_append
                                (MM_LIST (MM_DIALOG (dialog)->list),
                                                        copy_id3tag (id3tag0));
                        mm_list_show_delete
                                (MM_LIST (MM_DIALOG (dialog)->list), n1);
                        mm_list_set_active_delete
                                (MM_LIST (MM_DIALOG (dialog)->list), n1, TRUE);
                      }
                }
            }
          g_free (file);
        }
      g_dir_close (gdir);
    }
}


void
command_search (GtkWidget *widget,
                gpointer   user_data)
{
  gchar *path, *tmp;
  const gchar *file;
  gint page_num;
  GtkWidget *dialog;
  MmaidWindow *mmaid;

  page_num = gtk_notebook_get_current_page (GTK_NOTEBOOK (mdi));
  mmaid = orz_mdi_get_data (ORZ_MDI (mdi), page_num);
  file = orz_mdi_get_data (ORZ_MDI (mdi), page_num);
  path = g_path_is_absolute (file) ? g_path_get_dirname (file)
       : open_path ? g_strdup (open_path)
       : save_path ? g_strdup (save_path)
       : NULL;
#if GTK_CHECK_VERSION(2,4,0)
  dialog = gtk_file_chooser_dialog_new (_("Browse"),
                        GTK_WINDOW (window),
                        GTK_FILE_CHOOSER_ACTION_SELECT_FOLDER,
                        GTK_STOCK_CANCEL, GTK_RESPONSE_CANCEL,
                        GTK_STOCK_OK, GTK_RESPONSE_ACCEPT,
                        NULL);
  if (path)
    {
      gtk_file_chooser_set_current_folder (GTK_FILE_CHOOSER (dialog), path);
      g_free (path);
      path = NULL;
    }
  if (gtk_dialog_run (GTK_DIALOG (dialog)) == GTK_RESPONSE_ACCEPT)
    path = gtk_file_chooser_get_filename (GTK_FILE_CHOOSER (dialog));
#else /* not GTK_CHECK_VERSION(2,4,0) */
  dialog = gtk_file_selection_new (_("Browse"));
  if (path)
    {
      if (path[0] != '\0')
        {
          gchar *tmp;

          if (!G_IS_DIR_SEPARATOR (path[g_strlen (path) - 1]))
            {
              tmp = g_strconcat (path, G_DIR_SEPARATOR_S, NULL);
              g_free (path);
              path = tmp;
            }
          gtk_file_selection_set_filename (GTK_FILE_SELECTION (dialog), path);
        }
      g_free (path);
      path = NULL;
    }
  gtk_file_selection_hide_fileop_buttons (GTK_FILE_SELECTION (dialog));
  gtk_widget_hide
            (gtk_widget_get_parent (GTK_FILE_SELECTION (dialog)->file_list));
  gtk_widget_hide (GTK_FILE_SELECTION (dialog)->selection_entry);
  if (gtk_dialog_run (GTK_DIALOG (dialog)) == GTK_RESPONSE_OK)
    path = g_strdup (gtk_file_selection_get_filename
                                                (GTK_FILE_SELECTION (dialog)));
#endif /* not GTK_CHECK_VERSION(2,4,0) */
  gtk_widget_destroy (dialog);

  if (!path)
    return;
  tmp = g_filename_to_utf8 (path, -1, NULL, NULL, NULL);
  g_free (path);
  if (!tmp || tmp[0] =='\0')
    {
      g_free (tmp);
      return;
    }
  if (G_IS_DIR_SEPARATOR (tmp[g_strlen (tmp) - 1]))
    tmp[g_strlen (tmp) - 1] = '\0';
  path = g_filename_from_utf8 (tmp, -1, NULL, NULL, NULL);
  g_free (tmp);
  if (!path)
    return;
  userbreak = TRUE;
  dialog = mm_dialog_new ();
  gtk_window_set_title (GTK_WINDOW (dialog), _("Search"));
  mm_list_set_editable (MM_LIST (MM_DIALOG (dialog)->list), TRUE);
  g_signal_connect (G_OBJECT (dialog), "destroy",
                                            G_CALLBACK (gtk_main_quit), NULL);
  g_signal_connect (G_OBJECT (MM_DIALOG (dialog)->abort_button),
                        "clicked", G_CALLBACK (command_dialog_abort), dialog);
  g_signal_connect (G_OBJECT (MM_DIALOG (dialog)->ok_button),
                        "clicked", G_CALLBACK (command_dialog_ok), dialog);
  g_signal_connect_swapped (G_OBJECT (MM_DIALOG (dialog)->cancel_button),
                "clicked", G_CALLBACK (gtk_widget_destroy), G_OBJECT (dialog));
  g_object_set_data (G_OBJECT (dialog), "user_data", mmaid);
  gtk_widget_show (MM_DIALOG (dialog)->abort_button);
  gtk_widget_show (dialog);
  gtk_grab_add (dialog);
  command_search_recursive (path, dialog);
  g_free (path);
  gtk_widget_hide (MM_DIALOG (dialog)->abort_button);
  gtk_widget_show (MM_DIALOG (dialog)->ok_button);
  gtk_widget_show (MM_DIALOG (dialog)->cancel_button);
  gtk_main ();
}


void
command_cut (GtkWidget *widget,
             gpointer   user_data)
{
  command_copy (widget, user_data);
  command_delete (widget, user_data);
}


void
command_copy (GtkWidget *widget,
              gpointer   user_data)
{
#ifndef G_OS_WIN32
  if (gtk_selection_owner_set (window, GDK_SELECTION_CLIPBOARD,
                                                            GDK_CURRENT_TIME))
    {
#endif /* not G_OS_WIN32 */
      gint i, count;
      MmaidWindow *mmaid;

      mmaid = orz_mdi_get_data (ORZ_MDI (mdi), -1);
      g_free (clipboard_id3tag);
      clipboard_id3tag = NULL;
      count = mm_list_length (MM_LIST (mmaid->list));
      for (i = 0; i < count; i++)
        if (mm_list_is_selected (MM_LIST (mmaid->list), i))
          {
            gchar *str;
            const ID3Tag *id3tag;

            id3tag = mm_list_get_nth (MM_LIST (mmaid->list), i);
            str = g_strdup_printf ("%s\t%s\t%s\t%s\t%s\t%s\t%d\t%s\t%s",
                                    id3tag->file    ? id3tag->file     : "",
                                    id3tag->title   ? id3tag->title    : "",
                                    id3tag->artist  ? id3tag->artist   : "",
                                    id3tag->album   ? id3tag->album    : "",
                                    id3tag->year    ? id3tag->year     : "",
                                    id3tag->comment ? id3tag->comment  : "",
                                    id3tag->track,
                                    id3tag->genre   ? id3tag->genre    : "",
                                    id3tag->charset ? id3tag->charset  : "");
            if (clipboard_id3tag)
              {
                gchar *tmp;

                tmp = g_strconcat (clipboard_id3tag, "\n", str, NULL);
                g_free (clipboard_id3tag);
                g_free (str);
                clipboard_id3tag = tmp;
              }
            else
              {
                clipboard_id3tag = str;
              }
          }
#ifdef G_OS_WIN32
      if (clipboard_id3tag)
        {
          HGLOBAL hGlobal;
          LPSTR lpID3Tag;

          hGlobal = GlobalAlloc (GMEM_MOVEABLE,
                        (g_strlen (clipboard_id3tag) + 1) * sizeof (gchar));
          if (hGlobal)
            {
              lpID3Tag = GlobalLock (hGlobal);
              if (lpID3Tag)
                {
                  g_strcpy (lpID3Tag, clipboard_id3tag);
                  GlobalUnlock (hGlobal);
                  if (OpenClipboard (GDK_WINDOW_HWND (window->window)))
                    {
                      if (!EmptyClipboard ()
                                    || !SetClipboardData (uFormat, hGlobal))
                        GlobalFree (hGlobal);
                      CloseClipboard ();
                    }
                  else
                    {
                      GlobalFree (hGlobal);
                    }
                }
              else
                {
                  GlobalFree (hGlobal);
                }
            }
        }
#else /* not G_OS_WIN32 */
    }
#endif /* not G_OS_WIN32 */
}


void
command_paste (GtkWidget *widget,
               gpointer   user_data)
{
  /* ja:Windowsのセレクションの実装があやしいので暫定的処理 */
#ifdef G_OS_WIN32
  gchar *id3tag = NULL;
  HGLOBAL hGlobal;
  LPSTR lpID3Tag;

  if (OpenClipboard (GDK_WINDOW_HWND (window->window)))
    {
      if (hGlobal = GetClipboardData (uFormat))
        {
          if (lpID3Tag = GlobalLock (hGlobal))
            id3tag = g_strdup (lpID3Tag);
          GlobalUnlock (hGlobal);
        }
      CloseClipboard ();
      if (id3tag)
        {
          GtkSelectionData data;

          data.selection = GDK_SELECTION_CLIPBOARD;
          data.target = atom_id3tag;
          data.type = atom_id3tag;
          data.format = 8;
          data.data = id3tag;
          data.length = g_strlen (id3tag);
          signal_selection_received (window, &data, GDK_CURRENT_TIME, NULL);
          g_free (id3tag);
        }
    }
#else /* not G_OS_WIN32 */
  gtk_selection_convert (window, GDK_SELECTION_CLIPBOARD, atom_targets,
                                                            GDK_CURRENT_TIME);
#endif /* not G_OS_WIN32 */
}


void
command_delete (GtkWidget *widget,
                gpointer   user_data)
{
  gint i, page_num;
  MmaidWindow *mmaid;

  page_num = gtk_notebook_get_current_page (GTK_NOTEBOOK (mdi));
  mmaid = orz_mdi_get_data (ORZ_MDI (mdi), page_num);
  for (i = mm_list_length (MM_LIST (mmaid->list)) - 1; i >= 0 ; i--)
    if (mm_list_is_selected (MM_LIST (mmaid->list), i))
      mm_list_delete (MM_LIST (mmaid->list), i);
  orz_mdi_set_edited (ORZ_MDI (mdi), page_num, TRUE);
}


void
command_all (GtkWidget *widget,
             gpointer   user_data)
{
  gint i;
  MmaidWindow *mmaid;

  mmaid = orz_mdi_get_data (ORZ_MDI (mdi), -1);
  for (i = mm_list_length (MM_LIST (mmaid->list)) - 1; i >= 0 ; i--)
    if (!mm_list_is_selected (MM_LIST (mmaid->list), i))
      break;
  if (i >= 0)
    mm_list_select_all (MM_LIST (mmaid->list));
  else
    mm_list_unselect_all (MM_LIST (mmaid->list));
}


void
command_conf (GtkWidget *widget,
              gpointer   user_data)
{
  GtkWidget *dialog;

  dialog = orz_conf_new ();
  gtk_window_set_transient_for (GTK_WINDOW (dialog), GTK_WINDOW (window));
  orz_conf_set_history (ORZ_CONF (dialog),
                                orz_history_get_num (ORZ_HISTORY (history)));
  orz_conf_set_newfile (ORZ_CONF (dialog), newfile);
  orz_conf_set_second (ORZ_CONF (dialog), second);
  orz_conf_set_size (ORZ_CONF (dialog), def_width, def_height);
  orz_conf_set_tab (ORZ_CONF (dialog), n_pos);
  if (gtk_dialog_run (GTK_DIALOG (dialog)) == GTK_RESPONSE_OK)
    {
      orz_history_set_num (ORZ_HISTORY (history),
                                    orz_conf_get_history (ORZ_CONF (dialog)));
      newfile = orz_conf_get_newfile (ORZ_CONF (dialog));
      second = orz_conf_get_second (ORZ_CONF (dialog));
      orz_conf_get_size (ORZ_CONF (dialog), &def_width, &def_height);
      n_pos = orz_conf_get_tab (ORZ_CONF (dialog));
      /* ja:ノートの位置 */
      gtk_notebook_set_tab_pos (GTK_NOTEBOOK (mdi), n_pos);
    }
  gtk_widget_destroy (dialog);
}


void
command_genre (GtkWidget *widget,
               gpointer   user_data)
{
  genre_dialog ();
}


void
command_charset (GtkWidget *widget,
                 gpointer   user_data)
{
  GtkWidget *dialog;

  dialog = orz_charlist_new ();
  gtk_window_set_transient_for (GTK_WINDOW (dialog), GTK_WINDOW (window));
  orz_charlist_set_charlist (ORZ_CHARLIST (dialog), charset_list);
  if (gtk_dialog_run (GTK_DIALOG (dialog)) == GTK_RESPONSE_OK)
    {
      g_free (charset_list);
      charset_list = orz_charlist_get_charlist (ORZ_CHARLIST (dialog));
      charlist_renewal_all (charset_list);
    }
  gtk_widget_destroy (dialog);
}


void
command_process (GtkWidget *widget,
                 gpointer   user_data)
{
  process_dialog ();
}


void
command_version (GtkWidget *widget,
                 gpointer   user_data)
{
  version_dialog ();
}


void
command_play (GtkWidget *widget,
              gpointer   user_data)
{
  gchar *cmd;

  cmd = g_filename_from_utf8 (cmd_play, -1, NULL, NULL, NULL);
  if (cmd)
    {
      gint n;
      MmaidWindow *mmaid;

      mmaid = orz_mdi_get_data (ORZ_MDI (mdi), -1);
      mm_list_get_cursor (MM_LIST (mmaid->list), &n, NULL);
      if (n >= 0)
        {
          gchar *file;
          const ID3Tag *id3tag;

          id3tag = mm_list_get_nth (MM_LIST (mmaid->list), n);
          file = g_filename_from_utf8 (id3tag->file, -1, NULL, NULL, NULL);
          if (file)
            {
              gchar *quoted, *str;

              quoted = g_shell_quote (file);
              str = g_strconcat (cmd, " ", quoted, NULL);
              if (quoted && str)
                g_spawn_command_line_async (str, NULL);
              g_free (str);
              g_free (quoted);
              g_free (file);
            }
        }
      g_free (cmd);
    }
}


void
command_stop (GtkWidget *widget,
              gpointer   user_data)
{
  gchar *cmd;

  cmd = g_filename_from_utf8 (cmd_stop, -1, NULL, NULL, NULL);
  if (cmd)
    {
      g_spawn_command_line_async (cmd, NULL);
      g_free (cmd);
    }
}
