/*
    orzaccel
    copyright (c) 1998-2008 Kazuki IWAMOTO http://www.maid.org/ iwm@maid.org

    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program.  If not, see <http://www.gnu.org/licenses/>.
*/
#include "orzaccel.h"


/******************************************************************************
*                                                                             *
******************************************************************************/
G_DEFINE_TYPE (OrzAccel, orz_accel, G_TYPE_OBJECT);
static void orz_accel_dispose (GObject *object);


static void
orz_accel_class_init (OrzAccelClass *klass)
{
  GObjectClass *object_class;

  object_class = G_OBJECT_CLASS (klass);
  object_class->dispose = orz_accel_dispose;
}


/*  ja:新規作成
    RET,オブジェクト                                                        */
static void
orz_accel_init (OrzAccel *accel)
{
  accel->handler_destroy   = 0;
  accel->handler_key_press = 0;
  accel->gslist            = NULL;
  accel->window            = NULL;
}


static void
orz_accel_dispose (GObject *object)
{
  OrzAccel *accel;

  accel = ORZ_ACCEL (object);
  if (accel->window)
    {
      if (accel->handler_destroy)
        g_signal_handler_disconnect (G_OBJECT (accel->window),
                                                    accel->handler_destroy);
      if (accel->handler_key_press)
        g_signal_handler_disconnect (G_OBJECT (accel->window),
                                                    accel->handler_key_press);
      accel->window = NULL;
    }
  if (accel->gslist)
    {
      GSList *gslist;

      for (gslist = accel->gslist; gslist; gslist = gslist->next)
        g_object_unref (gslist->data);
      g_slist_free (accel->gslist);
      accel->gslist = NULL;
    }

  if (G_OBJECT_CLASS (orz_accel_parent_class)->dispose)
    G_OBJECT_CLASS (orz_accel_parent_class)->dispose (object);
}


static void
orz_accel_destroy (GtkWidget *widget,
                   OrzAccel *accel)
{
  accel->window = NULL;
}


static gboolean
orz_accel_key_press (GtkWidget   *widget,
                     GdkEventKey *event,
                     OrzAccel   *accel)
{
  gboolean result = FALSE;

  if (accel->gslist)
    {
      GSList *gslist;

      for (gslist = accel->gslist; gslist; gslist = gslist->next)
        gtk_window_add_accel_group (GTK_WINDOW (widget), gslist->data);
      result = gtk_accel_groups_activate (G_OBJECT (widget),
                                                event->keyval, event->state);
      for (gslist = accel->gslist; gslist; gslist = gslist->next)
        gtk_window_remove_accel_group (GTK_WINDOW (widget), gslist->data);
    }
  return result;
}


/******************************************************************************
*                                                                             *
* ja:アクセラレータ関数群                                                     *
*                                                                             *
******************************************************************************/
/*  ja:新規作成
    RET,オブジェクト                                                        */
GObject *
orz_accel_new (void)
{
  return g_object_new (ORZ_TYPE_ACCEL, NULL);
}


/*  ja:ウインドウを取得する
    accel,オブジェクト
      RET,ウインドウ                                                        */
GtkWindow *
orz_accel_get_window (OrzAccel *accel)
{
  return accel ? accel->window : NULL;
}


/*  ja:ウインドウを設定する
     accel,オブジェクト
    window,ウインドウ                                                       */
void
orz_accel_set_window (OrzAccel  *accel,
                      GtkWindow *window)
{
  if (accel)
    {
      if (accel->window)
        {
          if (accel->handler_destroy)
            g_signal_handler_disconnect (G_OBJECT (accel->window),
                                                    accel->handler_destroy);
          if (accel->handler_key_press)
            g_signal_handler_disconnect (G_OBJECT (accel->window),
                                                    accel->handler_key_press);
        }
      accel->window = window;
      if (accel->window)
        {
          accel->handler_destroy = g_signal_connect (G_OBJECT (window),
                "destroy", G_CALLBACK (orz_accel_destroy), accel);
          accel->handler_key_press = g_signal_connect_after (G_OBJECT (window),
                "key-press-event", G_CALLBACK (orz_accel_key_press), accel);
        }
    }
}


/*  ja:アクセルグループを追加する
          accel,オブジェクト
    accel_group,アクセルグループ                                            */
void
orz_accel_add_accel_group (OrzAccel      *accel,
                           GtkAccelGroup *accel_group)
{
  if (accel && accel_group)
    {
      g_object_ref (accel_group);
      accel->gslist = g_slist_append (accel->gslist, accel_group);
    }
}


/*  ja:アクセルグループを削除する
          accel,オブジェクト
    accel_group,アクセルグループ                                            */
void
orz_accel_remove_accel_group (OrzAccel      *accel,
                              GtkAccelGroup *accel_group)
{
  if (accel && accel_group)
    {
      accel->gslist = g_slist_remove (accel->gslist, accel_group);
      g_object_unref (accel_group);
    }
}


/*  ja:アクセルグループを取得する
    accel,オブジェクト
      RET,アクセルグループのリスト                                          */
GSList *
orz_accel_get_accel_group (OrzAccel *accel)
{
  return accel ? accel->gslist : NULL;
}
