/*
    gcommon
    copyright (c) 1998-2013 Kazuki Iwamoto http://www.maid.org/ iwm@maid.org

    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program.  If not, see <http://www.gnu.org/licenses/>.
*/
#include "gcommon.h"
#ifdef G_OS_WIN32
# include <windows.h>
# include <tchar.h>
#endif /* G_OS_WIN32 */


/******************************************************************************
* Date and Time Functions                                                     *
******************************************************************************/
#ifdef USE_GTK_EMULATE
void
g_get_current_time (GTimeVal *result)
{
  if (result)
    {
# ifdef G_OS_WIN32
      FILETIME ft;
      guint64 usec;

      GetSystemTimeAsFileTime (&ft);
      usec = (ft.dwLowDateTime | ((guint64)ft.dwHighDateTime << 32)) / 10
                                    - G_GINT64_CONSTANT (11644473600000000);
      result->tv_sec = usec / G_USEC_PER_SEC;
      result->tv_usec = usec % G_USEC_PER_SEC;
# else /* not G_OS_WIN32 */
#  ifdef HAVE_CLOCK_GETTIME
      struct timespec tp;
#  endif /* HAVE_CLOCK_GETTIME */
#  ifdef HAVE_GETTIMEOFDAY
      struct timeval tv;
#  endif /* HAVE_GETTIMEOFDAY */

#  ifdef HAVE_CLOCK_GETTIME
      if (clock_gettime (CLOCK_REALTIME, &tp) == 0)
        {
          result->tv_sec = tp.tv_sec;
          result->tv_usec = tp.tv_usec / 1000;
          return;
        }
#  endif /* HAVE_CLOCK_GETTIME */
#  ifdef HAVE_GETTIMEOFDAY
      if (gettimeofday (&tv, NULL) == 0)
        {
          result->tv_sec = tv.tv_sec;
          result->tv_usec = tv.tv_usec;
          return;
        }
#  endif /* HAVE_GETTIMEOFDAY */
      result->tv_sec = time (NULL);
      result->tv_usec = 0;
# endif /* not G_OS_WIN32 */
    }
}


void
g_usleep (gulong microseconds)
{
# ifdef G_OS_WIN32
  Sleep (microseconds / 1000);
# else /* not G_OS_WIN32 */
#  ifdef HAVE_NANOSLEEP
  struct timespec req, rem;

  req.tv_sec = microseconds / G_USEC_PER_SEC;
  req.tv_nsec = (microseconds % G_USEC_PER_SEC) * 1000;
  while (nanosleep (&req, &rem) == -1 && errno == EINTR)
    req = rem;
#  else /* not HAVE_NANOSLEEP */
#   ifdef HAVE_SLEEP
  unsigned int seconds;

  seconds = microseconds / G_USEC_PER_SEC;
  while (seconds > 0)
    seconds = sleep (seconds);
#   endif /* HAVE_SLEEP */
#   ifdef HAVE_USLEEP
#    ifdef HAVE_SLEEP
  usleep (microseconds % G_USEC_PER_SEC);
#    else /* not HAVE_SLEEP */
  usleep (microseconds);
#    endif /* not HAVE_SLEEP */
#   endif /* HAVE_USLEEP */
#  endif /* not HAVE_NANOSLEEP */
# endif /* not G_OS_WIN32 */
}
#endif /* USE_GTK_EMULATE */


#if ! GLIB_CHECK_VERSION(2,28,0)
gint64
g_get_monotonic_time (void)
{
# ifdef G_OS_WIN32
  gint64 monotonic = -1;
  HMODULE hLib;

  hLib = LoadLibrary (_T("kernel32.dll"));
  if (hLib)
    {
      ULONGLONG (WINAPI *pGetTickCount64)(VOID);

      pGetTickCount64 = (gpointer)GetProcAddress (hLib, "GetTickCount64");
      if (pGetTickCount64)
        monotonic = pGetTickCount64 ();
      FreeLibrary (hLib);
    }
  return (gint64)(monotonic >= 0 ? monotonic : GetTickCount ()) * 1000;
# else /* not G_OS_WIN32 */
#  if defined (HAVE_CLOCK_GETTIME) && defined (CLOCK_MONOTONIC)
  struct timespec tp;

  if (clock_gettime (CLOCK_MONOTONIC, &tp) == 0)
    return (gint64)tp.tv_sec * 1000000 + tp.tv_usec / 1000;
#  endif /* defined (HAVE_CLOCK_GETTIME) && defined (CLOCK_MONOTONIC) */
  return g_get_real_time ();
# endif /* not G_OS_WIN32 */
}


gint64
g_get_real_time (void)
{
  GTimeVal tv;

  g_get_current_time (&tv);
  return (gint64)tv.tv_sec * G_USEC_PER_SEC + tv.tv_usec;
}
#endif /* not GLIB_CHECK_VERSION(2,28,0) */
