/*
    gcommon
    copyright (c) 1998-2013 Kazuki Iwamoto http://www.maid.org/ iwm@maid.org

    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program.  If not, see <http://www.gnu.org/licenses/>.
*/
#include "gcommon.h"
#ifdef G_OS_WIN32
# include <windows.h>
# include <tchar.h>
#endif /* G_OS_WIN32 */


/******************************************************************************
* File Utilities                                                              *
******************************************************************************/
#ifdef USE_GTK_EMULATE
gboolean
g_file_get_contents (const gchar  *file,
                     gchar       **contents,
                     gsize        *length,
                     GError      **error)
{
  gboolean result = FALSE;
  glong fs = 0;

  if (file && contents)
    {
      gchar *buf = NULL;
      FILE *fp;

      fp = g_fopen (file, "rb");
      if (fp)
        {
          if (fseek (fp, 0, SEEK_END) == 0)
            {
              fs = ftell (fp);
              if (fs >= 0 && fseek (fp, 0, SEEK_SET) == 0)
                {
                  buf = g_malloc (fs + 1);
                  buf[fs] = '\0';
                  if (fread (buf, sizeof (gchar), fs, fp) == fs)
                    result = TRUE;
                }
            }
          if (fclose (fp) != 0)
            result = FALSE;
        }
      if (!result)
        {
          g_free (buf);
          buf = NULL;
        }
      *contents = buf;
    }
  if (length)
    *length = result ? fs : 0;
  return result;
}
#endif /* USE_GTK_EMULATE */


#if ! GLIB_CHECK_VERSION(2,8,0)
gboolean
g_file_set_contents (const gchar  *file,
                     const gchar  *contents,
                     gssize        length,
                     GError      **error)
{
  gboolean result = FALSE;

  if (file && contents)
    {
      FILE *fp;

      fp = g_fopen (file, "wb");
      if (fp)
        {
          gsize len;

          len = length >= 0 ? length : g_strlen (contents);
          if (fwrite (contents, sizeof (gchar), len, fp) == len)
            result = TRUE;
          if (fclose (fp) != 0)
            result = FALSE;
        }
    }
  return result;
}
#endif /* not GLIB_CHECK_VERSION(2,8,0) */


#ifdef USE_GTK_EMULATE
gboolean
g_file_test (const gchar *file,
             GFileTest    test)
{
# ifdef G_OS_WIN32
#  ifndef INVALID_FILE_ATTRIBUTES
#   define INVALID_FILE_ATTRIBUTES -1
#  endif /* not INVALID_FILE_ATTRIBUTES */
#  ifndef FILE_ATTRIBUTE_DEVICE
#   define FILE_ATTRIBUTE_DEVICE 64
#  endif /* not FILE_ATTRIBUTE_DEVICE */
  gboolean ret = FALSE;
  DWORD dwAttributes;
#  ifdef UNICODE
  gchar *utf8str;
  LPWSTR lpszFile;

  utf8str = g_filename_to_utf8 (file, -1, NULL, NULL, NULL);
  lpszFile = g_utf8_to_utf16 (utf8str, -1, NULL, NULL, NULL);
  g_free (utf8str);
  dwAttributes = GetFileAttributesW (lpszFile);
  if (dwAttributes != INVALID_FILE_ATTRIBUTES)
    {
      if ((test & G_FILE_TEST_EXISTS)
        || ((test & G_FILE_TEST_IS_REGULAR) && (dwAttributes
                    & (FILE_ATTRIBUTE_DIRECTORY | FILE_ATTRIBUTE_DEVICE)) == 0)
        || ((test & G_FILE_TEST_IS_DIR)
                            && (dwAttributes & FILE_ATTRIBUTE_DIRECTORY) != 0))
        ret = TRUE;
      if (test & G_FILE_TEST_IS_EXECUTABLE)
        {
          INT nLength;

          nLength = lstrlenW (lpszFile);
          if (nLength > 4
                        && lstrcmpiW (lpszFile + nLength - 4, L".exe") == 0)
            ret = TRUE;
        }
    }
  g_free (lpszFile);
#  else /* not UNICODE */
  dwAttributes = GetFileAttributesA (file);
  if (dwAttributes != INVALID_FILE_ATTRIBUTES)
    {
      if ((test & G_FILE_TEST_EXISTS)
        || ((test & G_FILE_TEST_IS_REGULAR) && (dwAttributes
                    & (FILE_ATTRIBUTE_DIRECTORY | FILE_ATTRIBUTE_DEVICE)) == 0)
        || ((test & G_FILE_TEST_IS_DIR)
                            && (dwAttributes & FILE_ATTRIBUTE_DIRECTORY) != 0))
        ret = TRUE;
      if (test & G_FILE_TEST_IS_EXECUTABLE)
        {
          gsize len;

          len = g_strlen (file);
          if (len > 4 && g_ascii_strcasecmp (file + len - 4, ".exe") == 0)
            ret = TRUE;
        }
    }
#  endif /* not UNICODE */
  return ret;
# else /* not G_OS_WIN32 */
  if ((test & G_FILE_TEST_EXISTS) && (g_access (file, F_OK) == 0))
    return TRUE;
  if ((test & G_FILE_TEST_IS_EXECUTABLE) && (g_access (file, X_OK) == 0))
    {
      if (getuid () != 0)
        return TRUE;
    }
  else
    {
      test &= ~G_FILE_TEST_IS_EXECUTABLE;
    }
  if (test & G_FILE_TEST_IS_SYMLINK)
    {
      struct stat s;

      if ((g_lstat (file, &s) == 0) && S_ISLNK (s.st_mode))
        return TRUE;
    }
  if (test & (G_FILE_TEST_IS_REGULAR | G_FILE_TEST_IS_DIR
                                                | G_FILE_TEST_IS_EXECUTABLE))
    {
      GStatBuf s;

      if (g_stat (file, &s) == 0
              && (((test & G_FILE_TEST_IS_REGULAR) && S_ISREG (s.st_mode))
              || ((test & G_FILE_TEST_IS_DIR) && S_ISDIR (s.st_mode))
              || ((test & G_FILE_TEST_IS_EXECUTABLE) && ((s.st_mode & S_IXOTH)
                        || (s.st_mode & S_IXUSR) || (s.st_mode & S_IXGRP)))))
        return TRUE;
    }
  return FALSE;
# endif /* not G_OS_WIN32 */
}
#endif /* USE_GTK_EMULATE */


#if ! GLIB_CHECK_VERSION(2,26,0)
gchar *
g_mkdtemp (gchar *tmpl)
{
  return g_mkdtemp_full (tmpl, 0700);
}


gchar *
g_mkdtemp_full (gchar *tmpl,
                gint   mode)
{
  gchar *xxxxxx;
  gint i;

  if (!tmpl)
    return NULL;
  xxxxxx = g_strrstr (tmpl, "XXXXXX");
  if (!xxxxxx)
    {
      errno = EINVAL;
      return NULL;
    }
  for (i = 0; i < 1000000; i++)
    {
      gchar *p;
      gint j, n;

      p = xxxxxx + 5;
      n = i;
      for (j = 0; j < 6; j++)
        {
          *p = n % 10 + '0';
          p--;
          n /= 10;
        }
      if (g_mkdir (tmpl, mode) == 0)
        return tmpl;
      else if (errno != EEXIST)
        return NULL;
    }
  errno = EEXIST;
  return NULL;
}
#endif /* not GLIB_CHECK_VERSION(2,26,0) */


#if ! GLIB_CHECK_VERSION(2,30,0)
gchar *
g_dir_make_tmp (const gchar  *tmpl,
                GError      **error)
{
  gchar *tmplate;
  const gchar *tmp;

  if (tmpl)
    {
      if (g_strchr (tmpl, G_DIR_SEPARATOR)
# ifdef G_OS_WIN32
        || g_strchr (tmpl, '/')
# endif /* G_OS_WIN32 */
        || !g_strstr (tmpl, "XXXXXX"))
        return NULL;
      tmp = tmpl;
    }
  else
    {
      tmp = ".XXXXXX";
    }
  tmplate = g_build_filename (g_get_tmp_dir (), tmp, NULL);
  if (!g_mkdtemp (tmplate))
    {
      g_free (tmplate);
      return NULL;
    }
  return tmplate;
}
#endif /* not GLIB_CHECK_VERSION(2,30,0) */


#if ! GLIB_CHECK_VERSION(2,6,0) && defined (G_OS_WIN32)
gint
g_mkdir (const gchar *filename,
         gint         mode)
{
  int err;
  gint ret;
  LPTSTR lpszFile;
# ifdef UNICODE
  gchar *utf8str;
# endif /* UNICODE */

# ifdef UNICODE
  utf8str = g_filename_to_utf8 (filename, -1, NULL, NULL, NULL);
  if (!utf8str)
    {
      errno = EINVAL;
      return -1;
    }
  lpszFile = g_utf8_to_utf16 (utf8str, -1, NULL, NULL, NULL);
  g_free (utf8str);
# else /* not UNICODE */
  lpszFile = g_win32_locale_filename_from_utf8 (filename);
# endif /* not UNICODE */
  if (!lpszFile)
    {
      errno = EINVAL;
      return -1;
    }
  ret = _tmkdir (lpszFile, mode);
  err = errno;
  g_free (lpszFile);
  errno = err;
  return ret;
}
#endif /* ! GLIB_CHECK_VERSION(2,6,0) && defined (G_OS_WIN32) */


#if ! GLIB_CHECK_VERSION(2,8,0) && defined (G_OS_WIN32)
gint
g_chmod (const gchar *filename,
         gint         mode)
{
  int err;
  gint ret;
  LPTSTR lpszFile;
# ifdef UNICODE
  gchar *utf8str;
# endif /* UNICODE */

# ifdef UNICODE
  utf8str = g_filename_to_utf8 (filename, -1, NULL, NULL, NULL);
  if (!utf8str)
    {
      g_free (path);
      return -1;
    }
  lpszFile = g_utf8_to_utf16 (utf8str, -1, NULL, NULL, NULL);
  g_free (utf8str);
# else /* not UNICODE */
  lpszFile = g_win32_locale_filename_from_utf8 (filename);
# endif /* not UNICODE */
  if (!lpszFile)
    {
      errno = EINVAL;
      return -1;
    }
  ret = _tchmod (lpszFile, mode);
  err = errno;
  g_free (lpszFile);
  errno = err;
  return ret;
}


gint
g_access (const gchar *filename,
          gint         mode)
{
  int err;
  gint ret;
  LPTSTR lpszFile;
# ifdef UNICODE
  gchar *utf8str;
# endif /* UNICODE */

# ifdef UNICODE
  utf8str = g_filename_to_utf8 (filename, -1, NULL, NULL, NULL);
  if (!utf8str)
    {
      errno = EINVAL;
      return -1;
    }
  lpszFile = g_utf8_to_utf16 (utf8str, -1, NULL, NULL, NULL);
  g_free (utf8str);
# else /* not UNICODE */
  lpszFile = g_win32_locale_filename_from_utf8 (filename);
# endif /* not UNICODE */
  if (!lpszFile)
    {
      errno = EINVAL;
      return -1;
    }
  ret = _taccess (lpszFile, mode);
  err = errno;
  g_free (lpszFile);
  errno = err;
  return ret;
}


gint g_creat (const gchar *filename,
              gint         mode)
{
  int err;
  gint ret;
  LPTSTR lpszFile;
# ifdef UNICODE
  gchar *utf8str;
# endif /* UNICODE */

# ifdef UNICODE
  utf8str = g_filename_to_utf8 (filename, -1, NULL, NULL, NULL);
  if (!utf8str)
    {
      errno = EINVAL;
      return -1;
    }
  lpszFile = g_utf8_to_utf16 (utf8str, -1, NULL, NULL, NULL);
  g_free (utf8str);
# else /* not UNICODE */
  lpszFile = g_win32_locale_filename_from_utf8 (filename);
# endif /* not UNICODE */
  if (!lpszFile)
    {
      errno = EINVAL;
      return -1;
    }
  ret = _tcreat (lpszFile, mode);
  err = errno;
  g_free (lpszFile);
  errno = err;
  return ret;
}


gint
g_chdir (const gchar *path)
{
  int err;
  gint ret;
  LPTSTR lpszPath;
# ifdef UNICODE
  gchar *utf8str;
# endif /* UNICODE */

# ifdef UNICODE
  utf8str = g_filename_to_utf8 (path, -1, NULL, NULL, NULL);
  if (!utf8str)
    {
      errno = EINVAL;
      return -1;
    }
  lpszPath = g_utf8_to_utf16 (utf8str, -1, NULL, NULL, NULL);
  g_free (utf8str);
# else /* not UNICODE */
  lpszPath = g_win32_locale_filename_from_utf8 (path);
# endif /* not UNICODE */
  if (!lpszPath)
    {
      errno = EINVAL;
      return -1;
    }
  ret = _tchdir (lpszPath);
  err = errno;
  g_free (lpszPath);
  errno = err;
  return ret;
}
#endif /* ! GLIB_CHECK_VERSION(2,8,0) && defined (G_OS_WIN32) */


#if ! GLIB_CHECK_VERSION(2,18,0)
gint
g_utime (const gchar    *filename,
         struct utimbuf *utb)
{
#ifdef G_OS_WIN32
  gint ret;
  HANDLE hFile;
  LPTSTR lpszFile;
# ifdef UNICODE
  gchar *utf8str;
# endif /* UNICODE */

# ifdef UNICODE
  utf8str = g_filename_to_utf8 (filename, -1, NULL, NULL, NULL);
  if (!utf8str)
    {
      errno = EINVAL;
      return -1;
    }
  lpszFile = g_utf8_to_utf16 (utf8str, -1, NULL, NULL, NULL);
  g_free (utf8str);
# else /* not UNICODE */
  lpszFile = g_win32_locale_filename_from_utf8 (filename);
# endif /* not UNICODE */
  if (!lpszFile)
    {
      errno = EINVAL;
      return -1;
    }
  ret = _tutime (lpszFile, (struct _utimbuf *)utb);
  g_free (lpszFile);
  return ret;
#else /* not G_OS_WIN32 */
# ifdef HAVE_UTIME
  return utime (filename, utb);
# else /* not HAVE_UTIME */
  return -1;
# endif /* not HAVE_UTIME */
#endif /* not G_OS_WIN32 */
}
#endif /* not GLIB_CHECK_VERSION(2,18,0) */
