/*
    orzcharset
    copyright (c) 1998-2013 Kazuki Iwamoto http://www.maid.org/ iwm@maid.org

    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program.  If not, see <http://www.gnu.org/licenses/>.
*/
#include "orzcharset.h"


enum
{
  ACTIVATE_SIGNAL,
  EDITED_SIGNAL,
  LAST_SIGNAL
};


static gint orz_charset_signals[LAST_SIGNAL] = {0};


/******************************************************************************
*                                                                             *
******************************************************************************/
#if GTK_CHECK_VERSION(2,24,0)
G_DEFINE_TYPE (OrzCharset, orz_charset, GTK_TYPE_COMBO_BOX_TEXT);
#elif GTK_CHECK_VERSION(2,4,0)
G_DEFINE_TYPE (OrzCharset, orz_charset, GTK_TYPE_COMBO_BOX_ENTRY);
#else /* not GTK_CHECK_VERSION(2,4,0) */
G_DEFINE_TYPE (OrzCharset, orz_charset, GTK_TYPE_COMBO);
#endif /* not GTK_CHECK_VERSION(2,4,0) */


static void
orz_charset_class_init (OrzCharsetClass *klass)
{
  klass->activate = NULL;
  klass->edited = NULL;

  orz_charset_signals[ACTIVATE_SIGNAL]
                = g_signal_new ("activate",
                                G_TYPE_FROM_CLASS (klass),
                                G_SIGNAL_RUN_LAST | G_SIGNAL_ACTION,
                                G_STRUCT_OFFSET (OrzCharsetClass, activate),
                                NULL, NULL,
                                g_cclosure_marshal_VOID__VOID,
                                G_TYPE_NONE, 0);
  orz_charset_signals[EDITED_SIGNAL]
                = g_signal_new ("edited",
                                G_TYPE_FROM_CLASS (klass),
                                G_SIGNAL_RUN_LAST | G_SIGNAL_ACTION,
                                G_STRUCT_OFFSET (OrzCharsetClass, edited),
                                NULL, NULL,
                                g_cclosure_marshal_VOID__VOID,
                                G_TYPE_NONE, 0);
}


static void
orz_charset_activate (GtkWidget  *widget,
                      OrzCharset *charset)
{
  g_signal_emit_by_name (G_OBJECT (charset), "activate");
}


static void
orz_charset_changed (GtkWidget  *widget,
                     OrzCharset *charset)
{
  g_signal_emit_by_name (G_OBJECT (charset), "edited");
}


#if ! GTK_CHECK_VERSION(2,4,0)
static void
orz_charset_grab_focus (GtkWidget *widget,
                        gpointer   user_data)
{
  gtk_widget_grab_focus (GTK_COMBO (widget)->entry);
}


static gboolean
orz_charset_mnemonic (GtkWidget *widget,
                      gboolean   arg1,
                      gpointer   user_data)
{
  gtk_widget_grab_focus (GTK_COMBO (widget)->entry);
  return TRUE;
}
#endif /* not GTK_CHECK_VERSION(2,4,0) */


static void
orz_charset_init (OrzCharset *charset)
{
#if GTK_CHECK_VERSION(2,24,0)
#elif GTK_CHECK_VERSION(2,4,0)
  gtk_combo_box_set_model (GTK_COMBO_BOX (charset),
                    GTK_TREE_MODEL (gtk_list_store_new (1, G_TYPE_STRING)));
  gtk_combo_box_entry_set_text_column (GTK_COMBO_BOX_ENTRY (charset), 0);
  g_signal_connect (G_OBJECT (gtk_bin_get_child (GTK_BIN (charset))),
                    "activate", G_CALLBACK (orz_charset_activate), charset);
  g_signal_connect (G_OBJECT (gtk_bin_get_child (GTK_BIN (charset))),
                    "changed", G_CALLBACK (orz_charset_changed), charset);
#else /* not GTK_CHECK_VERSION(2,4,0) */
  gtk_combo_set_value_in_list (GTK_COMBO (charset), FALSE, FALSE);
  gtk_combo_set_case_sensitive (GTK_COMBO (charset), TRUE);
  g_signal_connect (G_OBJECT (GTK_COMBO (charset)->entry), "activate",
                                G_CALLBACK (orz_charset_activate), charset);
  g_signal_connect (G_OBJECT (GTK_COMBO (charset)->entry), "changed",
                                G_CALLBACK (orz_charset_changed), charset);
  g_signal_connect (G_OBJECT (charset), "focus",
                                G_CALLBACK (orz_charset_grab_focus), NULL);
  g_signal_connect (G_OBJECT (charset), "mnemonic-activate",
                                G_CALLBACK (orz_charset_mnemonic), NULL);
#endif /* not GTK_CHECK_VERSION(2,4,0) */
}


/******************************************************************************
*                                                                             *
* ja:文字符号化方式関数群                                                     *
*                                                                             *
******************************************************************************/
/*  ja:新規作成
    RET,ウィジェット                                                        */
GtkWidget *
orz_charset_new (void)
{
#if GTK_CHECK_VERSION(2,24,0)
  GtkWidget *charset;

# if GTK_CHECK_VERSION(3,0,0)
  charset = GTK_WIDGET (g_object_new (ORZ_TYPE_CHARSET,
                                      "has-entry", TRUE,
                                      "entry-text-column", 0,
                                      "id-column", 1,
                                      NULL));
# else /* not GTK_CHECK_VERSION(3,0,0) */
  charset = GTK_WIDGET (g_object_new (ORZ_TYPE_CHARSET,
                                      "has-entry", TRUE,
                                      "entry-text-column", 0,
                                      NULL));
# endif /* not GTK_CHECK_VERSION(3,0,0) */
  g_signal_connect (G_OBJECT (gtk_bin_get_child (GTK_BIN (charset))),
                    "activate", G_CALLBACK (orz_charset_activate), charset);
  g_signal_connect (G_OBJECT (gtk_bin_get_child (GTK_BIN (charset))),
                    "changed", G_CALLBACK (orz_charset_changed), charset);
  return charset;
#else /* not GTK_CHECK_VERSION(2,24,0) */
  return GTK_WIDGET (g_object_new (ORZ_TYPE_CHARSET, NULL));
#endif /* not GTK_CHECK_VERSION(2,24,0) */
}


/*  ja:文字符号化方式を取得する
    charset,ウィジェット
        RET,文字符号化方式                                                  */
gchar *
orz_charset_get_charset (OrzCharset *charset)
{
#if GTK_CHECK_VERSION(2,4,0)
  return charset ? g_strdup (gtk_entry_get_text
                (GTK_ENTRY (gtk_bin_get_child (GTK_BIN (charset))))) : NULL;
#else /* not GTK_CHECK_VERSION(2,4,0) */
  return charset ? g_strdup (gtk_entry_get_text
                            (GTK_ENTRY (GTK_COMBO (charset)->entry))) : NULL;
#endif /* not GTK_CHECK_VERSION(2,4,0) */
}


/*  ja:文字符号化方式を設定する
    charset,ウィジェット
        str,文字符号化方式
        RET,TRUE:正常終了,FALSE:エラー                                      */
gboolean
orz_charset_set_charset (OrzCharset *charset,
                          const gchar *str)
{
  if (!charset)
    return FALSE;
#if GTK_CHECK_VERSION(2,4,0)
  gtk_entry_set_text (GTK_ENTRY (gtk_bin_get_child (GTK_BIN (charset))),
                                                            str ? str : "");
#else /* not GTK_CHECK_VERSION(2,4,0) */
  gtk_entry_set_text (GTK_ENTRY (GTK_COMBO (charset)->entry), str ? str : "");
#endif /* not GTK_CHECK_VERSION(2,4,0) */
  return TRUE;

}


/*  ja:文字符号化方式の候補を取得する
    charset,ウィジェット
        RET,登録されている文字符号化方式の候補                              */
gchar *
orz_charset_get_candidate (OrzCharset *charset)
{
  gchar *candidate = NULL;

  if (charset)
    {
#if GTK_CHECK_VERSION(2,4,0)
      gint i, column;
      GtkTreeModel *model;

      model = gtk_combo_box_get_model (GTK_COMBO_BOX (charset));
# if GTK_CHECK_VERSION(2,24,0)
      column = gtk_combo_box_get_entry_text_column (GTK_COMBO_BOX (charset));
# else /* not GTK_CHECK_VERSION(2,24,0) */
      for (column = gtk_tree_model_get_n_columns (model) - 1;
                                                        column > 0; column--)
        if (gtk_tree_model_get_column_type (model, column) == G_TYPE_STRING)
          break;
# endif /* not GTK_CHECK_VERSION(2,24,0) */
      for (i = gtk_tree_model_iter_n_children (model, NULL) - 1; i >= 0; i--)
        {
          gchar *str, *tmp;
          GtkTreeIter iter;

          gtk_tree_model_iter_nth_child (model, &iter, NULL, i);
          gtk_tree_model_get (model, &iter, column, &tmp, -1);
          str = g_strconcat (tmp, candidate ? "," : NULL, candidate, NULL);
          g_free (tmp);
          g_free (candidate);
          candidate = str;
        }
#else /* not GTK_CHECK_VERSION(2,4,0) */
      GList *glist;

      glist = gtk_container_get_children
                                (GTK_CONTAINER (GTK_COMBO (charset)->list));
      for (glist = g_list_first (glist);
                            glist; glist = g_list_delete_link (glist, glist))
        {
          const gchar *text;

          text = gtk_label_get_text
                    (GTK_LABEL (gtk_bin_get_child (GTK_BIN (glist->data))));
          if (candidate)
            {
              gchar *str;

              str = g_strconcat (candidate, ",", text, NULL);
              g_free (candidate);
              candidate = str;
            }
          else
            {
              candidate = g_strdup (text);
            }
        }
#endif /* not GTK_CHECK_VERSION(2,4,0) */
    }
  return candidate;
}


/*  ja:文字符号化方式の候補を追加する
      charset,ウィジェット
    candidate,文字符号化方式の候補
          RET,TRUE:正常終了,FALSE:エラー                                    */
gboolean
orz_charset_add_candidate (OrzCharset  *charset,
                           const gchar *candidate)
{
  gboolean result = TRUE;
  gchar **str;
  gint i;
#if GTK_CHECK_VERSION(2,4,0)
  gint column;
  GtkListStore *store;
#endif /* GTK_CHECK_VERSION(2,4,0) */

  if (!charset || !candidate)
    return FALSE;
  str = g_strsplit (candidate, ",", 0);
#if GTK_CHECK_VERSION(2,4,0)
  store = GTK_LIST_STORE (gtk_combo_box_get_model (GTK_COMBO_BOX (charset)));
# if GTK_CHECK_VERSION(2,24,0)
  column = gtk_combo_box_get_entry_text_column (GTK_COMBO_BOX (charset));
# else /* not GTK_CHECK_VERSION(2,24,0) */
  for (column = gtk_tree_model_get_n_columns (GTK_TREE_MODEL (store)) - 1;
                                                        column > 0; column--)
    if (gtk_tree_model_get_column_type (GTK_TREE_MODEL (store), column)
                                                            == G_TYPE_STRING)
      break;
# endif /* not GTK_CHECK_VERSION(2,24,0) */
#endif /* GTK_CHECK_VERSION(2,4,0) */
  for (i = 0; str[i]; i++)
    if (charuty_is_valid (str[i]))
      {
        gint j;

#if GTK_CHECK_VERSION(2,4,0)
        GtkTreeIter iter;

        for (j = gtk_tree_model_iter_n_children
                            (GTK_TREE_MODEL (store), NULL) - 1; j >= 0; j--)
          {
            gchar *tmp;
            gint cmp;

            gtk_tree_model_iter_nth_child (GTK_TREE_MODEL (store),
                                                            &iter, NULL, j);
            gtk_tree_model_get (GTK_TREE_MODEL (store),
                                                    &iter, column, &tmp, -1);
            cmp = g_ascii_strcasecmp (tmp, str[i]);
            g_free (tmp);
            if (cmp == 0)
              {
                gtk_list_store_remove (store, &iter);
                j--;
              }
            if (cmp <= 0)
              break;
          }
        gtk_list_store_insert (store, &iter, j + 1);
        gtk_list_store_set (store, &iter, column, str[i], -1);
#else /* not GTK_CHECK_VERSION(2,4,0) */
        GList *glist;
        GtkWidget *item;

        glist = gtk_container_get_children
                                (GTK_CONTAINER (GTK_COMBO (charset)->list));
        for (j = g_list_length (glist) - 1; j >= 0; j--)
          {
            gint cmp;

            cmp = g_ascii_strcasecmp (gtk_label_get_text (GTK_LABEL
                (gtk_bin_get_child (GTK_BIN (g_list_nth_data (glist, j))))),
                                                                    str[i]);
            if (cmp == 0)
              {
                gtk_list_clear_items (GTK_LIST (GTK_COMBO (charset)->list),
                                                                    j, j + 1);
                j--;
              }
            if (cmp <= 0)
              break;
          }
        g_list_free (glist);
        item = gtk_list_item_new_with_label (str[i]);
        gtk_widget_show (item);
        glist = g_list_append (NULL, item);
        gtk_list_insert_items (GTK_LIST (GTK_COMBO (charset)->list),
                                                                glist, j + 1);
        g_list_free (glist);
#endif /* not GTK_CHECK_VERSION(2,4,0) */
      }
    else
      {
        result = FALSE;
      }
  g_strfreev (str);
  return result;
}


/*  ja:文字符号化方式の候補を削除する
      charset,ウィジェット
    candidate,文字符号化方式の候補,NULL:すべて削除
          RET,TRUE:正常終了,FALSE:エラー                                    */
gboolean
orz_charset_remove_candidate (OrzCharset *charset,
                               const gchar *candidate)
{
  gboolean result = TRUE;
  gchar **str;
  gint i;
#if GTK_CHECK_VERSION(2,4,0)
  gint column;
  GtkListStore *store;
#endif /* GTK_CHECK_VERSION(2,4,0) */

  if (!charset)
    return FALSE;
#if GTK_CHECK_VERSION(2,4,0)
  store = GTK_LIST_STORE (gtk_combo_box_get_model (GTK_COMBO_BOX (charset)));
# if GTK_CHECK_VERSION(2,24,0)
  column = gtk_combo_box_get_entry_text_column (GTK_COMBO_BOX (charset));
# else /* not GTK_CHECK_VERSION(2,24,0) */
  for (column = gtk_tree_model_get_n_columns (GTK_TREE_MODEL (store)) - 1;
                                                        column > 0; column--)
    if (gtk_tree_model_get_column_type (GTK_TREE_MODEL (store), column)
                                                            == G_TYPE_STRING)
      break;
# endif /* not GTK_CHECK_VERSION(2,24,0) */
#endif /* GTK_CHECK_VERSION(2,4,0) */
  if (candidate)
    {
      str = g_strsplit (candidate, ",", 0);
      for (i = 0; str[i]; i++)
        {
          gint j;

#if GTK_CHECK_VERSION(2,4,0)
          for (j = gtk_tree_model_iter_n_children
                            (GTK_TREE_MODEL (store), NULL) - 1; j >= 0; j--)
            {
              gchar *tmp;
              gint cmp;
              GtkTreeIter iter;

              gtk_tree_model_iter_nth_child (GTK_TREE_MODEL (store),
                                                            &iter, NULL, j);
              gtk_tree_model_get (GTK_TREE_MODEL (store),
                                                    &iter, column, &tmp, -1);
              cmp = g_ascii_strcasecmp (tmp, str[i]);
              g_free (tmp);
              if (cmp == 0)
                {
                  gtk_list_store_remove (store, &iter);
                  break;
                }
            }
#else /* not GTK_CHECK_VERSION(2,4,0) */
          GList *glist;

          glist = gtk_container_get_children
                                (GTK_CONTAINER (GTK_COMBO (charset)->list));
          for (j = g_list_length (glist) - 1; j >= 0; j--)
            if (g_ascii_strcasecmp (gtk_label_get_text (GTK_LABEL
                (gtk_bin_get_child (GTK_BIN (g_list_nth_data (glist, j))))),
                                                                str[i]) == 0)
              {
                gtk_list_clear_items (GTK_LIST (GTK_COMBO (charset)->list),
                                                                    j, j + 1);
                break;
              }
          g_list_free (glist);
#endif /* not GTK_CHECK_VERSION(2,4,0) */
          if (j < 0)
            result = FALSE;
        }
      g_strfreev (str);
    }
  else
    {
#if GTK_CHECK_VERSION(2,4,0)
      gtk_list_store_clear (store);
#else /* not GTK_CHECK_VERSION(2,4,0) */
      gtk_list_clear_items (GTK_LIST (GTK_COMBO (charset)->list), 0, G_MAXINT);
#endif /* not GTK_CHECK_VERSION(2,4,0) */
    }
  return result;
}


/*  ja:文字符号化方式を検査する
    charset,ウィジェット
        RET,TRUE:正しい文字符号化方式,FALSE:不正な文字符号化方式            */
gboolean
orz_charset_is_valid (OrzCharset *charset)
{
#if GTK_CHECK_VERSION(2,4,0)
  return charset ? charuty_is_valid (gtk_entry_get_text
                (GTK_ENTRY (gtk_bin_get_child (GTK_BIN (charset))))) : FALSE;
#else /* not GTK_CHECK_VERSION(2,4,0) */
  return charset ? charuty_is_valid (gtk_entry_get_text
                            (GTK_ENTRY (GTK_COMBO (charset)->entry))) : FALSE;
#endif /* not GTK_CHECK_VERSION(2,4,0) */
}
