/*
 * The MIT License
 *
 * Copyright 2015 nazo.
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */
package jp.sourceforge.mmd.ik_solver;

import jp.sourceforge.mmd.motion.geo.Matrix;
import jp.sourceforge.mmd.motion.model.Bone;
import jp.sourceforge.mmd.motion.geo.Vector3D;

/**
 * IK をするためのクラス.
 * Inverse Kinetics (IK) とは, 腕など多節で回転しかできないようなもの先端を,
 * 目的の位置に合わせる用に各節の回転角を合わせる行為で, 反復計算が有効なので,
 * コンピューターにやらせることが多い.
 * 
 * @author nazo
 */
public class IKSolver {
    /**
     * はじめから制限軸がある場合のフラグ.
     * モデルに回転制限がある場合に自動的に付く.
     */
    static final public int RLIMIT_LIMITED=0x80;

    /**
     * グローバルで回転させない. bones[0] にしか効果ない.
     */
    static final public int RLIMIT_NOGlobalR=0x40;

    /**
     * ローカル x 軸周りの回転させないフラグ.
     */
    static final public int RLIMIT_NOX=0x3;

    /**
     * ローカル y 軸周りの回転させないフラグ.
     */
    static final public int RLIMIT_NOY=0xc;

    /**
     * ローカル z 軸周りの回転させないフラグ.
     */
    static final public int RLIMIT_NOZ=0x30;

    /**
     * 回転角度制限. 固定値で sine(pi/4).
     */
    static final protected double limit_sine=Math.sqrt(0.5);

    /**
     * 対象になるボーン.
     * [0]: target に合わせるボーン,
     * [1+]: 動かすボーン
     */
    protected Bone[] bones;

    /**
     * ボーンに対する回転リミットのフラグ.
     * {@link #setLimits(int, int) }などで指定するもの.
     */
    protected int [] limits;
    
    /**
     * 今のところ何もしないコンストラクター.
     * 必ず{@link solve}する前に, {@link setBones} をすること.
     */
    public IKSolver(){
    }

    /**
     * 対象になるボーンを設定.
     * @param arms [0]: target に合わせるボーン, [1+]: 動かすボーン.
     */
    public void setBones(Bone [] arms){
        bones=new Bone[arms.length];
        limits=new int[arms.length];
        int i;

        for(i=0;i<arms.length;i++){
            bones[i]=arms[i];
            if(arms[i].getLimitRot()!=null){
                limits[i]=RLIMIT_LIMITED;
            }
        }
    }

    /**
     * IKをする際に、回転に制限を加える.
     * @param index {@link #bones} の番号
     * @param flags {@link #RLIMIT_NOX} などを or 加算する。
     */
    public void setLimits(int index,int flags){
        limits[index]=flags;
    }

    /**
     * 回転制限を得る.
     * @param index 番目
     * @return 回転制限のフラグ
     */
    public int getLimits(int index){
        return limits[index];
    }

    /**
     * target にbones[0]を合わせるIKを開始する. 反復回数は100回で固定.
     * @param target bones[0]を合わせる目的の位置(グローバルベクトル).
     */
    public void solve(Vector3D target){
        int i,j,k;
        Vector3D center;
        Vector3D tr,ta;
        Vector3D ntr,nta;
        Vector3D rt;
        Matrix gmr=null;
        double co,sine,norm;

        if((limits[0]&RLIMIT_NOGlobalR)>0){
            gmr=bones[0].getGMatrix();
        }
        for(i=0;i<100;i++){
            for(k=1;k<bones.length;k++){
                center=bones[k].getPos();
                tr=target.sub(center);
                ta=bones[0].getPos().sub(center);
                if((norm=tr.norm())==0){continue;}
                ntr=tr.divide(norm);
                if((norm=ta.norm())==0){continue;}
                nta=ta.divide(norm);
                rt=nta.cross(ntr);

                if(limits[k]>0){
                    Vector3D axis;
                    if((limits[k] & RLIMIT_LIMITED)>0){
                        sine=rt.times(axis=bones[k].getLimitRot());
                        rt=axis.times(sine);
                        co=Math.sqrt(1-sine*sine);
                    }else {
                        if((limits[k]&RLIMIT_NOX)>0){
                            co=rt.times(axis=bones[k].getLx());
                            rt=rt.sub(axis.times(co));
                        }
                        if((limits[k]&RLIMIT_NOY)>0){
                            co=rt.times(axis=bones[k].getLy());
                            rt=rt.sub(axis.times(co));
                        }
                        if((limits[k]&RLIMIT_NOZ)>0){
                            co=rt.times(axis=bones[k].getLz());
                            rt=rt.sub(axis.times(co));
                        }
                        sine=rt.norm();
                        co=((sine>1.0)?0:Math.sqrt(1-sine*sine));
                    }
                } else {
                    sine=rt.norm();
                    if(sine>limit_sine){
                        rt=rt.times(limit_sine/sine);
                        co=limit_sine;
                    }else{
                        co=ntr.times(nta);
                    }
                }
                if(sine==0)continue;
                bones[k].rotate(rt, co);
            }
        }
        if((limits[0]&RLIMIT_NOGlobalR)>0){
            bones[0].rotateToG(gmr);
        }
    }
}
