/*
 * The MIT License
 *
 * Copyright 2015 nazo.
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */
package jp.sourceforge.mmd.midiMotion.gui;
import java.awt.datatransfer.DataFlavor;
import java.awt.datatransfer.Transferable;
import java.awt.datatransfer.UnsupportedFlavorException;
import java.awt.dnd.DnDConstants;
import java.awt.dnd.DropTarget;
import java.awt.dnd.DropTargetDropEvent;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.PrintStream;
import java.util.List;
import java.util.logging.Level;
import java.util.logging.Logger;
import javax.swing.JFileChooser;
import javax.swing.JFrame;
import javax.swing.JOptionPane;
import javax.swing.filechooser.FileNameExtensionFilter;
import javax.swing.table.DefaultTableModel;
import jp.sourceforge.mmd.midiMotion.MidiResolver;

/**
 * 複数の Standard MIDI ファイルを結合する手伝いをするGUI.
 * {@link MidiResolver#add(MidiResolver) } を利用する.
 * @author nazo
 */
public class MidiConcat extends JFrame {
    private MidiResolver mir;
    private File midiDir;

    /**
     * Creates new form MidiCat
     */
    public MidiConcat() {
        super("Midi Concatnate");
        String s=System.getProperty("midiMotion.midiDir");
        if(s!=null)
            midiDir=new File(s);
        initComponents();
        setDropTarget(new HteDropTarget(this));
    }

    /**
     * This method is called from within the constructor to initialize the form.
     * WARNING: Do NOT modify this code. The content of this method is always
     * regenerated by the Form Editor.
     */
    @SuppressWarnings("unchecked")
    // <editor-fold defaultstate="collapsed" desc="Generated Code">//GEN-BEGIN:initComponents
    private void initComponents() {
        java.awt.GridBagConstraints gridBagConstraints;

        jScrollPane2 = new javax.swing.JScrollPane();
        midiTable = new javax.swing.JTable();
        buttonAdd = new javax.swing.JButton();
        buttonDo = new javax.swing.JButton();
        jButton1 = new javax.swing.JButton();
        jButtonUp = new javax.swing.JButton();
        jButtonDown = new javax.swing.JButton();

        setDefaultCloseOperation(javax.swing.WindowConstants.DISPOSE_ON_CLOSE);
        getContentPane().setLayout(new java.awt.GridBagLayout());

        midiTable.setModel(new javax.swing.table.DefaultTableModel(
            new Object [][] {
            },
            new String [] {
                "ファイル名", "開始点(beat)", "長さ(beat)"
            }
        ));
        jScrollPane2.setViewportView(midiTable);

        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridx = 1;
        gridBagConstraints.gridy = 0;
        gridBagConstraints.gridheight = 5;
        gridBagConstraints.fill = java.awt.GridBagConstraints.BOTH;
        gridBagConstraints.weightx = 0.1;
        gridBagConstraints.weighty = 0.1;
        getContentPane().add(jScrollPane2, gridBagConstraints);

        buttonAdd.setText("add");
        buttonAdd.addActionListener(new java.awt.event.ActionListener() {
            public void actionPerformed(java.awt.event.ActionEvent evt) {
                buttonAddActionPerformed(evt);
            }
        });
        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridx = 2;
        gridBagConstraints.gridy = 0;
        getContentPane().add(buttonAdd, gridBagConstraints);

        buttonDo.setText("do");
        buttonDo.addActionListener(new java.awt.event.ActionListener() {
            public void actionPerformed(java.awt.event.ActionEvent evt) {
                buttonDoActionPerformed(evt);
            }
        });
        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridx = 2;
        gridBagConstraints.gridy = 4;
        getContentPane().add(buttonDo, gridBagConstraints);

        jButton1.setText("remove");
        jButton1.addActionListener(new java.awt.event.ActionListener() {
            public void actionPerformed(java.awt.event.ActionEvent evt) {
                jButton1ActionPerformed(evt);
            }
        });
        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridx = 2;
        gridBagConstraints.gridy = 2;
        getContentPane().add(jButton1, gridBagConstraints);

        jButtonUp.setText("↑");
        jButtonUp.addActionListener(new java.awt.event.ActionListener() {
            public void actionPerformed(java.awt.event.ActionEvent evt) {
                jButtonUpActionPerformed(evt);
            }
        });
        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridx = 2;
        gridBagConstraints.gridy = 1;
        getContentPane().add(jButtonUp, gridBagConstraints);

        jButtonDown.setText("↓");
        jButtonDown.addActionListener(new java.awt.event.ActionListener() {
            public void actionPerformed(java.awt.event.ActionEvent evt) {
                jButtonDownActionPerformed(evt);
            }
        });
        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridx = 2;
        gridBagConstraints.gridy = 3;
        getContentPane().add(jButtonDown, gridBagConstraints);

        pack();
    }// </editor-fold>//GEN-END:initComponents

    private void buttonAddActionPerformed(java.awt.event.ActionEvent evt) {//GEN-FIRST:event_buttonAddActionPerformed
        JFileChooser jfc=new JFileChooser(midiDir);
        jfc.setFileFilter(new FileNameExtensionFilter("MIDI File", "mid"));
        jfc.setMultiSelectionEnabled(true);
        if(jfc.showOpenDialog(this)!=JFileChooser.APPROVE_OPTION)
            return;
        add(jfc.getSelectedFiles());
    }//GEN-LAST:event_buttonAddActionPerformed

    private void add(File [] fs){
        DefaultTableModel m=(DefaultTableModel)midiTable.getModel();
        int offset=0;
        int n=m.getRowCount();
        for(int i=0;i<n;i++){
            offset+=(Integer)m.getValueAt(i, 2);
        }
        MidiResolver midiR;
        int l;
        for(File f:fs){
            midiR=new MidiResolver(f);
            l=(int)(midiR.getMidiSeq().getTickLength()/midiR.getReso());
            m.addRow(new Object []{f,offset,l});
            offset+=l;
        }
    }

    private void buttonDoActionPerformed(java.awt.event.ActionEvent evt) {//GEN-FIRST:event_buttonDoActionPerformed
        MidiResolver mr;
        int i,l;
        DefaultTableModel dtm=(DefaultTableModel)midiTable.getModel();
        l=dtm.getRowCount();
        mr=new MidiResolver((File)dtm.getValueAt(0,0));
        for(i=1;i<l;i++){
            mr.add(new MidiResolver((File)dtm.getValueAt(i,0)));
        }

        JFileChooser jfc=new JFileChooser(midiDir);
        jfc.setFileFilter(new FileNameExtensionFilter("MIDI File", "mid"));
        if(jfc.showSaveDialog(this)!=JFileChooser.APPROVE_OPTION)
            return;
        File f=jfc.getSelectedFile();
        FileOutputStream fos;
        if(!f.getName().toLowerCase().endsWith(".mid")){
            f=new File(f.getParentFile(),f.getName()+".mid");
        }
        try{
            fos=new FileOutputStream(f);
            mr.write(fos);
            fos.close();
        } catch (IOException ex) {
            JOptionPane.showMessageDialog(this, "IO エラー:"+ex.getLocalizedMessage());
        }
    }//GEN-LAST:event_buttonDoActionPerformed

    private void jButton1ActionPerformed(java.awt.event.ActionEvent evt) {//GEN-FIRST:event_jButton1ActionPerformed
        DefaultTableModel dtm=(DefaultTableModel)midiTable.getModel();
        int i=midiTable.getSelectedRow();
        if(i<0)return;
        int offset=(Integer)dtm.getValueAt(i, 1);
        int n=dtm.getRowCount();
        for(int j=i+i;j<n;j++){
            dtm.setValueAt(offset, j, 1);
            offset+=(Integer)dtm.getValueAt(j, 2);
        }
        dtm.removeRow(i);
    }//GEN-LAST:event_jButton1ActionPerformed

    private void jButtonUpActionPerformed(java.awt.event.ActionEvent evt) {//GEN-FIRST:event_jButtonUpActionPerformed
        int i=midiTable.getSelectedRow();
        if(i<1)return;
        DefaultTableModel dtm=(DefaultTableModel)midiTable.getModel();
        int n=dtm.getRowCount();
        int offset=(Integer)dtm.getValueAt(i-1, 1);
        dtm.moveRow(i,i,i-1);
        for(int j=i-1;j<n;j++){
            dtm.setValueAt(offset, j, 1);
            offset+=(Integer)dtm.getValueAt(j, 2);
        }

    }//GEN-LAST:event_jButtonUpActionPerformed

    private void jButtonDownActionPerformed(java.awt.event.ActionEvent evt) {//GEN-FIRST:event_jButtonDownActionPerformed
        int i=midiTable.getSelectedRow();
        if(i<0)return;
        DefaultTableModel dtm=(DefaultTableModel)midiTable.getModel();
        int n=dtm.getRowCount();
        if(i==n-1)return;
        int offset=(Integer)dtm.getValueAt(i, 1);
        dtm.moveRow(i,i,i+1);
        for(int j=i;j<n;j++){
            dtm.setValueAt(offset, j, 1);
            offset+=(Integer)dtm.getValueAt(j, 2);
        }
        
    }//GEN-LAST:event_jButtonDownActionPerformed

    static public void main(String [] args){
        FileInputStream fis;
        MidiResolver midiR1=null,midiRA;
        
        if(args.length<2){
            PrintStream e=System.err;
            e.println("Usage:");
            e.println("\t$ java -jar MID2Motion.jar (Output MIDI) (MIDI file) (MIDI file) ...\n");
            e.println("catted.mid will be outputed.");
            System.exit(1);
        }
        try {
            for(String s:args){
                fis=new FileInputStream(s);
                if(midiR1==null){
                    midiR1=new MidiResolver(new File(s));
                }else {
                    midiRA=new MidiResolver(new File(s));
                    midiR1.add(midiRA);
                }
            }
            midiR1.write(new FileOutputStream("catted.mid"));
        } catch (FileNotFoundException ex) {
            Logger.getLogger(MidiConcat.class.getName()).log(Level.SEVERE, null, ex);
        } catch (IOException ex) {
            Logger.getLogger(MidiConcat.class.getName()).log(Level.SEVERE, null, ex);
        }
    }

    // Variables declaration - do not modify//GEN-BEGIN:variables
    private javax.swing.JButton buttonAdd;
    private javax.swing.JButton buttonDo;
    private javax.swing.JButton jButton1;
    private javax.swing.JButton jButtonDown;
    private javax.swing.JButton jButtonUp;
    private javax.swing.JScrollPane jScrollPane2;
    private javax.swing.JTable midiTable;
    // End of variables declaration//GEN-END:variables
    private class HteDropTarget extends DropTarget {
        private MidiConcat parent;
        HteDropTarget(MidiConcat midiC){
            parent=midiC;
        }
        @Override
        public void drop(DropTargetDropEvent dtde) {
            Transferable t = dtde.getTransferable();
            if (t.isDataFlavorSupported(DataFlavor.javaFileListFlavor)) {
                dtde.acceptDrop(DnDConstants.ACTION_REFERENCE);
                try {
                    List<File> list = (List<File>)t.getTransferData(DataFlavor.javaFileListFlavor);
                    add(list.toArray(new File [list.size()]));
                } catch (UnsupportedFlavorException e) {
                } catch (IOException ex) {
                    JOptionPane.showMessageDialog(parent, "IO エラー:"+ex.getLocalizedMessage());
                }
            }
        }
    }
}
