/*
 * The MIT License
 *
 * Copyright 2015 nazo.
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */
/**
 * 3次元幾何学に関するクラスのパッケージ.
 */
package jp.sourceforge.mmd.motion.geo;

/**
 * 3次元行列クラス (右手系・左手系関係ない).
 * 3D matrix class (independent with right and left hand)
 * @author nazo
 */
public class Matrix implements Cloneable{
    /** 行列成分. double [9]で, 行が先に増える配置.
     * <pre>
     * 0 3 6
     * 1 4 7
     * 2 5 8
     * </pre>
     */
    protected double r[]=new double[9];

    /**
     * 単位行列.
     */
    public Matrix(){
        r[0]=r[4]=r[8]=1;
        r[1]=r[2]=r[3]=0;
        r[5]=r[6]=r[7]=0;
    }

    /** 成分を設定する.
     * @param r double[9]. それ以上の成分は無視される.
     */
    public Matrix(double []r){
        int i;
        for(i=0;i<9;i++){
            this.r[i]=r[i];
        }
    }

    /**
     * 等しいか比較する.
     * @param m 比較対象
     * @return 成分が全て同じならtrue.
     */
    public boolean equals(Matrix m){
        int i;
        for(i=0;i<9;i++){
           if(r[i]!=m.r[i])return false;
        }
        return true;
    }

    /**
     * クローン.
     * @param b クローン元.
     */
    public Matrix(Matrix b){
        int i;
        for(i=0;i<9;i++){
            r[i]=b.r[i];
        }
    }

    /**
     * クローン.
     * @return クローンされたMatrix.
     */
    @Override
    @SuppressWarnings({"CloneDeclaresCloneNotSupported", "CloneDoesntCallSuperClone"})
    public Matrix clone() {
        return new Matrix(r);
    }

    /**
     * ローカルx軸, z軸から(左手系)回転行列を作る.
     * @param lx ローカルx軸. 単位ベクトルである必要がある.
     * @param lz ローカルz軸(左手系). 単位ベクトルである必要がある.
     */
    public Matrix(Vector3D lx,Vector3D lz){
        int i;
        double [] v =lx.toDouble();
        for(i=0;i<3;i++){
            this.r[i]=v[i];
        }
        v =lz.toDouble();
        for(i=0;i<3;i++){
            this.r[i+6]=v[i];
        }
        
        v =lz.cross(lx).toDouble();
        for(i=0;i<3;i++){
            this.r[i+3]=v[i];
        }
    }

    /**
     * オイラー角から回転行列を生成する. MMDで採用されているのは, Y-X-Z 回転型で,
     * ヨーク・ピッチ・ロール角と呼ばれる方が普通の型.
     * @param rx x軸周りの回転角(度)
     * @param ry y軸周りの回転角(度)
     * @param rz z軸周りの回転角(度)
     * @return それらの回転を単位行列にしたあとの回転行列.
     */
    static public Matrix rotation(double rx,double ry,double rz){
        Matrix b,c,r;
        double []a=new double[9];

        a[0]=a[4]=Math.cos(rz*Math.PI/180);
        a[3]=Math.sin(rz*Math.PI/180);
        a[1]=-a[3];
        a[8]=1;
        a[2]=a[5]=a[6]=a[7]=0;
        c=new Matrix(a);

        a[4]=a[8]=Math.cos(rx*Math.PI/180);
        a[5]=Math.sin(rx*Math.PI/180);
        a[7]=-a[5];
        a[0]=1;
        a[1]=a[2]=a[3]=a[6]=0;
        b=new Matrix(a);
        r=b.times(c);

        a[0]=a[8]=Math.cos(ry*Math.PI/180);
        a[2]=Math.sin(ry*Math.PI/180);
        a[6]=-a[2];
        a[4]=1;
        a[1]=a[3]=a[5]=a[7]=0;
        
        c=new Matrix(a);
        return c.times(r);
    }

    /**
     * 回転用4元数を回転用マトリクスにする.
     * @param nx 極力ノーマライズしてあること.
     * @param ny 極力ノーマライズしてあること.
     * @param nz 極力ノーマライズしてあること.
     * @param w 極力ノーマライズしてあること.
     * @return 対応する回転行列. ノルムが0以下のとき, null.
     */
    static public Matrix rotationQ(double nx,double ny,double nz,double w){
        double []a=new double[9];
        double []s=new double[10];
        double norm=Math.sqrt(nx*nx +ny*ny +nz*nz +w*w);
        if(norm<=0){
            return null;
        }
        nx/=norm;
        ny/=norm;
        nz/=norm;
        w/=norm;

        s[0]=w*w;s[1]=nx*nx;s[2]=ny*ny;s[3]=nz*nz;
        s[4]=w*nx;s[5]=w*ny;s[6]=w*nz;
        s[7]=nx*ny;s[8]=nx*nz;s[9]=ny*nz;

        a[0]=s[0]+s[1]-s[2]-s[3];
        a[4]=s[0]-s[1]+s[2]-s[3];
        a[8]=s[0]-s[1]-s[2]+s[3];

        a[1]=2*(s[7]+s[6]);
        a[2]=2*(s[8]-s[5]);

        a[3]=2*(s[7]-s[6]);
        a[5]=2*(s[9]+s[4]);

        a[6]=2*(s[8]+s[5]);
        a[7]=2*(s[9]-s[4]);

        return new Matrix(a);
    }

    /** MMDで使われるオイラー角を返す.
     * @return [0] x軸周り回転(度), [1] y軸周り回転(度), [2] z軸周り回転(度). */
    public double [] angles(){
        Vector3D tx,ty,tz,pz,x;
        double rz;
        double []rv=new double[3];
        int i;
        pz=new Vector3D(r[6],0,r[8]);
        rz=pz.norm();
        rv[0]=Math.atan2(-r[7], rz);
        rv[1]=(r[7]<=-1 || r[7]>=1)?0:Math.atan2(-r[6],r[8]);

        tz=pz.divide(rz);
        tx=tz.cross(new Vector3D(0,1,0));
        ty=new Vector3D(r[6],r[7],r[8]).cross(tx);
        x=new Vector3D(r[0],r[1],r[2]);
        rv[2]=Math.atan2(x.times(ty),-x.times(tx));

        for(i=0;i<3;i++){
            rv[i]*=180/Math.PI;
        }
        return rv;
    }

    /**
     * 四元数を得る.
     * @return 四元数, [0] qx, [1]qy, [2] qz, [3] w.
     */
    public double [] getQuotanions(){
        double []q=new double[4];
        int i,biggest;
        q[3]= r[0]+r[4]+r[8]+1;
        q[0]= r[0]-r[4]-r[8]+1;
        q[1]=-r[0]+r[4]-r[8]+1;
        q[2]=-r[0]-r[4]+r[8]+1;

        biggest=0;
        for(i=1;i<4;i++){
            if(q[i]>q[biggest])biggest=i;
        }

        double mult=0.25/(q[biggest]=Math.sqrt(q[biggest])*0.5);
        if(biggest==0){
            q[1]=(r[1]+r[3])*mult;
            q[2]=(r[2]+r[6])*mult;
            q[3]=(r[5]-r[7])*mult;
        }else if(biggest==1){
            q[0]=(r[1]+r[3])*mult;
            q[2]=(r[5]+r[7])*mult;
            q[3]=(r[6]-r[2])*mult;
        }else if(biggest==2){
            q[0]=(r[2]+r[6])*mult;
            q[1]=(r[5]+r[7])*mult;
            q[3]=(r[1]-r[3])*mult;            
        }else {
            q[0]=(r[5]-r[7])*mult;
            q[1]=(r[6]-r[2])*mult;
            q[2]=(r[1]-r[3])*mult;
        }

        double norm=Math.sqrt(q[0]*q[0] +q[1]*q[1] +q[2]*q[2] +q[3]*q[3]);
        q[0]/=norm;
        q[1]/=norm;
        q[2]/=norm;
        q[3]/=norm;
        return q;
    }

    /**
     * 回転後のローカルx軸を返す。
     * @return ローカルx軸のグローバル側のベクトル.
     */
    public Vector3D getLx(){
        return new Vector3D(r);
    }

    /**
     * 回転後のローカルy軸を返す。
     * @return ローカルy軸のグローバル側のベクトル.
     */
    public Vector3D getLy(){
        return new Vector3D(r[3],r[4],r[5]);
    }

    /**
     * 回転後のローカルz軸を返す。
     * @return ローカルz軸のグローバル側のベクトル.
     */
    public Vector3D getLz(){
        return new Vector3D(r[6],r[7],r[8]);
    }

    /** ベクトル周りの回転をする.
     * @param axis_sin 回転軸でノルムが回転分のsine 成分を持つもの.
     *                  よってノルムは, 0以上, 1以下でなければならない.
     * @param co cosine 成分のスカラー -1以上, 1以下でなければならない.
     * 　　　　　　　　　またaxis_sin のノルムと対応しなければ、挙動は保障されない.
     */
    public void rotate(Vector3D axis_sin,double co){
        int i;
        double [] dt;
        Vector3D t;

        for(i=0;i<3;i++){
            t=new Vector3D(r[i*3],r[i*3+1],r[i*3+2]);
            dt=t.rotate_vector(axis_sin, co).toDouble();
            r[i*3]=dt[0];
            r[i*3+1]=dt[1];
            r[i*3+2]=dt[2];
        }
    }

    /**
     * 転値行列. 回転行列では逆回転行列になり, 逆行列と大体同義.
     * @return 転値行列.
     */
    public Matrix transpose(){
        Matrix ret=new Matrix();
        int i,j;
        for(i=0;i<3;i++){
            for(j=0;j<3;j++){
                ret.r[j*3+i]=r[i*3+j];
            }
        }
        return ret;
    }

    /**
     * 逆行列. 転値行列と大体同じだが、精度がこちらの方が高いので、
     * 繰り返し演算する部分ではこっちを使う.
     * ガウスの消去法による演算で, 計算時間は結構かかる.
     * @return 逆行列
     */
    public Matrix inverse(){
        Matrix ret=new Matrix();
        int i,j,k;
        double []temp=new double[9];
        System.arraycopy(r, 0, temp, 0, 9);

        for(i=0;i<3;i++){
            for(k=0;k<3;k++){
                ret.r[i+k*3]/=temp[i*4];
            }
            for(j=i+1;j<3;j++){
                temp[i+j*3]/=temp[i*4];
            }
//            temp[i*4]=1;
            for(j=i+1;j<3;j++){
                for(k=0;k<3;k++){
                    ret.r[j+k*3]-=temp[j+i*3]*ret.r[i+k*3];
                }
                for(k=i+1;k<3;k++){
                    temp[j+k*3]-=temp[j+i*3]*temp[i+k*3];
                }
//                temp[j+i*3]=0;
            }
        }
        
        for(k=0;k<3;k++){
            ret.r[0+k*3]-=temp[6]*ret.r[2+k*3];
        }
        for(k=0;k<3;k++){
            ret.r[1+k*3]-=temp[7]*ret.r[2+k*3];
        }
        for(k=0;k<3;k++){
            ret.r[0+k*3]-=temp[3]*ret.r[1+k*3];
        }

        return ret;
    }

    /**
     * 行列積
     * @param b 後ろから掛ける行列
     * @return 積の演算結果の行列.
     */
    public Matrix times(Matrix b){
        Matrix ret=new Matrix();
        int i,j;
        for(i=0;i<3;i++){
            for(j=0;j<3;j++){
                ret.r[i*3+j]=
                        r[j]*b.r[i*3]
                        +r[j+3]*b.r[i*3+1]
                        +r[j+6]*b.r[i*3+2];
            }
        }
        return ret;
    }

    /**
     * 対ベクトル積. 行列側が回転行列なので, 回転させる作業に相当する.
     * @param b 後ろから掛けるベクトル
     * @return 積の演算結果のベクトル.
     */
    public Vector3D times(Vector3D b)
    {
        Vector3D t;
        int i;
        double dr[]=new double[3];
        for(i=0;i<3;i++){
            t=new Vector3D(r[i],r[i+3],r[i+6]);
            dr[i]=t.times(b);
        }
        return new Vector3D(dr);
    }

    /**
     * 実数べき乗.
     * power(1)は元と同じ, power(0) は単位行列, power(-1) は逆行列になるが,
     * 計算精度・時間ともにそれぞれ劣るので, それらの計算は他のメソッドを使ったほうが
     * よい. 補間計算用. 内部で四元数対数をつかう.
     * @param p べき数
     * @return べき乗された行列
     */
    public Matrix power(double p){
        double []q=getQuotanions();
        Vector3D n=new Vector3D(q);
        double norm=n.norm();
        double angle=Math.atan2(norm,q[3]);
        if(norm>0){
            n=n.times(Math.sin(angle*p)/norm);
            q=n.toDouble();
        }else {
            q[0]=q[1]=q[2]=0;
        }
        return Matrix.rotationQ(q[0], q[1], q[2], Math.cos(angle*p));
    }

    /**
     * 文字列化. デバグ用.
     * @return 3行の文字列.
     */
    @Override
    public String toString(){
        return ""+r[0]+","+r[3]+","+r[6]+"\n"
                +r[1]+","+r[4]+","+r[7]+"\n"
                +r[2]+","+r[5]+","+r[8]+"\n";
    }
}