/*
 * The MIT License
 *
 * Copyright 2015 nazo.
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */
package jp.sourceforge.mmd.motion.model;

import java.util.ArrayList;
import java.util.Map;
import java.util.TreeMap;
import jp.sourceforge.mmd.motion.geo.Matrix;
import jp.sourceforge.mmd.motion.geo.Vector3D;
import jp.sfjp.mikutoga.bin.parser.MmdFormatException;
import jp.sfjp.mikutoga.bin.parser.ParseStage;
import jp.sfjp.mikutoga.pmd.parser.PmdBasicHandler;
import jp.sfjp.mikutoga.pmd.parser.PmdBoneHandler;

/**
 * PMD データを読み出すのに使用するハンドラ.
 * @author nazo
 */
public class PmdFileHander implements PmdBoneHandler,PmdBasicHandler{
    static protected Vector3D absoluteZ=Vector3D.unitZ;

    protected Model model;
    protected Bone bone;
    protected ArrayList<String> idToName;
    protected TreeMap<Integer,Integer> childToParent;
    protected TreeMap<Integer,Integer> parentToTail;

    public PmdFileHander(Model model){
        super();
        this.model=model;
    }
    @Override
    public void pmdParseStart(){
        bone=null;
    }

    @Override
    public void pmdHeaderInfo(byte[] header) throws MmdFormatException {
        if(header[0]!='P'||header[1]!='m'||header[2]!='d'){
            throw new MmdFormatException("Not PMD file.");
        }
    }

    @Override
    public void pmdModelInfo(String modelName, String description) {
        model.setName(modelName);
    }

    @Override
    public void loopStart(ParseStage stage, int loops) {
        if(stage==PmdBoneHandler.BONE_LIST){
            idToName=new ArrayList<String>();
            childToParent=new TreeMap<Integer, Integer>();
            parentToTail=new TreeMap<Integer, Integer>();
        }
    }

    @Override
    public void pmdBoneInfo(String boneName, byte boneKind) {
        bone=new Bone(model);
        bone.name=boneName;
        idToName.add(boneName);
        switch(boneKind){
            case 8:
                bone.limitRot=new Vector3D();
                break;
        }
    }

    @Override
    public void pmdBoneLink(int parentId, int tailId, int ikId) {
        if(parentId==0xffff){
            bone.parent=null;
        }else {
            childToParent.put(idToName.size()-1,parentId);
        }
        if(tailId!=0xffff){
            parentToTail.put(idToName.size()-1, tailId);
        }
    }

    @Override
    public void pmdBonePosition(float xPos, float yPos, float zPos) {
        bone.gv=new Vector3D(xPos,yPos,zPos);
        bone.mr=new Matrix();
        bone.ini_mr=new Matrix();
    }

    @Override
    public void pmdIKInfo(int boneId, int targetId, int depth, float weight) {
    }

    @Override
    public void pmdIKChainInfo(int childId) {
    }

    @Override
    public void pmdBoneGroupInfo(String groupName) {
    }

    @Override
    public void pmdGroupedBoneInfo(int boneId, int groupId) {
    }

    @Override
    public void loopNext(ParseStage stage) {
        if(bone!=null){
            model.put(bone);
        }
        bone=null;
    }

    /**
     * {@literal }
     * {@link PmdBoneHandler#BONE_LIST}のときだけ特殊なことをする.
     * 親への子ボーンの登録と, 矢印末端への軸を回転制限や腕や指のローカルx軸にするなど.
     * @param stage 
     */
    @Override
    public void loopEnd(ParseStage stage) {
        if(stage==PmdBoneHandler.BONE_LIST){
            for(Map.Entry<Integer,Integer> e:childToParent.entrySet()){
                Bone child=model.get(idToName.get(e.getKey()));
                child.parent=idToName.get(e.getValue());
                model.get(child.parent).addChild(child);
            }
            for(Map.Entry<Integer,Integer> e:parentToTail.entrySet()){
                String name=idToName.get(e.getKey());
                Bone parent=model.get(name);
                String tail=idToName.get(e.getValue());
                if(parent.limitRot!=null){
                    Vector3D arrow= model.get(tail).getPos().sub(parent.getPos());
                    double norm=arrow.norm();
                    if(norm>0){
                        parent.limitRot=arrow.divide(norm);
                    }
                }else if(name.matches("((左|右)(腕|ひじ|手首)|.*(親|人|中|薬|小)指.*)")){
                    Vector3D arrow= model.get(tail).getPos().sub(parent.getPos());
                    arrow=arrow.sub(absoluteZ.times(arrow.times(absoluteZ)));
                    double norm=arrow.norm();
                    if(norm>0){
                        parent.mr=new Matrix(arrow.divide(norm), absoluteZ);
                        parent.ini_mr=new Matrix(parent.mr);
                    }
                }
            }
        }
    }

    @Override
    public void pmdParseEnd(boolean hasMoreData) {
    }
    
}
