/*
 * The MIT License
 *
 * Copyright 2015 nazo.
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */
package jp.sourceforge.mmd.motion.swing;

import java.awt.Component;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.util.zip.GZIPInputStream;
import java.util.zip.ZipException;
import javax.swing.JFileChooser;
import javax.swing.JOptionPane;
import javax.swing.filechooser.FileNameExtensionFilter;
import jp.sourceforge.mmd.motion.model.Model;
import jp.sourceforge.mmd.motion.Motion;
import jp.sfjp.mikutoga.bin.parser.MmdFormatException;

/**
 * MMD モーションファイルを読み込む Swing Dialog を呼び出したりする.
 * 全ての static method 
 * @author nazo
 */
public class MmdDialog {
    static private File  modelDir;
    static private File motionDir;

    /**
     * System property の prefix (前置詞)
     */
    static public final String prefix="MMDMotion.";

    /**
     * 最後に選択されたファイル
     */
    static public File lastSelected;

    /**
     * System.getProperty の
     * MMDMotion.motionDir, MMDMotion.modelDir で指定した
     * ディレクトリをダイアログの初期ディレクトリにする.
     */
    static public void loadDefault(){
        String s=System.getProperty(prefix+"motionDir");
        if(s==null){
            s=".";
        }
        motionDir=new File(s);
        s=System.getProperty(prefix+"modelDir");
        if(s==null){
            s=".";
        }
        modelDir=new File(s);
    }

    /**
     * 現在のダイアログの初期ディレクトリにする
     * System.setProperty の
     * MMDMotion.motionDir, MMDMotion.modelDir に設定する.
     */
    static public void saveDefault(){
        try {
            System.setProperty(prefix+"modelDir", modelDir.getCanonicalPath());
            System.setProperty(prefix+"motionDir", motionDir.getCanonicalPath());
        } catch (IOException ex) {
            JOptionPane.showMessageDialog(null,"IOエラー:dir の正則名が得られません。\n"+ex.getLocalizedMessage());
        }
    }

    /**
     * モデルを読み込むダイアログを開く
     * @param parent ダイアログの呼び出し元
     * @return null じゃなかったら、読み込まれたモデル
     */
    static public Model loadModel(Component parent){
        Model ret=null;
        JFileChooser jfc=new JFileChooser(modelDir);
        jfc.setDialogTitle("モデルを開く");
        jfc.setFileFilter(new FileNameExtensionFilter(
                "MMD Model PMD, PMX or CSV", "pmd","pmx"));
        jfc.addChoosableFileFilter(new FileNameExtensionFilter(
                "MMD Model CSV","csv","csv.gz"));
        if(jfc.showOpenDialog(parent)!=JFileChooser.APPROVE_OPTION)
            return null;
        File f=jfc.getSelectedFile();
        lastSelected=f;
        modelDir=f.getParentFile();
        String name=f.getName().toLowerCase();
        FileInputStream fis=null;
        try {
            if(name.endsWith(".pmd")){
                ret=Model.fromPMD(fis=new FileInputStream(f));
                fis.close();
            } else if(name.endsWith(".pmx")){
                ret=Model.fromPMD(fis=new FileInputStream(f));
                fis.close();
            } else if(name.endsWith(".csv.gz")){
                fis=new FileInputStream(f);
                ret=Model.fromCSV(new GZIPInputStream(fis));
                fis.close();
            } else if(name.endsWith(".csv")){
                ret=Model.fromCSV(fis=new FileInputStream(f));
                fis.close();
            }
        } catch (FileNotFoundException ex){
            JOptionPane.showMessageDialog(parent,"ファイルが見つかりません\n"+ex.getLocalizedMessage());
        } catch (ZipException ex){
            JOptionPane.showMessageDialog(parent,"gzip圧縮がおかしいです\n"+ex.getLocalizedMessage());
        } catch (IOException ex) {
            JOptionPane.showMessageDialog(parent,"IOエラー:読み取りの失敗など\n"+ex.getLocalizedMessage());
        } catch (MmdFormatException ex) {
            try {
                if(fis!=null)fis.close();
            } catch (IOException ex1) {
                JOptionPane.showMessageDialog(parent,"IOエラー:読み取りの失敗など\n"+ex.getLocalizedMessage());
            }
            JOptionPane.showMessageDialog(parent,"MMDモデルじゃありません\n"+ex.getLocalizedMessage());
        }
        return ret;
    }

    /**
     * モーションを読み込むダイアログを開く
     * @param parent ダイアログの呼び出し元
     * @return null じゃなかったら、読み込まれたモーション
     */
    static public Motion loadMotion(Component parent){
        Motion ret=null;
        JFileChooser jfc=new JFileChooser(motionDir);
        jfc.setDialogTitle("モーションを開く");
        jfc.setFileFilter(new FileNameExtensionFilter("MMD Motion VMD", "vmd"));
        jfc.addChoosableFileFilter(new FileNameExtensionFilter("MMD Motion CSV","csv","csv.gz"));
        if(jfc.showOpenDialog(parent)!=JFileChooser.APPROVE_OPTION)
            return null;
        File f=jfc.getSelectedFile();
        lastSelected=f;
        motionDir=f.getParentFile();
        String name=f.getName().toLowerCase();
        FileInputStream fis=null;
        try {
            if(name.endsWith(".vmd")){
                ret=new Motion();
                ret.fromVMD(fis=new FileInputStream(f));
                fis.close();
            }else if(name.endsWith(".csv.gz")){
                ret=new Motion();
                fis=new FileInputStream(f);
                ret.fromCSV(new GZIPInputStream(fis));
                fis.close();
            }else if(name.endsWith(".csv")){
                ret=new Motion();
                ret.fromCSV(fis=new FileInputStream(f));
                fis.close();
            }
        } catch (FileNotFoundException ex){
            JOptionPane.showMessageDialog(parent,"ファイルが見つかりません\n"+ex.getLocalizedMessage());
        } catch (ZipException ex){
            JOptionPane.showMessageDialog(parent,"gzip圧縮がおかしいです\n"+ex.getLocalizedMessage());
        } catch (IOException ex) {
            JOptionPane.showMessageDialog(parent,"IOエラー:読み取りの失敗など\n"+ex.getLocalizedMessage());
        } catch (MmdFormatException ex) {
            try {
                if(fis!=null)fis.close();
            } catch (IOException ex1) {
                JOptionPane.showMessageDialog(parent,"IOエラー:読み取りの失敗など\n"+ex.getLocalizedMessage());
            }
            JOptionPane.showMessageDialog(parent,"MMDモーションじゃありません\n"+ex.getLocalizedMessage());
        }
        return ret;
    }
    
}
