/*
 * The MIT License
 *
 * Copyright 2015 nazo.
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */
package jp.sourceforge.mmd.motion;

import java.nio.ByteBuffer;
import java.nio.ByteOrder;
import java.util.ArrayList;
import java.util.Iterator;
import java.util.Map;
import java.util.TreeMap;
import jp.sourceforge.mmd.motion.geo.Vector3D;

/**
 * package limited なクラス,
 * ボーン(モーフも含む)ごとのモーションの管理.
 * @author nazo
 */
@SuppressWarnings("FieldMayBeFinal")
public class MoveOnBone <T extends Pose>{
    private String nameOfBone;
    private TreeMap<Integer,T> frametopose;

    /**
     * 
     * @param name ボーンの名前
     */
    public MoveOnBone(String name){
        nameOfBone=name;
        frametopose=new TreeMap<Integer,T>();
    }

    /**
     * ボーン名の hash (HashMap用)
     * @return ボーン名の hash
     */
    @Override
    public int hashCode(){
        return nameOfBone.hashCode();
    }

    /**
     * hashCode と対応する equals
     * @param obj 比較対象
     * @return 同じなら true
     */
    @Override
    public boolean equals(Object obj) {
        if (obj == null) {
            return false;
        }
        if (getClass() != obj.getClass()) {
            return false;
        }
        final MoveOnBone other = (MoveOnBone) obj;
        return (nameOfBone.equals(other.nameOfBone));
    }

    /**
     * Pose を登録する. フレーム番号はPose中からみる.
     * ポーズはクローンせずに, そのまま入る.
     * @param p 登録するポーズ
     */
    public void put(T p){
        frametopose.put(p.frame, p);
    }

    /**
     * i frame の Pose を読み取る.
     * @param i frame
     * @return i frame の Pose を読み取る. (reference)
     */
    public T get(int i){
        return frametopose.get(i);
    }

    /**
     * 補間モーション. キーポーズがある場合は、キーポーズが帰ってくる.
     * @param frame フレーム番.
     * @return 保管されたモーション.
     */
    public T getInterporate(int frame){
        T ret=frametopose.get(frame);
        if(ret!=null)
            return (T)ret.clone();
        T pre=null;
        T future=null;
        for(Map.Entry<Integer,T> e:frametopose.entrySet()){
            if(e.getKey()<frame){
                pre=e.getValue();
            }else {
                future=e.getValue();
                break;
            }
        }

        if(pre==null){// 初期値にする
            return (T)future.clone();
        }
        if(future==null){// 補間の必要なし
            return (T)pre.clone();
        }

        int df=future.frame-pre.frame;
        if(pre instanceof BonePose){
            BonePose bpre=(BonePose)pre;
            BonePose bfuture=(BonePose)future;
            if(bpre.equals(bfuture)){// 補間の必要なし
                return (T)bpre.clone();
            }
            BonePose bret=new BonePose();
            bret.frame=frame;
            bret.nameOfBone=new String(bpre.nameOfBone);

            double []rv=new double[3];
            double []prev=bpre.v.toDouble();
            double []futurev=bfuture.v.toDouble();
            double t=(double)(frame-bpre.frame)/df;
            double f=bezier(bfuture.interpX, t);
            rv[0]=futurev[0]*f+prev[0]*(1-f);
            f=bezier(bfuture.interpY, t);
            rv[1]=futurev[1]*f+prev[1]*(1-f);
            f=bezier(bfuture.interpZ, t);
            rv[2]=futurev[2]*f+prev[2]*(1-f);
            bret.v=new Vector3D(rv);

            f=bezier(bfuture.interpR, t);
            bret.mr=bfuture.mr.times(bpre.mr.inverse()).power(f)
                    .times(bpre.mr);
            return (T)bret;
        }else if(pre instanceof CameraPose){
            CameraPose cpre=(CameraPose)pre;
            CameraPose cfuture=(CameraPose)future;
            CameraPose cret=new CameraPose();
            cret.frame=frame;
            cret.nameOfBone=new String(cpre.nameOfBone);
            cret.perspective=cpre.perspective;
            
            double []rv=new double[3];
            double []prev=cpre.v.toDouble();
            double []futurev=cfuture.v.toDouble();
            double t=(double)(frame-cpre.frame)/df;
            double f=bezier(cfuture.interpX, t);
            rv[0]=futurev[0]*f+prev[0]*(1-f);
            f=bezier(cfuture.interpY, t);
            rv[1]=futurev[1]*f+prev[1]*(1-f);
            f=bezier(cfuture.interpZ, t);
            rv[2]=futurev[2]*f+prev[2]*(1-f);
            cret.v=new Vector3D(rv);

            f=bezier(cfuture.interpRange, t);
            cret.range=(float)(cfuture.range*f+cpre.range*(1-f));
            f=bezier(cfuture.interpProjection, t);
            cret.angle=(int)(cfuture.angle*f+cpre.angle*(1-f));

            return (T)cret;
        }else if(pre instanceof MorphPose){
            MorphPose mpre=(MorphPose)pre;
            MorphPose mfuture=(MorphPose)future;
            MorphPose mret=new MorphPose();
            
            mret.frame=frame;
            mret.nameOfBone=new String(mpre.nameOfBone);
            double f=(frame-mpre.frame)/df;
            mret.factor=(float)(mpre.factor*(1-f)+mfuture.factor*f);
            return (T)mret;
        }else if(pre instanceof LightPose){
            LightPose lpre=(LightPose)pre;
            LightPose lfuture=(LightPose)future;
            LightPose lret=new LightPose();

            lret.frame=frame;
            lret.nameOfBone=new String(lpre.nameOfBone);
            double f=(frame-lpre.frame)/df;
            lret.v=lpre.v.times(1-f).add(lfuture.v.times(f));

            lret.rgb[0]=(float)(lpre.rgb[0]*(1-f)+lfuture.rgb[0]*f);
            lret.rgb[1]=(float)(lpre.rgb[1]*(1-f)+lfuture.rgb[1]*f);
            lret.rgb[2]=(float)(lpre.rgb[2]*(1-f)+lfuture.rgb[2]*f);

            return (T)lret;
        }
        return pre;
    }

    /**
     * ベージェ補間関数.
     * 端点を(0,0),(interp[0],interp[1]),(interp[2],interp[3]),(1,1)とする
     * ベージェ補間曲線を作りy=f(t)の関数とする.
     * ベージェx軸を時間 t(定義域[0:1]), y軸(値域[0:1])を値として返す。
     * @param interp 補間パラメーター 4 byte.
     * @param t 入力の補間時間 [0:1]
     * @return 補間ファクター [0:1]
     */
    static private double bezier(byte [] interp,double t){
        double df=0.5;
        double f=0.5;
        double fi=1.0-f;
        double tt;

        while(df>1e-10){// 2分法で補間点を探す.
            tt=((interp[0]*fi +interp[2]*f)*fi*3.0/127 +f*f)*f;
            if(t==tt){
                break;
            } else if(t<tt){
                f-=df;
            } else {
                f+=df;
            }
            fi=1.0-f;
            df*=0.5;
        }
        return ((interp[1]*fi +interp[3]*f)*fi*3.0/127 +f*f)*f;
    }

    /**
     * i番 frame の Pose を削除。
     * @param i frame番
     */
    public void remove(int i){
        frametopose.remove(i);
    }

    /**
     * 登録ポーズ数を返す.
     * @return 登録ポーズ数
     */
    public int size(){
        return frametopose.size();
    }

    /**
     * Gerbage Collection ごみ掃除.
     * 3連続以上で同じものが続いている場合, 中を削除.
     * @return 削除されたフレームのリスト.
     */
    public Integer[] gc(){
        ArrayList<Integer> list=new ArrayList<Integer>();
        T pre=null,pre2=null;
        for(T p:frametopose.values()){
            if(pre==null){
                pre=p;continue;
            }
            if(pre2==null){
                pre2=pre;pre=p;continue;
            }
            if(pre2 instanceof MorphPose){
                MorphPose mp=(MorphPose)p;
                if(((MorphPose)pre2).factor==mp.factor 
                        && ((MorphPose)pre).factor==mp.factor){
                    list.add(pre.frame);
                }else {
                    pre2=pre;
                }
            } else {
                BonePose bp=(BonePose)p;
                if(bp.equals((BonePose)pre) && bp.equals((BonePose)pre2)){
                    list.add(pre.frame);
                } else {
                    pre2=pre;
                }
            }
            pre=p;
        }
        for(Integer i:list){
            frametopose.remove(i);
        }

        return list.toArray(new Integer[list.size()]);
    }
    
    /**
     * 登録されている Pose をクローンして返す.
     * @param poses 入れ物の配列. new T[this.size()] で用意してください.
     * @return Pose [] 登録されている全ポーズ.
     */
    @SuppressWarnings("unchecked")
    public T[]  toArray(T[] poses){
//      T [] poses = new T[this.size()];
        Iterator<T> ip=frametopose.values().iterator();
        T p;
        int i=0;
        while(ip.hasNext()){
            p=ip.next();
            poses[i]=(T)p.clone();
            i++;
        }
        return poses;
    }

    /**
     * CSV 文字列にする.
     * @return CSV 文字列(複数行)
     */
    public String toCSV(){
        StringBuilder ret=new StringBuilder(512);
        for(T p:frametopose.values()){
            ret.append(p.toCSV());
        }
        return ret.toString();
    }

    /**
     * VMD バイト列にする.
     * @return VMD バイト列
     */
    byte[] toVMD() {
        ArrayList<byte []> al=new ArrayList<byte []>();
        int length=0;
        byte [] a;
        for(T p:frametopose.values()){
            a=p.toVMD();
            al.add(a);
            length+=a.length;
        }
        ByteBuffer ret=ByteBuffer.allocate(length).order(ByteOrder.LITTLE_ENDIAN);
        for(byte [] line:al){
            ret.put(line);
        }
        return ret.array();
    }
}
