/*
 * The MIT License
 *
 * Copyright 2015 nazo.
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */
package jp.sourceforge.mmd.midiMotion;

import javax.sound.midi.MidiEvent;
import javax.sound.midi.ShortMessage;
import jp.sfjp.mikutoga.bin.parser.MmdFormatException;
import jp.sourceforge.mmd.ik_solver.IKSolver;
import jp.sourceforge.mmd.motion.model.Bone;
import jp.sourceforge.mmd.motion.BonePose;
import jp.sourceforge.mmd.motion.geo.Matrix;
import jp.sourceforge.mmd.motion.model.Model;
import jp.sourceforge.mmd.motion.Motion;
import jp.sourceforge.mmd.motion.Pose;
import jp.sourceforge.mmd.motion.geo.Vector3D;

/**
 * Standard MIDI File からモーションを作成する.
 * setPlayer を指定しなければ、鍵盤の動きしか生成しない.
 *
 * @author nazo
 */
public class KeyBoardMotionBuilder extends MotionBuilder {

    static private String[] rightHands = new String[]{"右人指３先", "右人指１", "右手首", "右ひじ", "右腕"};
    static private String[] leftHands = new String[]{"左人指３先", "左人指１", "左手首", "左ひじ", "左腕"};
    static private String[] keyName={"C%d","C%d#","D%d","D%d#","E%d","F%d","F%d#","G%d","G%d#","A%d","A%d#","B%d"};

    /** Midi2Vmd タイプ */
    static final public int PROTOCOL_MIDI2VMD = 0;

    /** voronoi 式ピアノ無人演奏マクロタイプ */
    static final public int PROTOCOL_VOLONOI = 1;

    private Motion player = null;
    private Model playerModel = null;
    private Model keyModel = null;
    private Vector3D offset = new Vector3D(0, 0, -3.2);

    protected String bendW = null;
    protected String programChange = null;
    protected String damper=null;

    private IKSolver iksR = null, iksL = null;
    private double keyAngle = -4.0;
    private int center = 61;
    private int protocol = PROTOCOL_MIDI2VMD;

    /**
     * コンストラクター. {@link MotionBuilder#motion} に "Keyboard"のモデル名を付ける.
     */
    public KeyBoardMotionBuilder() {
        super();
        motion.setModelName("Keyboard");
    }

    /**
     * コンストラクター, モーション付き.
     * @param init キーボードの初期モーション
     */
    public KeyBoardMotionBuilder(Motion init) {
        super(init);
    }

    /**
     * キーボードプレイヤーのモデルを指定.
     *
     * @param motion nullにすると初期状態になる。
     * @param model プレイヤーのモデル. nullにするとプレイヤーモーションは作らなくなる. 
     * @throws MmdFormatException キーボードプレイヤーに指に関するボーンが無いときに返す例外.
     */
    public void setPlayer(Motion motion, Model model) throws MmdFormatException {
        playerModel = model;

        if (playerModel == null) return;
        if (motion == null) {
            this.player = new Motion(model.getName());
        } else {
            this.player = motion;
        }

        try{
            iksR = new IKSolver();
            iksR.setBones(new Bone[]{playerModel.get(rightHands[2]),
                playerModel.get(rightHands[3]),
                playerModel.get(rightHands[4])});
            iksR.setLimits(0,IKSolver.RLIMIT_NOGlobalR);
            iksR.setLimits(1, IKSolver.RLIMIT_NOZ | IKSolver.RLIMIT_NOX);

            iksL = new IKSolver();
            iksL.setBones(new Bone[]{playerModel.get(leftHands[2]),
                playerModel.get(leftHands[3]),
                playerModel.get(leftHands[4])});
            iksL.setLimits(0,IKSolver.RLIMIT_NOGlobalR);
            iksL.setLimits(1, IKSolver.RLIMIT_NOZ | IKSolver.RLIMIT_NOX);
        }catch(NullPointerException ex){
            throw  new MmdFormatException("特定の指ボーンが見つかりません."+ex.getLocalizedMessage());
        }
    }

    /**
     * キーボードプレイヤーのモーションを取得.
     * @return プレイヤーのモーション.
     */
    public Motion getMotionPlayer() {
        return player;
    }

    /**
     * キーボードのモデルを指定.
     *
     * @param motion null にすると初期状態になる。
     * @param model null にできない。キーボードのモデル。
     */
    public void setKeyboard(Motion motion, Model model) {
        if (motion != null) {
            this.motion = motion;
        }
        keyModel = model;
        this.motion.setModelName(model.getName());
    }

    /**
     * 右手・左手の境目の note number を取得.
     *
     * @return 右手・左手の境目になるノートナンバー.
     */
    public int getCenter() {
        return center;
    }

    /**
     * 右手・左手の境目の note number を設定.
     *
     * @param aCenter 右手・左手の境目になるノートナンバー.
     */
    public void setCenter(int aCenter) {
        center = aCenter;
    }

    /**
     * 鍵盤押し込み角度を取得.
     * @return 押し込み角度.
     */
    public double getAngle() {
        return keyAngle;
    }

    /**
     * 鍵盤押し込み角度の設定.
     * @param angle 押し込み角度の設定.
     */
    public void setAngle(double angle) {
        keyAngle = angle;
    }

    /**
     * 自動演奏プロトコルの設定を取得.
     * @return プロトコル番号.
     */
    public int getProtocol() {
        return protocol;
    }

    /**
     * 自動演奏プロトコルを設定. デフォルトは{@link #PROTOCOL_MIDI2VMD}.
     * @param protocol プロトコル番号.
     */
    public void setProtocol(int protocol) {
        this.protocol = protocol;
    }

    /**
     * 鍵盤押し位置オフセットの取得.
     * @return the offset
     */
    public Vector3D getOffset() {
        return offset;
    }

    /**
     * 鍵盤押し位置オフセットの設定.
     * @param offset the offset to set
     */
    public void setOffset(Vector3D offset) {
        this.offset = offset;
    }

    public void setBendW(String s) {
        bendW = s;
    }

    public void setProgramChange(String s) {
        programChange = s;
    }

    /**
     * ダンプペダル設定. 
     * @param damper ペダルのボーン. {@code null} にするとペダルを踏まなくなる.
     */
    public void setDamper(String damper) {
        this.damper = damper;
    }

    /**
     * MIDIメッセージからモーションを作るのを開始する.
     *
     */
    @Override
    public void loadMessages() {
        MidiEvent[] keys=null;
        int i=0;
        while(keys==null){
            if(i>=16)return;
            keys= mm.getChannelMessage(i);
            i++;
        }
        if (keys == null)return;

        ShortMessage sm;
        int com;
        int note;
        int velo;
        int frame;
        int pitchBend = 0;
        int loaded = 0;
        BonePose p;
        Pose[] ps;

        if(playerModel!=null){
            playerModel.setPoses(player.get(0));
            playerModel.resetChanged();
        }
        if(keyModel!=null){
            keyModel.setPoses(motion.get(0));
            keyModel.resetChanged();
        }

        reportProgressStart();

        for (MidiEvent me : keys) {
            sm = (ShortMessage) me.getMessage();
            frame = ticsToframe(me.getTick());
            reportProgress(frame);
            if (loaded < frame) {
                if (keyModel != null) {
                    keyModel.setPoses(motion.getInterporate(frame));
                    keyModel.resetChanged();
                }
                if (playerModel != null) {
                    ps = player.getInterporate(frame);
                    playerModel.setPoses(ps);
                    playerModel.resetChanged();
                }
                loaded = frame;
            }

            if ((com = sm.getCommand()) == ShortMessage.NOTE_ON) {
                note = sm.getData1();
                velo = sm.getData2();
                keyOn(frame, note, velo);
            } else if (com == ShortMessage.NOTE_OFF) {
                note = sm.getData1();
                keyOff(frame,note);
            } else if (com == ShortMessage.PITCH_BEND && bendW != null) {
                int new_pitch = sm.getData1() + sm.getData2() * 128 - 8192;
                pitchBend=bendingWheel(frame, pitchBend, new_pitch);
            } else if(com == ShortMessage.PROGRAM_CHANGE && programChange != null){
                programChange(frame);
            } else if(com == ShortMessage.CONTROL_CHANGE && damper != null){
                if(sm.getData1()==64){// CC 64: hold1 (damper pedal)
                    damper(frame,sm.getData2());
                }
            }
        }
        reportProgressEnd();
    }

    private void keyOn(int frame, int note,int velo){
        BonePose p = new BonePose();
        p.nameOfBone = getNodeBone(note);
        p.frame = frame - 1;
        motion.put(p);
        p.frame++;
        p.mr = Matrix.rotation(keyAngle, 0, 0);
        motion.put(p);

        if(keyModel==null)return;
        Bone keyBone=keyModel.get(p.nameOfBone);
        if(keyBone==null)return;
        keyBone.setPose(p);

        if (playerModel == null) return;
        IKSolver iks;
        Vector3D finger;
        Pose [] ps;
        if (note > center) {
            iks = iksR;
            finger=playerModel.get(rightHands[2]).getPos()
                    .sub(playerModel.get(rightHands[0]).getPos());
        } else {
            iks = iksL;
            finger=playerModel.get(leftHands[2]).getPos()
                    .sub(playerModel.get(leftHands[0]).getPos());
        }
        playerModel.resetChanged();
        iks.solve(getVecter(note, 0.5D * velo / 100).add(finger));
        ps = playerModel.getChanged();
        player.putAll(ps, frame - 1);

        playerModel.resetChanged();
        iks.solve(getVecter(note,0).add(finger));
        ps = playerModel.getChanged();
        player.putAll(ps, frame);
    }

    private void keyOff(int frame,int note){
        BonePose p = new BonePose();
        p.nameOfBone = getNodeBone(note);
        p.frame = frame;
        motion.put(p);
        p.frame--;
        p.mr = Matrix.rotation(keyAngle, 0, 0);
        motion.put(p);

        if(keyModel==null)return;
        Bone keyBone=keyModel.get(p.nameOfBone);
        if(keyBone==null)return;
        keyBone.setPose(p);

        if (playerModel == null)return;
        IKSolver iks;
        Vector3D finger;
        Pose [] ps;
        if (note > center) {
            iks = iksR;
            finger=playerModel.get(rightHands[2]).getPos()
                    .sub(playerModel.get(rightHands[0]).getPos());
        } else {
            iks = iksL;
            finger=playerModel.get(leftHands[2]).getPos()
                    .sub(playerModel.get(leftHands[0]).getPos());
        }
        playerModel.resetChanged();
        iks.solve(getVecter(note,0).add(finger));
        ps = playerModel.getChanged();
        player.putAll(ps, frame - 1);

        playerModel.resetChanged();
        iks.solve(getVecter(note,0.3D).add(finger));
        ps = playerModel.getChanged();
        player.putAll(ps, frame);
    }

    /**
     * pitch bend Wheel をまわす.
     * @param frame 新しい入力のフレーム
     * @param old_pitch 古いピッチ値
     * @param new_pitch 新しいピッチ値
     * @return new_pitch そのものが帰る.
     */
    private int bendingWheel(int frame,int old_pitch,final int new_pitch){
        BonePose p = new BonePose();
        p.nameOfBone = bendW;
        p.frame = frame - 1;
        p.mr = Matrix.rotation(60D * old_pitch / 8192, 0, 0);
        motion.put(p);
        p.frame = frame;
        p.mr = Matrix.rotation(60D * new_pitch / 8192, 0, 0);
        motion.put(p);

        if (keyModel==null || playerModel == null)return new_pitch;

        Pose [] ps;
        Vector3D finger=playerModel.get(leftHands[2]).getPos()
                .sub(playerModel.get(leftHands[0]).getPos());
        Vector3D key_surface = keyModel.get(bendW).getPos();
        playerModel.resetChanged();
        iksL.solve(key_surface
                .add(new Vector3D(0.10, 0.35, 0)
                        .rotate_vector(new Vector3D(
                                Math.sin(-(double) old_pitch / 8192), 0, 0),
                                Math.cos((double) old_pitch / 8192)))
                .add(finger));
        ps = playerModel.getChanged();
        player.putAll(ps, frame - 1);

        playerModel.resetChanged();
        iksL.solve(key_surface
                .add(new Vector3D(0, 0.4, 0)
                        .rotate_vector(new Vector3D(
                                Math.sin(-(double)new_pitch / 8192), 0, 0),
                                Math.cos((double)new_pitch / 8192)))
                .add(finger));
        ps = playerModel.getChanged();
        player.putAll(ps, frame);

        return new_pitch;
    }

    private void programChange(int frame){
        if (keyModel == null || playerModel == null)return ;

        Pose [] ps;
        Vector3D finger =playerModel.get(rightHands[2]).getPos()
                .sub(playerModel.get(rightHands[0]).getPos());
        Bone key= keyModel.get(programChange);
        Vector3D key_surface = key.getPos();
        Matrix key_local_axis = key.getGMatrix();

        playerModel.resetChanged();
        iksR.solve(key_surface
                .add(key_local_axis.times(new Vector3D(0, 0.4, 0)))
                .add(finger));
        ps = playerModel.getChanged();
        player.putAll(ps, frame - 1);
        
        playerModel.resetChanged();
        iksR.solve(key_surface
                .add(finger));
        ps = playerModel.getChanged();
        player.putAll(ps, frame);
    }

    private void damper(int frame,int value){
        BonePose p = motion.getInterporateBone(frame,damper);
        p.nameOfBone = damper;
        p.frame = frame - 1;
        motion.put(p);
        p.frame = frame;
        p.mr = Matrix.rotation(((value<63)?0:((value<66)?-4D:-8D)), 0, 0);
        motion.put(p);

        if (keyModel==null || playerModel == null)return;

        Bone tsumasaki=playerModel.get("右つま先ＩＫ");
        Bone kakato=playerModel.get("右足ＩＫ");
        Bone pedal=keyModel.get(damper);

        Vector3D foot=kakato.getPos().sub(
                tsumasaki.getPos());
        Vector3D pedal_pre=pedal.getPos().add(foot);

        Vector3D aim=pedal.getLz().times(-1.0D)
                .add(pedal_pre);
        kakato.translationToG(aim);
        Pose kakato_p=kakato.getPose();
        kakato_p.frame=frame-1;
        player.put(kakato_p);

        pedal.setPose(p);
        aim=pedal.getLz().times(-1.0D)
                .add(pedal_pre);
        kakato.translationToG(aim);
        kakato_p=kakato.getPose();
        kakato_p.frame=frame;
        player.put(kakato_p);
    }

    /**
     * 鍵盤の押す位置を探す.
     * @param note note number 
     * @param y ローカルのy. (たたきつける前の位置)
     * @return 指先を置くグローバルベクトル 
     */
    private Vector3D getVecter(int note,double y) {
        Bone b = keyModel.get(getNodeBone(note));
        return b.getPos().add(b.getGMatrix().times(offset.add(new Vector3D(0,y,0))));
    }

    /**
     * ノートナンバーからkeyのボーン名を取得. {@link #setProtocol(int) }依存.
     * @param note
     * @return 
     */
    private String getNodeBone(int note) {
        if(protocol==PROTOCOL_MIDI2VMD){
            return String.format("nt%03d", note);
        }else {
            int octabe=note/12;
            return String.format(keyName[note%12],octabe-1);
        }
    }
}
