/*
 * The MIT License
 *
 * Copyright 2015 nazo.
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */
package jp.sourceforge.mmd.midiMotion.gui;

import java.io.File;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.FileOutputStream;
import java.io.IOException;
import javax.swing.JFileChooser;
import javax.swing.JOptionPane;
import javax.swing.filechooser.FileNameExtensionFilter;
import jp.sourceforge.mmd.midiMotion.DrumsMotionBuilder;
import jp.sourceforge.mmd.midiMotion.midi.MidiResolver;
import jp.sourceforge.mmd.midiMotion.ProgressLisner;
import jp.sourceforge.mmd.motion.Motion;
import jp.sourceforge.mmd.motion.swing.MmdDialog;
import jp.sfjp.mikutoga.bin.parser.MmdFormatException;
import jp.sourceforge.mmd.motion.swing.FileDropEvent;
import jp.sourceforge.mmd.motion.swing.FileDropListener;
import jp.sourceforge.mmd.motion.swing.FileDropTarget;

/**
 *
 * @author nazo
 */
public class DrumMotionPane extends MotionPane
implements ProgressLisner,FileDropListener{
    static final private String propertyHead="midiMotion.drum.";

    protected DrumsMotionBuilder dmb;
    protected Motion drumMotion;
    protected Motion playerMotion;
    protected File convertTable;

    /**
     * Creates new form DrumMotionPane
     */
    public DrumMotionPane() {
        super("drum");
        dmb=new DrumsMotionBuilder();

        String s=System.getProperty(propertyHead+"drumTemplate");
        drumMotion=null;
        if(s!=null){
            drumMotion=new Motion();
            FileInputStream fis;
            try {
                drumMotion.fromVMD(fis=new FileInputStream(s));
                fis.close();
            } catch (IOException ex) {
                log("IOエラー","drumテンプレート読み込み。\nFile:"+s+"\n"+ex);
            } catch (MmdFormatException ex) {
                log("MMDモーションじゃありません","drumテンプレート読み込み。\nFile:"+s+"\n"+ex);
            }
        }

        playerMotion=null;
        s=System.getProperty(propertyHead+"playerTemplate");
        if(s!=null){
            playerMotion=new Motion();
            FileInputStream fis;
            try {
                playerMotion.fromVMD(fis=new FileInputStream(s));
                fis.close();
            } catch (IOException ex) {
                log("IOエラー","drumテンプレート読み込み。\nFile:"+s+"\n"+ex);
            } catch (MmdFormatException ex) {
                log("MMDモーションじゃありません","drumテンプレート読み込み。\nFile:"+s+"\n"+ex);
            }
        }

        convertTable=null;
        s=System.getProperty(propertyHead+"convertTable");
        if(s!=null){
            convertTable=new File(s);
            if(!convertTable.exists()){
                log("ファイルが存在しない","drum convet table読み込み。\nFile:"+s);
                convertTable=null;
            }
        }
        initComponents();

        textDrum.setDropTarget(new FileDropTarget(this));
        textPlayer.setDropTarget(new FileDropTarget(this));
        textConvert.setDropTarget(new FileDropTarget(this));
    }

    @Override
    public void setMidiResolver(MidiResolver midiR) {
        super.setMidiResolver(midiR);
        buttonProgress.setEnabled(true);
    }

    @Override
    void setFPS(float fps) {
        dmb.setFPS(fps);
    }

    @Override
    public void progressStart(int end) {
        progressBar.setEnabled(true);
        progressBar.setMaximum(end);
        progressBar.setValue(0);
    }

    @Override
    public void progressMid(int s) {
        progressBar.setValue(s);
        progressBar.setString(String.valueOf(s));
    }

    @Override
    public void progressEnd() {
        progressBar.setEnabled(false);
        progressBar.setMaximum(0);
        buttonProgress.setEnabled(true);

        Motion drum=dmb.getMotion();
        Motion player=dmb.getMotionPlayer();

        String s=System.getProperty("midiMotion.outputDir");
        if(s==null){s=".";}
        JFileChooser jfc=new JFileChooser(s);
        jfc.setFileSelectionMode(JFileChooser.DIRECTORIES_ONLY);
        jfc.setToolTipText("フォルダーを選ぶ");
        if(jfc.showSaveDialog(this)!=JFileChooser.APPROVE_OPTION){
            return;
        }
        
        File dir=jfc.getSelectedFile();
        try {
            FileOutputStream fos=new FileOutputStream(new File(dir,"drums.vmd"));
            drum.toVMD(fos);
            fos.close();
            if(player!=null){
                fos=new FileOutputStream(new File(dir,"drumPlayer.vmd"));
                player.toVMD(fos);
                fos.close();
            }
            System.setProperty("midiMotion.outputDir",dir.getParent());
        }catch(FileNotFoundException ex){
            log("ファイルが見つかりません\n",ex.getLocalizedMessage());
        } catch (IOException ex) {
            log("IOエラー",ex.getLocalizedMessage());
        }
        dmb.resetMotion();
    }

    /**
     * This method is called from within the constructor to initialize the form.
     * WARNING: Do NOT modify this code. The content of this method is always
     * regenerated by the Form Editor.
     */
    @SuppressWarnings("unchecked")
    // <editor-fold defaultstate="collapsed" desc="Generated Code">//GEN-BEGIN:initComponents
    private void initComponents() {
        java.awt.GridBagConstraints gridBagConstraints;

        javax.swing.JLabel jLabel1 = new javax.swing.JLabel();
        javax.swing.JLabel jLabel2 = new javax.swing.JLabel();
        textDrum = new javax.swing.JTextField();
        textPlayer = new javax.swing.JTextField();
        buttonDrum = new javax.swing.JButton();
        buttonPlayer = new javax.swing.JButton();
        javax.swing.JLabel jLabel3 = new javax.swing.JLabel();
        textConvert = new javax.swing.JTextField();
        buttonConvert = new javax.swing.JButton();
        buttonProgress = new javax.swing.JButton();
        progressBar = new javax.swing.JProgressBar();

        setLayout(new java.awt.GridBagLayout());

        jLabel1.setText("drums モーションテンプレート");
        add(jLabel1, new java.awt.GridBagConstraints());

        jLabel2.setText("player モーションテンプレート");
        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridx = 0;
        gridBagConstraints.gridy = 1;
        add(jLabel2, gridBagConstraints);

        textDrum.setEditable(false);
        textDrum.setText((drumMotion==null)?"まだ":drumMotion.getModelName());
        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.fill = java.awt.GridBagConstraints.HORIZONTAL;
        gridBagConstraints.weightx = 0.1;
        add(textDrum, gridBagConstraints);

        textPlayer.setEditable(false);
        textPlayer.setText((playerMotion==null)?"まだ":playerMotion.getModelName());
        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridx = 1;
        gridBagConstraints.gridy = 1;
        gridBagConstraints.fill = java.awt.GridBagConstraints.HORIZONTAL;
        gridBagConstraints.weightx = 0.1;
        add(textPlayer, gridBagConstraints);

        buttonDrum.setText("open");
        buttonDrum.addActionListener(new java.awt.event.ActionListener() {
            public void actionPerformed(java.awt.event.ActionEvent evt) {
                buttonDrumActionPerformed(evt);
            }
        });
        add(buttonDrum, new java.awt.GridBagConstraints());

        buttonPlayer.setText("open");
        buttonPlayer.addActionListener(new java.awt.event.ActionListener() {
            public void actionPerformed(java.awt.event.ActionEvent evt) {
                buttonPlayerActionPerformed(evt);
            }
        });
        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridx = 2;
        gridBagConstraints.gridy = 1;
        add(buttonPlayer, gridBagConstraints);

        jLabel3.setText("換算表");
        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridx = 0;
        gridBagConstraints.gridy = 2;
        add(jLabel3, gridBagConstraints);

        textConvert.setEditable(false);
        textConvert.setText((convertTable==null)?"まだ":convertTable.getName());
        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridx = 1;
        gridBagConstraints.gridy = 2;
        gridBagConstraints.fill = java.awt.GridBagConstraints.HORIZONTAL;
        gridBagConstraints.weightx = 0.1;
        add(textConvert, gridBagConstraints);

        buttonConvert.setText("open");
        buttonConvert.addActionListener(new java.awt.event.ActionListener() {
            public void actionPerformed(java.awt.event.ActionEvent evt) {
                buttonConvertActionPerformed(evt);
            }
        });
        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridx = 2;
        gridBagConstraints.gridy = 2;
        add(buttonConvert, gridBagConstraints);

        buttonProgress.setText("do");
        buttonProgress.setEnabled(false);
        buttonProgress.addActionListener(new java.awt.event.ActionListener() {
            public void actionPerformed(java.awt.event.ActionEvent evt) {
                buttonProgressActionPerformed(evt);
            }
        });
        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridx = 0;
        gridBagConstraints.gridy = 3;
        add(buttonProgress, gridBagConstraints);

        progressBar.setString("0");
        progressBar.setStringPainted(true);
        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridx = 1;
        gridBagConstraints.gridy = 3;
        gridBagConstraints.gridwidth = 2;
        gridBagConstraints.fill = java.awt.GridBagConstraints.HORIZONTAL;
        gridBagConstraints.weightx = 0.1;
        add(progressBar, gridBagConstraints);
    }// </editor-fold>//GEN-END:initComponents

    private void buttonProgressActionPerformed(java.awt.event.ActionEvent evt) {//GEN-FIRST:event_buttonProgressActionPerformed
        dmb.setTemplate(drumMotion, playerMotion);
        FileInputStream fis;
        try {
            fis=new FileInputStream(convertTable);
            dmb.loadFrameMap(fis);
            fis.close();
        } catch (FileNotFoundException ex) {
            log("ファイルが見つかりません", "convert Table:"+convertTable.getAbsolutePath());
        } catch (IOException ex) {
            log("IOエラー", "convert Table 読み込み不良");
        }
        dmb.addProgressListener(this);
        buttonProgress.setEnabled(false);
        new Thread(new Runnable() {
            private ProgressLisner perent;
            public Runnable setP(ProgressLisner l){
                perent=l;
                return this;
            }
            @Override
            public void run() {
                dmb.setMidiResolver(midiR);
                dmb.loadMessages();
                dmb.removeProgressListener(perent);
            }
        }.setP(this)).start();
               
    }//GEN-LAST:event_buttonProgressActionPerformed

    private void buttonPlayerActionPerformed(java.awt.event.ActionEvent evt) {//GEN-FIRST:event_buttonPlayerActionPerformed
        Motion m=MmdDialog.loadMotion(this);
        if(m==null)return;
        openPlayerMotion(m);

        System.setProperty(propertyHead+"playerTemplate", MmdDialog.lastSelected.getAbsolutePath());
    }//GEN-LAST:event_buttonPlayerActionPerformed

    private void openPlayerMotion(Motion m){
        playerMotion=m;
        textPlayer.setText(playerMotion.getModelName());

    }
    
    private void buttonConvertActionPerformed(java.awt.event.ActionEvent evt) {//GEN-FIRST:event_buttonConvertActionPerformed
        JFileChooser jfc;
        String s=System.getProperty(propertyHead+"drumTemplate");
        if(s==null){
            jfc=new JFileChooser(new File("."));
        }else {
            jfc=new JFileChooser(new File(s).getParentFile());
        }
        jfc.setFileFilter(new FileNameExtensionFilter("CSV convert table", "csv"));
        if(jfc.showOpenDialog(this)!=JFileChooser.APPROVE_OPTION)
            return;
        openConvertTable(jfc.getSelectedFile());
    }//GEN-LAST:event_buttonConvertActionPerformed

    private void openConvertTable(File f)
    {
        convertTable=f;
        textConvert.setText(convertTable.getName());
        System.setProperty(propertyHead+"convertTable",convertTable.getAbsolutePath());
    }
    
    private void buttonDrumActionPerformed(java.awt.event.ActionEvent evt) {//GEN-FIRST:event_buttonDrumActionPerformed
        Motion m=MmdDialog.loadMotion(this);
        if(m==null)return;
        openDrumMotion(m);
        System.setProperty(propertyHead+"drumTemplate", MmdDialog.lastSelected.getAbsolutePath());        
    }//GEN-LAST:event_buttonDrumActionPerformed

    private void openDrumMotion(Motion m){
        drumMotion=m;
        textDrum.setText(drumMotion.getModelName());

    }

    // Variables declaration - do not modify//GEN-BEGIN:variables
    private javax.swing.JButton buttonConvert;
    private javax.swing.JButton buttonDrum;
    private javax.swing.JButton buttonPlayer;
    private javax.swing.JButton buttonProgress;
    private javax.swing.JProgressBar progressBar;
    private javax.swing.JTextField textConvert;
    private javax.swing.JTextField textDrum;
    private javax.swing.JTextField textPlayer;
    // End of variables declaration//GEN-END:variables

    @Override
    public void fileDrop(FileDropEvent e) {
        Object src=e.getSource();
        File file=e.getFiles()[0];
        String name=file.getName().toLowerCase();
        if(src==textDrum || src==textPlayer){
            Motion m=null;
            FileInputStream fis=null;
            try {
                fis=new FileInputStream(file);
                if(name.endsWith(".vmd")){
                    m=new Motion().fromVMD(fis);
                }else if(name.endsWith(".csv")){
                    m=new Motion().fromVMD(fis);
                }
                fis.close();
            }catch(FileNotFoundException ex){
                JOptionPane.showMessageDialog(this, "ファイルが見つかりません: "+ ex.getLocalizedMessage());
            }catch(IOException ex){
                JOptionPane.showMessageDialog(this, "ファイルが読み取れません: "+ ex.getLocalizedMessage());
            }catch(MmdFormatException ex){
                JOptionPane.showMessageDialog(this, "MMDのモーションファイルではありません: "+ ex.getLocalizedMessage());
                try{fis.close();}catch(IOException ex2){
                    JOptionPane.showMessageDialog(this, "ファイルが読み取れません: "+ ex2.getLocalizedMessage());
                }
            }
            if(m==null)return;
            if(src==textDrum){
                openDrumMotion(m);
                System.setProperty(propertyHead+"drumTemplate", file.getPath());        
            }else {
                openPlayerMotion(m);
                System.setProperty(propertyHead+"playerTemplate", file.getPath());
            }
        }else {// table
            if(!name.endsWith(".csv"))return;
            openConvertTable(file);
        }
    }
}
