/*
 * The MIT License
 *
 * Copyright 2015 nazo.
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */
package jp.sourceforge.mmd.midiMotion.gui;

import java.io.File;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.FileOutputStream;
import java.io.IOException;
import javax.swing.JFileChooser;
import javax.swing.JOptionPane;
import jp.sfjp.mikutoga.bin.parser.MmdFormatException;
import jp.sourceforge.mmd.midiMotion.KeyBoardMotionBuilder;
import jp.sourceforge.mmd.midiMotion.midi.MidiResolver;
import jp.sourceforge.mmd.midiMotion.ProgressLisner;
import jp.sourceforge.mmd.motion.model.Model;
import jp.sourceforge.mmd.motion.Motion;
import jp.sourceforge.mmd.motion.geo.Vector3D;
import jp.sourceforge.mmd.motion.swing.FileDropEvent;
import jp.sourceforge.mmd.motion.swing.FileDropListener;
import jp.sourceforge.mmd.motion.swing.FileDropTarget;
import jp.sourceforge.mmd.motion.swing.MmdDialog;

/**
 * キーボードモーションパネル.
 * @author nazo
 */
public class KeyboardMotionPane extends MotionPane
implements ProgressLisner, FileDropListener {
    private static final String propertyHead="midiMotion.keyboard.";

    protected Model keyModel;
    protected Motion keyMotion;
    protected Model playerModel;
    protected Motion playerMotion;
    protected KeyBoardMotionBuilder kbmb;
    protected Vector3D offset;

    /**
     * Creates new form KeyboardMotionPane
     */
    public KeyboardMotionPane() {
        super("keyboard");
        kbmb=new KeyBoardMotionBuilder();
        offset=kbmb.getOffset();
        initComponents();
        
        keysModelText.setDropTarget(new FileDropTarget(this));
        keysMotionText.setDropTarget(new FileDropTarget(this));
        playerModelText.setDropTarget(new FileDropTarget(this));
        playerMotionText.setDropTarget(new FileDropTarget(this));
    }

    @Override
    public void setMidiResolver(MidiResolver midiR) {
        super.setMidiResolver(midiR);
        buttonProgress.setEnabled(true);
    }

    
    /**
     * This method is called from within the constructor to initialize the form.
     * WARNING: Do NOT modify this code. The content of this method is always
     * regenerated by the Form Editor.
     */
    @SuppressWarnings("unchecked")
    // <editor-fold defaultstate="collapsed" desc="Generated Code">//GEN-BEGIN:initComponents
    private void initComponents() {
        java.awt.GridBagConstraints gridBagConstraints;

        radioGroupProto = new javax.swing.ButtonGroup();
        buttonProgress = new javax.swing.JButton();
        progressBar = new javax.swing.JProgressBar();
        keysModelText = new javax.swing.JTextField();
        keyModelOpenButton = new javax.swing.JButton();
        playerModelText = new javax.swing.JTextField();
        playerModelOpenButton = new javax.swing.JButton();
        keysMotionText = new javax.swing.JTextField();
        keyMotionOpenButton = new javax.swing.JButton();
        playerMotionText = new javax.swing.JTextField();
        playerMotionOpenButton = new javax.swing.JButton();
        textOffsetX = new javax.swing.JTextField();
        textOffsetY = new javax.swing.JTextField();
        textOffsetZ = new javax.swing.JTextField();
        textAngle = new javax.swing.JTextField();
        textCenter = new javax.swing.JTextField();
        radioMidi2Vmd = new javax.swing.JRadioButton();
        radioVornoi = new javax.swing.JRadioButton();
        comboWheel = new jp.sourceforge.mmd.motion.swing.BoneComboBox();
        comboProgram = new jp.sourceforge.mmd.motion.swing.BoneComboBox();
        comboDamper = new jp.sourceforge.mmd.motion.swing.BoneComboBox();
        javax.swing.JLabel jLabel1 = new javax.swing.JLabel();
        javax.swing.JLabel jLabel2 = new javax.swing.JLabel();
        javax.swing.JLabel jLabel3 = new javax.swing.JLabel();
        javax.swing.JLabel jLabel4 = new javax.swing.JLabel();
        javax.swing.JLabel jLabel5 = new javax.swing.JLabel();
        javax.swing.JLabel jLabel6 = new javax.swing.JLabel();
        javax.swing.JLabel jLabel7 = new javax.swing.JLabel();
        javax.swing.JLabel jLabel8 = new javax.swing.JLabel();
        javax.swing.JLabel jLabel9 = new javax.swing.JLabel();
        javax.swing.JLabel jLabel10 = new javax.swing.JLabel();

        setLayout(new java.awt.GridBagLayout());

        buttonProgress.setText("do");
        buttonProgress.setEnabled(false);
        buttonProgress.addActionListener(new java.awt.event.ActionListener() {
            public void actionPerformed(java.awt.event.ActionEvent evt) {
                buttonProgressActionPerformed(evt);
            }
        });
        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridx = 1;
        gridBagConstraints.gridy = 6;
        add(buttonProgress, gridBagConstraints);

        progressBar.setEnabled(false);
        progressBar.setString("0");
        progressBar.setStringPainted(true);
        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridx = 2;
        gridBagConstraints.gridy = 6;
        gridBagConstraints.gridwidth = 4;
        gridBagConstraints.fill = java.awt.GridBagConstraints.HORIZONTAL;
        add(progressBar, gridBagConstraints);

        keysModelText.setEditable(false);
        keysModelText.setText("まだ");
        keysModelText.setToolTipText("キーボードのモデル.PMD,PMXをドロップ可能.");
        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridx = 1;
        gridBagConstraints.gridy = 0;
        gridBagConstraints.fill = java.awt.GridBagConstraints.HORIZONTAL;
        gridBagConstraints.weightx = 0.1;
        add(keysModelText, gridBagConstraints);

        keyModelOpenButton.setText("open");
        keyModelOpenButton.addActionListener(new java.awt.event.ActionListener() {
            public void actionPerformed(java.awt.event.ActionEvent evt) {
                keyModelOpenButtonActionPerformed(evt);
            }
        });
        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridx = 2;
        gridBagConstraints.gridy = 0;
        gridBagConstraints.anchor = java.awt.GridBagConstraints.LINE_START;
        add(keyModelOpenButton, gridBagConstraints);

        playerModelText.setEditable(false);
        playerModelText.setText("まだ");
        playerModelText.setToolTipText("プレイヤーのモデル.PMD,PMXをドロップ可能.");
        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridx = 4;
        gridBagConstraints.gridy = 0;
        gridBagConstraints.fill = java.awt.GridBagConstraints.HORIZONTAL;
        gridBagConstraints.weightx = 0.1;
        add(playerModelText, gridBagConstraints);

        playerModelOpenButton.setText("open");
        playerModelOpenButton.addActionListener(new java.awt.event.ActionListener() {
            public void actionPerformed(java.awt.event.ActionEvent evt) {
                playerModelOpenButtonActionPerformed(evt);
            }
        });
        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridx = 5;
        gridBagConstraints.gridy = 0;
        gridBagConstraints.anchor = java.awt.GridBagConstraints.LINE_START;
        add(playerModelOpenButton, gridBagConstraints);

        keysMotionText.setEditable(false);
        keysMotionText.setText("まだ");
        keysMotionText.setToolTipText("キーボードのモーション.VMD,CSVをドロップ可能.");
        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridx = 1;
        gridBagConstraints.gridy = 1;
        gridBagConstraints.fill = java.awt.GridBagConstraints.HORIZONTAL;
        gridBagConstraints.weightx = 0.1;
        add(keysMotionText, gridBagConstraints);

        keyMotionOpenButton.setText("open");
        keyMotionOpenButton.addActionListener(new java.awt.event.ActionListener() {
            public void actionPerformed(java.awt.event.ActionEvent evt) {
                keyMotionOpenButtonActionPerformed(evt);
            }
        });
        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridx = 2;
        gridBagConstraints.gridy = 1;
        gridBagConstraints.anchor = java.awt.GridBagConstraints.LINE_START;
        add(keyMotionOpenButton, gridBagConstraints);

        playerMotionText.setEditable(false);
        playerMotionText.setText("まだ");
        playerMotionText.setToolTipText("プレイヤーのモーション.VMD,CSVをドロップ可能.");
        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridx = 4;
        gridBagConstraints.gridy = 1;
        gridBagConstraints.fill = java.awt.GridBagConstraints.HORIZONTAL;
        gridBagConstraints.weightx = 0.1;
        add(playerMotionText, gridBagConstraints);

        playerMotionOpenButton.setText("open");
        playerMotionOpenButton.addActionListener(new java.awt.event.ActionListener() {
            public void actionPerformed(java.awt.event.ActionEvent evt) {
                playerMotionOpenButtonActionPerformed(evt);
            }
        });
        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridx = 5;
        gridBagConstraints.gridy = 1;
        gridBagConstraints.anchor = java.awt.GridBagConstraints.LINE_START;
        add(playerMotionOpenButton, gridBagConstraints);

        textOffsetX.setText(String.valueOf(offset.toDouble()[0]));
        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridx = 1;
        gridBagConstraints.gridy = 2;
        gridBagConstraints.fill = java.awt.GridBagConstraints.HORIZONTAL;
        gridBagConstraints.weightx = 0.1;
        add(textOffsetX, gridBagConstraints);

        textOffsetY.setText(String.valueOf(offset.toDouble()[1]));
        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridx = 2;
        gridBagConstraints.gridy = 2;
        gridBagConstraints.fill = java.awt.GridBagConstraints.HORIZONTAL;
        add(textOffsetY, gridBagConstraints);

        textOffsetZ.setText(String.valueOf(offset.toDouble()[2]));
        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridx = 3;
        gridBagConstraints.gridy = 2;
        gridBagConstraints.fill = java.awt.GridBagConstraints.HORIZONTAL;
        add(textOffsetZ, gridBagConstraints);

        textAngle.setText("-4.0");
        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridx = 5;
        gridBagConstraints.gridy = 2;
        gridBagConstraints.fill = java.awt.GridBagConstraints.HORIZONTAL;
        add(textAngle, gridBagConstraints);

        textCenter.setText("60");
        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridx = 1;
        gridBagConstraints.gridy = 3;
        gridBagConstraints.fill = java.awt.GridBagConstraints.HORIZONTAL;
        gridBagConstraints.weightx = 0.1;
        add(textCenter, gridBagConstraints);

        radioGroupProto.add(radioMidi2Vmd);
        radioMidi2Vmd.setSelected(true);
        radioMidi2Vmd.setText("Midi2Vmd");
        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridx = 2;
        gridBagConstraints.gridy = 3;
        gridBagConstraints.anchor = java.awt.GridBagConstraints.LINE_START;
        add(radioMidi2Vmd, gridBagConstraints);

        radioGroupProto.add(radioVornoi);
        radioVornoi.setText("voronoi式");
        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridx = 3;
        gridBagConstraints.gridy = 3;
        gridBagConstraints.anchor = java.awt.GridBagConstraints.LINE_START;
        add(radioVornoi, gridBagConstraints);
        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridx = 1;
        gridBagConstraints.gridy = 4;
        gridBagConstraints.fill = java.awt.GridBagConstraints.HORIZONTAL;
        add(comboWheel, gridBagConstraints);
        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridx = 1;
        gridBagConstraints.gridy = 5;
        gridBagConstraints.fill = java.awt.GridBagConstraints.HORIZONTAL;
        add(comboProgram, gridBagConstraints);
        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridx = 3;
        gridBagConstraints.gridy = 4;
        gridBagConstraints.fill = java.awt.GridBagConstraints.HORIZONTAL;
        add(comboDamper, gridBagConstraints);

        jLabel1.setText("keys モデル");
        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridx = 0;
        gridBagConstraints.gridy = 0;
        gridBagConstraints.anchor = java.awt.GridBagConstraints.LINE_END;
        add(jLabel1, gridBagConstraints);

        jLabel2.setText("keys モーション");
        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridx = 0;
        gridBagConstraints.gridy = 1;
        gridBagConstraints.anchor = java.awt.GridBagConstraints.LINE_END;
        add(jLabel2, gridBagConstraints);

        jLabel3.setText("player モデル");
        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridx = 3;
        gridBagConstraints.gridy = 0;
        gridBagConstraints.anchor = java.awt.GridBagConstraints.LINE_END;
        add(jLabel3, gridBagConstraints);

        jLabel4.setText("player モーション");
        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridx = 3;
        gridBagConstraints.gridy = 1;
        gridBagConstraints.anchor = java.awt.GridBagConstraints.LINE_END;
        add(jLabel4, gridBagConstraints);

        jLabel5.setText("センター");
        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridx = 0;
        gridBagConstraints.gridy = 3;
        gridBagConstraints.anchor = java.awt.GridBagConstraints.LINE_END;
        add(jLabel5, gridBagConstraints);

        jLabel6.setText("ホイール");
        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridx = 0;
        gridBagConstraints.gridy = 4;
        gridBagConstraints.anchor = java.awt.GridBagConstraints.LINE_END;
        add(jLabel6, gridBagConstraints);

        jLabel7.setText("Program Change");
        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridx = 0;
        gridBagConstraints.gridy = 5;
        gridBagConstraints.anchor = java.awt.GridBagConstraints.LINE_END;
        add(jLabel7, gridBagConstraints);

        jLabel8.setText("オフセット");
        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridx = 0;
        gridBagConstraints.gridy = 2;
        gridBagConstraints.anchor = java.awt.GridBagConstraints.LINE_END;
        add(jLabel8, gridBagConstraints);

        jLabel9.setText("x 角度");
        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridx = 4;
        gridBagConstraints.gridy = 2;
        gridBagConstraints.anchor = java.awt.GridBagConstraints.LINE_END;
        add(jLabel9, gridBagConstraints);

        jLabel10.setText("ダンパーペダル");
        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridx = 2;
        gridBagConstraints.gridy = 4;
        gridBagConstraints.anchor = java.awt.GridBagConstraints.LINE_END;
        add(jLabel10, gridBagConstraints);
    }// </editor-fold>//GEN-END:initComponents

    private void keyModelOpenButtonActionPerformed(java.awt.event.ActionEvent evt) {//GEN-FIRST:event_keyModelOpenButtonActionPerformed
        Model m=MmdDialog.loadModel(this);
        if(m==null)return;
        openKeysModel(m);
    }//GEN-LAST:event_keyModelOpenButtonActionPerformed

    private void openKeysModel(Model m){
        keyModel=m;
        keysModelText.setText(keyModel.getName());
        comboWheel.setMMDModel(m);
        comboProgram.setMMDModel(m);
        comboDamper.setMMDModel(m);

        String proper=System.getProperty(propertyHead+keyModel.getName()+".wheel");
        if(proper!=null){
            comboWheel.setSelect(proper);
        }

        proper=System.getProperty(propertyHead+keyModel.getName()+".programChange");
        if(proper!=null){
            comboProgram.setSelect(proper);
        }

        proper=System.getProperty(propertyHead+keyModel.getName()+".damper");
        if(proper!=null){
            comboDamper.setSelect(proper);
        }

        proper=System.getProperty(propertyHead+keyModel.getName()+".center");
        if(proper!=null){
            textCenter.setText(proper);
        }

        proper=System.getProperty(propertyHead+keyModel.getName()+".angle");
        if(proper!=null){
            textAngle.setText(proper);
        }

        proper=System.getProperty(propertyHead+keyModel.getName()+".offset");
        if(proper!=null){
            String [] sp=proper.split(",");
            textOffsetX.setText(sp[0]);
            textOffsetY.setText(sp[1]);
            textOffsetZ.setText(sp[2]);
        }

        proper=System.getProperty(propertyHead+keyModel.getName()+".protocol");
        if(proper!=null){
            if("vornoi".equalsIgnoreCase(proper)){
                radioVornoi.setSelected(true);
            } else {
                radioMidi2Vmd.setSelected(true);
            }
        }

    }
    private void playerModelOpenButtonActionPerformed(java.awt.event.ActionEvent evt) {//GEN-FIRST:event_playerModelOpenButtonActionPerformed
        Model m=MmdDialog.loadModel(this);
        if(m==null)return;
        openPlayerModel(m);
    }//GEN-LAST:event_playerModelOpenButtonActionPerformed

    private void openPlayerModel(Model m){
        playerModel=m;
        playerModelText.setText(playerModel.getName());
    }

    private void keyMotionOpenButtonActionPerformed(java.awt.event.ActionEvent evt) {//GEN-FIRST:event_keyMotionOpenButtonActionPerformed
        Motion m=MmdDialog.loadMotion(this);
        if(m==null)return;
        openKeysMotion(m);
    }//GEN-LAST:event_keyMotionOpenButtonActionPerformed

    private void openKeysMotion(Motion m){
        keyMotion=m;
        keysMotionText.setText(keyMotion.getModelName());
    }

    private void playerMotionOpenButtonActionPerformed(java.awt.event.ActionEvent evt) {//GEN-FIRST:event_playerMotionOpenButtonActionPerformed
        Motion m=MmdDialog.loadMotion(this);
        if(m==null)return;
        openPlayerMotion(m);
    }//GEN-LAST:event_playerMotionOpenButtonActionPerformed

    private void openPlayerMotion(Motion m){
        playerMotion=m;
        playerMotionText.setText(playerMotion.getModelName());        
    }

    private void buttonProgressActionPerformed(java.awt.event.ActionEvent evt) {//GEN-FIRST:event_buttonProgressActionPerformed
        if(playerMotion!=null && playerModel!=null){
            try{
                kbmb.setPlayer(playerMotion, playerModel);
            }catch (MmdFormatException ex){
                log("プレイヤーモデルとして使えません",
                        "プレイヤーモデルとしてボーンが不足しています。"+ex);
                return;
            }
        }
        if(keyModel!=null){
            kbmb.setKeyboard(keyMotion, keyModel);

            try {
                kbmb.setCenter(Integer.parseInt(textCenter.getText()));
            }catch(NumberFormatException e){
                log("数字じゃありません","centerが数字じゃありません。");
                return;
            }
            System.setProperty(propertyHead+keyModel.getName()+".center",textCenter.getText());

            try{
                offset=new Vector3D(
                        Double.parseDouble(textOffsetX.getText()),
                        Double.parseDouble(textOffsetY.getText()),
                        Double.parseDouble(textOffsetZ.getText())
                );
                kbmb.setOffset(offset);
            }catch(NumberFormatException e){
                log("数字じゃありません","offset が数字じゃありません。");
                return;
            }
            System.setProperty(propertyHead+keyModel.getName()+".offset",offset.toString());

            try {
                kbmb.setAngle(Double.parseDouble(textAngle.getText()));
            } catch(NumberFormatException e){
                log("数字じゃありません","offset が数字じゃありません。");
                return;
            }
            System.setProperty(propertyHead+keyModel.getName()+".angle",textAngle.getText());

            kbmb.setProtocol(radioMidi2Vmd.isSelected()?
                    KeyBoardMotionBuilder.PROTOCOL_MIDI2VMD:
                    KeyBoardMotionBuilder.PROTOCOL_VOLONOI);
            System.setProperty(propertyHead+keyModel.getName()+".protocol",(radioMidi2Vmd.isSelected()?"Midi2Vmd":"vornoi"));

            String s=(String)comboWheel.getSelectedItem();
            if("(なし)".equalsIgnoreCase(s)){
                kbmb.setBendW(null);
                System.setProperty(propertyHead+keyModel.getName()+".wheel","(なし)");
            }else {
                kbmb.setBendW(s);
                System.setProperty(propertyHead+keyModel.getName()+".wheel",s);
            }

            s=(String)comboProgram.getSelectedItem();
            if("(なし)".equalsIgnoreCase(s)){
                kbmb.setProgramChange(null);
                System.setProperty(propertyHead+keyModel.getName()+".programChange","(なし)");
            }else {
                kbmb.setProgramChange(s);
                System.setProperty(propertyHead+keyModel.getName()+".programChange",s);
            }

            s=(String)comboDamper.getSelectedItem();
            if("(なし)".equalsIgnoreCase(s)){
                kbmb.setDamper(null);
                System.setProperty(propertyHead+keyModel.getName()+".damper","(なし)");
            }else {
                kbmb.setDamper(s);
                System.setProperty(propertyHead+keyModel.getName()+".damper",s);
            }
        }
        kbmb.addProgressListener(this);
        buttonProgress.setEnabled(false);
        kbmb.setMidiResolver(midiR);
        new Thread(new Runnable() {
            private ProgressLisner perent;
            public Runnable setP(ProgressLisner l){
                perent=l;
                return this;
            }
            @Override
            public void run() {
                kbmb.loadMessages();
                kbmb.removeProgressListener(perent);
            }
        }.setP(this)).start();
    }//GEN-LAST:event_buttonProgressActionPerformed

    @Override
    public void progressStart(int end) {
        progressBar.setEnabled(true);
        progressBar.setMaximum(end);
        progressBar.setValue(0);
    }

    @Override
    public void progressMid(int s) {
        progressBar.setValue(s);
        progressBar.setString(String.valueOf(s));
    }

    @Override
    public void progressEnd() {
        progressBar.setEnabled(false);
        progressBar.setMaximum(0);
        buttonProgress.setEnabled(true);
        // output
        Motion keyboardOutputMotion=kbmb.getMotion();
        Motion playerOutputMotion=kbmb.getMotionPlayer();

        String s=System.getProperty("midiMotion.outputDir");
        if(s==null){s=".";}
        JFileChooser jfc=new JFileChooser(s);
        jfc.setFileSelectionMode(JFileChooser.DIRECTORIES_ONLY);
        jfc.setToolTipText("フォルダーを選ぶ");
        if(jfc.showSaveDialog(this)!=JFileChooser.APPROVE_OPTION){
            return;
        }

        File dir=jfc.getSelectedFile();
        try {
            FileOutputStream fos=new FileOutputStream(new File(dir,"keyboard.vmd"));
            keyboardOutputMotion.toVMD(fos);
            fos.close();
            if(playerOutputMotion!=null){
                fos=new FileOutputStream(new File(dir,"keysPlayer.vmd"));
                playerOutputMotion.toVMD(fos);
                fos.close();
            }
            System.setProperty("midiMotion.outputDir",dir.getParent());
        }catch(FileNotFoundException ex){
            JOptionPane.showMessageDialog(this, "ファイルが見つかりません\n"+ex);
        } catch (IOException ex) {
            JOptionPane.showMessageDialog(this, "IOエラー\n"+ex);
        }
        kbmb.resetMotion();
    }

    @Override
    void setFPS(float fps) {
        kbmb.setFPS(fps);
    }

    @Override
    public void fileDrop(FileDropEvent e) {
        Object src=e.getSource();
        File file=e.getFiles()[0];
        String name=file.getName().toLowerCase();
        if(src==keysModelText || src==playerModelText){
            Model m=null;
            FileInputStream fis=null;
            try {
                fis=new FileInputStream(file);
                if(name.endsWith(".pmx")){
                    m=Model.fromPMX(fis);
                }else if(name.endsWith(".pmd")){
                    m=Model.fromPMD(fis);
                }else if(name.endsWith(".csv")){
                    m=Model.fromCSV(fis);
                }
                fis.close();
            }catch(FileNotFoundException ex){
                JOptionPane.showMessageDialog(this, "ファイルが見つかりません: "+ ex.getLocalizedMessage());
            }catch(IOException ex){
                JOptionPane.showMessageDialog(this, "ファイルが読み取れません: "+ ex.getLocalizedMessage());
            }catch(MmdFormatException ex){
                JOptionPane.showMessageDialog(this, "MMDのモデルファイルではありません: "+ ex.getLocalizedMessage());
                try{fis.close();}catch(IOException ex2){
                    JOptionPane.showMessageDialog(this, "ファイルが読み取れません: "+ ex2.getLocalizedMessage());
                }
            }
            if(m==null)return;
            if(src==keysModelText){
                openKeysModel(m);
            }else {
                openPlayerModel(m);
            }
        } else if(src==keysMotionText || src==playerMotionText){
            Motion m=null;
            FileInputStream fis=null;
            try {
                fis=new FileInputStream(file);
                if(name.endsWith(".vmd")){
                    m=new Motion().fromVMD(fis);
                }else if(name.endsWith(".csv")){
                    m=new Motion().fromVMD(fis);
                }
                fis.close();
            }catch(FileNotFoundException ex){
                JOptionPane.showMessageDialog(this, "ファイルが見つかりません: "+ ex.getLocalizedMessage());
            }catch(IOException ex){
                JOptionPane.showMessageDialog(this, "ファイルが読み取れません: "+ ex.getLocalizedMessage());
            }catch(MmdFormatException ex){
                JOptionPane.showMessageDialog(this, "MMDのモーションファイルではありません: "+ ex.getLocalizedMessage());
                try{fis.close();}catch(IOException ex2){
                    JOptionPane.showMessageDialog(this, "ファイルが読み取れません: "+ ex2.getLocalizedMessage());
                }
            }
            if(m==null)return;
            if(src==keysMotionText){
                openKeysMotion(m);
            }else {
                openPlayerMotion(m);
            }
        }
    }

    // Variables declaration - do not modify//GEN-BEGIN:variables
    private javax.swing.JButton buttonProgress;
    private jp.sourceforge.mmd.motion.swing.BoneComboBox comboDamper;
    private jp.sourceforge.mmd.motion.swing.BoneComboBox comboProgram;
    private jp.sourceforge.mmd.motion.swing.BoneComboBox comboWheel;
    private javax.swing.JButton keyModelOpenButton;
    private javax.swing.JButton keyMotionOpenButton;
    private javax.swing.JTextField keysModelText;
    private javax.swing.JTextField keysMotionText;
    private javax.swing.JButton playerModelOpenButton;
    private javax.swing.JTextField playerModelText;
    private javax.swing.JButton playerMotionOpenButton;
    private javax.swing.JTextField playerMotionText;
    private javax.swing.JProgressBar progressBar;
    private javax.swing.ButtonGroup radioGroupProto;
    private javax.swing.JRadioButton radioMidi2Vmd;
    private javax.swing.JRadioButton radioVornoi;
    private javax.swing.JTextField textAngle;
    private javax.swing.JTextField textCenter;
    private javax.swing.JTextField textOffsetX;
    private javax.swing.JTextField textOffsetY;
    private javax.swing.JTextField textOffsetZ;
    // End of variables declaration//GEN-END:variables

}
