/*
 * The MIT License
 *
 * Copyright 2015 nazo.
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */
package jp.sourceforge.mmd.midiMotion.midi;
import java.io.File;
import java.io.IOException;
import java.io.OutputStream;
import java.util.TreeSet;
import java.util.logging.Level;
import java.util.logging.Logger;
import javax.sound.midi.*;

/**
 * Standard MIDI ファイルを整理するためのクラス.
 * チャンネルごとにメッセージを分ける.
 * @author nazo
 */
public class MidiResolver {
    private String name;
    private Sequence midiSeq;
    private Track [] tracks;
    /** ticks for quota note */
    private int reso;
    /** mil sec for quota note */
    private int tempo=500000;
    private TreeSet<MidiEvent> [] channelList;
//    private long end=0;

    /** 空のMidiResolver を作る.
     * sequence も同時に作られる. 基本的に SMF Format 0 になる.
     * @param beats 終点の長さ.
     */
    public MidiResolver(int beats){
        name="EMPTY";
        try {
            midiSeq=new Sequence(Sequence.PPQ,96);
            reso=midiSeq.getResolution();
            tracks=new Track[1];
            tracks[0] =midiSeq.createTrack();
            tracks[0].add(new MidiEvent(
                    new MetaMessage(0x2f, new byte[0], 0), beats*96)
            );
            channelList=new TreeSet[16];
        } catch (InvalidMidiDataException ex) { // 無いはず
            System.err.println(ex);
            System.exit(-1);
        }
    }

    /**
     * Standard MIDI File をモーション生成用に重要な情報に分ける.
     * SMFはこんな構造をしている
     * <ul>
     * <li>resolution や ファイルサイズ などメタ情報</li>
     * <li>Track[1+] (format 0 なら 1つしかない)<ul>
     *   <li>MidiEvents<ul>
     *     <li>tics</li>
     *     <li>MidiMessage</li>
     *   </ul></li>
     * </ul></li>
     * </ul>
     * MidiMessage は ShortMessage と MetaMessage と SystemMessageに分かれるが、
     * ShortMessage 以外は、モーションに関係ないので、ShortMessageのみを
     * チャンネルごとに分け、分類をする。 MetaMessage からは Tempo情報を読み取る.
     * @param f SMF のファイル
     */
    public MidiResolver(File f){
        int i,j,ch;
        MidiEvent me;
        MidiMessage mm;
        ShortMessage sm;
        Track t;
        try {
            midiSeq=MidiSystem.getSequence(f);
            name=f.getName();
            tracks=midiSeq.getTracks();
            reso=midiSeq.getResolution();
            channelList=new TreeSet[16];

            for(i=0;i<tracks.length;i++){
                t=tracks[i];
                for(j=0;j<t.size();j++){
                    me=t.get(j);
                    if(me==null)break;
                    if((mm=me.getMessage())==null)continue;
                    if(mm instanceof ShortMessage){
                        sm=(ShortMessage)mm;
                        ch=sm.getChannel();
                        if(sm.getCommand()==ShortMessage.NOTE_ON && sm.getData2()==0){
                            sm.setMessage(ShortMessage.NOTE_OFF, ch, sm.getData1(), 0);
                        }
                        if(channelList[ch]==null){
                            channelList[ch]=new TreeSet<MidiEvent>(new MidiComparator());
                        }
                        channelList[ch].add(me);
                    } else if(mm instanceof MetaMessage){
                        MetaMessage mem=((MetaMessage)mm);
                        if(mem.getType()==0x51){ // set temp
                            byte []tempB=mem.getData();
                            tempo=(((int)tempB[0]&0xff)<<16)
                                    +(((int)tempB[1]&0xff)<<8)
                                    +((int)tempB[2]&0xff);
/*                        }else if(mem.getType()==0x2f){ // end of track
                            long end=me.getTick();
                            System.out.println("end: " + end);*/
                        }
                    }
                } // in a track
            } // track
        } catch (InvalidMidiDataException ex) {
            Logger.getLogger(MidiResolver.class.getName()).log(Level.SEVERE, null, ex);
        } catch (IOException ex) {
            Logger.getLogger(MidiResolver.class.getName()).log(Level.SEVERE, null, ex);
        }
    }

    /**
     * そのチャンネルのMidiメッセージが tics 順に帰ってくる.
     * @param ch 指定チャンネル
     * @return ch の全Midiメッセージイベント. このメッセージは全部{@link ShortMessage}.
     */
    public MidiEvent [] getChannelMessage(int ch){
        if(channelList[ch]==null)
            return null;
        return channelList[ch].toArray(new MidiEvent[channelList[ch].size()]);
    }

    /**
     * @return the midiSeq
     */
    public Sequence getMidiSeq() {
        return midiSeq;
    }

    /**
     * tics per beat.
     * @return the reso
     */
    public int getReso() {
        return reso;
    }

    /**
     * mili seconds per beat
     * @return the tempo
     */
    public int getTemp() {
        return tempo;
    }

    /**
     * {@link MidiResolver} をつなぐ.
     * テンポとレゾリューションは共通であると想定. 終点はビート区切りになる.
     * @param mr 後に続く {@link MidiResolver}.
     */
    public void add(MidiResolver mr){
        Track t=tracks[0];
        long length=midiSeq.getTickLength();
        if(length%reso >0){
            length=(length/reso+1)*reso;
        }

        MidiEvent [] mms;
        int ch;
        for(ch=0;ch<16;ch++){
            mms=mr.getChannelMessage(ch);
            if(mms!=null){
                if(channelList[ch]==null){
                    channelList[ch]=new TreeSet<MidiEvent>(new MidiComparator());                    
                }
                for(MidiEvent me:mms){
                    me.setTick(me.getTick()+length);
                    t.add(me);
                    channelList[ch].add(me);
                }
            }
        }
        long length2=mr.getMidiSeq().getTickLength();
        if(length2 % reso >0){
            length2=(length2/reso+1)*reso;
        }
        try {
            t.add(new MidiEvent(
                    new MetaMessage(0x2f, new byte[0], 0),
                    length+length2)
            );
        } catch (InvalidMidiDataException ex) { // ありえない
            Logger.getLogger(MidiResolver.class.getName()).log(Level.SEVERE, null, ex);
        }
    }

    /**
     * シーケンスの出力. Starndard MIDI File Format 0 でシーケンスを書き出し.
     * Output Sequence. 
     * @param os 書き込み先
     * @throws IOException 書き込みエラー
     */
    public void write(OutputStream os) throws IOException{
        MidiSystem.write(midiSeq, 0, os);
    }
    
    /**
     * 文字列化.
     * @return 元ファイル名 か EMPTY.
     */
    @Override
    public String toString(){
        return name;
    }
}
