/*
 * The MIT License
 *
 * Copyright 2015 nazo.
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */
package jp.sourceforge.mmd.motion;

import java.util.ArrayList;
import java.util.Iterator;
import java.util.List;
import java.util.TreeMap;

/**
 * package limited なクラス.
 * フレームごとのポーズの管理. ボーンとモーフだけは、名前のマップ.
 * @author nazo
 */
@SuppressWarnings("FieldMayBeFinal")
class MoveOnFrame {
    private int frame;
    protected TreeMap<String,BonePose> bonePoses;
    protected TreeMap<String,MorphPose> morphPoses;
    protected List<Pose<?>> otherPoses;

    public MoveOnFrame(int frame){
        this.frame=frame;
        bonePoses=new TreeMap<String,BonePose>();
        morphPoses=new TreeMap<String,MorphPose>();
        otherPoses=new ArrayList<Pose<?>>();
    }

    @Override
    public int hashCode(){
        return frame;
    }

    @Override
    public boolean equals(Object obj) {
        if (obj == null) {
            return false;
        }
        if (getClass() != obj.getClass()) {
            return false;
        }
        final MoveOnFrame other = (MoveOnFrame) obj;
        return this.frame == other.frame;
    }

    /**
     * ポーズを登録する.
     * @param p ポーズ.
     */
    public void put(Pose<?> p){
        if(p instanceof BonePose){
            bonePoses.put(p.nameOfBone, (BonePose)p);
        } else if(p instanceof MorphPose){
            morphPoses.put(p.nameOfBone,(MorphPose)p);
        } else {
            otherPoses.add(p);
        }
    }

    /**
     * そのボーン名の{@link BonePose}を得る.
     * @param s ボーン名
     * @return 存在すればそのポーズ. 無ければ{@code null}.
     */
    public BonePose get(String s){
        return bonePoses.get(s);
    }

    /**
     * そのモーフ名の{@link MorphPose}を得る.
     * @param s モーフ名
     * @return 存在すればそのポーズ. 無ければ{@code null}.
     */
    public MorphPose getMorph(String s){
        return morphPoses.get(s);
    }

    /**
     * 指定したボーン名のボーンを削除
     * @param s 削除するボーン名
     * @param 格納されていた {@link BonePose}.
     */
    public BonePose remove(String s){
        return bonePoses.remove(s);
    }

    /**
     * 指定したモーフ名のモーフを削除
     * @param s 削除するモーフ名
     * @param 格納されていた {@link MorphPose}.
     */
    public MorphPose removeMorph(String s){
        return morphPoses.remove(s);
    }

    public int size(){
        return bonePoses.size()+morphPoses.size()+otherPoses.size();
    }

    /** 
     * 同フレーム軸上のポーズを全て返す.
     * ポーズは全てクローンされているので書き換えても大丈夫.
     * @return 
     */
    public Pose<?> [] toArray(){
        Pose<?> [] ps=new Pose<?>[this.size()];
        Iterator<? extends Pose<?>> ip=bonePoses.values().iterator();
        Pose<?> p;
        int i=0;

        while(ip.hasNext()){// bonePoses
            p=ip.next();
            ps[i]=p.clone();
            i++;
        }

        ip=morphPoses.values().iterator();
        while(ip.hasNext()){
            p=ip.next();
            ps[i]=p.clone();
            i++;
        }

        ip=otherPoses.iterator();
        while(ip.hasNext()){
            p=ip.next();
            ps[i]=p.clone();
            i++;
        }
        return ps;
    }
}
