/*
 * The MIT License
 *
 * Copyright 2015 nazo.
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */
package jp.sourceforge.mmd.midiMotion.gui;

import java.awt.Color;
import java.awt.datatransfer.Transferable;
import java.awt.event.MouseEvent;
import java.awt.event.MouseListener;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.IOException;
import javax.swing.JComponent;
import javax.swing.JOptionPane;
import javax.swing.JTextField;
import javax.swing.TransferHandler;
import jp.sfjp.mikutoga.bin.parser.MmdFormatException;
import jp.sourceforge.mmd.motion.model.Model;
import jp.sourceforge.mmd.motion.swing.FileDropEvent;
import jp.sourceforge.mmd.motion.swing.FileDropListener;
import jp.sourceforge.mmd.motion.swing.FileDropTarget;
import jp.sourceforge.mmd.motion.swing.MmdDialog;

/**
 * MMDモデルデータを表示・ドラック&ドロップするための swing JTextField クラス.
 * @author nazo
 * @since 1.2
 */
public class ModelTextBox extends JTextField implements FileDropListener,MouseListener  {

    private class FileTransferHandler extends TransferHandler {
        @Override
        public int getSourceActions(JComponent c) {
            return TransferHandler.COPY;
        }
        @Override
        protected Transferable createTransferable(JComponent c){
            return new FileTransferable(orgFile);
        }
    }

    private Model model;

    /**
     * motion の元ファイル(得られる場合)
     * @since 1.2
     */
    protected File orgFile;

    public ModelTextBox(){
        super();
        model=null;
        orgFile=null;
        setEditable(false);
        setBackground(Color.PINK);
        setText("(まだ)");
        setToolTipText("MMDモデル.PMD, PMXをドロップ可能.");
        setDropTarget(new FileDropTarget(this));
        addMouseListener(this);
        setDragEnabled(false);
        setTransferHandler(new FileTransferHandler());
    }

    public void setModel(Model m){
        model=m;
        if(m==null){
            setDragEnabled(false);
            orgFile=null;
            setText("(まだ)");
            setToolTipText("MMDモデル.PMD, PMXをドロップ可能.");
        } else {
            setText(m.getName());
            setToolTipText("<html>"+m.getName()+":<br>file:"+((orgFile==null)?"なし":orgFile.getPath())+"</html>");
        }
    }

    public void setModel(File file){
        if(file==null){
            setModel((Model)null);
            return;
        }
        String name=file.getName().toLowerCase();
        Model m=null;
        FileInputStream fis=null;

        try {
            fis=new FileInputStream(file);
            if(name.endsWith(".pmx")){
                m=Model.fromPMX(fis);
            } else if(name.endsWith(".pmd")){
                m=Model.fromPMD(fis);
            } else if(name.endsWith(".csv")){
                m=Model.fromCSV(fis);
            } else {
                JOptionPane.showMessageDialog(this, "ファイル形式に対応してません:\n"+ file.toString());
            }
            fis.close();
        } catch(FileNotFoundException ex){
            JOptionPane.showMessageDialog(this, "ファイルが見つかりません: "+ ex.getLocalizedMessage());
        } catch(IOException ex){
            JOptionPane.showMessageDialog(this, "ファイルが読み取れません: "+ ex.getLocalizedMessage());
        } catch(MmdFormatException ex){
            JOptionPane.showMessageDialog(this, "MMDのモデルファイルではありません: "+ ex.getLocalizedMessage());
            try{if(fis!=null)fis.close();}catch(IOException ex2){
                JOptionPane.showMessageDialog(this, "ファイルが読み取れません: "+ ex2.getLocalizedMessage());
            }
        }
        if(m==null)return;
        orgFile=file;
        setDragEnabled(true);
        setModel(m);
    }

    public Model getModel(){
        return model;
    }

    /**
     * 得られる場合に限り, モデルファイルを取得する.
     * @return 得られないときは{@code null}.
     */
    public File getFile(){
        return orgFile;
    }

    public void openDialog(){
        Model m=MmdDialog.loadModel(getParent());
        if(m==null)return;
        orgFile=MmdDialog.lastSelected;
        setDragEnabled(true);
        setModel(m);
    }

    @Override
    public void fileDrop(FileDropEvent e) {
        File file=e.getFiles()[0];
        setModel(file);
    }

    @Override
    public void mouseClicked(MouseEvent e) {
        if(e.getClickCount()>1){// double clicked
            if(e.getButton()==MouseEvent.BUTTON3){// right 
                setModel((Model)null);
            } else if(e.getButton()==MouseEvent.BUTTON1){// left
                openDialog();
            }
        }
    }

    @Override
    public void mousePressed(MouseEvent e) {
        selectAll();
    }

    @Override
    public void mouseReleased(MouseEvent e) {}

    @Override
    public void mouseEntered(MouseEvent e) {}

    @Override
    public void mouseExited(MouseEvent e) {}
}
