package jp.sourceforge.mmd.ik_solver;

/*
 * The MIT License
 *
 * Copyright 2015 nazo.
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */


import jp.sourceforge.mmd.motion.geo.Matrix;
import jp.sourceforge.mmd.motion.model.Bone;
import jp.sourceforge.mmd.motion.geo.Vector3D;

/**
 * IK をするためのクラス.
 * Inverse Kinetics (IK) とは, 腕など多節で回転しかできないようなもの先端を,
 * 目的の位置に合わせる用に各節の回転角を合わせる作業で, 反復計算が有効なので,
 * コンピューターにやらせることが多い. このプログラムでは最大100回の接近試行を
 * 行い、10<sup>-8</sup>まで接近したらやめる.
 * 
 * @author nazo
 */
public class IKSolver {
    /**
     * はじめから制限軸がある場合のフラグ.
     * モデルに回転制限がある場合に自動的に付く.
     */
    static final public int RLIMIT_LIMITED=0x80;

    /**
     * グローバルで回転させない. bones[0] にしか効果ない.
     */
    static final public int RLIMIT_NOGlobalR=0x40;

    /**
     * ローカル x 軸周りの回転させないフラグ.
     */
    static final public int RLIMIT_NOX=0x3;

    /**
     * ローカル y 軸周りの回転させないフラグ.
     */
    static final public int RLIMIT_NOY=0xc;

    /**
     * ローカル z 軸周りの回転させないフラグ.
     */
    static final public int RLIMIT_NOZ=0x30;

    /**
     * ローカル x 軸周りの回転フラグ.
     */
    static final public int RLIMIT_ONLY_X=0x3c;

    /**
     * ローカル y 軸周りの回転フラグ.
     */
    static final public int RLIMIT_ONLY_Y=0x33;

    /**
     * ローカル z 軸周りの回転フラグ.
     */
    static final public int RLIMIT_ONLY_Z=0x0f;

    /**
     * 回転角度制限. 1度に動かす角度の限界, 固定値で sine(pi/4)=sqrt(1/2).
     */
    static final protected double limit_sine=Math.sqrt(0.5);

    /**
     * 回転角度制限 {@link #limit_sine} の2乗. (=0.5)
     */
    static final protected double limit_sine2=0.5;

    /**
     * 対象になるボーン.
     * [0]: target に合わせるボーン,
     * [1+]: 動かすボーン
     */
    protected Bone[] bones;

    /**
     * ボーンに対する回転リミットのフラグ.
     * {@link #setLimits(int, int) }などで指定するもの.
     */
    protected int [] limits;

    /**
     * 今のところ何もしないコンストラクター.
     * 必ず{@link #solve(Vector3D) }する前に, {@link #setBones(Bone[]) } をすること.
     */
    public IKSolver(){
    }

    /**
     * 対象になるボーンを設定.
     * @param arms [0]: target に合わせるボーン, [1+]: 動かすボーン.
     */
    public void setBones(Bone [] arms){
        bones=new Bone[arms.length];
        limits=new int[arms.length];
        int i;

        for(i=0;i<arms.length;i++){
            bones[i]=arms[i];
            if(arms[i].getLimitRot()!=null){
                limits[i]=RLIMIT_LIMITED;
            }
        }
    }

    /**
     * 対象になるボーン列を取得. クローンじゃないので、変更しないように,
     * @return arms [0]: target に合わせるボーン, [1+]: 動かすボーン.
     */
    public final Bone []  getBones(){
        return bones;
    }
    
    /**
     * IKをする際に、回転に制限を加える.
     * @param index {@link #bones} の番号
     * @param flags {@link #RLIMIT_NOX} などを or 加算する。
     */
    public void setLimits(int index,int flags){
        limits[index]=flags;
    }

    /**
     * 回転制限を得る.
     * @param index 番目
     * @return 回転制限のフラグ
     */
    public int getLimits(int index){
        return limits[index];
    }

    /**
     * target にbones[0]を合わせるIKを開始する. 反復回数は 40回で固定.
     * @param target bones[0]を合わせる目的の位置(グローバルベクトル).
     * @return 残り距離.
     */
    public double solve(Vector3D target){
        int i,j,k;
        Vector3D center;
        Vector3D tr,ta;
        Vector3D ntr,nta;
        Vector3D rt;
        Matrix gmr=null;
        double cosine,sine2,norm2;
        double distance2=1;

        if((limits[0]&RLIMIT_NOGlobalR)>0){
            gmr=bones[0].getGMatrix();
        }

        for(i=0;i<40;i++){
            distance2=target.sub(bones[0].getPos()).norm2();
            if(distance2<1e-15){
                break;
            }
            for(k=1;k<bones.length;k++){
                center=bones[k].getPos();
                tr=target.sub(center);
                ta=bones[0].getPos().sub(center);

                Vector3D axis=null;
                if(limits[k]>0){
                    if((limits[k] & RLIMIT_LIMITED)>0){
                        axis=bones[k].getLimitRot();
                    } else if((limits[k] & RLIMIT_ONLY_X)==RLIMIT_ONLY_X){
                        axis=bones[k].getLx();
                    } else if((limits[k] & RLIMIT_ONLY_Y)==RLIMIT_ONLY_Y){
                        axis=bones[k].getLy();
                    } else if((limits[k] & RLIMIT_ONLY_Z)==RLIMIT_ONLY_Z){
                        axis=bones[k].getLz();
                    }

                    if(axis!=null){
                        cosine=tr.times(axis);
                        tr=tr.sub(axis.times(cosine));
                        cosine=ta.times(axis);
                        ta=ta.sub(axis.times(cosine));
                    }
                }

                if((norm2=tr.norm2())==0){continue;}
                ntr=tr.divide(Math.sqrt(norm2));
                if((norm2=ta.norm2())==0){continue;}
                nta=ta.divide(Math.sqrt(norm2));
                rt=nta.cross(ntr);

                sine2=rt.norm2();
                cosine=ntr.times(nta);
                if(cosine<0 || sine2>limit_sine2){
                    rt=rt.times(limit_sine/Math.sqrt(sine2));
                    cosine=limit_sine;
                }

                if(limits[k]>0 && axis==null) {
                    if((limits[k] & RLIMIT_NOX)>0){
                        cosine=rt.times(axis=bones[k].getLx());
                        rt=rt.sub(axis.times(cosine));
                    }

                    if((limits[k] & RLIMIT_NOY)>0){
                        cosine=rt.times(axis=bones[k].getLy());
                        rt=rt.sub(axis.times(cosine));
                    }

                    if((limits[k] & RLIMIT_NOZ)>0){
                        cosine=rt.times(axis=bones[k].getLz());
                        rt=rt.sub(axis.times(cosine));
                    }

                    sine2=rt.norm2();
                    cosine=(sine2>1.0)?0:Math.sqrt(1-sine2);
                }

                bones[k].rotate(rt, cosine);
            }
        }

        if((limits[0]&RLIMIT_NOGlobalR)>0){
            bones[0].rotateToG(gmr);
        }
        return Math.sqrt(distance2);
    }
}
