/*
 * The MIT License
 *
 * Copyright 2015 nazo.
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */
package jp.sourceforge.mmd.motion;

import java.io.BufferedReader;
import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.io.OutputStream;
import java.io.OutputStreamWriter;
import java.io.UnsupportedEncodingException;
import java.nio.ByteBuffer;
import java.nio.ByteOrder;
import java.util.ArrayList;
import java.util.Map;
import java.util.Set;
import java.util.TreeMap;
import java.util.TreeSet;
import jp.sfjp.mikutoga.bin.parser.MmdFormatException;
import jp.sfjp.mikutoga.vmd.parser.VmdParser;

/**
 * MMD モーションの管理.
 * モーションはフレーム番とボーン名の両方で管理されている.
 * @author nazo
 * @since 1.0
 */
@SuppressWarnings("FieldMayBeFinal")
public class Motion {
    private String modelName;
    private int max_frame;
    private TreeMap<Integer,MoveOnFrame> frame_map;
    private TreeMap<String,MoveOnBone<BonePose>> bone_map;
    private TreeMap<String,MoveOnBone<MorphPose>> morph_map;
    private MoveOnBone<CameraPose> cameraArray;
    private MoveOnBone<LightPose> lightArray;
    private MoveOnBone<ShadowPose> shadowArray;
    private TreeSet<BooleanPose> booleanArray;

    /**
     * 空で名前のないモーションをつくる.
     * @since 1.0
     */
    public Motion(){
        modelName="";
        max_frame=0;
        frame_map = new TreeMap<Integer,MoveOnFrame> ();
        bone_map = new TreeMap<String,MoveOnBone<BonePose>>();
        morph_map = new TreeMap<String,MoveOnBone<MorphPose>>();
        cameraArray=new MoveOnBone<CameraPose>("カメラ");
        lightArray=new MoveOnBone<LightPose>("照明");
        shadowArray=new MoveOnBone<ShadowPose>("セルフ影");
        booleanArray=new TreeSet<BooleanPose>();
    }

    /**
     * 空でモデル名があるモーションをつくる.
     * @param model_name モデル名.
     * @since 1.0
     */
    public Motion(String model_name){
        this();
        this.modelName=model_name;
    }

    /**
     * Poseを登録する.
     * フレーム番と名前は, Pose から読み取られる.
     * クローンが登録されるので, 元Poseを書き換えても大丈夫.
     * @param pose 登録するPose
     * @since 1.0
     */
    public void put(Pose <?> pose){
        Pose<?> p=pose.clone();
        if(p instanceof BonePose){
            MoveOnBone<BonePose> mob=bone_map.get(p.nameOfBone);
            if(mob==null){
                mob=new MoveOnBone<BonePose>(p.nameOfBone);
                bone_map.put(p.nameOfBone, mob);
            }
            mob.put((BonePose)p);
        }else if(p instanceof MorphPose){
            MoveOnBone<MorphPose> mob=morph_map.get(p.nameOfBone);
            if(mob==null){
                mob=new MoveOnBone<MorphPose>(p.nameOfBone);
                morph_map.put(p.nameOfBone, mob);
            }
            mob.put((MorphPose)p);
        }else if(p instanceof CameraPose){
            cameraArray.put((CameraPose)p);
        }else if(p instanceof LightPose){
            lightArray.put((LightPose)p);
        }else if(p instanceof ShadowPose){
            shadowArray.put((ShadowPose)p);
        }else if(p instanceof BooleanPose){
            booleanArray.add((BooleanPose)p);
        }

        MoveOnFrame mof=frame_map.get(p.frame);
        if(mof==null){
            mof=new MoveOnFrame(p.frame);
            frame_map.put(p.frame, mof);
            if(p.frame>max_frame){
                max_frame=p.frame;
            }
        }
        mof.put(p);
    }

    /**
     * 複数のPoseを登録する.
     * 名前は, Pose から読み取られる. フレーム番は書き換えられる.
     * @param ps 登録するPoseのアレイ. null でも可.
     * @param frame 登録するフレーム番. 負を指定すると 0 になる.
     * @since 1.0
     */
    public void putAll(Pose<?> [] ps,int frame){
        if(ps==null)return;
        if(frame<0)frame=0;
        for(Pose<?> p:ps){
            p.frame=frame;
            this.put(p);
        }
    }

    /**
     * 指定したボーン名のポーズを全て得る.
     * @param bone ボーン名.
     * @return あてはまるポーズのクローンがアレイで帰る. ないときは{@code null}.
     * @since 1.0
     */
    public BonePose [] get(String bone){
        if(bone==null)return null;
        MoveOnBone<BonePose> mob=bone_map.get(bone);
        if(mob==null){
                return null;
        }
        return mob.toArray(new BonePose[mob.size()]);
    }

    /**
     * 指定したフレーム番・ボーン名のキーポーズがあれば得る.
     * @param frame フレーム番.
     * @param bone ボーン名.
     * @return あてはまるポーズのクローンが帰る. ないときは{@code null}.
     * @since 1.4
     */
    public BonePose get(int frame, String bone){
        if(bone==null)return null;
        MoveOnBone<BonePose> mob=bone_map.get(bone);
        if(mob==null)
            return null;

        BonePose bp=mob.get(frame);
        if(bp!=null)
            bp=new BonePose(bp);

        return bp;
    }

    /**
     * 指定したフレーム番・ボーン名のキーポーズがあれば削除.
     * @param frame フレーム番.
     * @param bone ボーン名.
     * @return あてはまるポーズのクローンが帰る. ない(何も削除してない)ときは{@code null}.
     * @since 1.5
     */
    public BonePose remove(int frame, String bone){
        if(bone==null)return null;
        MoveOnBone<BonePose> mob=bone_map.get(bone);
        if(mob==null)
            return null;

        BonePose bp=mob.remove(frame);
        if(bp==null)
            return null;
        if(mob.size()==0){
            bone_map.remove(bone);
        }

        MoveOnFrame mof=frame_map.get(frame);
        if(mof==null)// 無いと思う
            return bp;
        BonePose bp2=mof.remove(bone);
        if(bp2==null)
            return bp;
        if(mof.size()==0){
            frame_map.remove(frame);
        }
        return bp;
    }

    /**
     * 指定したモーフ名のポーズを全て得る.
     * @param morph モーフ名.
     * @return あてはまるモーフのクローンがアレイで帰る. ないときは{@code null}.
     * @since 1.4
     */
    public MorphPose [] getMorph(String morph){
        if(morph==null)return null;
        MoveOnBone<MorphPose> mob=morph_map.get(morph);
        if(mob==null){
            return null;
        }
        return  mob.toArray(new MorphPose[mob.size()]);
    }

    /**
     * 指定したフレーム番・モーフ名のキーポーズがあれば得る.
     * @param frame フレーム番.
     * @param morph モーフ名.
     * @return あてはまるポーズのクローンが帰る. ないときは{@code null}.
     * @since 1.4
     */
    public MorphPose getMorph(int frame, String morph){
        if(morph==null)return null;
        MoveOnBone<MorphPose> mob=morph_map.get(morph);
        if(mob==null)
            return null;

        MorphPose mp=mob.get(frame);
        if(mp!=null)
            mp=new MorphPose(mp);

        return mp;
    }

    /**
     * 指定したフレーム番・モーフ名のキーポーズがあれば削除.
     * @param frame フレーム番.
     * @param morph モーフ名.
     * @return あてはまるポーズのクローンが帰る. ない(何も削除してない)ときは{@code null}.
     * @since 1.5
     */
    public MorphPose removeMorph(int frame, String morph){
        if(morph==null)return null;
        MoveOnBone<MorphPose> mob=morph_map.get(morph);
        if(mob==null)
            return null;

        MorphPose bp=mob.remove(frame);
        if(bp==null)
            return null;
        if(mob.size()==0){
            morph_map.remove(morph);
        }

        MoveOnFrame mof=frame_map.get(frame);
        if(mof==null)// 無いと思う
            return bp;
        MorphPose bp2=mof.removeMorph(morph);
        if(bp2==null)
            return bp;
        if(mof.size()==0){
            frame_map.remove(frame);
        }
        return bp;
    }

    /**
     * 指定したフレームのポーズを全て得る.
     * ポーズは全てクローンされているので書き換えても大丈夫.
     * @param frame フレーム番
     * @return あてはまるポーズのクローンがアレイで帰る. ないときは{@code null}.
     * @since 1.0
     */
    public Pose<?> [] get(int frame){
        MoveOnFrame mof=frame_map.get(frame);
        if(mof==null)
            return null;
        return mof.toArray();
    }

    /**
     * 指定したフレームの補間ポーズを全て得る.
     * @param frame フレーム番
     * @return 補間したポーズ(ボーンとモーフのみ)がアレイで帰る. 補間不要なキーポーズもクローンして入っている.
     * @since 1.0
     */
    public Pose<?> [] getInterporate(int frame) {
        int i=0;
        Pose <?>[] ret=new Pose<?>[bone_map.size()+morph_map.size()];
        for(MoveOnBone <BonePose>e:bone_map.values()){
            ret[i]=e.getInterporate(frame);
            i++;
        }
        for(MoveOnBone <MorphPose> e:morph_map.values()){
            ret[i]=e.getInterporate(frame);
            i++;
        }
        return ret;
    }

    /**
     * 指定したフレームの指定した名前のボーンポーズを全て得る.
     * @param frame フレーム番
     * @param boneName ボーン名
     * @return 補間したポーズがアレイで帰る. 補間不要なキーポーズもクローンして入る. 
     * @since 1.4
     */
    public BonePose getInterporateBone(int frame, String boneName) {
        MoveOnBone<BonePose> e=bone_map.get(boneName);
        BonePose ret;
        if(e==null){
            ret=new BonePose();
            ret.frame=frame;
            ret.nameOfBone=boneName;
        } else {
            ret=e.getInterporate(frame);
        }
        return ret;
    }

    /**
     * 指定したフレームの指定した名前のモーフポーズを全て得る.
     * @param frame フレーム番
     * @param morphName モーフ名
     * @return 補間したポーズがアレイで帰る. 補間不要なキーポーズもクローンして入る.
     * @since 1.4
     */
    public MorphPose getInterporateMorph(int frame, String morphName) {
        MorphPose ret;
        MoveOnBone<MorphPose> e=morph_map.get(morphName);
        if(e==null){
            ret=new MorphPose();   
            ret.frame=frame;
            ret.nameOfBone=morphName;
        } else {
            ret=e.getInterporate(frame);
        }
        return ret;
    }

    /**
     * ごみ掃除. 無意味なポーズを削除する.
     * @since 1.0
     */
    public void gc(){
        Integer [] list;
        for(Map.Entry<String,MoveOnBone<BonePose>> mob:bone_map.entrySet()){
            list=mob.getValue().gc();
            for(Integer i : list){
                MoveOnFrame mof=frame_map.get(i);
                mof.remove(mob.getKey());
                if(mof.size()==0){
                    frame_map.remove(i);
                }
            }
        }
        for(Map.Entry<String,MoveOnBone<MorphPose>> mob:morph_map.entrySet()){
            list=mob.getValue().gc();
            for(Integer i : list){
                MoveOnFrame mof=frame_map.get(i);
                mof.removeMorph(mob.getKey());
                if(mof.size()==0){
                    frame_map.remove(i);
                }
            }
        }
    }

    /**
     * CSV化
     * @param os CSV化された文字列の出力先.
     * @throws java.io.IOException 書き込みできないなど.
     * @since 1.0
     */
    public void toCSV(OutputStream os) throws IOException{
        OutputStreamWriter osw=null;
        try {
            osw=new OutputStreamWriter(os,"MS932");
        } catch (UnsupportedEncodingException e) {
            /* never called. */
        }
        osw.write("Vocaloid Motion Data 0002,0\n"+modelName+"\n");

        int num_bone=0;
        StringBuilder list=new StringBuilder(4096);
        for(Map.Entry<String,MoveOnBone<BonePose>> e:bone_map.entrySet()){
            MoveOnBone<BonePose> mob=e.getValue();
            num_bone+=mob.size();
            list.append(mob.toCSV());
        }
        osw.write(num_bone+"\n"+list);

        num_bone=0; //morph
        list=new StringBuilder(4096);
        for(Map.Entry<String,MoveOnBone<MorphPose>> e:morph_map.entrySet()){
            MoveOnBone<MorphPose> mob=e.getValue();
            num_bone+=mob.size();
            list.append(mob.toCSV());
        }
        osw.write(num_bone+"\n"+list);
        
        // camera
        osw.write(cameraArray.size()+"\n"+cameraArray.toCSV());

        // light
        osw.write(lightArray.size()+"\n"+lightArray.toCSV());

        // shadow
        osw.write(shadowArray.size()+"\n"+shadowArray.toCSV());

        // boolean
        osw.write(booleanArray.size()+"\n");
        for(BooleanPose e:booleanArray){
            osw.write(e.toCSV());
        }
        osw.flush();
        os.flush();
    }

    /**
     * VMD化.
     * @param os 書き込み先のOutputStream.
     * @throws IOException 書き込み時に不具合が発生したとき.
     * @since 1.0
     */
    public void toVMD(OutputStream os) throws IOException{
        ByteBuffer bbInt=ByteBuffer.allocate(4).order(ByteOrder.LITTLE_ENDIAN);
        byte [] temp;
        int i;
        try {
            os.write("Vocaloid Motion Data 0002".getBytes("MS932"));
        } catch (UnsupportedEncodingException ex) {
            System.err.println("Syntax error in toVMD in Motion class.");
            System.exit(-1);
        }
        os.write(0);
        os.write(bbInt.putInt(0).array());
        temp=modelName.getBytes("MS932");
        for(i=0;i<temp.length&&i<20;i++){
            os.write(temp[i]);
        }
        for(;i<20;i++){
            os.write(0);
        }

        int number=0;
        ArrayList<byte[]> list=new ArrayList<byte[]>();
        for(Map.Entry<String,MoveOnBone<BonePose>> e:bone_map.entrySet()){
            MoveOnBone<BonePose> mob;// bone
            mob=e.getValue();
            number+=mob.size();
            list.add(mob.toVMD());
        }
        os.write(bbInt.putInt(0,number).array());
        for(byte [] line:list){
            os.write(line);
        }

        number=0;// morph
        list.clear();
        for(Map.Entry<String,MoveOnBone<MorphPose>> e:morph_map.entrySet()){
            MoveOnBone<MorphPose> mob;// morph
            mob=e.getValue();
            number+=mob.size();
            list.add(mob.toVMD());
        }
        os.write(bbInt.putInt(0,number).array());
        for(byte [] line:list){
            os.write(line);
        }

        // camera
        os.write(bbInt.putInt(0,cameraArray.size()).array());
        os.write(cameraArray.toVMD());

        // light
        os.write(bbInt.putInt(0,lightArray.size()).array());
        os.write(lightArray.toVMD());

        // shadow
        os.write(bbInt.putInt(0,shadowArray.size()).array());
        os.write(shadowArray.toVMD());

        // boolean
        os.write(bbInt.putInt(0,booleanArray.size()).array());
        for(BooleanPose e:booleanArray){
            os.write(e.toVMD());
        }
        os.flush();
    }

    /**
     * CSVからMotion を追加する.
     * @param is CSVのInputStream
     * @return 追加後のモーション. {@code this}と同じ.
     * @throws IOException 書き込み時に不具合が発生したとき.
     * @throws MmdFormatException MMDモーションのフォーマットじゃないとき.
     * @since 1.0
     */
    public Motion fromCSV(InputStream is)throws IOException,MmdFormatException{
        return fromCSV(is,0);
    }

    /**
     * CSVからMotion をフレーム番ずらして追加する.
     * @param is CSVのInputStream
     * @param offsetF ずらすオフセット. 負数は推奨されない.
     * @return 追加後のモーション. {@code this}と同じ.
     * @throws IOException 書き込み時に不具合が発生したとき.
     * @throws MmdFormatException MMDモーションのフォーマットじゃないとき.
     * @since 1.0
     */
    public Motion fromCSV(InputStream is,int offsetF) throws IOException,MmdFormatException{
        BufferedReader br=null;
        int numOfLine=1;
        try {
            br = new BufferedReader(new InputStreamReader(is,"MS932"));
        } catch (UnsupportedEncodingException ex) {
            System.err.println("Syntax err in class Motion.fromCSV.");
            System.exit(-1);
        }
        String line;
        String [] parts;
        int l,i;

        line=br.readLine();
        if(!line.startsWith("Vocaloid Motion Data")){
            throw new MmdFormatException("Header error at line "+numOfLine,0);
        }

        numOfLine++;
        line=br.readLine();
        parts=CsvSpliter.split(line);
        if(modelName.length()==0){
            modelName=parts[0];
        } else if(modelName.compareTo(parts[0])!=0){
            throw new MmdFormatException("Not the same model. memory:"
                    +modelName+", file:"+parts[0]);
        }

        Pose <?> p;
        numOfLine++;
        line=br.readLine(); // Bones
        parts=CsvSpliter.split(line);
        try {
            l=Integer.parseInt(parts[0]);
        }catch (NumberFormatException e){
            throw new MmdFormatException("Illegal number for bone lines: "+e.getMessage()+" at line:"+numOfLine);
        }
        for(i=0;i<l;i++){
            numOfLine++;
            line=br.readLine();
            try{
                p=BonePose.fromCSV(line);
            }catch (NumberFormatException e){
                throw new MmdFormatException("Illegal number for bone line: "+e.getMessage()+" at line:"+numOfLine);
            }
            if(p!=null){
                p.frame+=offsetF;
                this.put(p);
            }
        }

        numOfLine++;
        line=br.readLine();  //Morphs
        if(line==null)return this;
        parts=CsvSpliter.split(line);
        try{
            l=Integer.parseInt(parts[0]);
        }catch (NumberFormatException e){
            throw new MmdFormatException("Illegal number for morph lines: "+e.getMessage()+" at line:"+numOfLine);
        }
        for(i=0;i<l;i++){
            numOfLine++;
            line=br.readLine();
            try{
                p=MorphPose.fromCSV(line);
            }catch (NumberFormatException e){
                throw new MmdFormatException("Illegal number for morph line: "+e.getMessage()+" at line:"+numOfLine);
            }
            if(p!=null){
                p.frame+=offsetF;
                this.put(p);
            }
        }

        numOfLine++;
        line=br.readLine();  //camera
        if(line==null)return this;
        parts=CsvSpliter.split(line);
        try{
            l=Integer.parseInt(parts[0]);
        }catch (NumberFormatException e){
            throw new MmdFormatException("Illegal number for camera lines: "+e.getMessage()+" at line:"+numOfLine);
        }
        for(i=0;i<l;i++){
            numOfLine++;
            line=br.readLine();
            try{
                p=CameraPose.fromCSV(line);
            }catch (NumberFormatException e){
                throw new MmdFormatException("Illegal number for camera line: "+e.getMessage()+" at line:"+numOfLine);
            }
            if(p!=null){
                p.frame+=offsetF;
                this.put(p);
            }
        }

        numOfLine++;
        line=br.readLine();  //light
        if(line==null)return this;
        parts=CsvSpliter.split(line);
        try{
            l=Integer.parseInt(parts[0]);
        }catch (NumberFormatException e){
            throw new MmdFormatException("Illegal number for light lines: "+e.getMessage()+" at line:"+numOfLine);
        }
        for(i=0;i<l;i++){
            numOfLine++;
            line=br.readLine();
            try{
                p=LightPose.fromCSV(line);
            }catch (NumberFormatException e){
                throw new MmdFormatException("Illegal number for light line: "+e.getMessage()+" at line:"+numOfLine);
            }
            if(p!=null){
                p.frame+=offsetF;
                this.put(p);
            }
        }

        numOfLine++;
        line=br.readLine();  //shadow
        if(line==null)return this;
        parts=CsvSpliter.split(line);
        try{
            l=Integer.parseInt(parts[0]);
        }catch (NumberFormatException e){
            throw new MmdFormatException("Illegal number for shadow lines: "+e.getMessage()+" at line:"+numOfLine);
        }
        for(i=0;i<l;i++){
            numOfLine++;
            line=br.readLine();
            try{
                p=ShadowPose.fromCSV(line);
            }catch (NumberFormatException e){
                throw new MmdFormatException("Illegal number for shadow line: "+e.getMessage()+" at line:"+numOfLine);
            }
            if(p!=null){
                p.frame+=offsetF;
                this.put(p);
            }
        }

        numOfLine++;
        line=br.readLine();  //boolean
        if(line==null)return this;
        parts=CsvSpliter.split(line);
        try{
            l=Integer.parseInt(parts[0]);
        }catch (NumberFormatException e){
            throw new MmdFormatException("Illegal number for boolean lines: "+e.getMessage()+" at line:"+numOfLine);
        }
        for(i=0;i<l;i++){
            numOfLine++;
            line=br.readLine();
            try{
                p=BooleanPose.fromCSV(line);
            }catch (NumberFormatException e){
                throw new MmdFormatException("Illegal number for boolean line: "+e.getMessage()+" at line:"+numOfLine);
            }
            if(p!=null){
                p.frame+=offsetF;
                this.put(p);
            }
        }
        return this;
    }

    /**
     * VMDからMotion を追加する.
     * @param is VMDのInputStream
     * @return 追加後のモーション. {@code this}と同じ.
     * @throws IOException 書き込み時に不具合が発生したとき.
     * @throws MmdFormatException MMDモーションのフォーマットじゃないとき.
     * @since 1.0
     */
    public Motion fromVMD(InputStream is) throws IOException,MmdFormatException{
        return fromVMD(is,0);
    }

    /**
     * VMDからMotion をフレーム番ずらして追加する.
     * @param is VMDのInputStream
     * @param frame ずらすオフセット. 負数は推奨されない.
     * @return 追加後のモーション. {@code this}と同じ.
     * @throws IOException 書き込み時に不具合が発生したとき.
     * @throws MmdFormatException MMDモーションのフォーマットじゃないとき.
     * @since 1.0
     */
    public Motion fromVMD(InputStream is,int frame) throws IOException,MmdFormatException{
        VmdParser vmdp=new VmdParser(is);
        VMDMotionHander parser = new VMDMotionHander(this,frame);

        vmdp.setBasicHandler(parser);
        vmdp.setCameraHandler(parser);
        vmdp.setLightingHandler(parser);
        vmdp.setBoolHandler(parser);

        vmdp.parseVmd();

        return this;
    }

    /**
     * モデル名を得る.
     * @return モデル名.
     * @since 1.0
     */
    public String getModelName() {
        return modelName;
    }

    /**
     * 最終フレーム番を得る.
     * @return 最終フレーム番.
     * @since 1.0
     */
    public int getMaxFrame(){
        return max_frame;
    }

    /**
     * モデル名を書き換える.
     * @param modelName 新しいモデル名.
     * @since 1.0
     */
    public void setModelName(String modelName) {
        this.modelName = modelName;
    }

    /**
     * ボーン名のSetを返す.
     * @return ボーン名のSet.
     * @since 1.0
     */
    public Set<String> listOfBone() {
        return bone_map.keySet();
    }
}
