/*
 * The MIT License
 *
 * Copyright 2015 nazo.
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */
package jp.sourceforge.mmd.motion.model;

import java.util.ArrayList;
import java.util.Map;
import java.util.TreeMap;
import jp.sourceforge.mmd.motion.geo.Matrix;
import jp.sourceforge.mmd.motion.geo.Vector3D;
import jp.sfjp.mikutoga.bin.parser.MmdFormatException;
import jp.sfjp.mikutoga.bin.parser.ParseStage;
import jp.sfjp.mikutoga.pmx.parser.PmxBasicHandler;
import jp.sfjp.mikutoga.pmx.parser.PmxBoneHandler;
import jp.sfjp.mikutoga.pmx.BoneFlags;
import jp.sfjp.mikutoga.pmx.parser.PmxMorphHandler;
import jp.sourceforge.mmd.motion.BonePose;
import jp.sourceforge.mmd.motion.Pose;

/**
 * PMX データを読み出すのに使用するハンドラ.
 * @author nazo
 */
public class PmxFileHander implements PmxBoneHandler,PmxBasicHandler,PmxMorphHandler{
    static protected Vector3D absoluteZ=Vector3D.unitZ;

    /** パースして構築するモデルオブジェクト. {@link #PmxFileHander(Model)}. */
    protected Model model;
    /** パース作業中のボーン */
    protected Bone bone;
    /** パース作業中のモーフ */
    protected Morph morph;
    protected short boneFlags;
    /** idを名前に関連付けるアレイ */
    protected ArrayList<String> idToName;

   /** 子ボーンからの親ボーン指名 */
    protected TreeMap<Integer,Integer> childToParent;
   /** 子ボーンからの関連親ボーン指名 */
    protected TreeMap<Integer,Number []> childToLinkParent;
    /** 親ボーンからの矢先子ボーン指名 */
    protected TreeMap<Integer,Integer> parentToTail;
    /** 親ボーンからの矢先オフセット */
    protected TreeMap<Integer,Vector3D> parentToOffset;
    
    public PmxFileHander(Model model){
        super();
        this.model=model;
    }

    @Override
    public void pmxParseStart() {
        bone=null;
        boneFlags=0;
        morph=null;
    }

    @Override
    public void pmxHeaderInfo(float version, int encode, int uv) throws MmdFormatException {
        if(version<2){
            throw new MmdFormatException("Too old PMX file.");
        }
    }

    @Override
    public void pmxModelName(String name, String nameE) {
        model.setName(name);
    }

    @Override
    public void pmxModelDescription(String description, String descriptionE) {
    }

    @Override
    public void loopStart(ParseStage stage, int loops) {
        if(stage==PmxBoneHandler.BONE_LIST){
            idToName=new ArrayList<String>();
            childToParent=new TreeMap<Integer, Integer>();
            childToLinkParent=new TreeMap<Integer, Number[]>();
            parentToTail=new TreeMap<Integer, Integer>();
            parentToOffset=new TreeMap<Integer, Vector3D>();
        }
    }

    @Override
    public void pmxBoneInfo(String name, String nameE) {
        bone=new Bone(model);
        bone.name=name;
        idToName.add(name);
        bone.id=idToName.size()-1;
    }

    @Override
    public void pmxBonePosition(float xPos, float yPos, float zPos) {
        bone.gv=new Vector3D(xPos,yPos,zPos);
    }

    @Override
    public void pmxBoneStructure(int parentId, int depth) {
        if(parentId!=-1){
            childToParent.put(idToName.size()-1,parentId);
        }
    }

    @Override
    public void pmxBoneFlags(short flags) {
        this.boneFlags=flags;
        bone.flags=flags;
    }

    @Override
    public void pmxBoneOffset(float offX, float offY, float offZ) {
        if(offX!=0 || offY!=0 || offZ!=0){
            parentToOffset.put(idToName.size()-1, new Vector3D(offX,offY,offZ));
        }
    }

    @Override
    public void pmxBoneArrowhead(int arrowId) {
        if(arrowId>=0){
            parentToTail.put(idToName.size()-1, arrowId);
        }
    }

    @Override
    public void pmxBoneLink(int linkParent, float ratio) throws MmdFormatException {
        if(linkParent>=0){
            childToLinkParent.put(idToName.size()-1, new Number[]{
                linkParent,
                (BoneFlags.MOVE_LINK.check(boneFlags)?ratio:0),
                (BoneFlags.ROTATE_LINK.check(boneFlags)?ratio:0)
            });
        }
    }

    @Override
    public void pmxBoneLocalAxis(float xx, float xy, float xz, float zx, float zy, float zz) {
        bone.setInicialLocalCoordinate(new Vector3D(xx, xy, xz),new Vector3D(-zx, -zy, -zz));
    }

    @Override
    public void pmxBoneRotateAxe(float x, float y, float z) {
        bone.limitRot=new Vector3D(x,y,z);
    }

    @Override
    public void pmxBoneExtraParent(int extraParent) {
    }

    @Override
    public void pmxBoneIKInfo(int targetId, int depth, float weight) {
    }

    @Override
    public void pmxIKChainInfo(int childId, float[] limit_rotation) {
    }

    /**
     * ループの1順後. ボーンループでは完成したボーンをモデルに追加する.
     * @param stage ループのステージ. 使用されない.
     */
    @Override
    public void loopNext(ParseStage stage) {
        if(stage == PmxBoneHandler.BONE_LIST){
            model.put(bone);
            bone=null;
        } else if(stage == PmxMorphHandler.MORPH_LIST){
            model.put(morph);
            morph=null;
        }
    }

    /**
     * ループの最後. ボーンループでは, 親子, リンク関係を構築する.
     * @param stage ループのステージ. ボーンループ判別に使用.
     * @throws MmdFormatException 不正なPMX フォーマットだった場合.
     * 大体はボーンIDが不明.
     */
    @Override
    public void loopEnd(ParseStage stage) throws MmdFormatException {
        if(stage==PmxBoneHandler.BONE_LIST){
            for(Map.Entry<Integer,Integer> e:childToParent.entrySet()){
                Bone child=model.get(idToName.get(e.getKey()));
                try {
                    model.get(idToName.get(e.getValue())).addChild(child);
                } catch (ArrayIndexOutOfBoundsException ex){
                    throw new MmdFormatException("Parent ID is not found. id:"
                            +e.getValue()+" from bone:"+child.name);
                }
            }
            for(Map.Entry<Integer,Number[]> e:childToLinkParent.entrySet()){
                Bone child=model.get(idToName.get(e.getKey()));
                Number [] link=e.getValue();
                try {
                model.get(idToName.get((Integer)link[0]))
                        .addLinkChild(child,(Float)link[1],(Float)link[2]);
                } catch (ArrayIndexOutOfBoundsException ex){
                    throw new MmdFormatException("Link parent ID is not found. id:"
                            +link[0]+" from bone:"+child.name);
                }
            }
            for(Map.Entry<Integer,Vector3D> e:parentToOffset.entrySet()){
                String name=idToName.get(e.getKey());
                if(model.get(name+"先")==null){
                    Bone parent=model.get(name);
                    Bone arrowhead=new Bone(model);
                    arrowhead.setName(name+"先");
                    arrowhead.gv=parent.getPos().add(e.getValue());
                    parent.addChild(arrowhead);

                    idToName.add(arrowhead.getName());
                    parentToTail.put(e.getKey(), idToName.size()-1);
                    model.put(arrowhead);
                }
            }
            for(Map.Entry<Integer,Integer> e:parentToTail.entrySet()){
                String name=idToName.get(e.getKey());
                if(name.matches("((左|右)(腕|ひじ|手首)|.*(親|人|中|薬|小)指.*)")){
                    Bone parent=model.get(name);
                    if(parent.getMatrix().equals(new Matrix())){ // Local Axis 最優先
                        String tail;
                        try {
                            tail=idToName.get(e.getValue());
                        } catch (ArrayIndexOutOfBoundsException ex){
                            throw new MmdFormatException("Arrowhead is not found. id:"
                                    +e.getValue()+" from bone: "+name);
                        }
                        Vector3D arrow= model.get(tail).getPos().sub(parent.getPos());
                        arrow=arrow.sub(absoluteZ.times(arrow.times(absoluteZ)));
                        double norm=arrow.norm();
                        if(norm>0){
                            parent.setInicialLocalCoordinate(arrow.divide(norm), absoluteZ);
                        }
                    }
                }
            }
        } // end of BONE_LIST
        }
    
    @Override
    public void pmxMorphInfo(String morphName, String morphNameE, byte panelType) throws MmdFormatException {
        morph=new Morph(model);
        morph.setName(morphName);
    }    
    
    @Override
    public void pmxMorphElementListHeader(byte morphType, int numOfOffsets) throws MmdFormatException {
    }

    @Override
    public void pmxMorphVertexInfo(int vertexId, float xPos, float yPos, float zPos) throws MmdFormatException {
}

    @Override
    public void pmxMorphUVInfo(int vertexId, float uPos, float vPos, float zPos, float wPos) throws MmdFormatException {
    }

    @Override
    public void pmxMorphBoneInfo(int boneId, float xPos, float yPos, float zPos,
            float qx, float qy, float qz, float qw) throws MmdFormatException {
        Bone b=model.get(idToName.get(boneId));
        BonePose bp=new BonePose();
        bp.nameOfBone=b.getName();
        bp.v=new Vector3D(xPos,yPos,zPos);
        bp.mr=Matrix.rotationQ(qx, qy, qz, qw);
        morph.addChild(b, bp);
    }

    @Override
    public void pmxMorphMaterialInfo(int materialId, byte type) throws MmdFormatException {
    }

    @Override
    public void pmxMorphMaterialDiffuse(float red, float green, float blue, float alpha) throws MmdFormatException {
    }

    @Override
    public void pmxMorphMaterialSpecular(float red, float green, float blue, float shininess) throws MmdFormatException {
    }

    @Override
    public void pmxMorphMaterialAmbient(float red, float green, float blue) throws MmdFormatException {
    }

    @Override
    public void pmxMorphMaterialEdges(float red, float green, float blue, float alpha, float thick) throws MmdFormatException {
    }

    @Override
    public void pmxMorphMaterialTexture(float red, float green, float blue, float alpha) throws MmdFormatException {
    }

    @Override
    public void pmxMorphMaterialSphere(float red, float green, float blue, float alpha) throws MmdFormatException {
    }

    @Override
    public void pmxMorphMaterialToon(float red, float green, float blue, float alpha) throws MmdFormatException {
    }

    @Override
    public void pmxMorphGroupInfo(int morphId, float ratio) throws MmdFormatException {
    }

    @Override
    public void pmxParseEnd(boolean hasMoreData) {
    }
}
