﻿//-----------------------------------------------------------------------------
// MMDModel.fx
//
// MikuMikuDance for XNA
// Copyright (C) Wilfrem and Microsoft.
//
// このソースコードはひにけにXNAの「頂点テクスチャでスキンアニメーション」の
// SkinnedModel.fxのソースと
// http://blogs.msdn.com/ito/archive/2009/05/07/more-bones-07.aspx
// XNAクリエイターズオンラインにあったBasicEffect.fxを混ぜた上で
// エフェクトMMDに合わせて変更したもの
// 他、表情、トゥーンやらいろいろいじってる
// 
//-----------------------------------------------------------------------------

//既知の問題(XNAチームに報告済み)
//XBox用にfxファイルをコンパイルする際に、
//ピクセルシェーダの構造体内に未使用のデータ、または未初期化変数があるとコンパイルエラーを起こす
//http://forums.xna.com/forums/t/42630.aspx

//-----------------------------------------------------------------------------
// 定数レジスタ宣言
//=============================================================================
uniform shared const float3	EyePosition				: register(c0);		// in world space

//-----------------------------------------------------------------------------
// ライト設定用レジスタ。
// All directions and positions are in world space and must be unit vectors
//-----------------------------------------------------------------------------
uniform shared const float3	AmbientLightColor		: register(c1);
uniform shared const float3	DirLight0Direction		: register(c2);

//-----------------------------------------------------------------------------
// スフィア
//-----------------------------------------------------------------------------
uniform const int  UseSphere						: register(c3);

//-----------------------------------------------------------------------------
// ボーン及びモーフィング
//-----------------------------------------------------------------------------
uniform const float2 BoneTextureSize				: register(c4);	// ボーン用頂点テクスチャのサイズ
uniform const float2 FaceRateTextureSize			: register(c5);//フェイス割合用の頂点テクスチャのサイズ
uniform const float2 FaceVertTextureSize			: register(c6);//フェイス頂点データ用の頂点テクスチャのサイズ
uniform const float2 FaceVertTextureColLines		: register(c7);//フェイス頂点データ用の頂点テクスチャの行数
uniform const bool UseGPUAnime						: register(c8);//GPU表情アニメーションフラグ

//-----------------------------------------------------------------------------
// シャドウマップ
//-----------------------------------------------------------------------------
uniform shared const float2 ShadowMapOffset			: register(c9);//シャドウマップオフセット
const	float	ShadowBias=0.001f;

//-----------------------------------------------------------------------------
// その他マテリアル設定用レジスタ
//-----------------------------------------------------------------------------
uniform const bool UseToon							: register(c11);

//-----------------------------------------------------------------------------
// マトリックス
//-----------------------------------------------------------------------------
// オブジェクトのワールド座標
uniform const float4x4	World						: register(vs, c12);	// 12 - 15
// ビューのトランスフォーム
uniform shared const float4x4	View				: register(vs, c16);	// 16 - 19
// プロジェクションのトランスフォーム
uniform shared const float4x4	Projection			: register(vs, c20);	// 20 - 23
// シャドウマップ用のトランスフォーム
uniform shared const float4x4 ShadowViewProj		: register(vs, c24);	// 24 - 27

//-----------------------------------------------------------------------------
// マテリアルパレット設定用レジスタ
//-----------------------------------------------------------------------------
//レジスタ120個使用。付属モデルはリンモデルの材質が27なので40とれば大体のモデルはいける
#define MaxPalette 40
uniform const float4 DiffuseColorPalette[MaxPalette];//wはAlpha
uniform const float4 EmissiveColorPalette[MaxPalette];//wはToonIndex
uniform const float4 SpecularColorPalette[MaxPalette];//wはSpecularPower

//-----------------------------------------------------------------------------
// テスクチャ
//-----------------------------------------------------------------------------
texture Texture;		// テクスチャ
texture SphereTexture;	// スフィアマップ
shared texture ShadowMap;//シャドウマップ
// トゥーン(アホっぽいけど、材質減らすにはこれしか手がない……)
uniform const texture ToonTexture1;
uniform const texture ToonTexture2;
uniform const texture ToonTexture3;
uniform const texture ToonTexture4;
uniform const texture ToonTexture5;
uniform const texture ToonTexture6;
uniform const texture ToonTexture7;
uniform const texture ToonTexture8;
uniform const texture ToonTexture9;
uniform const texture ToonTexture10;
texture BoneRotationTexture;
texture BoneTranslationTexture;
texture FaceRateTexture;
texture FaceVertTexture;

//-----------------------------------------------------------------------------
// テスクチャサンプラ
//-----------------------------------------------------------------------------
//テクスチャのサンプラー
sampler Sampler = sampler_state
{
    Texture = <Texture>;

    MinFilter = Linear;
    MagFilter = Linear;
    MipFilter = Linear;
};
//スフィアマップのサンプラー
sampler SphereSampler = sampler_state
{
    Texture = <SphereTexture>;

    MinFilter = Linear;
    MagFilter = Linear;
    MipFilter = Linear;
};
sampler ShadowSampler = sampler_state
{
    Texture = <ShadowMap>;

    MipFilter = NONE;
    MinFilter = POINT;
    MagFilter = POINT;
    AddressU  = CLAMP;
    AddressV  = CLAMP;
};
//トゥーンのサンプラー
sampler ToonSampler1 = sampler_state
{
    Texture = <ToonTexture1>;

    MinFilter = Linear;
    MagFilter = Linear;
    MipFilter = Linear;
};
sampler ToonSampler2 = sampler_state
{
    Texture = <ToonTexture2>;

    MinFilter = Linear;
    MagFilter = Linear;
    MipFilter = Linear;
};
sampler ToonSampler3 = sampler_state
{
    Texture = <ToonTexture3>;

    MinFilter = Linear;
    MagFilter = Linear;
    MipFilter = Linear;
};
sampler ToonSampler4 = sampler_state
{
    Texture = <ToonTexture4>;

    MinFilter = Linear;
    MagFilter = Linear;
    MipFilter = Linear;
};
sampler ToonSampler5 = sampler_state
{
    Texture = <ToonTexture5>;

    MinFilter = Linear;
    MagFilter = Linear;
    MipFilter = Linear;
};
sampler ToonSampler6 = sampler_state
{
    Texture = <ToonTexture6>;

    MinFilter = Linear;
    MagFilter = Linear;
    MipFilter = Linear;
};
sampler ToonSampler7 = sampler_state
{
    Texture = <ToonTexture7>;

    MinFilter = Linear;
    MagFilter = Linear;
    MipFilter = Linear;
};
sampler ToonSampler8 = sampler_state
{
    Texture = <ToonTexture8>;

    MinFilter = Linear;
    MagFilter = Linear;
    MipFilter = Linear;
};
sampler ToonSampler9 = sampler_state
{
    Texture = <ToonTexture9>;

    MinFilter = Linear;
    MagFilter = Linear;
    MipFilter = Linear;
};
sampler ToonSampler10 = sampler_state
{
    Texture = <ToonTexture10>;

    MinFilter = Linear;
    MagFilter = Linear;
    MipFilter = Linear;
};
// ボーン用頂点テクスチャサンプラー宣言
sampler BoneRotationSampler : register(vs,s0) = sampler_state
{
	Texture = <BoneRotationTexture>;
	// 殆どのGPUでは以下のようなステート設定にしないと
	// 頂点テクスチャのフェッチがうまくいかない
	MinFilter = Point;
	MagFilter = Point;
	MipFilter = None;
	AddressU = Clamp;
	AddressV = Clamp;
};
sampler BoneTranslationSampler : register(vs,s1) = sampler_state
{
	Texture = <BoneTranslationTexture>;
	// 殆どのGPUでは以下のようなステート設定にしないと
	// 頂点テクスチャのフェッチがうまくいかない
	MinFilter = Point;
	MagFilter = Point;
	MipFilter = None;
	AddressU = Clamp;
	AddressV = Clamp;
};
//表情の適応割合をセットする頂点テクスチャ
//フォーマット：表情番号-割合がfloat配列で格納されている
//毎フレーム転送される
sampler FaceRateSampler : register(vs,s2) = sampler_state
{
	Texture = <FaceRateTexture>;
	// 殆どのGPUでは以下のようなステート設定にしないと
	// 頂点テクスチャのフェッチがうまくいかない
	MinFilter = Point;
	MagFilter = Point;
	MipFilter = None;
	AddressU = Clamp;
	AddressV = Clamp;
};
sampler FaceVertSampler : register(vs,s3) = sampler_state
{
	Texture = <FaceVertTexture>;
	// 殆どのGPUでは以下のようなステート設定にしないと
	// 頂点テクスチャのフェッチがうまくいかない
	MinFilter = Point;
	MagFilter = Point;
	MipFilter = None;
	AddressU = Clamp;
	AddressV = Clamp;
};

//表情のbase頂点番号から移動量が格納されているテクスチャ(CPUアニメーション時)
//表情のbase頂点番号から移動量を計算するまでに必要なデータが格納されているテクスチャ(GPUアニメーション時)
//フォーマット：
//頂点表情セクタのヘッダ(Vector4に割り当て)
//struct VertToFace{
//	float start;//base頂点→表情番号が書かれたフィールドの開始点(Vector4個数単位)
//	float count;//base頂点→表情番号が書かれたフィールドの個数(表情数ではなくVector4の個数)
//  float vstart;//base頂点→表情ごとの移動量が書かれたフィールドの開始点
//  float vcount;//base頂点→表情ごとの移動量が書かれたフィールドの個数(count*4-αと同じになるはずだが)
//};
//頂点→表情番号フィールド(float:Vector4に4個づつ割り当て)
//base頂点→表情ごとの移動量(Vector4。wは0)。indexは頂点→表情番号フィールドと一致


//-----------------------------------------------------------------------------
// 構造体宣言
//=============================================================================
struct ColorPair
{
	float3 Diffuse;
	float3 Specular;
	float2 ToonTex;
};
struct CommonVSOutput
{
	float4 Position;
	float4 Diffuse;
	float4 Specular;
	float3 ToonTexCoord;
	float2 SphereCoord; 
};
struct CommonVSOutputSd
{
	float4 Position;
	float4 Diffuse;
	float4 Specular;
	float3 ToonTexCoord;
	float2 SphereCoord; 
	float4 ShadowColor;
	float4 Pos_Light;
};
// 頂点シェーダー入力構造体
struct VSInput
{
	float4 Position : POSITION0;
    float3 Normal	: NORMAL0;
    float4 BoneIndices : BLENDINDICES0;//ボーン
    float4 BoneWeights : BLENDWEIGHT0;//ボーン
    float2 VectorIndex : TEXCOORD1;//(頂点番号,パレット番号)
};
struct VSInputTx
{
    float4 Position : POSITION0;
    float3 Normal	: NORMAL0;
    float2 TexCoord : TEXCOORD0;
    float4 BoneIndices : BLENDINDICES0;//ボーン
    float4 BoneWeights : BLENDWEIGHT0;//ボーン
    float2 VectorIndex : TEXCOORD1;//(頂点番号,パレット番号)
};

// 頂点シェーダー出力構造体
struct VSOutput
{
	float4	Position	: POSITION;
	float4	Diffuse		: COLOR0;
	float4	Specular	: COLOR1;
	float3	ToonTexCoord: TEXCOORD2;
};
struct VSOutputTx
{
	float4	Position	: POSITION;
	float4	Diffuse		: COLOR0;
	float4	Specular	: COLOR1;
	float2	TexCoord	: TEXCOORD0;
	float3	ToonTexCoord: TEXCOORD2;
};
struct VSOutputSp
{
	float4	Position	: POSITION;
	float4	Diffuse		: COLOR0;
	float4	Specular	: COLOR1;
	float2	SphereCoord	: TEXCOORD1;
	float3	ToonTexCoord: TEXCOORD2;
};
struct VSOutputSd
{
	float4	Position	: POSITION;
	float4	Diffuse		: COLOR0;
	float4	Specular	: COLOR1;
	float4	ShadowColor	: COLOR2;
	float3	ToonTexCoord: TEXCOORD2;
	float4	Pos_Light	: TEXCOORD3;
};
struct VSOutputTxSp
{
	float4	Position	: POSITION;
	float4	Diffuse		: COLOR0;
	float4	Specular	: COLOR1;
	float2	TexCoord	: TEXCOORD0;
	float2	SphereCoord	: TEXCOORD1;
	float3	ToonTexCoord: TEXCOORD2;
};
struct VSOutputTxSd
{
	float4	Position	: POSITION;
	float4	Diffuse		: COLOR0;
	float4	Specular	: COLOR1;
	float4	ShadowColor	: COLOR2;
	float2	TexCoord	: TEXCOORD0;
	float3	ToonTexCoord: TEXCOORD2;
	float4	Pos_Light	: TEXCOORD3;
};
struct VSOutputSpSd
{
	float4	Position	: POSITION;
	float4	Diffuse		: COLOR0;
	float4	Specular	: COLOR1;
	float4	ShadowColor	: COLOR2;
	float2	SphereCoord	: TEXCOORD1;
	float3	ToonTexCoord: TEXCOORD2;
	float4	Pos_Light	: TEXCOORD3;
};
struct VSOutputTxSpSd
{
	float4	Position	: POSITION;
	float4	Diffuse		: COLOR0;
	float4	Specular	: COLOR1;
	float4	ShadowColor	: COLOR2;
	float2	TexCoord	: TEXCOORD0;
	float2	SphereCoord	: TEXCOORD1;
	float3	ToonTexCoord: TEXCOORD2;
	float4	Pos_Light	: TEXCOORD3;
};
//シャドウマップ用頂点シェーダー出力構造体
struct VS_SHADOW_OUTPUT
{
	float4	Position	: POSITION;
	float4	Pos_Light	: TEXCOORD0;
};
// ピクセルシェーダー入力構造体
struct PSInput
{
	float4	Diffuse		: COLOR0;
	float4	Specular	: COLOR1;
	float3	ToonTexCoord: TEXCOORD2;
};
struct PSInputTx
{
	float4	Diffuse		: COLOR0;
	float4	Specular	: COLOR1;
	float2	TexCoord	: TEXCOORD0;
	float3	ToonTexCoord: TEXCOORD2;
};
struct PSInputSp
{
	float4	Diffuse		: COLOR0;
	float4	Specular	: COLOR1;
	float2	SphereCoord	: TEXCOORD1;
	float3	ToonTexCoord: TEXCOORD2;
};
struct PSInputSd
{
	float4	Diffuse		: COLOR0;
	float4	Specular	: COLOR1;
	float4	ShadowColor	: COLOR2;
	float3	ToonTexCoord: TEXCOORD2;
	float4	Pos_Light	: TEXCOORD3;
};
struct PSInputTxSp
{
	float4	Diffuse		: COLOR0;
	float4	Specular	: COLOR1;
	float2	TexCoord	: TEXCOORD0;
	float2	SphereCoord	: TEXCOORD1;
	float3	ToonTexCoord: TEXCOORD2;
};
struct PSInputTxSd
{
	float4	Diffuse		: COLOR0;
	float4	Specular	: COLOR1;
	float4	ShadowColor	: COLOR2;
	float2	TexCoord	: TEXCOORD0;
	float3	ToonTexCoord: TEXCOORD2;
	float4	Pos_Light	: TEXCOORD3;
};
struct PSInputSpSd
{
	float4	Diffuse		: COLOR0;
	float4	Specular	: COLOR1;
	float4	ShadowColor	: COLOR2;
	float2	SphereCoord	: TEXCOORD1;
	float3	ToonTexCoord: TEXCOORD2;
	float4	Pos_Light	: TEXCOORD3;
};
struct PSInputTxSpSd
{
	float4	Diffuse		: COLOR0;
	float4	Specular	: COLOR1;
    float4	ShadowColor	: COLOR2;
	float2	TexCoord	: TEXCOORD0;
    float2	SphereCoord	: TEXCOORD1;
    float3	ToonTexCoord: TEXCOORD2;
    float4	Pos_Light	: TEXCOORD3;
};
//-----------------------------------------------------------------------------
// クォータニオンヘルパーメソッド
//=============================================================================
// 2つクォータニオンと平行移動から行列に変換する
// ひにけにさんのサンプルをMMD用に調整したもの
float4x4 CreateTransformFromQuaternionTransforms(
		float4 q1, float3 t1,
		float4 q2, float3 t2,
		float2 weights )
{
	float ww = q1.w * q1.w - 0.5f;
	float3 row10 = float3( ww         , q1.x * q1.y, q1.x * q1.z ) +
				   float3( q1.x * q1.x, q1.w * q1.z,-q1.w * q1.y );
	float3 row11 = float3( q1.x * q1.y, ww,          q1.y * q1.z ) +
	               float3(-q1.w * q1.z, q1.y * q1.y, q1.w * q1.x );
	float3 row12 = float3( q1.x * q1.z, q1.y * q1.z, ww          ) +
	               float3( q1.w * q1.y,-q1.w * q1.x, q1.z * q1.z );
	
	ww = q2.w * q2.w - 0.5f;
	float3 row20 = float3( ww,          q2.x * q2.y, q2.x * q2.z ) +
	               float3( q2.x * q2.x, q2.w * q2.z,-q2.w * q2.y );
	float3 row21 = float3( q2.x * q2.y, ww,          q2.y * q2.z ) +
	               float3(-q2.w * q2.z, q2.y * q2.y, q2.w * q2.x );
	float3 row22 = float3( q2.x * q2.z, q2.y * q2.z, ww          ) +
	               float3( q2.w * q2.y,-q2.w * q2.x, q2.z * q2.z );
	              
	float2 w2 = 2.0f * weights;
	
	return float4x4(
		row10 * w2.x + row20 * w2.y, 0,
		row11 * w2.x + row21 * w2.y, 0,
		row12 * w2.x + row22 * w2.y, 0, 
		t1 * weights.x + t2 * weights.y, 1
	);
	
}

//-----------------------------------------------------------------------------
// 頂点テクスチャからボーン情報のフェッチ(MMDの影響ボーンは2つまで)
//=============================================================================
float4x4 CreateTransformFromBoneTexture( float2 boneIndices, float2 boneWeights )
{
	float2 uv = 1.0f / BoneTextureSize;
	uv.y *= 0.5f;
	float4 texCoord0 = float4( ( 0.5f + boneIndices.x ) * uv.x, uv.y, 0, 1 );
	float4 texCoord1 = float4( ( 0.5f + boneIndices.y ) * uv.x, uv.y, 0, 1 );
	
	// 回転部分のフェッチ
	float4 q1 = tex2Dlod( BoneRotationSampler, texCoord0 );
	float4 q2 = tex2Dlod( BoneRotationSampler, texCoord1 );
	
	// 平行移動部分のフェッチ
	float4 t1 = tex2Dlod( BoneTranslationSampler, texCoord0 );
	float4 t2 = tex2Dlod( BoneTranslationSampler, texCoord1 );
	
	return CreateTransformFromQuaternionTransforms(
					q1, t1,
					q2, t2,
					boneWeights );
}

//-----------------------------------------------------------------------------
// 頂点テクスチャから表情情報のフェッチ及び計算
//=============================================================================
float4 GetFVTexCoord(float pos,float2 uv)
{
	return float4(fmod(pos+0.5f,FaceVertTextureColLines.x)* uv.x,(0.5f+floor(pos*FaceVertTextureColLines.y))* uv.y, 0, 1 );
}
float4 InnerCalcFace(int faceNum,float vertPos,float2 uv,float2 uv2)
{
	float4 result=0;
	if(faceNum>=0)
	{//余り部分には-1が入っている
		//faceNumに表情番号が入っている
		//ここから表情量を取得
		float4 texCoord2=float4((0.5f+faceNum) * uv2.x, uv2.y, 0, 1 );
		float4 faceRate=tex2Dlod(FaceRateSampler,texCoord2);
		if(faceRate.x>0)
		{
			//次に対応する頂点の移動量をフェッチ
			//vertPosの位置に来るようになっている
			float4 texCoord3=GetFVTexCoord(vertPos,uv);
			float4 faceMove=tex2Dlod( FaceVertSampler, texCoord3 );
			//表情の適応量を掛け合わせて加える
			result+=faceMove*faceRate.x;
		}
	}
	
	return result;
}
float4 CalcFace( float VectorIndex)
{
	//FaceVertTextureのVector4 1データの目盛取得(複数行形式)
	float2 uv = 1.0f / FaceVertTextureSize;
	//頂点番号から(表情が書かれたエリア/表情頂点位置)の取得
	//Vector4ごとにに2つ含まれている(GPU)
	float4 texCoord0=GetFVTexCoord(VectorIndex,uv);
	// 関連表情データエリア部分/頂点位置のフェッチ
	float4 fa = tex2Dlod( FaceVertSampler, texCoord0 );
	float4 result=float4(0,0,0,0);
	if(UseGPUAnime==true)
	{//GPU処理時はシェーダ内で計算する
		//FaceRateTextureのfloat 1データの目盛取得
		float2 uv2=1.0f/FaceRateTextureSize;
		uv2.y*=0.5f;
		
		//表情ごとに処理
		for(int i=0;i<fa.y;i++)
		{
			float4 texCoord1=GetFVTexCoord(i+fa.x,uv);
			//表情番号データエリア部分のフェッチ
			float4 faces=tex2Dlod( FaceVertSampler, texCoord1 );
			result+=InnerCalcFace(faces.x,fa.z+i*4.0f,uv,uv2);
			result+=InnerCalcFace(faces.y,fa.z+i*4.0f+1,uv,uv2);
			result+=InnerCalcFace(faces.z,fa.z+i*4.0f+2,uv,uv2);
			result+=InnerCalcFace(faces.w,fa.z+i*4.0f+3,uv,uv2);
			
		}
	}
	else
	{//CPU処理時は処理済みデータが格納されている
		result=fa;
	}
	return result;
}


//MMDの色計算式
//DiffuseRGB出力=マテリアルディフューズ*環境光アンビエント
//				+マテリアルエミッシブ
//DiffuseA出力=マテリアルα
//SpecularRGB出力=マテリアルスペキュラ*環境光アンビエント*pow(dot(法線,ハーフベクトル),マテリアルShininess) (但し dot(法線,ライト)>0。そうでない場合は0)
//出力色=(Diffuse*テクスチャ色+スペキュラ)
//		*or+スフィアマップ色
//		*トゥーン色
//		テクスチャ、スフィア、トゥーンは場合によって存在しない

//-----------------------------------------------------------------------------
// Compute lighting
// E: Eye-Vector
// N: Unit vector normal in world space
//-----------------------------------------------------------------------------
ColorPair ComputeLights(float3 E, float3 N,float PalleteNum)
{
	ColorPair result;
	
	result.Diffuse = AmbientLightColor;
	result.Specular = 0;
	// Directional Light 0
	float3 L = normalize(-DirLight0Direction);
	float3 H = normalize(E+L);
	float2 ret = lit(dot(N, L), dot(N, H), SpecularColorPalette[PalleteNum].w).yz;//VectorIndex.y=パレット番号
	result.Specular+=AmbientLightColor*ret.y;
	
	//MMDではEmissiveを足してからsaturateするのが正解らしい。
	result.Diffuse *= DiffuseColorPalette[PalleteNum].xyz;
	result.Diffuse	+= EmissiveColorPalette[PalleteNum].xyz;
	result.Diffuse	= saturate(result.Diffuse);
	result.Specular	*= SpecularColorPalette[PalleteNum].xyz;
	
	//トゥーンテクスチャ用のサンプル位置を計算
	result.ToonTex.x=clamp(0.5f-dot(normalize(N),normalize(E))*0.5f,0,1);
	result.ToonTex.y=clamp(0.5f-dot(normalize(N),normalize(L))*0.5f,0,1);
	
	return result;
}
//-----------------------------------------------------------------------------
// 共用関数
//=============================================================================
CommonVSOutput ComputeCommonVSOutput(float4 Position, float3 Normal, float2 BoneIndices, float2 BoneWeights, float2 VectorIndex)
{
	CommonVSOutput output;
	// スキン変換行列の取得
    float4x4 skinTransform =
				CreateTransformFromBoneTexture( BoneIndices, BoneWeights );
	//スキン変換
	skinTransform = mul( skinTransform, World );
	
	//表情移動分の取得
	float4 faceTransform=float4(0,0,0,0);
	if(VectorIndex.x>=0)
		faceTransform=CalcFace(VectorIndex.x);
	//表情による頂点移動を合算
	float4 facePos=Position+faceTransform;
	
    // 頂点変換
    float4 position = mul(facePos, skinTransform);
    output.Position = mul(mul(position, View), Projection);
    // 法線変換
    float3 normal = normalize( mul( Normal, skinTransform));
    //視線ベクトル取得
    float3 posToEye = EyePosition - position;
    float3 E = normalize(posToEye);
    //視線ベクトルと法線を元にライトを計算
    ColorPair lightResult = ComputeLights(E, normal,VectorIndex.y);
    //色計算
    output.Diffuse = float4(lightResult.Diffuse.rgb, DiffuseColorPalette[VectorIndex.y].w);//DColor.w=Alpha
    output.Specular = float4(lightResult.Specular,1);
    //トゥーンテクスチャ取得位置をコピー
	output.ToonTexCoord=float3(lightResult.ToonTex.x,lightResult.ToonTex.y,EmissiveColorPalette[VectorIndex.y].w);
	//スフィア計算
	output.SphereCoord=float2(normal.x/2+0.5,normal.y/2+0.5);
	
	return output;
}
CommonVSOutputSd ComputeCommonVSOutputSd(float4 Position, float3 Normal, float2 BoneIndices, float2 BoneWeights, float2 VectorIndex)
{
	CommonVSOutputSd output;
	// スキン変換行列の取得
    float4x4 skinTransform =
				CreateTransformFromBoneTexture( BoneIndices, BoneWeights );
	//スキン変換
	skinTransform = mul( skinTransform, World );
	
	//表情移動分の取得
	float4 faceTransform=float4(0,0,0,0);
	if(VectorIndex.x>=0)
		faceTransform=CalcFace(VectorIndex.x);
	//表情による頂点移動を合算
	float4 facePos=Position+faceTransform;
	
    // 頂点変換
    float4 position = mul(facePos, skinTransform);
    output.Position = mul(mul(position, View), Projection);
    // 法線変換
    float3 normal = normalize( mul( Normal, skinTransform));
    //視線ベクトル取得
    float3 posToEye = EyePosition - position;
    float3 E = normalize(posToEye);
    //視線ベクトルと法線を元にライトを計算
    ColorPair lightResult = ComputeLights(E, normal,VectorIndex.y);
    //色計算
    output.Diffuse = float4(lightResult.Diffuse.rgb, DiffuseColorPalette[VectorIndex.y].w);//DColor.w=Alpha
    output.Specular = float4(lightResult.Specular,1);
    //トゥーンテクスチャ取得位置をコピー
	output.ToonTexCoord=float3(lightResult.ToonTex.x,lightResult.ToonTex.y,EmissiveColorPalette[VectorIndex.y].w);
	//スフィア計算
	output.SphereCoord=float2(normal.x/2+0.5,normal.y/2+0.5);
	//シャドウマップ計算
	output.Pos_Light=mul(position,ShadowViewProj);
	output.ShadowColor=float4(DiffuseColorPalette[VectorIndex.y].xyz,1);
	return output;
}
//-----------------------------------------------------------------------------
// 頂点シェーダー
//=============================================================================
VSOutput VSBasic(VSInput input)
{
	VSOutput output;
	CommonVSOutput vout = ComputeCommonVSOutput
					(input.Position,input.Normal,input.BoneIndices.xy, input.BoneWeights.xy,input.VectorIndex);
	
	output.Position = vout.Position;
    output.Diffuse = vout.Diffuse;
    output.Specular = vout.Specular;
    output.ToonTexCoord = vout.ToonTexCoord;
	return output;
}
VSOutputTx VSBasicTx(VSInputTx input)
{
	VSOutputTx output;
	CommonVSOutput vout = ComputeCommonVSOutput
					(input.Position,input.Normal,input.BoneIndices.xy, input.BoneWeights.xy,input.VectorIndex);
	
	output.Position = vout.Position;
    output.Diffuse = vout.Diffuse;
    output.Specular = vout.Specular;
    output.ToonTexCoord = vout.ToonTexCoord;
	output.TexCoord = input.TexCoord;
	return output;
}
VSOutputSp VSBasicSp(VSInput input)
{
	VSOutputSp output;
	CommonVSOutput vout = ComputeCommonVSOutput
					(input.Position,input.Normal,input.BoneIndices.xy, input.BoneWeights.xy,input.VectorIndex);
	
	output.Position = vout.Position;
    output.Diffuse = vout.Diffuse;
    output.Specular = vout.Specular;
    output.ToonTexCoord = vout.ToonTexCoord;
    output.SphereCoord=vout.SphereCoord;
	return output;
}
VSOutputTxSp VSBasicTxSp(VSInputTx input)
{
	VSOutputTxSp output;
	CommonVSOutput vout = ComputeCommonVSOutput
					(input.Position,input.Normal,input.BoneIndices.xy, input.BoneWeights.xy,input.VectorIndex);
	
	output.Position = vout.Position;
    output.Diffuse = vout.Diffuse;
    output.Specular = vout.Specular;
    output.ToonTexCoord = vout.ToonTexCoord;
	output.TexCoord = input.TexCoord;
	output.SphereCoord=vout.SphereCoord;
	return output;
}
VSOutputSd VSBasicSd(VSInput input)
{
	VSOutputSd output;
	CommonVSOutputSd vout = ComputeCommonVSOutputSd
					(input.Position,input.Normal,input.BoneIndices.xy, input.BoneWeights.xy,input.VectorIndex);
	
	output.Position = vout.Position;
    output.Diffuse = vout.Diffuse;
    output.Specular = vout.Specular;
    output.ToonTexCoord = vout.ToonTexCoord;
    output.Pos_Light=vout.Pos_Light;
	output.ShadowColor=vout.ShadowColor;
	return output;
}
VSOutputTxSd VSBasicTxSd(VSInputTx input)
{
	VSOutputTxSd output;
	CommonVSOutputSd vout = ComputeCommonVSOutputSd
					(input.Position,input.Normal,input.BoneIndices.xy, input.BoneWeights.xy,input.VectorIndex);
	
	output.Position = vout.Position;
    output.Diffuse = vout.Diffuse;
    output.Specular = vout.Specular;
    output.ToonTexCoord = vout.ToonTexCoord;
	output.TexCoord = input.TexCoord;
	output.Pos_Light=vout.Pos_Light;
	output.ShadowColor=vout.ShadowColor;
	return output;
}
VSOutputSpSd VSBasicSpSd(VSInput input)
{
	VSOutputSpSd output;
	CommonVSOutputSd vout = ComputeCommonVSOutputSd
					(input.Position,input.Normal,input.BoneIndices.xy, input.BoneWeights.xy,input.VectorIndex);
	
	output.Position = vout.Position;
    output.Diffuse = vout.Diffuse;
    output.Specular = vout.Specular;
    output.ToonTexCoord = vout.ToonTexCoord;
    output.SphereCoord=vout.SphereCoord;
	output.Pos_Light=vout.Pos_Light;
	output.ShadowColor=vout.ShadowColor;
	return output;
}
VSOutputTxSpSd VSBasicTxSpSd(VSInputTx input)
{
    VSOutputTxSpSd output;
    
    CommonVSOutputSd vout=ComputeCommonVSOutputSd
    				(input.Position,input.Normal,input.BoneIndices.xy, input.BoneWeights.xy,input.VectorIndex);
	
    output.Position = vout.Position;
    output.Diffuse = vout.Diffuse;
    output.Specular = vout.Specular;
    output.ToonTexCoord = vout.ToonTexCoord;
	output.TexCoord = input.TexCoord;
	if(UseSphere>0)
		output.SphereCoord=vout.SphereCoord;//仮
	else
		output.SphereCoord=0;
    //光源から見た頂点位置(シャドウマップ参照用)
	output.Pos_Light=vout.Pos_Light;
	output.ShadowColor=vout.ShadowColor;
	return output;
}

VS_SHADOW_OUTPUT VSShadow(VSInputTx input)
{
    VS_SHADOW_OUTPUT output;
    
    // スキン変換行列の取得
    float4x4 skinTransform =
				CreateTransformFromBoneTexture( input.BoneIndices, input.BoneWeights );
	//スキン変換
	skinTransform = mul( skinTransform, World );
	
	//表情移動分の取得
	float4 faceTransform=float4(0,0,0,0);
	if(input.VectorIndex.x>=-0.001f)
		faceTransform=CalcFace(input.VectorIndex.x);
	
	//表情による頂点移動を合算
	float4 facePos=input.Position+faceTransform;
	
    // 頂点変換
    float4 position = mul(facePos, skinTransform);
    output.Position = mul(mul(position, View), Projection);
	
	//深度出力
	output.Pos_Light=output.Position;
    
	return output;
}
//-----------------------------------------------------------------------------
// ピクセルシェーダー
//=============================================================================
float4 ApplyToon(float4 color, float3 ToonTexCoord)
{
	if(ToonTexCoord.z==0)
		color*=tex2D(ToonSampler1,ToonTexCoord.xy);
	else if(ToonTexCoord.z==1)
		color*=tex2D(ToonSampler2,ToonTexCoord.xy);
	else if(ToonTexCoord.z==2)
		color*=tex2D(ToonSampler3,ToonTexCoord.xy);
	else if(ToonTexCoord.z==3)
		color*=tex2D(ToonSampler4,ToonTexCoord.xy);
	else if(ToonTexCoord.z==4)
		color*=tex2D(ToonSampler5,ToonTexCoord.xy);
	else if(ToonTexCoord.z==5)
		color*=tex2D(ToonSampler6,ToonTexCoord.xy);
	else if(ToonTexCoord.z==6)
		color*=tex2D(ToonSampler7,ToonTexCoord.xy);
	else if(ToonTexCoord.z==7)
		color*=tex2D(ToonSampler8,ToonTexCoord.xy);
	else if(ToonTexCoord.z==8)
		color*=tex2D(ToonSampler9,ToonTexCoord.xy);
	else if(ToonTexCoord.z==9)
		color*=tex2D(ToonSampler10,ToonTexCoord.xy);
	return color;
}
float4 PSBasic(PSInput pin) : COLOR
{
	float4 color;
	color = pin.Diffuse + float4(pin.Specular.rgb, 0);
	//トゥーンテクスチャの適応
	if(UseToon)
		color=ApplyToon(color, pin.ToonTexCoord);
	return color;
}
float4 PSBasicTx(PSInputTx pin) : COLOR
{
	float4 color;
	color = tex2D(Sampler, pin.TexCoord) * pin.Diffuse + float4(pin.Specular.rgb, 0);
	//トゥーンテクスチャの適応
	if(UseToon)
		color=ApplyToon(color, pin.ToonTexCoord);
	return color;
}
float4 PSBasicSp(PSInputSp pin) : COLOR
{
	float4 color;
	color = pin.Diffuse + float4(pin.Specular.rgb, 0);
	//スフィアマップ
	if(UseSphere==1)
		color*=tex2D(SphereSampler,pin.SphereCoord);//スフィアマップを乗算
	else if(UseSphere==2)//スフィアマップを加算
		color+=tex2D(SphereSampler,pin.SphereCoord);
	//トゥーンテクスチャの適応
	if(UseToon)
		color=ApplyToon(color, pin.ToonTexCoord);
	return color;
}
float4 PSBasicTxSp(PSInputTxSp pin) : COLOR
{
	float4 color;
	color = tex2D(Sampler, pin.TexCoord) * pin.Diffuse + float4(pin.Specular.rgb, 0);
	//スフィアマップ
	if(UseSphere==1)
		color*=tex2D(SphereSampler,pin.SphereCoord);//スフィアマップを乗算
	else if(UseSphere==2)//スフィアマップを加算
		color+=tex2D(SphereSampler,pin.SphereCoord);
	//トゥーンテクスチャの適応
	if(UseToon)
		color=ApplyToon(color, pin.ToonTexCoord);
	return color;
}
float4 PSBasicSd(PSInputSd pin) : COLOR
{
	bool IsShadow=false;
	//シャドウマップ(α値が0.98でもシャドウマップが描画されないらしい: http://www.nicovideo.jp/watch/sm9395644)
	if(abs(pin.Diffuse.a-0.98f)>0.001f)
	{
		//シャドウマップ取得位置計算
		float2 shadow_uv=0.5*pin.Pos_Light.xy/pin.Pos_Light.w+float2(0.5f,0.5f);
		shadow_uv.y=1.0f-shadow_uv.y;
		shadow_uv+=ShadowMapOffset;
		//シャドウマップから光視点の深度情報を取得
		float shadow_val=tex2D(ShadowSampler,shadow_uv).r;
		//影判定。
		float depth=pin.Pos_Light.z/pin.Pos_Light.w;
		//原理は光から見た現在の深度情報が、シャドウマップの深度情報より大きいなら、
		//それはシャドウマップ描画時により光から見た深度が浅い＝光から見て手前にあるピクセルに上書きされている
		//上書きされている→他のピクセルが光から見て手前にある→影となる。
		if(depth-shadow_val>ShadowBias)
		{
			IsShadow=true;
			//トゥーンテクスチャの取得位置を左下のピクセルにする(解説動画より:http://www.nicovideo.jp/watch/sm9395644)
			if(pin.ToonTexCoord.z>=0)
			{
				pin.ToonTexCoord.x=0;
				pin.ToonTexCoord.y=1;
			}
		}
	}
	float4 color;
	if(IsShadow)
		color = pin.ShadowColor;
	else
		color = pin.Diffuse + float4(pin.Specular.rgb, 0);
	//トゥーンテクスチャの適応
	if(UseToon)
		color=ApplyToon(color, pin.ToonTexCoord);
	return color;
}
float4 PSBasicTxSd(PSInputTxSd pin) : COLOR
{
	bool IsShadow=false;
	//シャドウマップ(α値が0.98でもシャドウマップが描画されないらしい: http://www.nicovideo.jp/watch/sm9395644)
	if(abs(pin.Diffuse.a-0.98f)>0.001f)
	{
		//シャドウマップ取得位置計算
		float2 shadow_uv=0.5*pin.Pos_Light.xy/pin.Pos_Light.w+float2(0.5f,0.5f);
		shadow_uv.y=1.0f-shadow_uv.y;
		shadow_uv+=ShadowMapOffset;
		//シャドウマップから光視点の深度情報を取得
		float shadow_val=tex2D(ShadowSampler,shadow_uv).r;
		//影判定。
		float depth=pin.Pos_Light.z/pin.Pos_Light.w;
		//原理は光から見た現在の深度情報が、シャドウマップの深度情報より大きいなら、
		//それはシャドウマップ描画時により光から見た深度が浅い＝光から見て手前にあるピクセルに上書きされている
		//上書きされている→他のピクセルが光から見て手前にある→影となる。
		if(depth-shadow_val>ShadowBias)
		{
			IsShadow=true;
			//トゥーンテクスチャの取得位置を左下のピクセルにする(解説動画より:http://www.nicovideo.jp/watch/sm9395644)
			if(pin.ToonTexCoord.z>=0)
			{
				pin.ToonTexCoord.x=0;
				pin.ToonTexCoord.y=1;
			}
		}
	}
	float4 color;
	if(IsShadow)
		color = tex2D(Sampler, pin.TexCoord) * pin.ShadowColor;
	else
		color = tex2D(Sampler, pin.TexCoord) * pin.Diffuse + float4(pin.Specular.rgb, 0);
	//トゥーンテクスチャの適応
	if(UseToon)
		color=ApplyToon(color, pin.ToonTexCoord);
	return color;
}
float4 PSBasicSpSd(PSInputSpSd pin) : COLOR
{
	bool IsShadow=false;
	//シャドウマップ(α値が0.98でもシャドウマップが描画されないらしい: http://www.nicovideo.jp/watch/sm9395644)
	if(abs(pin.Diffuse.a-0.98f)>0.001f)
	{
		//シャドウマップ取得位置計算
		float2 shadow_uv=0.5*pin.Pos_Light.xy/pin.Pos_Light.w+float2(0.5f,0.5f);
		shadow_uv.y=1.0f-shadow_uv.y;
		shadow_uv+=ShadowMapOffset;
		//シャドウマップから光視点の深度情報を取得
		float shadow_val=tex2D(ShadowSampler,shadow_uv).r;
		//影判定。
		float depth=pin.Pos_Light.z/pin.Pos_Light.w;
		//原理は光から見た現在の深度情報が、シャドウマップの深度情報より大きいなら、
		//それはシャドウマップ描画時により光から見た深度が浅い＝光から見て手前にあるピクセルに上書きされている
		//上書きされている→他のピクセルが光から見て手前にある→影となる。
		if(depth-shadow_val>ShadowBias)
		{
			IsShadow=true;
			//トゥーンテクスチャの取得位置を左下のピクセルにする(解説動画より:http://www.nicovideo.jp/watch/sm9395644)
			if(pin.ToonTexCoord.z>=0)
			{
				pin.ToonTexCoord.x=0;
				pin.ToonTexCoord.y=1;
			}
		}
	}
	float4 color;
	if(IsShadow)
		color = pin.ShadowColor;
	else
		color = pin.Diffuse + float4(pin.Specular.rgb, 0);
	//スフィアマップ
	if(UseSphere==1)
		color*=tex2D(SphereSampler,pin.SphereCoord);//スフィアマップを乗算
	else if(UseSphere==2)//スフィアマップを加算
		color+=tex2D(SphereSampler,pin.SphereCoord);
	//トゥーンテクスチャの適応
	if(UseToon)
		color=ApplyToon(color, pin.ToonTexCoord);
	return color;
}
float4 PSBasicTxSpSd(PSInputTxSpSd pin) : COLOR
{
	bool IsShadow=false;
	//シャドウマップ(α値が0.98でもシャドウマップが描画されないらしい: http://www.nicovideo.jp/watch/sm9395644)
	if(abs(pin.Diffuse.a-0.98f)>0.001f)
	{
		//シャドウマップ取得位置計算
		float2 shadow_uv=0.5*pin.Pos_Light.xy/pin.Pos_Light.w+float2(0.5f,0.5f);
		shadow_uv.y=1.0f-shadow_uv.y;
		shadow_uv+=ShadowMapOffset;
		//シャドウマップから光視点の深度情報を取得
		float shadow_val=tex2D(ShadowSampler,shadow_uv).r;
		//影判定。
		float depth=pin.Pos_Light.z/pin.Pos_Light.w;
		//原理は光から見た現在の深度情報が、シャドウマップの深度情報より大きいなら、
		//それはシャドウマップ描画時により光から見た深度が浅い＝光から見て手前にあるピクセルに上書きされている
		//上書きされている→他のピクセルが光から見て手前にある→影となる。
		if(depth-shadow_val>ShadowBias)
		{
			IsShadow=true;
			//トゥーンテクスチャの取得位置を左下のピクセルにする(解説動画より:http://www.nicovideo.jp/watch/sm9395644)
			if(pin.ToonTexCoord.z>=0)
			{
				pin.ToonTexCoord.x=0;
				pin.ToonTexCoord.y=1;
			}
		}
	}
	float4 color;
	//通常の色計算
	if(IsShadow)
		color = tex2D(Sampler, pin.TexCoord) * pin.ShadowColor;
	else
		color = tex2D(Sampler, pin.TexCoord) * pin.Diffuse + float4(pin.Specular.rgb, 0);
	//スフィアマップ
	if(UseSphere==1)
		color*=tex2D(SphereSampler,pin.SphereCoord);//スフィアマップを乗算
	else if(UseSphere==2)
		color+=tex2D(SphereSampler,pin.SphereCoord);//スフィアマップを加算
	//トゥーンテクスチャの適応
	if(UseToon)
		color=ApplyToon(color, pin.ToonTexCoord);
	return color;
}
//シャドウマップ用
float4 PSShadow(float4 Pos_Light : TEXCOORD0) : COLOR
{
	float depth=Pos_Light.z/Pos_Light.w;
	return float4(depth,depth,depth,1);
}
//-----------------------------------------------------------------------------
// 描画手法の宣言
//=============================================================================
int ShaderIndex = 0;

VertexShader VSArray[4] =
{
	compile vs_3_0 VSBasic(),
	compile vs_3_0 VSBasicTx(),
	compile vs_3_0 VSBasicSp(),
	compile vs_3_0 VSBasicTxSp(),
};

PixelShader PSArray[4] =
{
	compile ps_3_0 PSBasic(),
	compile ps_3_0 PSBasicTx(),
	compile ps_3_0 PSBasicSp(),
	compile ps_3_0 PSBasicTxSp(),
};
VertexShader VSArraySd[4] =
{
	compile vs_3_0 VSBasicSd(),
	compile vs_3_0 VSBasicTxSd(),
	compile vs_3_0 VSBasicSpSd(),
	compile vs_3_0 VSBasicTxSpSd(),
};

PixelShader PSArraySd[4] =
{
	compile ps_3_0 PSBasicSd(),
	compile ps_3_0 PSBasicTxSd(),
	compile ps_3_0 PSBasicSpSd(),
	compile ps_3_0 PSBasicTxSpSd(),
};

Technique MMDBasicEffect
{
	Pass
	{
		VertexShader = (VSArray[ShaderIndex]);
		PixelShader	 = (PSArray[ShaderIndex]);
	}
}
Technique MMDBasicEffectWithShadow
{
	Pass
	{
		VertexShader = (VSArraySd[ShaderIndex]);
		PixelShader	 = (PSArraySd[ShaderIndex]);
	}
}

//シャドウマップ用
Technique MMDShadowMap
{
	Pass
	{
		VertexShader = (compile vs_3_0 VSShadow());
		PixelShader  = (compile ps_3_0 PSShadow());
	}
}