#!/usr/bin/ruby -Ku
#
# mmon
#
# Copyright (c) 2007 sanpo
#
# This program is free software.
# You can redistribute it and/or modify it under the terms of the GPL.

require 'mmon/item'
require 'mmon/util'

class ItemVol < Item
    # linux/soundcard.h
    SOUND_MIXER_VOLUME = 0
    SOUND_MASK_VOLUME = 1 << SOUND_MIXER_VOLUME

    # man ioctl_list
    SOUND_MIXER_READ_STEREODEVS = 0x80044DFB
    SOUND_MIXER_READ_DEVMASK    = 0x80044DFE

    SOUND_MIXER_READ_VOLUME     = 0x80044D00  
    SOUND_MIXER_WRITE_VOLUME    = 0xC0044D00

    NAME = 'Volume'

    @@color_knob = Color.new(1.0, 0.0, 0.0, 1.0)
    @@color_knob_mute = Color.new(0.5, 0.0, 0.0, 1.0)

    def initialize(mmon)
        super(mmon)

        @interval = 1.0
        @use_bg = true
        @use_fg = true

        @width = 72
        @height = 20

        self.add_events(Gdk::Event::Mask::BUTTON_PRESS_MASK | Gdk::Event::Mask::SCROLL_MASK)
		signal_connect("button-press-event")    { |w, e| button_press_event(e) }
		signal_connect("scroll-event")          { |w, e| scroll_event(e) }

        @vol = 0
        @mute = false
    end

    private

    def render_mask(cc, w, h)
        Util.simple_bg(cc, w, h)
        Util.simple_fg(cc, w, h)
    end

    def render_bg(cc, w, h)
        Util.simple_bg(cc, w, h)
    end

    def render_fg(cc, w, h)
        Util.simple_fg(cc, w, h)
    end

    def render(cc, w, h)
        #puts "vol render vol:#{@vol}"

        l = 2.0

        x = l + (w - 2.0 * l) * @vol / 100

        cc.antialias = Cairo::ANTIALIAS_NONE

        cc.rectangle(x - 2, 0, 5, h)
        if @mute 
            cc.set_source_rgba(@@color_knob_mute)
        else
            cc.set_source_rgba(@@color_knob)
        end
        cc.fill
    end

    def prepare
        return false if @mute

        ret = false

        begin
            f = open("/dev/mixer")
            new_vol = read_volume(f)
            if @vol == new_vol 
                retval = false
            else
                @vol = new_vol
                retval = true
            end
        rescue
            p $!
        ensure
            f.close
        end
        
        return retval
    end

    def read_volume(fd)
        devmask = read_int(fd, SOUND_MIXER_READ_DEVMASK)
        if devmask | SOUND_MASK_VOLUME
            level = read_int(fd, SOUND_MIXER_READ_VOLUME)

            left = level & 0xff
            right = (level & 0xff00) >> 8
            #puts "l:#{left} r:#{right}"

            volume = level & 0xff
        end

        return volume
    end

    def write_volume(volume)
        volume = 100 if volume > 100
        volume = 0 if volume < 0

        begin
            f = open("/dev/mixer")
            left = volume
            right = volume << 8
            stereo = right | left
            #puts "l:#{left} r:#{right}  #{stereo}"

            a = []
            a[0] = stereo
            arg = a.pack("i")

            f.ioctl(SOUND_MIXER_WRITE_VOLUME, arg)
        rescue
            p $!
        ensure
            f.close
        end
    end

    def read_int(fd, request)
        begin
            arg = [0].pack("i")
            fd.ioctl(request, arg)
            retval = arg.unpack("i")[0]
        rescue
            p $!
            retval = 0
            raise
        end

        return retval
    end

    def button_press_event(e)
        if e.button == 1 && e.event_type == Gdk::Event::Type::BUTTON2_PRESS 
            if @mute 
                @mute = false
                write_volume(@vol)
            else
                @mute = true
                write_volume(0)
            end

            self.queue_draw
            
            return true
        end

        return false
    end

    def scroll_event(e)
        case e.direction
        when Gdk::EventScroll::Direction::UP
            @vol += 5
            @vol = 100 if @vol > 100
            write_volume(@vol) unless @mute
            self.queue_draw()
            return true
        when Gdk::EventScroll::Direction::DOWN
            @vol -= 5
            @vol = 0 if @vol < 0
            write_volume(@vol) unless @mute
            self.queue_draw()
            return true
        end

        return false
    end
end

if __FILE__ == $0
    require 'mmon/appmmon'

    app = AppMmon.new
    item = ItemVol.new(app)
    app.add_item(item)

    Gtk::main()
end
