#!/usr/bin/ruby -Ku
#
# mmon
#
# Copyright (c) 2007 sanpo
#
# This program is free software.
# You can redistribute it and/or modify it under the terms of the GPL.

require 'gtk2'
require 'gtkglext'
require 'rsvg2'
require 'rexml/document'

require 'mmon/color'

module GL
    TEXTURE_RECTANGLE_ARB = 0x84F5
end


class GLItem < Gtk::Window
    attr_reader :item_name

    def initialize(mmon)
        super(Gtk::Window::TOPLEVEL)

        @mmon = mmon 
        # サイズが必要な時に毎回 window から取ってたが、リサイズしてから window サイズが変更に
        # なるまでの間にサイズを取ると古いサイズが帰るのでメモることにする。
        @width = 72
        @height = 72

        self.decorated = false
        self.skip_pager_hint = true
        self.skip_taskbar_hint = true

        Gtk::GL.init
        @glconfig = Gdk::GLConfig.new(Gdk::GLConfig::MODE_RGB |
                                     Gdk::GLConfig::MODE_DEPTH |
                                     Gdk::GLConfig::MODE_DOUBLE)
        if !@glconfig
            @glconfig = Gdk::GLConfig.new(Gdk::GLConfig::MODE_RGBA | Gdk::GLConfig::MODE_DEPTH)
            if !@glconfig
                puts "glconfig.new error\n"
                exit 1
            end
        end

        @darea = Gtk::DrawingArea.new
#@darea.set_size_request(@width, @height)
        @darea.set_gl_capability(@glconfig)

        @darea.app_paintable = true

        signal_connect("destroy")               { |w, e| destroy_handler(w, e) }
        @darea.signal_connect("expose-event")          { |w, e| expose_handler(w, e) }
        @darea.signal_connect("realize")               { |w| realize_handler(w) }
        @darea.signal_connect("configure-event")       { |w, e| configure_handler(w, e) }
        @darea.signal_connect("button-press-event")    { |w, e| button_press_handler(e) }

        self.add(@darea)

        screen = self.screen
        colormap = screen.rgba_colormap
        if colormap.nil?
            #puts 'colormap is nil'
            self.colormap = screen.rgb_colormap
        else
            #puts 'colormap is not nil'
            self.colormap = colormap
        end

        @interval = 1  # sec
        @timeout_id = 0

        @size_changed = false

        @use_bg = false
        @surface_bg = nil
        @texture_bg = []

        @use_fg = false
        @surface_fg = nil
        @texture_fg = []

        # 子クラスが初期化されてない状態で set_item_size() を呼ぶと不幸がおこる
        # item 自身は set_item_size() 内でやってる処理が必要ないのでここでは単に resize()
        #self.set_item_size(@width, @height)
        self.resize(@width, @height)
    end

    def item_size
        return @width, @height
    end

    def set_item_size(w, h)
#        puts self.class.to_s + " set_item_size w:#{w} h:#{h}"
        @width = w
        @height = h
        create_bg(w, h) if @use_bg && @darea.realized?
        create_fg(w, h) if @use_fg && @darea.realized?
        set_shape_mask(w, h)
        @size_changed = true
        self.resize(w, h)
    end

    def destroy_handler(widget, event)
        #puts 'item destroy_handler'
        GLib::Source.remove(@timeout_id) unless @timeout_id[0].zero?

        glcontext = widget.gl_context
        gldrawable = widget.gl_drawable
        gldrawable.gl_begin(glcontext) do
            GL.DeleteTextures(@texture_fg) if @texture_fg[0] != nil && @texutre_fg[0] != 0
            GL.DeleteTextures(@texture_bg) if @texture_bg[0] != nil && @texutre_bg[0] != 0
        end
    end

    def expose_handler(widget, event)
        e_x = event.area.x
        e_y = event.area.y
        e_w = event.area.width
        e_h = event.area.height

#        puts "expose #{self.class.to_s} #{e_x} #{e_y} #{e_w} #{e_h}"

        w, h = self.window.size
        #cc = self.window.create_cairo_context

        glcontext = widget.gl_context
        gldrawable = widget.gl_drawable
        gldrawable.gl_begin(glcontext) do
            GL.Viewport(0, 0, w, h)

            GL.MatrixMode(GL::PROJECTION)
            GL.LoadIdentity

            GL.Ortho(0.0, @width, 0.0, @height, -1.0, 1.0)

            GL.MatrixMode(GL::MODELVIEW)
            GL.LoadIdentity

            GL.Clear(GL::COLOR_BUFFER_BIT | GL::DEPTH_BUFFER_BIT)

            if @use_bg 
                GL.Enable(GL::TEXTURE_RECTANGLE_ARB)
                GL.BlendFunc(GL::ONE, GL::ZERO)

                GL.BindTexture(GL::TEXTURE_RECTANGLE_ARB, @texture_bg[0])

                GL.Begin(GL::QUADS)

                GL.TexCoord(0.0, 0.0);  GL.Vertex(0.0, 0.0, 0.0);
                GL.TexCoord(0.0, h);    GL.Vertex(0.0, h, 0.0);
                GL.TexCoord(w, h);      GL.Vertex(w, h, 0.0);
                GL.TexCoord(w, 0.0);    GL.Vertex(w, 0.0, 0.0);
                GL.End()

                GL.Disable(GL::TEXTURE_RECTANGLE_ARB)
            end

=begin
            GL.LineWidth(5.0)
            GL.Begin(GL::LINES)
            GL.Color(0.0, 1.0, 0.0)
            GL.Vertex(0.0, 20.0, 0.0)
            GL.Vertex(65.0, 20.0, 0.0)
            GL.End

            GL.LineWidth(5.0)
            GL.Begin(GL::LINES)
            GL.Color(0.0, 0.0, 1.0)
            GL.Vertex(0.0, 50.0, 0.0)
            GL.Vertex(65.0, 50.0, 0.0)
            GL.End
=end

            GL.BlendFunc(GL::SRC_ALPHA, GL::ONE_MINUS_SRC_ALPHA)
            render(nil, w, h)

            if @use_fg
                GL.Enable(GL::TEXTURE_RECTANGLE_ARB)

                GL.BindTexture(GL::TEXTURE_RECTANGLE_ARB, @texture_fg[0])

                GL.Begin(GL::QUADS);
                GL.TexCoord(0.0, 0.0); GL.Vertex(0.0, 0.0, 0.0);
                GL.TexCoord(0.0, h); GL.Vertex(0.0, h, 0.0);
                GL.TexCoord(w, h); GL.Vertex(w, h, 0.0);
                GL.TexCoord(w, 0.0); GL.Vertex(w, 0.0, 0.0);
                GL.End();

                GL.Disable(GL::TEXTURE_RECTANGLE_ARB)
            end

            if gldrawable.double_buffered?
                gldrawable.swap_buffers
            else
                GL.Flush
            end
        end

        return true
    end

    def button_press_handler(event)
        #puts "button_press #{event.button}"

        case event.button
        when 3 
            show_menu(event)
            return true
        end

        return false
    end

    def show_menu(event)
        x = event.x
        y = event.y

        menuitem_setting = Gtk::MenuItem.new('Settings')
        menuitem_setting.signal_connect('activate') { show_dialog(x, y) }

        menuitem_quit = Gtk::MenuItem.new('Quit \'' + get_item_name() + '\'') 
        menuitem_quit.signal_connect('activate') { self.destroy() }

        menu = Gtk::Menu.new
        menu.append(menuitem_setting)
        menu.append(menuitem_quit)
        @mmon.add_menu(menu) 
        menu.show_all
        menu.popup(nil, nil, event.button, event.time)
    end

    def get_item_name
        if self.class.const_defined?(:NAME)
            return self.class.const_get(:NAME)
        else
            return self.class.to_s
        end
    end

    def show_dialog(x, y)
        mmon = @mmon.setting_widget

        title = Gtk::Label.new
        title.set_markup('<big><b>' + get_item_name() + '</b></big>')

        w = setting_widget(x, y)

        vbox = Gtk::VBox.new
        vbox.pack_start(title, false)
        vbox.pack_start(Gtk::HSeparator.new, false)
        vbox.pack_start(w, true) unless w.nil?

        notebook = Gtk::Notebook.new
        notebook.append_page(mmon, Gtk::Label.new('Mmon'))
        notebook.append_page(vbox, Gtk::Label.new(get_item_name())) unless w.nil?

        notebook.show_all

        n = notebook.page_num(vbox)
        notebook.page = n

        dialog = Gtk::Dialog.new('Settings', nil, Gtk::Dialog::MODAL, 
                [Gtk::Stock::CLOSE, Gtk::Dialog::ResponseType::CLOSE])
        dialog.vbox.pack_start(notebook)
        dialog.set_default_size(400, 400)
        dialog.run
        dialog.destroy
    end

    def realize_handler(widget)
        puts "realize #{self.class.to_s}"

        gl_init()
        create_bg(@width, @height) if @use_bg
        create_fg(@width, @height) if @use_fg

        unless @interval.zero?
            prepare()
            @timeout_id = GLib::Timeout.add(@interval * 1000) { update() }
        end
    end

    def configure_handler(widget, event)
        x = event.x
        y = event.y
        w = event.width
        h = event.height
#        puts "configure #{self.class.to_s} (x,y)=(#{x} #{y}) :w:#{w} h:#{h}"

        gl_init()
    end

    def update()
        return if @interval == 0.0

        if prepare() || @size_changed
            #puts 'update interval:' + @interval.to_s
            @size_changed = false
            self.queue_draw()
        end

        return true
    end

    def redraw_all
        w, h = item_size()
        rect = Gdk::Rectangle.new(0, 0, w, h)
        self.window.invalidate(rect, false)
    end

    def redraw(x, y, w, h)
        puts "redraw (#{x}, #{y}) #{w} #{h}"
        rect = Gdk::Rectangle.new(x, y, w, h)
        self.window.invalidate(rect, false)
    end

    def set_shape_mask(w, h)
        puts 'set_shape_mask'

        bitmap = Gdk::Pixmap.new(nil, w, h, 1)

        cc = bitmap.create_cairo_context
        cc.operator = Cairo::OPERATOR_CLEAR
        cc.paint
        cc.operator = Cairo::OPERATOR_SOURCE

        render_mask(cc, w, h)

        self.shape_combine_mask(bitmap, 0, 0)
    end

    def gl_init
        glcontext = @darea.gl_context
        gldrawable = @darea.gl_drawable

        gldrawable.gl_begin(glcontext) do
            GL.ClearColor(0.0, 0.0, 0.0, 0.0)
            GL.ClearDepth(1.0)

            GL.Disable(GL::DEPTH_TEST)
            #GL.Disable(GL::LIGHTING)

            GL.Enable(GL::TEXTURE_RECTANGLE_ARB)

            GL.Enable(GL::BLEND)
            #GL.BlendFunc(GL::SRC_ALPHA, GL::ONE_MINUS_SRC_ALPHA)
#GL.BlendFunc(GL::ONE, GL::ONE)

            GL.ColorMaterial(GL::FRONT_AND_BACK, GL::AMBIENT_AND_DIFFUSE)
            GL.Enable(GL::COLOR_MATERIAL)


            p @width
            p @height
            p gldrawable.size
            GL.Viewport(0, 0, @width, @height)

            GL.MatrixMode(GL::PROJECTION)
            GL.LoadIdentity

            GL.Ortho(0.0, @width, 0.0, @height, -1.0, 1.0)

            GL.MatrixMode(GL::MODELVIEW)
            GL.LoadIdentity
        end
    end

    def create_bg(w, h)
        puts 'create_bg'
        cc = create_surface_context(w, h)
        render_bg(cc, w, h)
        @surface_bg = cc.target

        glcontext = @darea.gl_context
        gldrawable = @darea.gl_drawable
        gldrawable.gl_begin(glcontext) do
                
            GL.DeleteTextures(@texture_bg) if @texture_bg[0] != nil && @texutre_bg[0] != 0
            @texture_bg = GL.GenTextures(1)
            GL.BindTexture(GL::TEXTURE_RECTANGLE_ARB, @texture_bg[0])
            GL.TexEnvf(GL::TEXTURE_ENV, GL::TEXTURE_ENV_MODE, GL::REPLACE)

            GL.TexParameteri(GL::TEXTURE_2D, GL::TEXTURE_WRAP_S, GL::CLAMP)
            GL.TexParameteri(GL::TEXTURE_2D, GL::TEXTURE_WRAP_T, GL::CLAMP)
            GL.TexParameteri(GL::TEXTURE_2D, GL::TEXTURE_MAG_FILTER, GL::NEAREST)
            GL.TexParameteri(GL::TEXTURE_2D, GL::TEXTURE_MIN_FILTER, GL::NEAREST)
            GL.TexImage2D(GL::TEXTURE_RECTANGLE_ARB, 0, GL::RGBA, @width, @height, 0,
            GL::BGRA, GL::UNSIGNED_BYTE, @surface_bg.data)
        end
    end

    def create_fg(w, h)
        cc = create_surface_context(w, h)
        render_fg(cc, w, h)
        surface = cc.target
        @surface_fg = cc.target

        glcontext = @darea.gl_context
        gldrawable = @darea.gl_drawable
        gldrawable.gl_begin(glcontext) do
            GL.DeleteTextures(@texture_fg) if @texture_fg[0] != nil && @texutre_fg[0] != 0
            @texture_fg = GL.GenTextures(1)
            GL.BindTexture(GL::TEXTURE_RECTANGLE_ARB, @texture_fg[0])
            GL.TexEnvf(GL::TEXTURE_ENV, GL::TEXTURE_ENV_MODE, GL::REPLACE)

            GL.TexParameteri(GL::TEXTURE_2D, GL::TEXTURE_WRAP_S, GL::CLAMP)
            GL.TexParameteri(GL::TEXTURE_2D, GL::TEXTURE_WRAP_T, GL::CLAMP)
            GL.TexParameteri(GL::TEXTURE_2D, GL::TEXTURE_MAG_FILTER, GL::NEAREST)
            GL.TexParameteri(GL::TEXTURE_2D, GL::TEXTURE_MIN_FILTER, GL::NEAREST)
            GL.TexImage2D(GL::TEXTURE_RECTANGLE_ARB, 0, GL::RGBA, @width, @height, 0,
            GL::BGRA, GL::UNSIGNED_BYTE, @surface_fg.data)
        end
    end

    def create_surface_context(w, h)
        surface = Cairo::ImageSurface.new(Cairo::FORMAT_ARGB32, w, h)
        cc = Cairo::Context.new(surface)
        cc.operator = Cairo::OPERATOR_CLEAR
        cc.paint
        cc.operator = Cairo::OPERATOR_SOURCE

        return cc
    end

    def read_element(e)
        x = e.elements['x'].text.to_i
        y = e.elements['y'].text.to_i
        
        self.move(x, y)

        read_element_local(e)
    end

    def write_element
        x, y = self.position

        e = REXML::Element.new(self.class.to_s)
        
        e_x = REXML::Element.new('x')
        e_x.text = x.to_s

        e_y = REXML::Element.new('y')
        e_y.text = y.to_s

        e.add_element(e_x)
        e.add_element(e_y)

        write_element_local(e)

        return e
    end

    private

    def prepare
        true
    end

    def render_bg(cc, w, h)
        puts 'render_bg'
        cc.set_source_rgba(1.0, 0.8, 0.8, 1.0)
        cc.paint
        #cc.rectangle(0.0, 0.0, 40.0, 40.0)
    end

    def render_fg(cc, w, h)
        puts 'render_fg'
        cc.set_source_rgba(0.0, 0.0, 1.0, 0.8)
        cc.paint
    end

    def render(cc, w, h)
        puts 'render'

        return

        GL.LineWidth(5.0)
        GL.Begin(GL::LINES)
        GL.Color(0.0, 1.0, 0.0, 1.0)
        GL.Vertex(0.0, 10.0, 0.0)
        GL.Vertex(65.0, 10.0, 0.0)
        GL.End
    end

    def render_mask(cc, w, h)
        cc.paint
    end

    def read_element_local(e)
    end
    
    def write_element_local(e)
    end

    def setting_widget(x, y)
        return nil
    end
end
    
if __FILE__ == $0
    require 'mmon/appmmon'

    app = AppMmon.new
    item = GLItem.new(app)
    app.add_item(item)

    Gtk::main()
end
           
