/******************************************************************************
 * Copyright (C) 2006 Tetsuya Kimata <kimata@acapulco.dyndns.org>
 *
 * All rights reserved.
 *
 * This software is provided 'as-is', without any express or implied
 * warranty.  In no event will the authors be held liable for any
 * damages arising from the use of this software.
 *
 * Permission is granted to anyone to use this software for any
 * purpose, including commercial applications, and to alter it and
 * redistribute it freely, subject to the following restrictions:
 *
 * 1. The origin of this software must not be misrepresented; you must
 *    not claim that you wrote the original software. If you use this
 *    software in a product, an acknowledgment in the product
 *    documentation would be appreciated but is not required.
 *
 * 2. Altered source versions must be plainly marked as such, and must
 *    not be misrepresented as being the original software.
 *
 * 3. This notice may not be removed or altered from any source
 *    distribution.
 *
 * $Id: BBSThreadVariableCreator.cpp 1945 2006-11-07 14:54:54Z svn $
 *****************************************************************************/

#include "Environment.h"

#include "BBSThreadVariableCreator.h"
#include "BBSCommentVariableCreator.h"
#include "BBSThreadManager.h"
#include "BBSThreadList.h"
#include "SourceInfo.h"

SOURCE_INFO_ADD("$Id: BBSThreadVariableCreator.cpp 1945 2006-11-07 14:54:54Z svn $");

/******************************************************************************
 * public メソッド
 *****************************************************************************/
BBSThreadVariableCreator::BBSThreadVariableCreator(const char **keys)
  : keys_(keys)
{
    init();
}

TemplateVariable::variable_t *
BBSThreadVariableCreator::create(apr_pool_t *pool,
                                 BBSCommentVariableCreator *var_creator,
                                 BBSCommentIterator *bcomment_iter,
                                 BBSThreadManager *thread_manager,
                                 BBSThread::info_t *info_list,
                                 apr_size_t info_list_size,
                                 apr_size_t comment_count) const
{
    void *memory;
    variable_t *var;
    variable_t *thread_array_memory;
    BBSCommentIterator::range_t ranges[2];
    BBSCommentIterator *bcomment_iter_begin;
    apr_size_t range_count;
    apr_size_t read_thread_count;

    // メモリを一気に確保
    APR_PALLOC(memory, void *, pool,
               sizeof(variable_t) +
               get_thread_array_memory_size(info_list_size));

    thread_array_memory = AS_VARIABLE(memory);

    var = thread_array_memory++;
    var->type = TemplateVariable::ARRAY;
    var->v = thread_array_memory;

    ranges[0].start_no = 1;
    ranges[0].stop_no = 1;

    bcomment_iter_begin = bcomment_iter;
    read_thread_count = 0;
    try {
        for (apr_size_t i = 0; i < info_list_size; i++) {
            set_range(info_list + i, comment_count,
                      ranges + 1, &range_count);

            thread_manager->get_thread_by_id(pool, info_list[i].id,
                                             bcomment_iter, ranges, range_count);
            read_thread_count++;

            memcpy(thread_array_memory, var_creator->create(pool, bcomment_iter),
                   sizeof(variable_t));

            thread_array_memory++;
            bcomment_iter = AS_BCOMMENT_ITER(AS_CHAR(bcomment_iter) +
                                             BBSCommentIterator::MEMORY_SIZE);
        }

        thread_array_memory->type = TemplateVariable::END;

        return var;
    } catch(const char *) {
        // 後始末
        bcomment_iter = bcomment_iter_begin;
        for (apr_size_t i = 0; i < read_thread_count; i++) {
            bcomment_iter->~BBSCommentIterator();
            bcomment_iter =
                AS_BCOMMENT_ITER(AS_CHAR(bcomment_iter) +
                                 BBSCommentIterator::MEMORY_SIZE);
        }

        throw;
    }
}

TemplateVariable::variable_t *
BBSThreadVariableCreator::create_info(apr_pool_t *pool,
                                      BBSThread::info_t *info_list,
                                      apr_size_t info_list_size) const
{
    void *memory;
    variable_t *info_array_memory;
    scalar_t *info_memory;
    variable_t *var;

    // メモリを一気に確保
    APR_PALLOC(memory, void *, pool,
               sizeof(variable_t) +
               get_info_array_memory_size(info_list_size) +
               get_info_memory_size() * info_list_size);

    info_array_memory = AS_VARIABLE(memory);
    info_memory
        = AS_SCALAR(AS_CHAR(info_array_memory) +
                    sizeof(variable_t) +
                    get_info_array_memory_size(info_list_size));

    var = info_array_memory++;
    var->type = TemplateVariable::ARRAY;
    var->v = info_array_memory;

    for (apr_size_t i = 0; i < info_list_size; i++) {
        create_info(pool, info_list + i, info_array_memory++, info_memory);
        info_memory += info_index_max_ + 1; // +1 でインデックスから個数に変換
    }

    info_array_memory->type = TemplateVariable::END;

    return var;
}

TemplateVariable::variable_t *
BBSThreadVariableCreator::create_info(apr_pool_t *pool,
                                      BBSThreadList *thread_list) const
{
    BBSThread::info_t *info_list;
    apr_size_t info_list_size;

    info_list_size = thread_list->get_size();

    APR_PALLOC(info_list, BBSThread::info_t *, pool,
               sizeof(BBSThread::info_t) * info_list_size);
    thread_list->get_info_list(info_list, &info_list_size);

    return create_info(pool, info_list, info_list_size);
}

BBSThreadVariableCreator *BBSThreadVariableCreator::get_instance(void *memory,
                                                                 const char **keys)
{
    new(memory) BBSThreadVariableCreator(keys);

    return reinterpret_cast<BBSThreadVariableCreator *>(memory);
}


/******************************************************************************
 * private メソッド
 *****************************************************************************/
void BBSThreadVariableCreator::init()
{
    info_index_max_ = TemplateVariableCreator::calc_index
        (keys_, AS_KEY_INDEX(&info_index_),
         sizeof(info_key_index_t)/sizeof(key_index_t));
}

inline TemplateVariable::variable_t *
BBSThreadVariableCreator::create_info(apr_pool_t *pool,
                                      BBSThread::info_t *thread_info,
                                      void *var_memory,
                                      void *sca_memory) const throw()
{
    variable_t *info_var;
    scalar_t *info_sca;
    scalar_t *sca;

    info_var = AS_VARIABLE(var_memory);
    info_sca = AS_SCALAR(sca_memory);

    info_var->type = TemplateVariable::HASH;
    info_var->s = info_sca;

    sca         = info_sca + info_index_.id.index;
    sca->type   = TemplateVariable::INTEGER;
    sca->i      = static_cast<int>(thread_info->id);

    sca         = info_sca + info_index_.subject.index;
    sca->type   = TemplateVariable::STRING;
    sca->s      = escape_html(pool, thread_info->subject);
    sca->l      = 0;

    sca         = info_sca + info_index_.comment_count.index;
    sca->type   = TemplateVariable::INTEGER;
    sca->i      = static_cast<int>(thread_info->comment_count);

    return info_var;
}

inline void
BBSThreadVariableCreator::set_range(BBSThread::info_t *thread_info,
                                    apr_size_t comment_count,
                                    BBSCommentIterator::range_t *range,
                                    apr_size_t *range_count) const throw()
{
    if (thread_info->comment_count == 1) {
        *range_count = 1;
    } else {
        *range_count = 2;
        if (thread_info->comment_count < comment_count) {
            range->start_no = 2;
            range->stop_no =
                static_cast<apr_uint16_t>(thread_info->comment_count);
        } else {
            range->start_no =
                static_cast<apr_uint16_t>(thread_info->comment_count -
                                          comment_count + 2);
            range->stop_no =
                static_cast<apr_uint16_t>(thread_info->comment_count);
        }
    }
}

apr_size_t BBSThreadVariableCreator::get_thread_array_memory_size(apr_size_t list_size) const
{
    return sizeof(variable_t) * (list_size + 1);
}

apr_size_t BBSThreadVariableCreator::get_info_array_memory_size(apr_size_t list_size) const
{
    return sizeof(variable_t) * (list_size + 1);
}

apr_size_t BBSThreadVariableCreator::get_info_memory_size() const
{
    return sizeof(scalar_t) * (info_index_max_ + 1);
}


/******************************************************************************
 * テスト
 *****************************************************************************/
#ifdef DEBUG_BBSThreadVariableCreator
#include "TestRunner.h"
#include "TemplateParser.h"
#include "BBSThreadListReader.h"
#include "Message.h"

static const apr_size_t INFO_LIST_SIZE  = 10;

void show_usage(const char *prog_name)
{
    cerr << "Usage: " << prog_name << " <DATA_DIR_PATH> <TMPL_FILE_PATH>" << endl;
}

void run_create(apr_pool_t *pool, BBSThreadList *thread_list,
                const char *data_dir_path, const char *tmpl_file_path)
{
    const char **ids;
    const char **keys;

    show_test_name("create variable");

    TemplateParser::parse(pool, tmpl_file_path, &ids, &keys);

    BBSThreadVariableCreator var_creator(keys);

    var_creator.create_info(pool, thread_list);

    show_spacer();
}

void run_all(apr_pool_t *pool, int argc, const char * const *argv)
{
    const char *data_dir_path;
    const char *tmpl_file_path;
    apr_shm_t *thread_list_shm;
    BBSThreadList *thread_list;

    if (argc != 3) {
        THROW(MESSAGE_ARGUMENT_INVALID);
    }

    data_dir_path = argv[1];
    tmpl_file_path = argv[2];

    if (!is_exist(pool, data_dir_path)) {
        THROW(MESSAGE_DAT_DIR_NOT_FOUND);
    }
    if (!is_exist(pool, tmpl_file_path)) {
        THROW(MESSAGE_TMPL_FILE_NOT_FOUND);
    }

    thread_list_shm = create_shm(pool, BBSThreadList::get_memory_size());
    thread_list = BBSThreadListReader::read(pool, data_dir_path,
                                            thread_list_shm);

    show_item("data dir", data_dir_path);
    show_item("template file", tmpl_file_path);
    show_line();

    run_create(pool, thread_list, data_dir_path, tmpl_file_path);
}

#endif

// Local Variables:
// mode: c++
// coding: utf-8-dos
// End:
