/******************************************************************************
 * Copyright (C) 2006 Tetsuya Kimata <kimata@acapulco.dyndns.org>
 *
 * All rights reserved.
 *
 * This software is provided 'as-is', without any express or nolockied
 * warranty.  In no event will the authors be held liable for any
 * damages arising from the use of this software.
 *
 * Permission is granted to anyone to use this software for any
 * purpose, including commercial applications, and to alter it and
 * redistribute it freely, subject to the following restrictions:
 *
 * 1. The origin of this software must not be misrepresented; you must
 *    not claim that you wrote the original software. If you use this
 *    software in a product, an acknowledgment in the product
 *    documentation would be appreciated but is not required.
 *
 * 2. Altered source versions must be plainly marked as such, and must
 *    not be misrepresented as being the original software.
 *
 * 3. This notice may not be removed or altered from any source
 *    distribution.
 *
 * $Id: BBSThreadManager.h 1958 2006-11-11 02:54:28Z svn $
 *****************************************************************************/

#ifndef BBS_THREAD_MANAGER_H
#define BBS_THREAD_MANAGER_H

#include "Environment.h"

#include "apr_shm.h"
#include "apr_atomic.h"

#include "BBSCommentIterator.h"
#include "BBSThreadReader.h"
#include "BBSThreadWriter.h"
#include "Uncopyable.h"

using namespace std;

class BBSThread;
class BBSCommentIterator;
class Locker;
class BBSThreadList;

/**
 * @brief BBS のスレッドを管理するクラス．
 *
 * 更新日時の新しい MGR_ON_MEMORY_THREAD_COUNT 個のスレッドの内容をメモリ上で
 * 管理します.
 */
class BBSThreadManager: public Uncopyable
{
public:
    BBSThreadManager(apr_pool_t *pool, const char *data_dir_path);
    /**
     * 初期化を行います．
     *
     * @param[in] thread_list スレッドのリスト
     */
    void init(BBSThreadList *thread_list);
    /**
     * 表示順にソートされたスレッド ID の配列を返します．
     * 配列の末尾には 0 が入っています．
     *
     * @param[in] pool プール
     * @param[in,out] thread_count スレッドの個数
     * @param[out] ids_sort_by_view スレッド ID の配列
     * @param[out] indexes_sort_by_view  インデックスの配列
     */
    void get_thread_info_sort_by_view(apr_pool_t *pool,
                                      apr_uint16_t *thread_count,
                                      apr_size_t **ids_sort_by_view,
                                      apr_uint16_t **indexes_sort_by_view) const;
    /**
     * thread_id と一致する ID を持つスレッドデータを返します．
     *
     * @param[in] pool プール
     * @param[in] thread_id スレッド ID
     * @param[out] bcomment_iter スレッドデータ
     * @param[in] ranges コメントの範囲のリスト
     * @param[in] range_count コメントの範囲のリストサイズ
     */
    void get_thread_by_id(apr_pool_t *pool, apr_size_t thread_id,
                          BBSCommentIterator *bcomment_iter,
                          BBSCommentIterator::range_t *ranges=NULL,
                          apr_size_t range_count=0) const;
    /**
     * 表示順で i 番目のスレッドデータを返します．
     *
     * @param[in] pool プール
     * @param[in] i スレッドのインデックス
     * @param[in] thread_id スレッド ID
     * @param[out] bcomment_iter スレッドデータ
     * @param[in] start_no コメントの開始番号
     * @param[in] stop_no コメントの終了番号
     * @return スレッドデータ
     */
    void get_thread_by_index(apr_pool_t *pool, apr_uint16_t i,
                             apr_size_t thread_id,
                             BBSCommentIterator *bcomment_iter,
                             BBSCommentIterator::range_t *ranges=NULL,
                             apr_size_t range_count=0) const;
    /**
     * スレッドを作成します．
     *
     * @param[in] subject スレッドの題名
     * @param[in] name 名前
     * @param[in] trip トリップ
     * @param[in] email メールアドレス
     * @param[in] message メッセージ
     * @param[in] image 添付画像
     * @param[in] time 書き込み時刻
     * @param[in] is_age age かどうか
     * @return スレッド ID
     */
    apr_size_t create_thread(const char *subject, const char *name,
                             const char *trip, const char *email,
                             const char *message, BBSThread::image_t *image,
                             apr_time_t time, bool is_age);
    /**
     * スレッドを作成します．
     *
     * @param[in] subject スレッドの題名
     * @param[in] subject スレッドの題名
     * @param[in] name 名前
     * @param[in] trip トリップ
     * @param[in] email メールアドレス
     * @param[in] message メッセージ
     * @param[in] time 書き込み時刻
     * @param[in] is_age age かどうか
     * @return スレッド ID
     */
    apr_size_t create_thread(const char *subject,
                             const char *name,
                             const char *trip, const char *email,
                             const char *message, apr_time_t time,
                             bool is_age);
    /**
     * スレッドにコメントを書き込みます．
     *
     * @param[in] thread_id スレッド ID
     * @param[in] name 名前
     * @param[in] trip トリップ
     * @param[in] email メールアドレス
     * @param[in] message メッセージ
     * @param[in] image 添付画像
     * @param[in] time 書き込み時刻
     * @param[in] is_age age かどうか
     * @return コメント番号
     */
    apr_uint16_t add_comment(apr_size_t thread_id,
                           const char *name, const char *trip,
                           const char *email, const char *message,
                           BBSThread::image_t *image, apr_time_t time,
                           bool is_age);
    /**
     * メモリ上にあるスレッドデータをディスクに書き出します．
     */
    void sync_all_thread();
    apr_uint16_t get_thread_count() const
    {
        return *thread_count_;
    };
    /**
     * 共有メモリを使ってインスタンスを生成します．
     *
     * @param[in] shm 共有メモリ
     * @return BBSThreadManager のインスタンス
     */
    static BBSThreadManager *get_instance(apr_shm_t *shm, apr_pool_t *pool,
                                          const char *data_dir_path);
    /**
     * 共有メモリを使って子プロセス用の初期化を行います．
     *
     * @param[in] shm 共有メモリ
     * @param[in] thread_list スレッドのリスト
     * @return BBSThreadManager のインスタンス
     */
    static BBSThreadManager *child_init(apr_shm_t *shm,
                                        BBSThreadList *thread_list);

    /**
     * マネージャのデータの最大サイズを計算します．
     *
     * @return マネージャのデータの最大サイズ
     */
    static apr_size_t get_memory_size();
    /**
     * 終了状態のチェックを行います．
     * 掲示板がロックされていたり，スレッドへの参照がある場合は例外を投げます．
     *
     * @param[in] thread_manager マネージャ
     */
    static void check_finalize_state(BBSThreadManager *thread_manager);

private:
    static const apr_uint16_t NULL_INDEX;

    /**
     * 子プロセス用の初期化を行います．
     *
     * @param[in] thread_list スレッドのリスト
     */
    void child_init_impl(BBSThreadList *thread_list);
    /**
     * 共有メモリをクリアします．
     */
    void clear_shm();
    /**
     * 共有メモリを初期化します．
     */
    void init_shm();
    /**
     * 共有メモリを子プロセス用に初期化します．
     */
    void attach_shm();
    /**
     * スレッドデータをファイルから読み出します．
     */
    void init_threads();
    /**
     * thread_id と一致する ID を持つスレッドデータを返します．
     * 見つからなかった場合は NULL を返します．
     *
     * @param[in] thread_id スレッド ID
     * @return スレッドデータ
     */
    BBSThread *get_thread_by_id(apr_size_t thread_id) const;
    /**
     * thread_id と一致する ID を持つスレッドデータを返します．
     * 掲示板全体のロックは行いません．
     *
     * @param[in] pool プール
     * @param[in] thread_id スレッド ID
     * @param[out] bcomment_iter スレッドデータ
     * @param[in] ranges コメントの範囲のリスト
     * @param[in] range_count コメントの範囲のリストサイズ
     */
    void get_thread_by_id_without_lock(apr_pool_t *pool,
                                       apr_size_t thread_id,
                                       BBSCommentIterator *bcomment_iter,
                                       BBSCommentIterator::range_t *ranges,
                                       apr_size_t range_count) const;
    /**
     * 指定されたインデックスのスレッドデータを返します．
     *
     * @param[in] i スレッドのインデックス
     * @param[in] is_check_index インデックスの値をチェックするかどうか
     * @return スレッドデータ
     */
    BBSThread *get_thread_by_index(apr_uint16_t i,
                                   bool is_check_index=true) const;
    /**
     * スレッドデータのインデックスを返します．
     *
     * @param[in] bbs_thread スレッドデータ
     * @return スレッドデータのインデックス
     */
    apr_uint16_t get_index_by_thread(BBSThread *bbs_thread) const;
    /**
     * コメントを書き込みます．
     *
     * @param[out] bbs_thread スレッドデータ
     * @param[in] i スレッドのインデックス
     * @param[in] name 名前
     * @param[in] trip トリップ
     * @param[in] email メールアドレス
     * @param[in] message メッセージ
     * @param[in] image 添付画像
     * @param[in] time 書き込み時刻
     * @param[in] is_age age かどうか
     * @param[in] is_insert 挿入かどうか (true: 挿入)
     */
    void exec_add_comment(BBSThread *bbs_thread, apr_uint16_t i,
                          const char *name, const char *trip,
                          const char *email, const char *message,
                          BBSThread::image_t *image, apr_time_t time,
                          bool is_age, bool is_insert=false);
    /**
     * スレッドをファイルからロードした後，コメントを書き込みます．
     *
     * @param[in] thread_id スレッド ID
     * @param[in] i スレッドのインデックス
     * @param[in] name 名前
     * @param[in] trip トリップ
     * @param[in] email メールアドレス
     * @param[in] message メッセージ
     * @param[in] image 添付画像
     * @param[in] time 書き込み時刻
     * @param[in] is_age age かどうか
     * @param[in] is_need_write 古いスレッドをファイルに書き込む必要があるかどうか
     * @return スレッド ID
     */
    apr_uint16_t read_and_add_comment(apr_size_t thread_id, apr_uint16_t i,
                                      const char *name, const char *trip,
                                      const char *email, const char *message,
                                      BBSThread::image_t *image,
                                      apr_time_t time, bool is_age,
                                      bool is_need_write=false);
    /**
     * スレッドをファイルからロードした後，コメントを書き込みます．
     *
     * @param[in] thread_id スレッド ID
     * @param[in] i スレッドのインデックス
     * @param[in] subject スレッドの題名
     * @param[in] name 名前
     * @param[in] trip トリップ
     * @param[in] email メールアドレス
     * @param[in] message メッセージ
     * @param[in] image 添付画像
     * @param[in] time 書き込み時刻
     * @param[in] is_age age かどうか
     * @param[in] is_need_write 古いスレッドをファイルに書き込む必要があるかどうか
     * @return スレッドデータ
     */
    BBSThread *create_and_add_comment(apr_size_t thread_id, apr_uint16_t i,
                                      const char *subject, const char *name,
                                      const char *trip, const char *email,
                                      const char *message,
                                      BBSThread::image_t *image,
                                      apr_time_t time, bool is_age,
                                      bool is_need_write=false);
    /**
     * スレッドを書き出します．
     *
     * @param[in] bbs_thread スレッドデータ
     */
    void write_thread(BBSThread *bbs_thread);
    /**
     * ユニークなスレッド ID を生成します．
     */
    apr_size_t create_thread_id() const;
    /**
     * ids_sort_by_view_ と indexes_sort_by_view_ を更新します．
     *
     * @param[in] i インデックス
     * @param[in] thread_id スレッド ID
     * @param[in] is_insert 挿入かどうか (true: 挿入)
     */
    void update_indexes_sort_by_view(apr_uint16_t i, apr_size_t thread_id,
                                     bool is_insert=false);
    /**
     * indexes_sort_by_mtime_ を更新します．
     *
     * @param[in] i インデックス
     * @param[in] is_insert 挿入かどうか (true: 挿入)
     */
    void update_indexes_sort_by_mtime(apr_uint16_t i, bool is_insert=false);
    /**
     * 値を配列の先頭に移動させます．
     *
     * @param[out] array 配列
     * @param[in] i 値
     * @param[in] is_insert 挿入かどうか (true: 挿入)
     */
    template<class T> void update_array(T *array, T i, bool is_insert);
    void sort_indexes_by_mtime(apr_uint16_t left=0,
                               apr_uint16_t right=MGR_ON_MEMORY_THREAD_COUNT - 1);
    int cmp_indexes_by_mtime(apr_uint16_t left, apr_uint16_t right) const;

    apr_pool_t *pool_;

    const char *data_dir_path_;

    BBSThreadReader thread_reader_;
    BBSThreadWriter thread_writer_;

    BBSThreadList *thread_list_;

    /** スレッドデータの配列．(in 共有メモリ) */
    BBSThread *threads_;
    /**
     * スレッド ID の配列．表示順にソートされています．
     * (in 共有メモリ)
     */
    apr_size_t *ids_sort_by_view_;
    /**
     * スレッドデータのインデックスの配列．表示順にソートされています．
     * get_thread_by_id の効率化のために使用します．
     * (in 共有メモリ)
     */
    apr_uint16_t *indexes_sort_by_view_;
    /**
     * スレッドデータのインデックスの配列．更新日時の新しいもの順にソー
     * トされています．(in 共有メモリ)
     */
    apr_uint16_t *indexes_sort_by_mtime_;
    /** 掲示板への参照の数 (in in 共有メモリ) */
    apr_atomic_t *manager_ref_count_;
    /** スレッドへの参照の数 (in in 共有メモリ) */
    apr_atomic_t *threads_ref_count_;
    /** 管理しているスレッドの個数 (in 共有メモリ) */
    apr_uint16_t *thread_count_;
};

#endif

// Local Variables:
// mode: c++
// coding: utf-8-dos
// End:
