/******************************************************************************
 * Copyright (C) 2006 Tetsuya Kimata <kimata@acapulco.dyndns.org>
 *
 * All rights reserved.
 *
 * This software is provided 'as-is', without any express or implied
 * warranty.  In no event will the authors be held liable for any
 * damages arising from the use of this software.
 *
 * Permission is granted to anyone to use this software for any
 * purpose, including commercial applications, and to alter it and
 * redistribute it freely, subject to the following restrictions:
 *
 * 1. The origin of this software must not be misrepresented; you must
 *    not claim that you wrote the original software. If you use this
 *    software in a product, an acknowledgment in the product
 *    documentation would be appreciated but is not required.
 *
 * 2. Altered source versions must be plainly marked as such, and must
 *    not be misrepresented as being the original software.
 *
 * 3. This notice may not be removed or altered from any source
 *    distribution.
 *
 * $Id: BBSThread.h 1994 2006-11-12 06:53:49Z svn $
 *****************************************************************************/

#ifndef BBS_THREAD_H
#define BBS_THREAD_H

#include "Environment.h"

#include <cstdlib>

#include "AtomicWrapper.h"
#include "apr_time.h"

#include "TemplateVariable.h"
#include "Uncopyable.h"
#include "Macro.h"

#define AS_BTHREAD(pointer)         reinterpret_cast<BBSThread *>(pointer)
#define AS_BTHREAD_P(pointer)       reinterpret_cast<BBSThread **>(pointer)

using namespace std;

/**
 * @brief BBS のスレッドを表すクラス．
 *
 * スレッドの生成，コメントの追記等はこのクラスを用いて行います．
 */
class BBSThread: public Uncopyable
{
public:
    /** スレッドのデータの最大サイズ */
    static const apr_size_t MAX_THREAD_DATA_SIZE;
    static const apr_size_t MAX_THREAD_DATA_SHIFT;

    static const char COMMENT_LINK_PREFIX[];

    typedef struct bbs_comment_flag {
        /** メッセージが age かどうか */
        apr_size_t is_age: 1;
        /** メッセージが URL の自動リンク処理を必要とするかどうか */
        apr_size_t is_need_url_link: 1;
        /** メッセージがコメントへの自動リンク処理を必要とするかどうか */
        apr_size_t is_need_comment_link: 1;
        /** メッセージが HTML エスケープ処理を必要とするかどうか */
        apr_size_t is_need_html_escape: 1;
    } bbs_comment_flag_t;

    /**
     * @brief BBS の書き込み情報を表す構造体．
     *
     * 複数の文字列ポインタを束ねるために使用します．
     */
    typedef struct bbs_comment_p {
        /** メッセージ番号 */
        apr_uint16_t no;
        /** 書き込み日時 */
        apr_time_t time;
        /** 名前 */
        const char *name;
        /** 書き込み日時 (フォーマット済み) */
        const char *date;
        /** トリップ */
        const char *trip;
        /** メールアドレス */
        const char *email;
        /** メッセージ */
        const char *message;
        /** メッセージの長さ */
        apr_size_t message_length;
        /** 添付画像の拡張子 */
        const char *image_ext;
        /** サムネイル画像の横サイズ */
        apr_uint16_t thumb_width;
        /** サムネイル画像の縦サイズ */
        apr_uint16_t thumb_height;
        union {
            bbs_comment_flag_t flag;
            apr_size_t flag_data;
        };

        bbs_comment_p()
          : no(0),
            time(0),
            name(NULL),
            date(NULL),
            trip(NULL),
            email(NULL),
            message(NULL),
            message_length(0),
            image_ext(""),
            thumb_width(0),
            thumb_height(0)
        {
            flag_data = 0;
        };
    } bbs_comment_p_t;

    typedef struct Image {
        /** 拡張子 */
        char ext[TRD_MAX_IMAGE_EXT_SIZE];
        /** サイズ */
        apr_size_t size;
        struct Thumnbnail {
            /** 横サイズ */
            apr_uint16_t width;
            /** 縦サイズ */
            apr_uint16_t height;
            Thumnbnail()
              : width(0),
                height(0)
            {

            }
        } thumbnail;

        Image()
          : size(0)
        {
            ext[0] = '\0';
        };
    } image_t;

    /**
     * @brief BBS のスレッドの情報を表す構造体．
     *
     * bbs_thread_t のサブセットです．
     */
    typedef struct thread_info {
        /** スレッドの ID */
        apr_size_t      id;
        /** スレッドの題名 */
        char            subject[TRD_MAX_SUBJECT_SIZE];
        /** スレッドの最終更新日時 */
        apr_time_t      mtime;
        /** スレッドの最終 age 日時 */
        apr_time_t      atime;
        /** スレッドのコメントの個数 */
        apr_size_t      comment_count;
        /** 参照された回数 */
        apr_size_t      view_count;
    } info_t;

    BBSThread();

    /**
     * コメントを書き込みます．
     *
     * @param[in] name 名前
     * @param[in] trip トリップ
     * @param[in] email メールアドレス
     * @param[in] message メッセージ
     * @param[in] image 添付画像
     * @param[in] time 書き込み時刻
     * @param[in] is_age age かどうか
     */
    void add_comment(const char *name, const char *trip, const char *email,
                     const char *message, BBSThread::image_t *image,
                     apr_time_t time, bool is_age);
    /**
     * コメントを読み出します．
     *
     * @param[in] no 番号
     * @param[out] bcomment_p コメント
     * @retval true 読み出せた場合
     * @retval false 指定された番号のコメントが存在しない場合
     */
    bool get_comment(apr_uint16_t no, bbs_comment_p_t *bcomment_p) const
    {
        bbs_comment_t *bcomment;

        if (UNLIKELY(no > bthread_.comment_count)) {
            return false;
        }

        bcomment = get_bcomment(no);
        fill_bcomment_p(no, bcomment, bcomment_p);

        return true;
    }

    const char *get_identifier() const
    {
        return bthread_.identifier;
    }
    const char *get_version() const
    {
        return bthread_.version;
    }
    apr_size_t get_id() const
    {
        return bthread_.id;
    }
    apr_time_t get_mtime() const
    {
        return bthread_.mtime;
    }
    apr_time_t get_atime() const
    {
        return bthread_.atime;
    }
    const char *get_subject() const
    {
        return bthread_.subject;
    }
    apr_uint16_t get_comment_count() const
    {
        return bthread_.comment_count;
    }
    void reset_unsynced_comment_count()
    {
        bthread_.unsynced_comment_count = 0;
    }
    apr_uint16_t get_unsynced_comment_count() const
    {
        return bthread_.unsynced_comment_count;
    }
    apr_size_t get_view_count()
    {
        return apr_atomic_read(&(bthread_.view_count));
    }
    bool can_add_comment() const
    {
        return bthread_.status == 0;
    }
    bool is_removed() const
    {
        return ((bthread_.status & STATUS_REMOVED_BIT) != 0);
    }
    void increment_view_count()
    {
        apr_atomic_inc(&(bthread_.view_count));
    }
    apr_size_t get_valid_size() const;
    /**
     * メモリを使ってインスタンスを生成します．
     *
     * @paramp[out] memory メモリ
     * @paramp[in] thread_id スレッドの ID
     * @paramp[in] subject スレッドの題名
     * @paramp[in] time スレッドの生成日時
     * @paramp[in] is_need_validate 内容をチェックするかどうか
     * @return インスタンス
     */
    static BBSThread *get_instance(void *memory, apr_size_t thread_id,
                                   const char *subject, apr_time_t time,
                                   bool is_need_validate=true);
    /**
     * プールを使ってインスタンスを生成します．
     *
     * @paramp[in] pool プール
     * @paramp[in] thread_id スレッドの ID
     * @paramp[in] subject スレッドの題名
     * @paramp[in] time スレッドの生成日時
     * @return インスタンス
     */
    static BBSThread *get_instance(apr_pool_t *pool, apr_size_t thread_id=0,
                                   const char *subject="", apr_time_t time=0);

#ifdef DEBUG
    /**
     * コメント内容をダンプします．
     *
     * @param[in] bcomment_p コメント
     */
    static void dump_comment(bbs_comment_p_t *bcomment_p);
#endif

private:
    static const apr_size_t STATUS_REMOVED_BIT;

    /**
     * @brief BBS の書き込みを表す構造体．
     *
     * プログラム中では通常，この構造体の後にコメントに関係する文字列データ
     * がが続きます．
     *
     * name_offset は常に 0 なので省略．
     */
    typedef struct bbs_comment {
        /** 書き込み日時 */
        apr_time_t      time;
        /** メッセージのデータサイズ */
        apr_uint16_t    size;
        /** 書き込み日時 (フォーマット済み) の開始位置のオフセット */
        apr_uint16_t    date_offset;
        /** トリップの開始位置のオフセット */
        apr_uint16_t    trip_offset;
        /** メールアドレス開始位置のオフセット */
        apr_uint16_t    email_offset;
        /** メッセージ開始位置のオフセット */
        apr_uint16_t    message_offset;
        /** メッセージの長さ */
        apr_uint16_t    message_length;
        /** 添付画像 */
        image_t         image;
        union {
            bbs_comment_flag_t flag;
            apr_size_t flag_data;
        };
#ifndef WIN32
        // VisualStudio 2005 が挿入する謎のパディング
        apr_size_t      padding;
#endif
    } bbs_comment_t;

    /**
     * @brief BBS のスレッドを表す構造体．
     *
     * プログラム中では通常，この構造体の後に comment_count 個の
     * bbs_comment_t が続きます．
     */
    typedef struct bbs_thread {
        /** 構造体の識別子 */
        char            identifier[TRD_MAX_IDENTIFIER_SIZE];
        /** 構造体のバージョン */
        char            version[TRD_MAX_VERSION_SIZE];
        /** スレッドの題名 */
        char            subject[TRD_MAX_SUBJECT_SIZE];
        /** ステータス (ビットマップ) */
        apr_size_t      status;
        /** スレッドの ID */
        apr_size_t      id;
        /** スレッドの最終更新日時 */
        apr_time_t      mtime;
        /** スレッドの最終 age 日時 */
        apr_time_t      atime;
        /** スレッドの生成日時 */
        apr_time_t      ctime;
        /** コメント末尾のオフセット */
        apr_size_t      comment_end_offset;
        /** スレッドのコメントの個数 */
        apr_uint16_t    comment_count;
        /** ディスクと同期していないコメントの数． */
        apr_uint16_t    unsynced_comment_count;
        /** 参照された回数 */
        apr_atomic_t    view_count;
        /**
         * bbs_comment_t のオフセットの配列．bbs_comment_tが可変長なので，
         * 頭出しを早くするために利用します．
         */
        apr_size_t      comment_offsets[((TRD_MAX_COMMENT_COUNT-1) /
                                         TRD_COMMENT_OFFSET_PERIOD) + 1 + 1];

        bbs_thread() {
            // ゼロクリア
            memset(identifier, 0, sizeof(struct bbs_thread));

            strncpy(identifier, PACKAGE_NAME, TRD_MAX_IDENTIFIER_SIZE-1);
            strncpy(version, PACKAGE_VERSION, TRD_MAX_VERSION_SIZE-1);
        }
    } bbs_thread_t;

    static const apr_size_t COMMENT_OFFSET_MASK;
    static const apr_size_t COMMENT_OFFSET_SHIFT;

#define AS_BCOMMENT(pointer)        reinterpret_cast<bbs_comment_t *>(pointer)
    /**
     * コメントへのポインタを返します．
     *
     * @param[in] no インデックス
     * @return コメントへのポインタ
     */
    bbs_comment_t *get_bcomment(apr_size_t no) const
#ifndef DEBUG
        throw()
#endif
    {
        bbs_comment_t *bcomment;
        apr_size_t index;

#ifdef DEBUG
        validate_no(no);
#endif
        index = no - 1;
        bcomment
            = AS_BCOMMENT(AS_CHAR(const_cast<bbs_thread_t *>(&bthread_) + 1) +
                          bthread_.comment_offsets[index >> COMMENT_OFFSET_SHIFT]);

        for (apr_size_t i = (index & COMMENT_OFFSET_MASK); i > 0; i--) {
            bcomment = AS_BCOMMENT(AS_BYTE(bcomment) + bcomment->size);
        }

        return bcomment;
    }
#undef AS_BCOMMENT

    /**
     * 次のコメントを読み出します．
     *
     * @param[in] bcomment コメント
     * @return コメントへのポインタ
     */
    bbs_comment_t *get_next_bcomment(bbs_comment_t *bcomment) const throw()
    {
        return reinterpret_cast<bbs_comment_t *>(AS_BYTE(bcomment) + bcomment->size);
    }
    /**
     * コメントのデータを書き込みます．
     *
     * @param[in] name 名前
     * @param[in] trip トリップ
     * @param[in] email メールアドレス
     * @param[in] message メッセージ
     * @param[in] image 添付画像
     * @param[in] time 書き込み時刻
     * @param[in] is_age age かどうか
     * @return コメント
     */
    bbs_comment_t *add_comment_data(const char *name, const char *trip,
                                    const char *email, const char *message,
                                    BBSThread::image_t *image, apr_time_t time,
                                    bool is_age);
    /**
     * コメントの番号をチェックします．
     *
     * @param[in] no 番号
     */
    void validate_no(apr_size_t no) const;
    /**
     * 日付をフォーマットします．
     *
     * @param[in] pool プール
     * @param[in] time 日付
     * @return フォーマット済みの日付
     */
    static const char *get_formatted_date(apr_pool_t *pool, apr_time_t time);
    /**
     * コメント内容をセットします．
     *
     * @param[in] no コメント番号
     * @param[in] bcomment コメント
     * @param[out] bcomment_p コメント
     */
    static void fill_bcomment_p(apr_int16_t no, bbs_comment_t *bcomment,
                                bbs_comment_p_t *bcomment_p) throw()
    {
        bcomment_p->no      = no;
        bcomment_p->time    = bcomment->time;
        bcomment_p->name    = AS_CHAR(bcomment + 1);
        bcomment_p->date    = bcomment_p->name + bcomment->date_offset;
        bcomment_p->trip    = bcomment_p->name + bcomment->trip_offset;
        bcomment_p->email   = bcomment_p->name + bcomment->email_offset;
        bcomment_p->message = bcomment_p->name + bcomment->message_offset;
        bcomment_p->message_length  = bcomment->message_length;

        bcomment_p->image_ext       = bcomment->image.ext;
        bcomment_p->thumb_width     = bcomment->image.thumbnail.width;
        bcomment_p->thumb_height    = bcomment->image.thumbnail.height;

        bcomment_p->flag_data = bcomment->flag_data;
    }
#ifdef DEBUG
    static void dump_item(const char *name, const char *value);
#endif

    friend class BBSCommentIterator;
    friend class BBSThreadReader;
    friend class BBSThreadList;
    friend void run_size();

    /** スレッドのデータ */
    bbs_thread_t bthread_;
};

#endif

// Local Variables:
// mode: c++
// coding: utf-8-dos
// End:
