/******************************************************************************
 * Copyright (C) 2006 Tetsuya Kimata <kimata@acapulco.dyndns.org>
 *
 * All rights reserved.
 *
 * This software is provided 'as-is', without any express or implied
 * warranty.  In no event will the authors be held liable for any
 * damages arising from the use of this software.
 *
 * Permission is granted to anyone to use this software for any
 * purpose, including commercial applications, and to alter it and
 * redistribute it freely, subject to the following restrictions:
 *
 * 1. The origin of this software must not be misrepresented; you must
 *    not claim that you wrote the original software. If you use this
 *    software in a product, an acknowledgment in the product
 *    documentation would be appreciated but is not required.
 *
 * 2. Altered source versions must be plainly marked as such, and must
 *    not be misrepresented as being the original software.
 *
 * 3. This notice may not be removed or altered from any source
 *    distribution.
 *
 * $Id: BBSThreadIO.cpp 2068 2006-11-24 15:06:07Z svn $
 *****************************************************************************/

#include "Environment.h"

#include "apr_file_info.h"
#include "apr_file_io.h"
#include "apr_strings.h"

#include "BBSThreadIO.h"
#include "Message.h"
#include "Macro.h"
#include "SourceInfo.h"

SOURCE_INFO_ADD("$Id: BBSThreadIO.cpp 2068 2006-11-24 15:06:07Z svn $");

/******************************************************************************
 * public メソッド
 *****************************************************************************/
BBSThreadIO::BBSThreadIO(apr_pool_t *pool, const char *dir_path)
  : pool_(pool),
    dir_path_(dir_path)
{

}

BBSThreadIO::~BBSThreadIO()
{

}

 const char *BBSThreadIO::get_file_path(apr_pool_t *pool, const char *dir_path,
                                        apr_size_t thread_id,
                                        const char *file_name)
{
    char *file_path;

    if (apr_filepath_merge(&file_path,
                           get_sub_dir_path(pool, dir_path, thread_id),
                           file_name,
                           APR_FILEPATH_NOTABOVEROOT, pool) != APR_SUCCESS) {
        THROW(MESSAGE_BBS_THREAD_FILE_PATH_CREATION_FAILED);
    }

    return file_path;
}

const char *BBSThreadIO::get_sub_dir_path(apr_pool_t *pool,
                                          const char *dir_path,
                                          apr_size_t thread_id)
{
    static const char HEX_CHAR[]    = "0123456789abcdef";

    char *sub_dir_path;
    char *sub_dir_name;

    APR_PALLOC(sub_dir_name, char *, pool, 3);
    sub_dir_name[0] = HEX_CHAR[(thread_id >> 4) & 0xf];
    sub_dir_name[1] = HEX_CHAR[(thread_id     ) & 0xf];
    sub_dir_name[2] = '\0';

    if (apr_filepath_merge(&sub_dir_path, dir_path, sub_dir_name,
                           APR_FILEPATH_NOTABOVEROOT, pool) != APR_SUCCESS) {
        THROW(MESSAGE_BBS_THREAD_SUB_DIR_PATH_CREATION_FAILED);
    }

    return sub_dir_path;
}

void BBSThreadIO::prepare_subdir(apr_pool_t *pool, const char *dir_path,
                                 apr_size_t thread_id)
{
    apr_finfo_t info;
    const char *sub_dir_path;

    sub_dir_path = get_sub_dir_path(pool, dir_path, thread_id);

    if (!APR_STATUS_IS_ENOENT(apr_stat(&info, sub_dir_path, APR_FINFO_MIN,
                                       pool))) {
        return;
    }

    if (apr_dir_make(sub_dir_path, APR_OS_DEFAULT, pool) != APR_SUCCESS) {
        THROW(MESSAGE_BBS_THREAD_SUB_DIR_CREATION_FAILED);
    }
}


/******************************************************************************
 * protected メソッド
 *****************************************************************************/
const char *BBSThreadIO::get_file_path(apr_pool_t *pool,
                                       apr_size_t thread_id) const
{
    return get_file_path(pool, dir_path_, thread_id,
                         apr_itoa(pool, static_cast<int>(thread_id)));
}

void BBSThreadIO::prepare_subdir(apr_pool_t *pool, apr_size_t thread_id) const
{
    prepare_subdir(pool, dir_path_, thread_id);
}

// Local Variables:
// mode: c++
// coding: utf-8-dos
// End:
