/******************************************************************************
 * Copyright (C) 2006 Tetsuya Kimata <kimata@acapulco.dyndns.org>
 *
 * All rights reserved.
 *
 * This software is provided 'as-is', without any express or implied
 * warranty.  In no event will the authors be held liable for any
 * damages arising from the use of this software.
 *
 * Permission is granted to anyone to use this software for any
 * purpose, including commercial applications, and to alter it and
 * redistribute it freely, subject to the following restrictions:
 *
 * 1. The origin of this software must not be misrepresented; you must
 *    not claim that you wrote the original software. If you use this
 *    software in a product, an acknowledgment in the product
 *    documentation would be appreciated but is not required.
 *
 * 2. Altered source versions must be plainly marked as such, and must
 *    not be misrepresented as being the original software.
 *
 * 3. This notice may not be removed or altered from any source
 *    distribution.
 *
 * $Id: MmapCommentIterator.cpp 2047 2006-11-23 18:57:33Z svn $
 *****************************************************************************/

#include "Environment.h"

#include "MmapCommentIterator.h"
#include "BBSThread.h"
#include "SourceInfo.h"

SOURCE_INFO_ADD("$Id: MmapCommentIterator.cpp 2047 2006-11-23 18:57:33Z svn $");

/******************************************************************************
 * public メソッド
 *****************************************************************************/
MmapCommentIterator::MmapCommentIterator(BBSThread *bbs_thread,
                                         File& bthread_file,
                                         BBSCommentIterator::range_t *ranges,
                                         apr_size_t range_count)
  : BBSCommentIterator(bbs_thread, ranges, range_count),
    bthread_file_(bthread_file)
{

}


/******************************************************************************
 * テスト
 *****************************************************************************/
#ifdef DEBUG_MmapCommentIterator
#include <memory>

#include "TestRunner.h"
#include "BBSThreadReader.h"
#include "TemporaryPool.h"
#include "File.h"
#include "Message.h"

void show_usage(const char *prog_name)
{
    cerr << "Usage: " << prog_name << " <DATA_DIR_PATH> <THREAD_ID>" << endl;
}

static void run_read(apr_pool_t *pool, const char *data_dir_path,
                     apr_size_t thread_id,
                     BBSCommentIterator **bcomment_iter)
{
    TemporaryPool temp_pool(pool);
    BBSThreadReader reader(temp_pool.get(), data_dir_path);
    File bthread_file(temp_pool.get());
    BBSThread *bbs_thread;

    show_test_name("read");

    reader.read(thread_id, &bbs_thread, pool, bthread_file);
    *bcomment_iter = new MmapCommentIterator(bbs_thread,
                                             bthread_file);
    show_item("size", (*bcomment_iter)->get_size());
}

static void run_iter(BBSCommentIterator *bcomment_iter)
{
    apr_size_t total_message_size;
    BBSThread::bbs_comment_p_t bcomment_p;

    show_test_name("iter");

    total_message_size = 0;
    do {
        bcomment_iter->get_comment(&bcomment_p);

        total_message_size += strlen(bcomment_p.message);
    } while (bcomment_iter->go_next());

    show_item("total message size", total_message_size);
}

void run_all(apr_pool_t *pool, int argc, const char * const *argv)
{
    const char *data_dir_path;
    apr_size_t thread_id;
    BBSCommentIterator *bcomment_iter;

    if (argc != 3) {
        THROW(MESSAGE_ARGUMENT_INVALID);
    }

    data_dir_path = argv[1];
    thread_id = atoi(argv[2]);

    if (!is_exist(pool, data_dir_path)) {
        THROW(MESSAGE_DATA_DIR_NOT_FOUND);
    }

    show_item("data dir", data_dir_path);
    show_item("thread id", thread_id);
    show_line();

    run_read(pool, data_dir_path, thread_id, &bcomment_iter);

    show_line();

    auto_ptr<BBSCommentIterator> bcomment_iter_ap(bcomment_iter);
    run_iter(bcomment_iter);
}

#endif

// Local Variables:
// mode: c++
// coding: utf-8-dos
// End:
