/******************************************************************************
 * Copyright (C) 2006 Tetsuya Kimata <kimata@acapulco.dyndns.org>
 *
 * All rights reserved.
 *
 * This software is provided 'as-is', without any express or implied
 * warranty.  In no event will the authors be held liable for any
 * damages arising from the use of this software.
 *
 * Permission is granted to anyone to use this software for any
 * purpose, including commercial applications, and to alter it and
 * redistribute it freely, subject to the following restrictions:
 *
 * 1. The origin of this software must not be misrepresented; you must
 *    not claim that you wrote the original software. If you use this
 *    software in a product, an acknowledgment in the product
 *    documentation would be appreciated but is not required.
 *
 * 2. Altered source versions must be plainly marked as such, and must
 *    not be misrepresented as being the original software.
 *
 * 3. This notice may not be removed or altered from any source
 *    distribution.
 *
 * $Id: NichDatParser.cpp 2063 2006-11-24 14:09:34Z svn $
 *****************************************************************************/

#include "Environment.h"

#include <algorithm>
#ifdef DEBUG
#include <iostream>
#endif

#include "apr_strings.h"

#include "NichDatParser.h"
#include "TemporaryPool.h"
#include "Auxiliary.h"
#include "Message.h"
#include "Macro.h"
#include "SourceInfo.h"

SOURCE_INFO_ADD("$Id: NichDatParser.cpp 2063 2006-11-24 14:09:34Z svn $");

const char NichDatParser::LINE_SEPARATOR[]       = "\n";
const char NichDatParser::COLUMN_SEPARATOR[]     = "<>";
const apr_size_t NichDatParser::NAME_PARAM_INDEX = 0;
const apr_size_t NichDatParser::EMAIL_PARAM_INDEX= 1;
const apr_size_t NichDatParser::MESSAGE_PARAM_INDEX= 3;

/******************************************************************************
 * public メソッド
 *****************************************************************************/
NichDatParser::NichDatParser(apr_pool_t *pool)
    : pool_(pool)
{

}

BBSThread *NichDatParser::parse(apr_size_t thread_id, const char *data_data,
                                apr_size_t data_size)
{
    BBSThread *bbs_thread;
    const char *data_line_start;
    const char *data_line_end;

    bbs_thread = BBSThread::get_instance(pool_, thread_id,
                                         MESSAGE_TEST_SUBJECT);

    for (data_line_start = data_data;
         (data_line_start < (data_data + data_size)) &&
             ((data_line_end = strstr(data_line_start, LINE_SEPARATOR)) != NULL);
         data_line_start = data_line_end + LITERAL_STRLEN(LINE_SEPARATOR)) {
        parse_line(bbs_thread, data_line_start, data_line_end - data_line_start);
    }

    return bbs_thread;
}

void NichDatParser::parse_line(BBSThread *bbs_thread,
                               const char *data_line,
                               apr_size_t line_size)
{
    const char *line_param_start;
    const char *line_param_end;
    const char *name = MESSAGE_TEST_NAME;
    const char *trip = MESSAGE_TEST_TRIP;
    const char *email = MESSAGE_TEST_EMAIL;
    const char *message = MESSAGE_TEST_MESSAGE;
    BBSThread::image_t image;
    apr_size_t i;

    for (line_param_start = data_line, i = 0;
         (line_param_start < (data_line + line_size)) &&
             ((line_param_end = strstr(line_param_start, COLUMN_SEPARATOR)) != NULL);
         line_param_start = line_param_end + LITERAL_STRLEN(COLUMN_SEPARATOR), i++) {

        if (i == NAME_PARAM_INDEX) {
#ifdef strndupa
            name = strndupa(line_param_start, line_param_end - line_param_start);
#else
            name = strndup(line_param_start, line_param_end - line_param_start);
#endif
        } else if (i == EMAIL_PARAM_INDEX) {
#ifdef strndupa
            email = strndupa(line_param_start, line_param_end - line_param_start);
#else
            email = strndup(line_param_start, line_param_end - line_param_start);
#endif
        } else if (i == MESSAGE_PARAM_INDEX) {
#ifdef strndupa
            message = strndupa(line_param_start,
                               min(line_param_end - line_param_start,
                                   TRD_MAX_MESSAGE_SIZE - 1));
#else
            message = strndup(line_param_start,
                              min(line_param_end - line_param_start,
                                  TRD_MAX_MESSAGE_SIZE - 1));
#endif
        }
    }

    bbs_thread->add_comment(name, trip, email, message, &image,
                            apr_time_now(), true);


#ifndef strndupa
    if (name    != MESSAGE_TEST_NAME   ) free(const_cast<char *>(name));
    if (email   != MESSAGE_TEST_EMAIL  ) free(const_cast<char *>(email));
    if (message != MESSAGE_TEST_MESSAGE) free(const_cast<char *>(message));
#endif
}


/******************************************************************************
 * テスト
 *****************************************************************************/
#ifdef DEBUG_NichDatParser
#include <cstdlib>
#include <ctime>

#include "TestRunner.h"
#include "BBSThreadWriter.h"
#include "BBSThreadReader.h"
#include "Auxiliary.h"
#include "File.h"

static const apr_size_t RUN_ITER_COUNT  = 200;

void show_usage(const char *prog_name)
{
    cerr << "Usage: " << prog_name << " <DATA_FILE_PATH>" << endl;
}

static void run_liner_access(BBSThread *bbs_thread)
{
    volatile double start_time;
    volatile double end_time;
    apr_size_t comment_count;
    BBSThread::bbs_comment_p_t bcomment_p;

    comment_count = bbs_thread->get_comment_count();

    start_time = get_usage_sec();
    for (apr_size_t i = 0; i < RUN_ITER_COUNT; i++) {
        for (apr_uint16_t index = 0; index < comment_count; index++) {
            bbs_thread->get_comment(index + 1, &bcomment_p);
        }
    }
    end_time = get_usage_sec();

    show_item("liner_access",
              (end_time - start_time) * 1000/RUN_ITER_COUNT/comment_count,
              " msec");
}

static void run_unit_access(BBSThread *bbs_thread)
{
    static const apr_size_t UNIT_SIZE   = 100;

    volatile double start_time;
    volatile double end_time;
    apr_size_t comment_count;
    apr_size_t unit_count;
    BBSThread::bbs_comment_p_t bcomment_p;
    apr_uint16_t *random_unit;

    comment_count = bbs_thread->get_comment_count();

    // ランダムな番号列を用意
    unit_count = comment_count/UNIT_SIZE;
    if (unit_count == 0) {
        cerr << "unit_access skipped (comment_count is too small)" << endl;
        return;
    }

    random_unit = new apr_uint16_t[unit_count];
    srand(static_cast<unsigned int>(time(NULL)));
    for (apr_size_t i = 0; i < unit_count; i++) {
        random_unit[i] = static_cast<apr_uint16_t>(rand() % unit_count);
    }

    start_time = get_usage_sec();
    for (apr_size_t i = 0; i < RUN_ITER_COUNT; i++) {
        for (apr_size_t j = 0; j < unit_count; j++) {
            apr_uint16_t k = random_unit[j]*UNIT_SIZE;
            for (apr_uint16_t index = k; index < k + UNIT_SIZE; index++) {
                bbs_thread->get_comment(index + 1, &bcomment_p);
            }
        }
    }
    end_time = get_usage_sec();

    show_item("unit_access",
              (end_time - start_time) * 1000/RUN_ITER_COUNT/comment_count,
              " msec");
}

static void run_reverse_access(BBSThread *bbs_thread)
{
    volatile double start_time;
    volatile double end_time;
    apr_uint16_t comment_count;
    BBSThread::bbs_comment_p_t bcomment_p;

    comment_count = bbs_thread->get_comment_count();

    start_time = get_usage_sec();
    for (apr_size_t i = 0; i < RUN_ITER_COUNT; i++) {
        for (apr_uint16_t index = comment_count; index > 0; index--) {
            bbs_thread->get_comment(index + 1, &bcomment_p);
        }
    }
    end_time = get_usage_sec();

    show_item("reverse_access",
              (end_time - start_time) * 1000/RUN_ITER_COUNT/comment_count,
              " msec");
}

static void run_random_access(BBSThread *bbs_thread)
{
    volatile double start_time;
    volatile double end_time;
    apr_size_t comment_count;
    BBSThread::bbs_comment_p_t bcomment_p;
    apr_uint16_t *random_index;

    comment_count = bbs_thread->get_comment_count();

    // ランダムな番号列を用意
    random_index = new apr_uint16_t[comment_count];
    srand(static_cast<unsigned int>(time(NULL)));
    for (apr_size_t i = 0; i < comment_count; i++) {
        random_index[i] = static_cast<apr_uint16_t>(rand() % comment_count);
    }

    start_time = get_usage_sec();
    for (apr_size_t i = 0; i < RUN_ITER_COUNT; i++) {
        for (apr_size_t j = 0; j < comment_count; j++) {
            bbs_thread->get_comment(random_index[j] + 1, &bcomment_p);
        }
    }
    end_time = get_usage_sec();

    show_item("random_access",
              (end_time - start_time) * 1000/RUN_ITER_COUNT/comment_count,
              " msec");
}

static void run_all_access(const char *target_name, BBSThread *bbs_thread)
{
    show_test_name(target_name);

    run_liner_access(bbs_thread);
    run_unit_access(bbs_thread);
    run_reverse_access(bbs_thread);
    run_random_access(bbs_thread);
    show_spacer();
}

static BBSThread *run_write_read(apr_pool_t *pool, const char *data_file_path,
                          BBSThread *bbs_thread1)
{
    BBSThread *bbs_thread2;
    TemporaryPool temp_pool(pool);

    BBSThreadWriter writer(temp_pool.get(),
                           dirname_ex(temp_pool.get(), data_file_path));
    writer.write(bbs_thread1);

    BBSThreadReader reader(temp_pool.get(),
                           dirname_ex(temp_pool.get(), data_file_path));
    bbs_thread2 = BBSThread::get_instance(pool);
    reader.read(bbs_thread1->get_id(), bbs_thread2);

    if (memcmp(bbs_thread1, bbs_thread2,
               BBSThread::MAX_THREAD_DATA_SIZE) == 0) {
        THROW(MESSAGE_BBS_THREAD_DATA_MISMATCH);
    }

    return bbs_thread2;
}

static void run_read(apr_pool_t *pool, const char *data_dir_path, apr_size_t id)
{
    BBSThread *bbs_thread;
    volatile double start_time;
    volatile double end_time;
    TemporaryPool temp_pool(pool);

    BBSThreadReader reader(temp_pool.get(), data_dir_path);
    bbs_thread = BBSThread::get_instance(temp_pool.get());

    start_time = get_usage_sec();
    for (apr_size_t i = 0; i < RUN_ITER_COUNT; i++) {
        reader.read(id, bbs_thread);
    }
    end_time = get_usage_sec();

    show_item("mmap", (end_time - start_time) * 1000/RUN_ITER_COUNT, " msec");
}

static void run_all_read(apr_pool_t *pool, const char *data_dir_path, apr_size_t id)
{
    show_test_name("read");
    run_read(pool, data_dir_path, id);
}

static double get_average_message_length(BBSThread *bbs_thread)
{
    apr_size_t total_message_length;
    BBSThread::bbs_comment_p_t bcomment_p;

    total_message_length = 0;
    for (apr_uint16_t i = 0; i < bbs_thread->get_comment_count(); i++) {
        bbs_thread->get_comment(i + 1, &bcomment_p);
        total_message_length += strlen(bcomment_p.message);
    }

    return total_message_length / static_cast<double>(bbs_thread->get_comment_count());
}

static apr_size_t create_thread_id()
{
    apr_size_t thread_id;

    // 0 以外の適等な値にする
    thread_id = static_cast<apr_size_t>(apr_time_now() % 10000) + 1;

    return thread_id;
}

void run_all(apr_pool_t *pool, int argc, const char * const *argv)
{
    apr_size_t thread_id;
    const char *data_file_path;
    BBSThread *bbs_thread1;
    BBSThread *bbs_thread2;
    apr_mmap_t *data_file_map;

    if (argc != 2) {
        THROW(MESSAGE_ARGUMENT_INVALID);
    }

    data_file_path = argv[1];

    if (!is_exist(pool, data_file_path)) {
        THROW(MESSAGE_DATA_FILE_NOT_FOUND);
    }

    File data_file(pool, data_file_path);
    data_file.open(APR_READ);
    data_file_map = data_file.mmap();

    thread_id = create_thread_id();

    NichDatParser data_parser(pool);
    bbs_thread1 = data_parser.parse(thread_id,
                                    reinterpret_cast<char *>(data_file_map->mm),
                                   data_file_map->size);

    show_item("data size", bbs_thread1->get_valid_size()/1024.0, " KB");
    show_item("comment count", bbs_thread1->get_comment_count(), "");
    show_item("average message", get_average_message_length(bbs_thread1), " byte");
    show_line();

    run_all_access("access 1", bbs_thread1);
    bbs_thread2 = run_write_read(pool, data_file_path, bbs_thread1);
    run_all_access("access 2", bbs_thread2);
    run_all_read(pool, dirname_ex(pool, data_file_path), bbs_thread1->get_id());
}

#endif

// Local Variables:
// mode: c++
// coding: utf-8-dos
// End:
