/******************************************************************************
 * Copyright (C) 2006 Tetsuya Kimata <kimata@acapulco.dyndns.org>
 *
 * All rights reserved.
 *
 * This software is provided 'as-is', without any express or implied
 * warranty.  In no event will the authors be held liable for any
 * damages arising from the use of this software.
 *
 * Permission is granted to anyone to use this software for any
 * purpose, including commercial applications, and to alter it and
 * redistribute it freely, subject to the following restrictions:
 *
 * 1. The origin of this software must not be misrepresented; you must
 *    not claim that you wrote the original software. If you use this
 *    software in a product, an acknowledgment in the product
 *    documentation would be appreciated but is not required.
 *
 * 2. Altered source versions must be plainly marked as such, and must
 *    not be misrepresented as being the original software.
 *
 * 3. This notice may not be removed or altered from any source
 *    distribution.
 *
 * $Id: fast_bbs_util.cpp 2328 2006-12-29 18:52:20Z svn $
 *****************************************************************************/

#include "Environment.h"

#include "apr_strings.h"

#include "fast_bbs_util.h"

#include "BBSConfig.h"
#include "BBSThreadIO.h"
#include "PostFlowController.h"
#include "Auxiliary.h"
#include "SourceInfo.h"

SOURCE_INFO_ADD("$Id: fast_bbs_util.cpp 2328 2006-12-29 18:52:20Z svn $");

const char *get_word(apr_pool_t *pool, const char **input,
                     const char delimiter)
{
    const char *start;
    const char *end;

    start = end = *input;

    while ((*end != '\0') && (*end != delimiter)) {
        end++;
    }

    if (*end == '\0') {
        *input = end;
    } else {
        *input = end + 1;
    }

    if (end != start) {
        return apr_pstrmemdup(pool, start, end - start);
    } else {
        return "";
    }
}

apr_size_t get_range_count(const char *ranges_arg)
{
    apr_size_t range_count;

    if (*ranges_arg == '\0') {
        return 0;
    }

    range_count = 1;
    do {
        if (*ranges_arg == ARG_COMMENT_RANGE_SEPARATE_CHAR) {
            range_count++;
        }
    } while (*(++ranges_arg) != '\0');

    return range_count;
}

void get_thread_id(apr_pool_t *pool, const char *arg, apr_size_t *thread_id)
{
    *thread_id = atosize(get_word(pool, &arg, ARG_SEPARATE_STR[0]));
}

void get_thread_param(apr_pool_t *pool, const char *arg, apr_size_t *thread_id,
                      BBSCommentIterator::range_t **ranges,
                      apr_size_t *range_count)
{
    const char *ranges_arg;
    const char *range_arg;

    // MEMO: PATH_INFO の長さ制限によって，range_count の大きさも制限される

    *thread_id = atosize(get_word(pool, &arg, ARG_SEPARATE_STR[0]));

    ranges_arg = get_word(pool, &arg, ARG_SEPARATE_STR[0]);
    *range_count = get_range_count(ranges_arg);

    if (*range_count == 0) {
        *ranges = NULL;
        return;
    }

    APR_PCALLOC(*ranges, BBSCommentIterator::range_t *, pool,
                sizeof(BBSCommentIterator::range_t) * (*range_count));

    // 末尾からの個数指定
    if (*ranges_arg == ARG_COMMENT_RANGE_LAST_STR[0]) {
        (*ranges)[0].start_no = BBSCommentIterator::RANGE_FROM_LAST;
        (*ranges)[0].stop_no = atoi(ranges_arg + 1);
        return;
    }

    for (apr_size_t i = 0; i < *range_count; i++) {
        range_arg = get_word(pool, &ranges_arg,
                             ARG_COMMENT_RANGE_SEPARATE_CHAR);
        (*ranges)[i].start_no = atoi(get_word(pool, &range_arg,
                                              ARG_COMMENT_RANGE_INTERVAL_CHAR));

        if ((*range_arg == '\0') &&
            (*(range_arg-1) != ARG_COMMENT_RANGE_INTERVAL_CHAR)) {
            (*ranges)[i].stop_no = (*ranges)[i].start_no;
        } else {
            (*ranges)[i].stop_no = atoi(range_arg);
        }
    }
}

void get_thumbnail_size(ImageFile *image_file,
                        apr_uint16_t *width, apr_uint16_t *height)
{
    static const double THUMBAIL_ASPECT =
        static_cast<double>(VIW_THUMBNAIL_WIDTH_LIMIT)/VIW_THUMBNAIL_HEIGHT_LIMIT;

    *width = image_file->get_width();
    *height = image_file->get_height();

    if (((*width) == 0) || ((*height) == 0)) {
        THROW(MESSAGE_POST_IMAGE_SIZE_INVALID);
    }

    if ((static_cast<double>(*width)/(*height)) > THUMBAIL_ASPECT) {
        // 横長の場合
        if ((*width) > VIW_THUMBNAIL_WIDTH_LIMIT) {
            *height = (*height) * VIW_THUMBNAIL_WIDTH_LIMIT / (*width);
            *width = VIW_THUMBNAIL_WIDTH_LIMIT;
        }
    } else {
        // 縦長の場合
        if ((*height) > VIW_THUMBNAIL_HEIGHT_LIMIT) {
            *width = (*width) * VIW_THUMBNAIL_HEIGHT_LIMIT / (*height);
            *height = VIW_THUMBNAIL_HEIGHT_LIMIT;
        }
    }

    if (((*width) == 0) || ((*height) == 0)) {
        THROW(MESSAGE_POST_IMAGE_SIZE_INVALID);
    }
}

void create_thumbnail(apr_pool_t *pool, const char *file_dir_path,
                      apr_size_t thread_id, apr_uint16_t comment_no,
                      BBSThread::image_t *image, const char *image_temp_path)
{
    const char *thumbnail_path;

    thumbnail_path = BBSThreadIO::get_file_path
        (pool, file_dir_path, thread_id,
         apr_pstrcat(pool, apr_itoa(pool, static_cast<int>(thread_id)),
                     FILE_TOKEN_SEPARATOR,
                     apr_itoa(pool, static_cast<int>(comment_no)),
                     VIW_THUMBNAIL_SUFFIX,
                     FILE_EXT_SEPARATOR, VIW_THUMBNAIL_EXT,
                     NULL));

    // 念のため
    apr_file_remove(thumbnail_path, pool);

    ImageFile image_file(pool, image_temp_path);
    image_file.create_thumbnail(thumbnail_path,
                                image->thumbnail.width,
                                image->thumbnail.height);
}

void commit_image_file(apr_pool_t *pool, const char *file_dir_path,
                       apr_size_t thread_id, apr_uint16_t comment_no,
                       BBSThread::image_t *image, const char *image_temp_path)
{
    const char *image_path;

    BBSThreadIO::prepare_subdir(pool, file_dir_path, thread_id);
    image_path = BBSThreadIO::get_file_path
        (pool, file_dir_path, thread_id,
         apr_pstrcat(pool, apr_itoa(pool, static_cast<int>(thread_id)),
                     FILE_TOKEN_SEPARATOR,
                     apr_itoa(pool, static_cast<int>(comment_no)),
                     FILE_EXT_SEPARATOR, image->ext, NULL));

    if (apr_file_rename(image_temp_path, image_path, pool) != APR_SUCCESS) {
        THROW(MESSAGE_POST_IMAGE_COMMIT_FAILED);
    }

    create_thumbnail(pool, file_dir_path,
                     thread_id, comment_no, image, image_path);
}

bool can_post(BBSConfig *config, apr_sockaddr_t *ip_address)
{
    if (config->is_debug_mode) {
        return true;
    }

    return config->get_flow_controller()->can_post(ip_address);
}

void regist_post(BBSConfig *config, apr_sockaddr_t *ip_address)
{
    if (config->is_debug_mode) {
        return;
    }

    config->get_flow_controller()->regist_post(ip_address);
}

// Local Variables:
// mode: c++
// coding: utf-8-dos
// End:
