package MojoWiki::Storage::MySQL;

use strict;
use warnings;

use Encode;
use FindBin;
use Time::Local;

use lib join( '/', $FindBin::Bin, '..', '..' );
use MojoWiki::Storage::MySQL::Schema;

use base 'MojoWiki::Storage::AbstractStorage';

##
## コンストラクタ
##
sub new {
    my ( $class, @args ) = @_;

    my $self = {};
    bless( $self, $class );

    $self->_init(@args);

    return $self;
}

##
## フィールドの初期化
##
sub _init {
    my ( $self, $args ) = @_;

    my $db_dsn
        = 'DBI:mysql:'
        . $args->{'DBName'}
        . ';mysql_socket='
        . $args->{'SocketPath'};

    my $db_user = $args->{'User'};
    my $db_pass = $args->{'Pass'};

    $self->{'schema'}
        = MojoWiki::Storage::MySQL::Schema->connect( $db_dsn, $db_user,
        $db_pass );

    return $self;
}

##
## ページを取得する
##
sub get_page {
    my ( $self, $path ) = @_;

    # トランザクション開始
    $self->{'schema'}->txn_begin();

    # ページ情報を PAGE テーブルから取得
    my $page_rs = $self->{'schema'}->resultset('Page')->find($path);

    my $page;

    if ( defined($page_rs) ) {
        $page->{'Title'}        = Encode::decode_utf8( $page_rs->title );
        $page->{'Body'}         = Encode::decode_utf8( $page_rs->body );
        $page->{'Revision'}     = $page_rs->revision;
        $page->{'CreatedTime'}  = &_datetime_to_utime( $page_rs->ctime );
        $page->{'ModifiedTime'} = &_datetime_to_utime( $page_rs->mtime );
    }

    # トランザクション終了、コミット
    $self->{'schema'}->txn_commit();

    return $page;
}

##
## ページを保存する
##
sub set_page {
    my ( $self, $path, $page ) = @_;

    # トランザクション開始
    $self->{'schema'}->txn_begin();

    # 保存済みのページ情報を PAGE テーブルから取得
    my $page_rs = $self->{'schema'}->resultset('Page')->find($path);

    if ( defined($page_rs) ) {
        $page_rs->update(
            {   'title'    => $page->{'Title'},
                'body'     => $page->{'Body'},
                'revision' => $page_rs->revision + 1,
                'mtime'    => &_utime_to_datetime( time() )
            }
        );
    }
    else {
        $self->{'schema'}->resultset('Page')->create(
            {   'path'     => $path,
                'title'    => $page->{'Title'},
                'body'     => $page->{'Body'},
                'revision' => 1,
                'ctime'    => &_utime_to_datetime( time() ),
                'mtime'    => &_utime_to_datetime( time() )
            }
        );
    }

    # トランザクション終了、コミット
    $self->{'schema'}->txn_commit();
}

##
## ページを削除する
##
sub delete_page {
    my ( $self, $path ) = @_;

    # トランザクション開始
    $self->{'schema'}->txn_begin();

    my $page_rs = $self->{'schema'}->resultset('Page')->find($path);
    if ( defined($page_rs) ) {
        $page_rs->delete;
    }

    # トランザクション終了、コミット
    $self->{'schema'}->txn_commit();
}

##
## 最終更新日が最新のページを取得する
##
sub get_newest_page {
    my ( $self, $n ) = @_;

    # トランザクション開始
    $self->{'schema'}->txn_begin();

    my $pages_rs = $self->{'schema'}->resultset('Page')->search(
        {},
        {   'order_by' => 'mtime DESC',
            'rows'     => $n
        }
    );

    my @pages;
    while ( my $page_rs = $pages_rs->next ) {
        my $page;
        $page->{'Path'}         = $page_rs->path;
        $page->{'Title'}        = Encode::decode_utf8( $page_rs->title );
        $page->{'Body'}         = Encode::decode_utf8( $page_rs->body );
        $page->{'Revision'}     = $page_rs->revision;
        $page->{'CreatedTime'}  = &_datetime_to_utime( $page_rs->ctime );
        $page->{'ModifiedTime'} = &_datetime_to_utime( $page_rs->mtime );

        push( @pages, $page );
    }

    # トランザクション終了、コミット
    $self->{'schema'}->txn_commit();

    return @pages;
}

##
## プラグインが利用可能な Key-Value Store 機能
##
sub plugin_set {
    my ( $self, $plugin_name, $key, @values ) = @_;

    # トランザクション開始
    $self->{'schema'}->txn_begin();

    if ( scalar(@values) == 0 ) {

        # 空の配列が渡されたら、Key を削除して終了
        my $key_rs = $self->{'schema'}->resultset('KVS_Key')
            ->find( { 'plugin_name' => $plugin_name, 'name' => $key } );
        if ( defined($key_rs) ) {
            $key_rs->delete;
        }

        # トランザクション終了、コミット
        $self->{'schema'}->txn_commit();

        return;
    }

    # Key 情報を取得
    my $key_rs = $self->{'schema'}->resultset('KVS_Key')
        ->find_or_create( { 'plugin_name' => $plugin_name, 'name' => $key } );

    # Value 情報を取得
    my $values_rs = $key_rs->kvs_values;

    # 既存の Value が存在する場合は書き換える
    while ( my $value_rs = $values_rs->next ) {
        if ( scalar(@values) == 0 ) {
            $value_rs->delete;
            next;
        }
        $value_rs->update( { 'value' => shift(@values) } );
    }
    while ( scalar(@values) > 0 ) {

        # 既存の Value に格納できなかった分は、新たに作成
        $self->{'schema'}->resultset('KVS_Value')
            ->create(
            { 'key_id' => $key_rs->id, 'value' => shift(@values) } );
    }

    # トランザクション終了、コミット
    $self->{'schema'}->txn_commit();
}

##
## プラグインが利用可能な Key-Value Store 機能
##
sub plugin_get {
    my ( $self, $plugin_name, $key ) = @_;

    # トランザクション開始
    $self->{'schema'}->txn_begin();

    # Key 情報を取得
    my $key_rs = $self->{'schema'}->resultset('KVS_Key')
        ->find( { 'plugin_name' => $plugin_name, 'name' => $key } );

    my @values;
    if ( defined($key_rs) ) {

        # Value 情報を取得
        my $values_rs = $key_rs->kvs_values;

        while ( my $value_rs = $values_rs->next ) {
            push( @values, $value_rs->value );
        }
    }

    # トランザクション終了、コミット
    $self->{'schema'}->txn_commit();

    return @values;
}

##
## UNIX Time を MySQL の DATETIME 型に変換する
##
sub _utime_to_datetime {
    my ($utime) = @_;

    my ( $sec, $min, $hour, $mday, $month, $year ) = localtime($utime);

    my $datetime = sprintf(
        "%4d-%02d-%02d %02d:%02d:%02d",
        $year + 1900,
        $month + 1, $mday, $hour, $min, $sec
    );

    return $datetime;
}

##
## MySQL の DATETIME 型を UNIX Time に変換する
##
sub _datetime_to_utime {
    my ($datetime) = @_;

    if ( $datetime =~ /^(\d{4})-(\d{2})-(\d{2})\s(\d{2}):(\d{2}):(\d{2})$/ ) {
        my $year  = $1 - 1900;
        my $month = $2 - 1;
        my $mday  = $3;
        my $hour  = $4;
        my $min   = $5;
        my $sec   = $6;

        my $utime = timelocal( $sec, $min, $hour, $mday, $month, $year );

        return $utime;
    }
    else {
        return 0;
    }
}

1;
