package MojoWiki::Storage::Storable;

use strict;
use warnings;

use Encode;
use FindBin;
use Storable;

use lib join( '/', $FindBin::Bin, '..', '..' );

use base 'MojoWiki::Storage::AbstractStorage';

use constant 'DB_PATH' => '/var/mojowiki/var/db';

##
## コンストラクタ
##
sub new {
    my ( $class, @args ) = @_;

    my $self = {};
    bless( $self, $class );

    $self->_init(@args);

    return $self;
}

##
## フィールドの初期化
##
sub _init {
    my ( $self, $args ) = @_;

    eval { $self->{'data'} = Storable::retrieve( $args->{'Path'} ); };
}

##
## ページを取得する
##
sub get_page {
    my ( $self, $path ) = @_;

    my $db_page = $self->{'data'}->{'Page'}->{$path};

    if ( !defined($db_page) ) {
        return undef;
    }

    my %page = %$db_page;

    $page{'Title'} = Encode::decode_utf8( $page{'Title'} );
    $page{'Body'}  = Encode::decode_utf8( $page{'Body'} );

    return \%page;
}

##
## ページを保存する
##
sub set_page {
    my ( $self, $path, $page ) = @_;

    if ( !defined($page) ) {
        die 'page not specified';
    }

    $self->{'data'}->{'Page'}->{$path}->{'Title'} = $page->{'Title'};
    $self->{'data'}->{'Page'}->{$path}->{'Body'}  = $page->{'Body'};
    $self->{'data'}->{'Page'}->{$path}->{'Revision'}
        = $page->{'Revision'} + 1;
    $self->{'data'}->{'Page'}->{$path}->{'CreatedTime'}
        = $self->{'data'}->{'Page'}->{$path}->{'CreatedTime'} || time();
    $self->{'data'}->{'Page'}->{$path}->{'ModifiedTime'} = time();

    Storable::nstore( $self->{'data'}, DB_PATH );
}

##
## ページを削除する
##
sub delete_page {
    my ( $self, $path ) = @_;

    delete( $self->{'data'}->{'Page'}->{$path} );

    Storable::nstore( $self->{'data'}, DB_PATH );
}

##
## 最終更新日が最新のページを取得する
##
sub get_newest_page {
    my ( $self, $n ) = @_;

    my $db_pages = $self->{'data'}->{'Page'};
    my %pages    = %$db_pages;

    my @newest_pages = ();
    while ( scalar(@newest_pages) < $n ) {
        if ( scalar( keys(%pages) ) ) {
            last;
        }

        my $newest_path;
        my $newest_mtime = 0;
        foreach my $path ( keys(%pages) ) {
            if ( $pages{$path}->{'ModifiedTime'} > $newest_mtime ) {
                $newest_mtime = $pages{$path}->{'ModifiedTime'};
                $newest_path  = $path;
            }
        }
        my $newest_page = $pages{$newest_path};
        push( @newest_pages, $newest_page );
        delete( $pages{$newest_path} );
    }

    return @newest_pages;
}

##
## プラグインが利用可能な key-value store 機能
##
sub plugin_set {
    my ( $self, $plugin_name, $key, @values ) = @_;

    if ( scalar(@values) == 0 ) {
        delete( $self->{'data'}->{'Plugin'}->{$plugin_name}->{$key} );
    }
    else {
        $self->{'data'}->{'Plugin'}->{$plugin_name}->{$key} = \@values;
    }

    Storable::nstore( $self->{'data'}, DB_PATH );
}

##
## プラグインが利用可能な key-value store 機能
sub plugin_get {
    my ( $self, $plugin_name, $key ) = @_;

    my $values = $self->{'data'}->{'Plugin'}->{$plugin_name}->{$key};
    if ( !defined($values) ) {
        return ();
    }

    return @$values;
}

1;
