package MojoWiki::WikiFormatter;
##
## wikiフォーマットの文字列からHTMLを生成するクラス(YukiWikiフォーマット)
## $Id: WikiFormatter.pm 11 2011-07-03 11:25:11Z bobunderson $
##

# 下記URLを参考にさせていただきました。
# http://www.akatsukinishisu.net/wiki.cgi?%BC%AB%CE%CF%A4%C7Wiki%A4%CE%CA%D1%B4%B9%BD%E8%CD%FD%A4%F2%BA%EE%A4%C3%A4%C6%A4%DF%A4%EB

use strict;
use warnings;

use FindBin;

use lib join( '/', $FindBin::Bin, '..' );
use MojoWiki::Constants;
use MojoWiki::Plugins;

##
## コンストラクタ
##
sub new {
    my ( $class, @args ) = @_;

    my $self = {};

    bless( $self, $class );

    $self->_init(@args);

    return $self;
}

##
## フィールドの初期化
## 引数
##   $controller MojoWiki::MojoDaemon::Page クラスのインスタンス
##
sub _init {
    my ( $self, $controller ) = @_;

    my @block_modules  = MojoWiki::Plugins::get_block_plugins($controller);
    my @inline_modules = MojoWiki::Plugins::get_inline_plugins($controller);

    $self->{'block_modules'}  = \@block_modules;
    $self->{'inline_modules'} = \@inline_modules;
    $self->{'controller'}     = $controller;
}

##
## YukiWiki形式で書かれたテキストからHTMLを生成する
## 引数
##   $arg wikiフォーマットの文字列
## 戻り値
##   HTMLソース
##
sub format {
    my ( $self, $arg, %option ) = @_;

    my $text             = '';
    my $in_verbatim_hard = 0;

    $arg =~ s/\x0D\x0A/\n/g;
    $arg =~ tr/\x0D\x0A/\n\n/;

    foreach my $line ( split( /\n/, $arg ) ) {
        if ( $line =~ /^---\($/ ) {
            $in_verbatim_hard = 1;
        }
        if ( $line =~ /^---\)$/ ) {
            $in_verbatim_hard = 0;
        }

        if ($in_verbatim_hard) {
            $text .= $self->escape($line);
        }
        elsif ( $line =~ /^(>{1,3})(.*)/ ) {
            $text .= $1 . $self->inline($2);
        }
        else {
            $text .= $self->inline($line);
        }
        $text .= "\n";
    }
    undef $arg;

    $self->verbatim( \$text );    # 最初に処理
    $self->horizontal_rule( \$text );
    $self->headding( \$text );
    $self->quotation( \$text );
    $self->unordered_list( \$text );
    $self->difinition( \$text );    # definition ?
    $self->make_table( \$text );
    $self->preformat( \$text );

    $self->block_plugin( \$text );
    $self->hyperlink( \$text );

    $self->paragraph( \$text );     # 最後から2番目に処理
    $self->line_break( \$text );    # 最後に処理

    return $text;
}

# verbatim機能
sub verbatim {
    my ( $self, $ref ) = @_;

    ${$ref} =~ s{^(---?)\(\n(.+?)\n\1\)$}{
        q{<pre class="verbatim-}
        . (($1 eq '---') ? 'hard' : 'soft')
        . '">'
        . join('<!--break->>', split(/\n/, $2))
        . q{</pre>}
    }emsg;
}

# リンク
sub hyperlink {
    my ( $self, $ref ) = @_;

    ${$ref} =~ s{\[\[([^\]]+)\]\]}{
        my $link_text = $1;
        if ( $link_text =~ /^(.+)\s*&gt;\s*(.+)$/ ) {
            my $label = $1;
            my $target = $2;

            if ( $target =~ /^http:\/\// ) {
                '<a href="' . $target . '">' . $label . '</a>';
            } else {
                '<a href="' . URL_PREFIX . $target . '">' . $label . '</a>';
            }
        } else {
            my $label = $link_text;
            my $target = $link_text;

            if ( $target =~ /^http:\/\// ) {
                '<a href="' . $target .'">' . $label . '</a>';
            } else {
            '<a href="' . URL_PREFIX . $target . '">' . $label . '</a>';
            }
        }
    }emsg;
}

# 水平線
sub horizontal_rule {
    my ( $self, $ref ) = @_;

    ${$ref} =~ s{^----}{<hr>\n}mg;
}

# 見出し
sub headding {
    my ( $self, $ref ) = @_;

    ${$ref} =~ s{^\*\*\*(.+)\s*\[#(\d+)\]}{<h4 id=$2>$1</h4>}mg;
    ${$ref} =~ s{^\*\*(.+)\s*\[#(\d+)\]}{<h3 id=$2>$1</h3>}mg;
    ${$ref} =~ s{^\*(.+)\s*\[#(\d+)\]}{<h2 id=$2>$1</h2>}mg;
}

# 引用
sub quotation {
    my ( $self, $ref ) = @_;

    # 3段
    ${$ref} =~ s{^>>>(.*)}{<blockquote>$1</blockquote>}mg;
    ${$ref} =~ s{</blockquote>\n<blockquote>}{<!--bqbr-->}g;
    ${$ref} =~ s{^(>.*)\n<blockquote>}{$1<blockquote>}mg;

    # 2段
    ${$ref} =~ s{^>>(.*)}{<blockquote>$1</blockquote>}mg;
    ${$ref} =~ s{</blockquote>\n<blockquote>}{<!--bqbr-->}g;
    ${$ref} =~ s{^(>.*)\n<blockquote}{$1<blockquote>}mg;

    # 1段
    ${$ref} =~ s{^>(.*)}{<blockquote>$1</blockquote>}mg;
    ${$ref} =~ s{</blockquote>\n<blockquote>}{<!--bqbr-->}g;

    # 引用文中に他のブロック要素を含めるための処理
    ${$ref} =~ s{(</?blockquote>)(.)}{$1\n$2}g;
    ${$ref} =~ s{(.)(</?blockquote>)}{$1\n$2}g;
    ${$ref} =~ s{<!--bqbr-->}{\n}g;
}

# 順不動リスト
sub unordered_list {
    my ( $self, $ref ) = @_;

    # 3段
    ${$ref} =~ s{^---(.*)}{<ul><li>$1</li></ul>}mg;
    ${$ref} =~ s{</ul>\n<ul>}{}g;
    ${$ref} =~ s{^-(.*)\n<ul>}{-$1<ul>}mg;

    # 2段
    ${$ref} =~ s{^--(.*)}{<ul><li>$1</li></ul>}mg;
    ${$ref} =~ s{</ul>\n<ul>}{}g;
    ${$ref} =~ s{^-(.*)\n<ul>}{-$1<ul>}mg;

    # 1段
    ${$ref} =~ s{^-(.*)}{<ul>\n<li>$1</li>\n</ul>}mg;
    ${$ref} =~ s{</ul>\n<ul>\n}{}g;
}

# 定義
sub difinition {
    my ( $self, $ref ) = @_;

    ${$ref} =~ s{^:([^:]+):(.+)}{<dl><dt>$1</dt><dd>$</dd></dl>}mg;
    ${$ref} =~ s{</dl>\n<dl>}{\n}g;
}

# 表
sub make_table {
    my ( $self, $ref ) = @_;

    ${$ref} =~ s{^\|\^(.+)}{
        "<table>\n<tr><th>"
        . join('</th><th>', split(/\|/, $1))
        . "</th></tr>\n</table>"
    }emg;

    ${$ref} =~ s{^\|(.+)}{
        "<table>\n<tr><td>"
        . join('</td><td>', split(/\|/, $1))
        . "</td></tr>\n</table>"
    }emg;

    ${$ref} =~ s{</table>\n<table>\n}{}g;
    ${$ref} =~ s{<td>\s+(.+?)\s*</td>}{<th>$1</th>}g;
}

# 整形済みテキスト
sub preformat {
    my ( $self, $ref ) = @_;

    ${$ref} =~ s{^ (.*)}{<pre>$1</pre>}mg;
    ${$ref} =~ s{</pre>\n<pre>}{<!--break-->}g;
}

# ブロック型プラグイン
sub block_plugin {
    my ( $self, $ref ) = @_;

    my $path = $self->{'controller'}->stash('path');

    my $block_modules = $self->{'block_modules'};

    ${$ref} =~ s{^#(\w+)(\((.*)\))?;}{
        my $original = "#$1$2";
        my $plugin_name = $1;
        my $argument = $self->escape($3);

        my $result = '';

        foreach my $module (@$block_modules) {
            if ( $module->is_processable_block($plugin_name) ) {
                $result .= $module->process_block_plugin($plugin_name, $path, $argument);
            }
        }

        $result;
    }emg;
}

# 段落
sub paragraph {
    my ( $self, $ref ) = @_;

    # 行頭がインライン要素タグ
    ${$ref} =~ s{^(<(?:[ib]|em|strong)>.*)}{<p>$1</p>}mg;
    ${$ref} =~ s{^(<(?:a |span).*)}{<p>$1</p>}mg;

    # 先頭が"<"と改行以外
    ${$ref} =~ s{^([^<\n].*)}{<p>$1</p>}mg;
    ${$ref} =~ s{</p>\n<p>}{<!--break-->}g;
}

# 改行の後処理
sub line_break {
    my ( $self, $ref ) = @_;

    ${$ref} =~ s{<!--break-->}{\n}g;
}

# タグ削除
sub remove_tag {
    my ( $self, $line ) = @_;

    $line =~ s{</?[A-Za-z][^>]*>}{}g;
    return $line;
}

# インライン要素の変換
sub inline {
    my ( $self, $arg ) = @_;

    my $path = $self->{'controller'}->stash('path');

    my $inline_modules = $self->{'inline_modules'};

    $arg = $self->escape($arg);
    $arg =~ s{'''([^']+?)'''}{<em><i>$1</i></em>}g;          # Italic
    $arg =~ s{''([^']+?)''}{<strong><b>$1</b></strong>}g;    # Bold
    $arg =~ s{&amp;br;}{<br>}g;

    $arg =~ s{&amp;(\w+)(\((.*)\))?;}{
        my $plugin_name = $1;
        my $argument = $3;

        my $result = '';

        foreach my $module (@$inline_modules) {
            if ( $module->is_processable_inline($plugin_name) ) {
                $result .= $module->process_inline_plugin($plugin_name, $path, $argument);
            }
        }

        $result;
    }emg;

    return $arg;
}

# 実体参照に変換
sub escape {
    my ( $self, $arg ) = @_;

    $arg =~ s/&/&amp;/g;
    $arg =~ s/</&lt;/g;
    $arg =~ s/>/&gt;/g;
    $arg =~ s/"/&quot;/g;

    return $arg;
}

##
## 見出しにIDをつける
##
sub head_index {
    my ( $self, $raw_body ) = @_;

    my $matched_str;
    my $id;
    my $random;

    $raw_body =~ s{^(\*\*\*.+)}{
        $self->_add_head_id($1);
    }emg;

    $raw_body =~ s{^(\*\*.+)}{
        $self->_add_head_id($1);
    }emg;

    $raw_body =~ s{^(\*.+)}{
        $self->_add_head_id($1);
    }emg;

    return $raw_body;
}

##
## IDつき見出しを生成する
##
sub _add_head_id {
    my ( $self, $head_str ) = @_;

    $head_str =~ s/[\r\n]*$//;

    my $id;

    if ( $head_str =~ /\s*\[#\w+\]\s*$/ ) {

        # IDがすでに付いていればそのまま利用
        $id = '';
    }
    else {

        # 新規IDをつける
        my $random = int( rand(10000) );
        $id = sprintf( " [#%04d]", $random );
    }

    return $head_str . $id;
}

1;
