#!/bin/bash
#===================================================#
# Basic System Configuration and Hardware Setup     #
#  (C)2006 Keicho Kondo <dgel@users.sourceforge.jp> #
#===================================================#
# This script automatically configures MOL System.
#

## === Setup Environment === ##
## Set command search path ##
PATH="/exec:/bin:/usr/bin:/sbin:/usr/sbin"
export PATH

## Set umask ##
umask 0022

## Set color ##
RESET="]R"       # Reset fb color mode
CRE="
[K"       # Erase to end of line
CLEAR="c"        # Clear and reset Screen
NORMAL="[0;39m"  # Normal Color
RED="[1;31m"     # Failure or error message
GREEN="[1;32m"   # Success message
YELLOW="[1;33m"  # Descriptions
BLUE="[1;34m"    # System message
MAGENTA="[1;35m" # Found devices or drivers
CYAN="[1;36m"    # Questions
WHITE="[1;37m"   # Hint

## Check boot command line ##
CMDLINE="$(cat /proc/cmdline)"

## Set functions ##
# Get Boot Parameter
getbootparam()
{
	case "$CMDLINE" in
		*\ $1=*) ;;
		$1=*) ;;
		*) return 1 ;;
	esac
	result="${CMDLINE##*$1=}"
	result="${result%%[     ]*}"
	echo "$result"
	return 0
}

# Check Boot Parameter
checkbootparam()
{
	case "$CMDLINE" in
		*\ $1*) return 0 ;;
		$1*)    return 0 ;;
	esac
	return 1
}

## Disable kernel messages ##
echo "0" > /proc/sys/kernel/printk

## Check boot command line ##
if checkbootparam "noaudio" ; then NOAUDIO="yes"   ; fi
if checkbootparam "noscsi"  ; then NOSCSI="yes"    ; fi
if checkbootparam "nonet"   ; then NONETWORK="yes" ; fi
if checkbootparam "novideo" ; then NOVIDEO="yes"   ; fi
if checkbootparam "noraid"  ; then NORAID="yes"    ; fi
if checkbootparam "nodhcp"  ; then NODHCP="yes"    ; fi
if checkbootparam "ipaddr"  ; then IPADDR="$(getbootparam ipaddr 2>/dev/null)"  ; fi
if checkbootparam "netmask" ; then NETMASK="$(getbootparam ipaddr 2>/dev/null)" ; fi
if checkbootparam "gateway" ; then GATEWAY="$(getbootparam ipaddr 2>/dev/null)" ; fi
if checkbootparam "netdev"  ; then NETDEV="$(getbootparam ipaddr 2>/dev/null)"  ; fi


## === Pre execute rc script === ##
## Executing hwsetup-mol ##
# check /etc/sysconfig/mol directory
if [ -d /etc/sysconfig/mol ]
then
	for f in $(ls /etc/sysconfig/mol 2>/dev/null)
	do
		for conf in floppy mol mouse netcard sound xserver
		do
			[ "$f" = "$conf" ] && mv -f /etc/sysconfig/mol/$f /etc/sysconfig/mol/$f.backup
		done
	done
else
	mkdir -p /etc/sysconfig/mol
fi
# check boot parameter
if [ -n "${NOAUDIO}" ]   ; then HWSETUP_OPT="-a ${HWSETUP_OPT}" ; fi
if [ -n "${NOSCSI}" ]    ; then HWSETUP_OPT="-s ${HWSETUP_OPT}" ; fi
if [ -n "${NONETWORK}" ] ; then HWSETUP_OPT="-net ${HWSETUP_OPT}"   ; fi
if [ -n "${NOVIDEO}" ]   ; then HWSETUP_OPT="-video ${HWSETUP_OPT}" ; fi
if [ -n "${NORAID}" ]    ; then HWSETUP_OPT="-raid ${HWSETUP_OPT}"  ; fi

#hwsetup-mol -p -skip-ld ${HWSETUP_OPT}
hwsetup-mol -p ${HWSETUP_OPT}

## XkbLayout ##
if [ -z "${NOVIDEO}" ]
then
	case $(grep -v "^#" /etc/X11/xorg.conf | awk '/XkbLayout/{print $3}') in *jp*) echo "XKEYBOARD=\"jp\"" >> /etc/sysconfig/mol/keyboard ;; esac
fi

## Read in what hwsetup-mol has found ##
[ -f /etc/sysconfig/mol/mol ] && . /etc/sysconfig/mol/mol

## Videocard ##
if [ -z "${NOVIDEO}" ]
then
	[ -f /etc/X11/xorg.conf ] && mv /etc/X11/xorg.conf /etc/X11/xorg.conf.backup
	xsetup-mol
else
	echo " ${BLUE}Skipping X Configuration.${NORMAL}"
fi

## Soundcard ##
if [ -z "${NOAUDIO}" ] && [ -n "${SOUND_FULLNAME}" -a -n "${SOUND_DRIVER}" ]
then
	echo " ${GREEN}Soundcard is ${YELLOW}${SOUND_FULLNAME}${GREEN} driver=${MAGENTA}${SOUND_DRIVER}${NORMAL}"
	sed -e 's|@@SOUND@@|'"${SOUND_DRIVER}"'|g' /etc/modprobe.conf.in > /etc/modprobe.conf.tmp
	mv -f /etc/modprobe.conf.tmp /etc/modprobe.conf.in
	# device check
	DEVNAME="/dev/snd /dev/audio /dev/dsp /dev/mixer"
	for chkdev in ${DEVNAME}
	do
		if [ ! -e ${chkdev} ]
		then	
			/sbin/MAKEDEV -d /dev ${chkdev#/dev/}
		fi
	done
else
	echo " ${BLUE}Skipping Soundcard detection.${NORMAL}"
	sed -e 's|@@SOUND@@||g' /etc/modprobe.conf.in > /etc/modprobe.conf.tmp
	mv -f /etc/modprobe.conf.tmp /etc/modprobe.conf.in
fi

## NIC ##
if [ -z "${NONETWORK}" ] && [ -n "${NETCARD_FULLNAME}" -a -n "${NETCARD_DRIVER}" ]
then
	NETCARD_DEVICE="$(awk -F: '/eth.:|tr.:/{print $1}' /proc/net/dev | awk -F' ' '{print $1}' 2>/dev/null)"
	echo " ${GREEN}Netcard is ${YELLOW}${NETCARD_FULLNAME}${GREEN} at ${MAGENTA}${NETCARD_DEVICE} ${GREEN}driver=${MAGENTA}${NETCARD_DRIVER}${NORMAL}"
	sed -e 's|@@NET@@|'"alias ${NETCARD_DEVICE} ${NETCARD_DRIVER}"'|g' /etc/modprobe.conf.in > /etc/modprobe.conf.tmp
	mv -f /etc/modprobe.conf.tmp /etc/modprobe.conf.in

	# for Static IP address
	if [ -n "${IPADDR}" ]
	then
		[ -n "${NETMASK}" ] || NETMASK="255.255.255.0"
		[ -n "${GATEWAY}" ] || GATEWAY="${IPADDR%.*}.1"
		[ -n "${NETDEV}" ]  || NETDEV="eth0"
		STATIC_NETDEV="${NETDEV}"
		echo " ${GREEN}Network device ${MAGENTA}${STATIC_NETDEV} ${GREEN}detected, IP is ${YELLOW}${IPADDR}${GREEN}.${NORMAL}"
#-----------------------------------------------------------------
cat > /etc/sysconfig/network-scripts/ifcfg-${STATIC_NETDEV} << EOF
DEVICE=${STATIC_NETDEV}
ONBOOT=yes
BOOTPROTO=static
IPADDR=${IPADDR}
NETMASK=${NETMASK}
EOF
#-----------------------------------------------------------------
		sed "s/GATEWAY=/#GATEWAY=/g" /etc/sysconfig/network > /tmp/network && echo "GATEWAY=${GATEWAY}" >> /tmp/network && mv -f /tmp/network /etc/sysconfig/network
	fi

	# for DHCP
	if [ -n "${NODHCP}" ]
	then
		echo " ${BLUE}Skipping DHCP broadcast/network detection as requested on boot commandline.${NORMAL}"
	else
		for DHCP_NETDEV in ${NETCARD_DEVICE}
		do
			[ "${STATIC_NETDEV}" = "${DHCP_NETDEV}" ] && continue
			echo " ${GREEN}Network device ${MAGENTA}${DHCP_NETDEV} ${GREEN}detected, DHCP broadcasting for IP.${NORMAL}"
#---------------------------------------------------------------
cat > /etc/sysconfig/network-scripts/ifcfg-${DHCP_NETDEV} << EOF
DEVICE=${DHCP_NETDEV}
ONBOOT=yes
BOOTPROTO=dhcp
EOF
#---------------------------------------------------------------
		done
	fi
else
	echo " ${BLUE}Skipping Netcard detection.${NORMAL}"
	sed -e 's|@@NET@@||g' /etc/modprobe.conf.in > /etc/modprobe.conf.tmp
	mv -f /etc/modprobe.conf.tmp /etc/modprobe.conf.in
fi

## Mouse ##
if [ -n "${MOUSE_FULLNAME}" -o -n "${MOUSE_DEVICE}" ]
then
	echo " ${GREEN}Mouse is ${YELLOW}${MOUSE_FULLNAME}${GREEN} at ${MAGENTA}${MOUSE_DEVICE}${NORMAL}"
fi

## Set /etc/modprobe.conf for MOL ##
cp -a /etc/modprobe.conf.in /etc/modprobe.conf
for tmpf in $(ls /etc/sysconfig/mol/*.tmp 2>/dev/null)
do
	mv -f ${tmpf%%.tmp} ${tmpf%%.tmp}.mol
	mv -f ${tmpf} ${tmpf%%.tmp}
done

## Enable kernel messages ##
echo "6" > /proc/sys/kernel/printk


