/*!
    \file  Fat12FileSystem.cpp
    \brief Fat12FileSystem

    Copyright (c) 2005 HigePon
    WITHOUT ANY WARRANTY

    \author  HigePon
    \version $Revision$
    \date   create:2005/03/26 update:$Date$
*/

#include <string.h>
#include <stdio.h>
#include "Fat12FileSystem.h"
#include "Fat12.h"
#include "Assert.h"

using namespace MonAPI;

//#define DEBUG_READ_TRACE

namespace FatFS
{

/*----------------------------------------------------------------------
    Fat12FileSystem
----------------------------------------------------------------------*/
Fat12FileSystem::Fat12FileSystem(IStorageDevice* fd)
{
    this->fd        = fd;
    this->lastError = NO_ERROR;
}

Fat12FileSystem::~Fat12FileSystem()
{
    delete[] this->sectorBuffer;
    delete this->fat;

    DeleteEntry(this->rootDirectory);
}

bool Fat12FileSystem::Initialize()
{
    /* allocate sector buffer */
    this->sectorBuffer = new byte[SECTOR_SIZE];

    if (this->sectorBuffer == NULL)
    {
        this->lastError = MEMORY_ALLOCATE_ERROR;
        return false;
    }

    /* read bpb */
    if (!ReadBpb()) return false;

    /* read Fat */
    if (!ReadAndSetFat()) return false;

    /* read root directory */
    if (!ReadRootDirectory()) return false;

    /* create directory cache */
    if (!CreateDirectoryCache()) return false;

    return true;
}

bool Fat12FileSystem::Read(dword lba, byte* buffer, dword size)
{
    return this->fd->read(lba, buffer, size) == 0;
}

File* Fat12FileSystem::Open(const CString& path, int mode)
{
    Fat12Directory* directoryEntry;
    CString fileName;

    int lastIndexOfSlash = path.lastIndexOf('/');

    if (lastIndexOfSlash == -1)
    {
        directoryEntry = this->rootDirectory;
        fileName = path;
    }
    else
    {
        CString directoryPath = path.substring(0, lastIndexOfSlash);
        fileName  = path.substring(lastIndexOfSlash + 1, path.getLength() - lastIndexOfSlash);
        directoryEntry = FindDirectoryEntry(this->rootDirectory, directoryPath);

        if (directoryEntry == NULL) return NULL;
    }

    Fat12File* file = FindFileEntry(directoryEntry, fileName);
    if (file == NULL) return NULL;

    return file;
}

bool Fat12FileSystem::Close(File* file)
{
    delete file;
    return false;
}

bool Fat12FileSystem::CreateFile(const CString& path)
{
    return false;
}

bool Fat12FileSystem::RemoveFile(const CString& path)
{
    return false;
}

bool Fat12FileSystem::CreateDirectory(const CString& path)
{
    return false;
}

bool Fat12FileSystem::RemoveDirectory(const CString& path)
{
    return false;
}

_A<FileSystemEntry*> Fat12FileSystem::GetFileSystemEntries(const CString& path)
{
    _A<FileSystemEntry*> noResult(0);

    Fat12Directory* entry = FindDirectoryEntry(this->rootDirectory, path);
    if (entry == NULL)
    {
        return noResult;
    }

    _A<FileSystemEntry*> ret(entry->children.size());

    for (int i = 0; i < ret.get_Length(); i++)
    {
        ret[i] = entry->children[i];
    }

    return ret;
}

/*----------------------------------------------------------------------
    Fat12FileSystem private functions
----------------------------------------------------------------------*/
Fat12Directory* Fat12FileSystem::FindDirectoryEntry(Fat12Directory* root, const CString& path)
{
    bool found;
    _A<CString> elements = path.split('/');

    FOREACH (CString, element, elements)
    {
        if (element == ".")
        {
            continue;
        }
        else if (element == "..")
        {
            root = root->parent;
            continue;
        }
        else if (element == "" || element == NULL)
        {
            continue;
        }

        found = false;
        for (int i = 0; i < root->children.size(); i++)
        {
            if (root->children[i]->GetName() == element)
            {
                root = (Fat12Directory*)root->children[i];
                found = true;
                break;
            }
        }
        if (!found) return NULL;
    }
    END_FOREACH

    return root;
}

// return value should be delete, when close
Fat12File* Fat12FileSystem::FindFileEntry(Fat12Directory* directory, const CString& fileName)
{
    for (int i = 0; i < directory->children.size(); i++)
    {
        FileSystemEntry* entry = directory->children[i];

        if (entry->GetName() == fileName && !entry->IsDirectory())
        {
            Fat12File* file = (Fat12File*)entry;

            /* use default copy constructor */
            return new Fat12File(*file);
        }
    }
    return NULL;
}

// should be size % sector size = 0
int Fat12FileSystem::ReadAtOnce(dword lba, void* buffer, dword size)
{
    dword sectors = size / SECTOR_SIZE;
    for (dword i = 0; i < sectors; i++)
    {
        int result = this->fd->read(i + lba, (void*)((byte*)buffer + SECTOR_SIZE * i), SECTOR_SIZE);
        if (result)
        {
            return result;
        }
    }
    return 0;
}

bool Fat12FileSystem::ReadBpb()
{
    if (this->fd->read(0, this->sectorBuffer, SECTOR_SIZE))
    {
        this->lastError = READ_ERROR;
        return false;
    }

    memcpy(&this->bpb, this->sectorBuffer, sizeof(Bpb));

    if (this->bpb.sectorsPerCluster != 1 || this->bpb.bytesPerSector != 512)
    {
        printf("fat unknown format\n");
        printf("bpb.sectorsPerCluster=%d\n", bpb.sectorsPerCluster);
        printf("bpb.bytesPerSector=%d\n", bpb.bytesPerSector);
        return false;
    }

    this->notClusterArea = 1 + bpb.sectorsPerFAT * bpb.fatCount + bpb.maxRootEntries * sizeof(DirectoryEntry) / SECTOR_SIZE;

    return true;
}

bool Fat12FileSystem::ReadAndSetFat()
{
    dword fatBytes = bpb.sectorsPerFAT * SECTOR_SIZE;
    dword fatStart = 1;

    /* allocate Fat buffer */
    byte* buffer = new byte[fatBytes];

    if (buffer == NULL)
    {

        this->lastError = MEMORY_ALLOCATE_ERROR;
        return false;
    }

    /* read Fat */
    if (ReadAtOnce(fatStart, buffer, fatBytes))
    {
        this->lastError = READ_ERROR;
        delete[] buffer;
        return false;
    }

    /* create Fat12 instance */
    this->fat = new Fat12(buffer);

    if (this->fat == NULL)
    {
        this->lastError = MEMORY_ALLOCATE_ERROR;
        return false;
    }

    return true;
}

bool Fat12FileSystem::ReadRootDirectory()
{
    dword rootEntrySectors  = this->bpb.maxRootEntries * sizeof(DirectoryEntry) / SECTOR_SIZE;
    dword rootEntryStart    = this->bpb.sectorsPerFAT * this->bpb.fatCount + 1;
    dword rootEntryBytes    = rootEntrySectors * SECTOR_SIZE;

    byte* rootEntryBuffer = new byte[rootEntryBytes];

    if (rootEntryBuffer == NULL)
    {
        this->lastError = MEMORY_ALLOCATE_ERROR;
        return false;
    }

    /* read */
    if (ReadAtOnce(rootEntryStart, rootEntryBuffer, rootEntryBytes))
    {
        delete[] rootEntryBuffer;
        this->lastError = READ_ERROR;
        return false;
    }

    /* read root directory */
    this->rootDirectory = new Fat12Directory();

    DirectoryEntry* entry = (DirectoryEntry*)rootEntryBuffer;
    for (int j = 0; j <  bpb.maxRootEntries; j++, entry++)
    {
        if (entry->filename[0] == 0x00 || entry->filename[0] == 0xe5)
        {
            continue;
        }

        /* add to list of children */
        FileSystemEntry* fileSystemEntry = SetupFileSystemEntryInformation(this->rootDirectory, entry);
        this->rootDirectory->children.add(fileSystemEntry);
    }

    delete[] rootEntryBuffer;

    return true;
}

FileSystemEntry* Fat12FileSystem::SetupFileSystemEntryInformation(Fat12Directory* parent, DirectoryEntry* entry)
{
    if (entry->attributes & ATTR_DIRECTORY)
    {
        Fat12Directory* directory = new Fat12Directory();
        FileDate* date= directory->GetCreateDate();

        SetupDate(date, entry);
        directory->SetName(GetProperName((const char*)entry->filename, (const char*)entry->extension));
        directory->SetCluster(entry->startCluster);
        directory->parent = parent;

        return directory;
    }
    else
    {
        Fat12File* file = new Fat12File(this);
        FileDate* date = file->GetCreateDate();

        SetupDate(date, entry);
        file->SetName(GetProperName((const char*)entry->filename, (const char*)entry->extension));
        file->SetSize(entry->fileSize);
        file->SetCluster(entry->startCluster);
        file->parent = parent;
        return file;
    }
}

void Fat12FileSystem::SetupDate(FileDate* date, DirectoryEntry* entry)
{
    date->SetHour((entry->time >> 11) & 0x1F);
    date->SetMinute((entry->time >> 5) & 0x3F);

    date->SetSecond((entry->time & 0x1F) * 2);

    date->SetYear((entry->date >> 9) + 1980);
    date->SetMonth((entry->date >> 5) & 0x0f);
    date->SetDay(entry->date & 0x1F);
}

CString Fat12FileSystem::Trim(const char* str, int maxLength)
{
    int length = maxLength;

    for (int i = 0; i < maxLength; i++)
    {
        if (str[i] == ' ')
        {
            length = i;
            break;
        }
    }

    return CString(str, length);
}

CString Fat12FileSystem::GetProperName(const char* filename, const char* extension)
{
    CString name = Trim(filename, 8);
    CString ext  = Trim(extension, 3);

    if (ext.getLength() == 0)
    {
        return name;
    }
    else
    {
        return name + "." + ext;
    }
}

bool Fat12FileSystem::CreateDirectoryCache()
{
    for (int i = 0; i < this->rootDirectory->children.size(); i++)
    {
        FileSystemEntry* entry = this->rootDirectory->children[i];

        if (!entry->IsDirectory()) continue;

        Fat12Directory* directory = (Fat12Directory*)entry;

        bool result = CreateDirectoryCache(directory, directory->GetCluster());

        if (!result) return false;
    }
    return true;
}

bool Fat12FileSystem::CreateDirectoryCache(Fat12Directory* startDirectory, dword cluster)
{
    if (cluster >= 0xff8 && cluster <= 0xfff) return true;

    if (startDirectory->GetName() == "." || startDirectory->GetName() == "..") return true;

    byte* buffer = new byte[SECTOR_SIZE];

    if (ReadAtOnce(GetLbaFromCluster(cluster), buffer, SECTOR_SIZE))
    {
        this->lastError = READ_ERROR;
        delete[] buffer;
        return false;
    }

    DirectoryEntry* entry = (DirectoryEntry*)buffer;
    for (dword j = 0; j <  SECTOR_SIZE / sizeof(DirectoryEntry); j++, entry++)
    {
        if (entry->filename[0] == 0x00 || entry->filename[0] == 0xe5)
        {
            continue;
        }

        /* add to list of children */
        FileSystemEntry* fileSystemEntry = SetupFileSystemEntryInformation(startDirectory, entry);
        startDirectory->children.add(fileSystemEntry);

        /* directory? */
        if (!fileSystemEntry->IsDirectory()) continue;

        Fat12Directory* directory = (Fat12Directory*)fileSystemEntry;

        if (!CreateDirectoryCache(directory, directory->GetCluster()))
        {
            delete[] buffer;
            return false;
        }
    }

    /* has next cluster? */
    dword next = this->fat->GetNextCluster(cluster);
    if (next != cluster)
    {
        if (!CreateDirectoryCache(startDirectory, next))
        {
            delete[] buffer;
            return false;
        }
    }

    delete[] buffer;
    return true;
}

dword Fat12FileSystem::GetLbaFromCluster(dword cluster)
{
    return this->notClusterArea + bpb.sectorsPerCluster * (cluster - 2);
}

void Fat12FileSystem::DeleteEntry(FileSystemEntry* entry)
{
    if (!entry->IsDirectory())
    {
        delete entry;
        return;
    }

    Fat12Directory* directory = (Fat12Directory*)entry;

    for (int i = 0; i < directory->children.size(); i++)
    {
        DeleteEntry(directory->children[i]);
    }

    delete entry;
}

bool Fat12FileSystem::ReadClusterChain(dword startCluster, dword postion, void* buffer, dword size)
{
    dword clusterCountToDataStart = postion / SECTOR_SIZE;
    dword clusterCountToDataEnd   = (postion + size) / SECTOR_SIZE;
    dword restSize = size;

#ifdef DEBUG_READ_TRACE
    printf("startCluster=%d\n", startCluster);
    printf("postion=%d\n", postion);
    printf("size=%d\n", size);
    printf("clusterCountToDataStart=%d\n", clusterCountToDataStart);
    printf("clusterCountToDataEnd=%d\n", clusterCountToDataEnd);
    fflush(stdout);
#endif

    dword readTargetCluster;

    byte* buf = (byte*)buffer;

    if (!this->fat->GetClusterFromChain(startCluster, clusterCountToDataStart, &readTargetCluster))
    {
        return false;
    }

    int result = this->fd->read(GetLbaFromCluster(readTargetCluster), this->sectorBuffer, SECTOR_SIZE);
    if (result)
    {
        return false;
    }

    dword firstReadSize;
    dword firstSectorStart = postion % SECTOR_SIZE;
    dword firstReadStart;

    if (firstSectorStart == 0)
    {
        firstReadSize = size < SECTOR_SIZE ? size : SECTOR_SIZE;
        firstReadStart = 0;
    }
    else
    {
        if (firstSectorStart + size < SECTOR_SIZE)
        {
            firstReadStart = firstSectorStart;
            firstReadSize = SECTOR_SIZE - firstSectorStart;
        }
        else
        {
            firstReadStart = firstSectorStart;
            firstReadSize = size;
        }
    }

    memcpy(buf, this->sectorBuffer + firstSectorStart, firstReadSize);

    restSize -= firstReadSize;
    buf += firstReadSize;

    for (dword i = clusterCountToDataStart + 1; i <= clusterCountToDataEnd; i++)
    {
        if (!this->fat->GetClusterFromChain(startCluster, i, &readTargetCluster))
        {
            return false;
        }

        int result = this->fd->read(GetLbaFromCluster(readTargetCluster), this->sectorBuffer, SECTOR_SIZE);
        if (result)
        {
            return false;
        }

        if (restSize < SECTOR_SIZE)
        {
            memcpy(buf, this->sectorBuffer, restSize);
            return true;
        }
        else
        {
            memcpy(buf, this->sectorBuffer, SECTOR_SIZE);
            buf += SECTOR_SIZE;
            restSize -= SECTOR_SIZE;
        }
    }

    return true;
}

}; /* namespace FatFS */
