/*
 * CoinAPIJsonRPCClient.java
 *
 * Copyright 2014 the Monache.org.
 * http://www.monache.org/
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package org.monache.lib.api;

import java.nio.charset.Charset;

import org.monache.lib.httpclient.HttpClient;
import org.monache.lib.httpclient.HttpClientException;
import org.monache.lib.httpclient.HttpClientResult;
import org.monache.lib.httpclient.JsonPostParams;
import org.monache.lib.jsonrpc.AbstractJsonRPCClient;
import org.monache.lib.jsonrpc.JsonRPCException;
import org.monache.lib.jsonrpc.JsonRPCv1ResponseObject;
import org.monache.lib.jsonrpc.JsonRPCv2ResponseObject;

/**
 * 
 * 
 * @author monache.org
 * @since 1.0
 */
public class CoinAPIJsonRPCClient extends AbstractJsonRPCClient {

	private String uri = null;
	private String schema = null;
	private Charset responseCharset = null;
	private HttpClient httpClient = null;

	public static final String DEFAULT_URI = "/";
	public static final String DEFAULT_SCHEMA = "http";
	public static final Charset DEFAULT_RESPONSE_CHARSET = null;

	/**
	 * 
	 * @throws JsonRPCException
	 * @since 1.0
	 */
	public CoinAPIJsonRPCClient () throws JsonRPCException {
		this(null,null,null,null);
	}

	/**
	 * 
	 * @param httpClient
	 * @throws JsonRPCException
	 * @since 1.0
	 */
	public CoinAPIJsonRPCClient (HttpClient httpClient) throws JsonRPCException {
		this(httpClient, null, null, null);
	}

	/**
	 * 
	 * @param httpClient
	 * @param uri
	 * @param schema
	 * @param responseCharset
	 * @throws JsonRPCException
	 * @since 1.0
	 */
	public CoinAPIJsonRPCClient (HttpClient httpClient, String uri, String schema, Charset responseCharset) throws JsonRPCException {
		super();
		this.setHttpClient(httpClient);
		this.setUri(uri);
		this.setSchema(schema);
		this.setResponseCharset(responseCharset);
	}

	/*
	 * (非 Javadoc)
	 * @see org.monache.lib.jsonrpc.AbstractJsonRPCClient#request(java.lang.String)
	 * @since 1.0
	 */
	@Override
	protected String request(String jsonRequestString) throws JsonRPCException {
		try {
			if (getHttpClient() == null) {
				throw new NullPointerException("HttpClient is null.");
			}
			JsonPostParams postParams = new JsonPostParams();
			postParams.setUri(getUri() != null ? getUri() : DEFAULT_URI);
			postParams.setSchema(getSchema() != null ? getSchema() : DEFAULT_SCHEMA);
			postParams.setData(jsonRequestString);
			postParams.setResponseCharset(getResponseCharset() != null ? getResponseCharset() : DEFAULT_RESPONSE_CHARSET);
			HttpClientResult result = getHttpClient().post(postParams);
			return result.getContent();
		} catch (Exception e) {
			throw new CoinAPIJsonRPCException("JSON-RPC error: " + e.getMessage(), e);
		}
	}

	/*
	 * (非 Javadoc)
	 * @see org.monache.lib.jsonrpc.AbstractJsonRPCClient#call(java.lang.Object, java.lang.Class)
	 * @since 1.0
	 */
	@Override
	public <T1, T2> T2 call(T1 requestObject, Class<? extends T2> responseObjectClass) throws JsonRPCException {
		T2 responseObject = super.call(requestObject, responseObjectClass);
		if (responseObject instanceof JsonRPCv1ResponseObject) {
			JsonRPCv1ResponseObject responseObject2 = (JsonRPCv1ResponseObject) responseObject;
			if (responseObject2.getError() != null) {
				throw new CoinAPIJsonRPCException("Server return error, code = " + responseObject2.getError().getCode() + ", message = " + responseObject2.getError().getMessage(), responseObject2.getError());
			}
		} else if (responseObject instanceof JsonRPCv2ResponseObject) {
			JsonRPCv2ResponseObject responseObject2 = (JsonRPCv2ResponseObject) responseObject;
			if (responseObject2.getError() != null) {
				throw new CoinAPIJsonRPCException("Server return error, code = " + responseObject2.getError().getCode() + ", message = " + responseObject2.getError().getMessage(), responseObject2.getError());
			}
		}
		return responseObject;
	}

	/**
	 * @return httpClient
	 * @since 1.0
	 */
	public HttpClient getHttpClient() {
		return httpClient;
	}

	/**
	 * @param httpClient セットする httpClient
	 * @since 1.0
	 */
	public void setHttpClient(HttpClient httpClient) {
		this.httpClient = httpClient;
	}

	/**
	 * @return uri
	 * @since 1.0
	 */
	public String getUri() {
		return uri;
	}

	/**
	 * @param uri セットする uri
	 * @since 1.0
	 */
	public void setUri(String uri) {
		this.uri = uri;
	}

	/**
	 * @return schema
	 * @since 1.0
	 */
	public String getSchema() {
		return schema;
	}

	/**
	 * @param schema セットする schema
	 * @since 1.0
	 */
	public void setSchema(String schema) {
		this.schema = schema;
	}

	/**
	 * @return responseCharset
	 * @since 1.0
	 */
	public Charset getResponseCharset() {
		return responseCharset;
	}

	/**
	 * @param responseCharset セットする responseCharset
	 * @since 1.0
	 */
	public void setResponseCharset(Charset responseCharset) {
		this.responseCharset = responseCharset;
	}

	/*
	 * (非 Javadoc)
	 * @see org.monache.lib.jsonrpc.JsonRPCClient#close()
	 * @since 1.0
	 */
	public void close() throws JsonRPCException {
		if (getHttpClient() != null) {
			try {
				getHttpClient().close();
			} catch (HttpClientException e) {
				throw new CoinAPIJsonRPCException(e);
			}
		}
	}
	
	

}
