/*
 * GetTransactionTestKuma.java
 *
 * Copyright 2014 the Monache.org.
 * http://www.monache.org/
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */

package org.monache.lib.api;

import java.math.BigDecimal;

import org.junit.After;
import org.junit.AfterClass;
import org.junit.Before;
import org.junit.BeforeClass;
import org.junit.Test;
import org.monache.lib.TestResources;
import org.monache.lib.api.impl.DefaultCoinAPIFactory;
import org.monache.lib.api.result.Details;
import org.monache.lib.api.result.GetTransactionResult;
import org.monache.lib.api.result.Vin;
import org.monache.lib.api.result.Vout;
import org.monache.lib.jsonrpc.JsonRPCClient;
import org.seasar.framework.container.S2Container;
import org.seasar.framework.container.factory.S2ContainerFactory;

import static org.junit.Assert.*;


/**
 * GetTransactionのテスト(kumacoin)。
 * 
 * @version 
 * @author monache.org
 * @since 
 */
public class GetTransactionTestKuma {

	static S2Container container = null;
	static DefaultCoinAPIFactory factory = null;
	static JsonRPCClient jsonRPC = null;
	
	/**
	 * 
	 * @throws java.lang.Exception
	 * @since 
	 */
	@BeforeClass
	public static void setUpBeforeClass() throws Exception {
		container = S2ContainerFactory.create("test-kumacoin.dicon");
		jsonRPC = (JsonRPCClient)container.getComponent(JsonRPCClient.class);
		factory = DefaultCoinAPIFactory.newInstance();
	}

	/**
	 * @throws java.lang.Exception
	 * @since 
	 */
	@AfterClass
	public static void tearDownAfterClass() throws Exception {
		jsonRPC.close();
	}

	/**
	 * @throws java.lang.Exception
	 * @since 
	 */
	@Before
	public void setUp() throws Exception {
	}

	/**
	 * @throws java.lang.Exception
	 * @since 
	 */
	@After
	public void tearDown() throws Exception {
	}

	/**
	 * Test
	 * GetTransactionResult.call(JsonRPCClient jsonRPCClient, String transactionId)
	 * 
	 * 【テスト仕様】
	 * 受信の場合、コンソールでの実行結果とJSON-RPCでの実行結果を比較する。
	 * gettransaction ce2e2691ff718cf0553759195096958760f11e778f6b90194575b904a23f298d
	 * 
	 * 【確認事項】
	 * 結果が一致すること。
	 * 
	 * @throws Exception
	 * @since 1.0
	 */
	@Test
	public void test1() throws Exception {

		GetTransaction obj = (GetTransaction)container.getComponent(GetTransaction.class);
		TestResources resources = (TestResources)container.getComponent(TestResources.class);
		if (resources.isTestGetTransaction_test1() == false) {
			System.out.println("test skip GetTransactionTestKuma.test1");
			return;
		}

		System.out.println("***** GetTransactionTestKuma.test1 *********************************");
		GetTransactionResult result = obj.call(jsonRPC, "ce2e2691ff718cf0553759195096958760f11e778f6b90194575b904a23f298d");
		out(result);
		
		assertEquals("ce2e2691ff718cf0553759195096958760f11e778f6b90194575b904a23f298d", result.getTxid());
		assertEquals(new Long(1), result.getVersion());
		assertEquals(new Long(1400064211), result.getTime());
		assertEquals(new Long(0), result.getLocktime());
		
		Vin[] vin = result.getVin();
		assertEquals("93a239c7bad1177ab95ea5812463f77d5fd58c18c57a11319dd15bb6847f074f", vin[0].getTxid());
		assertEquals(new Long(1), vin[0].getVout());
		assertEquals("3045022100ae2f0cba383d6fe092419cbcbc01c7e4e2c4ede9119d77657be1347cdbe5a018022010b3ec64d344f0a23326c05135f00a75208ae7326a849efde7c5c5a6343c7f7101 036c172e7c01419a063763653f1105284c350876fb846a3eea5d5137171883a514", vin[0].getScriptSig().getAsm());
		assertEquals("483045022100ae2f0cba383d6fe092419cbcbc01c7e4e2c4ede9119d77657be1347cdbe5a018022010b3ec64d344f0a23326c05135f00a75208ae7326a849efde7c5c5a6343c7f710121036c172e7c01419a063763653f1105284c350876fb846a3eea5d5137171883a514", vin[0].getScriptSig().getHex());
		assertEquals(new Long("4294967295"), vin[0].getSequence());

		assertEquals("cfec848bbd300bcd37c0230a96bd20c5ae68ef5cfd2a8f0b5b1b203ec9fae0eb", vin[1].getTxid());
		assertEquals(new Long(1), vin[1].getVout());
		assertEquals("3046022100871098791b3fe08fc9f00c8eae99f6c2321cb08c1465ef4e1cf20fa5df045f67022100d32fe5f73fe926fa984135d1788d194b6668acc149d831472d972e533e4a41ce01 036c172e7c01419a063763653f1105284c350876fb846a3eea5d5137171883a514", vin[1].getScriptSig().getAsm());
		assertEquals("493046022100871098791b3fe08fc9f00c8eae99f6c2321cb08c1465ef4e1cf20fa5df045f67022100d32fe5f73fe926fa984135d1788d194b6668acc149d831472d972e533e4a41ce0121036c172e7c01419a063763653f1105284c350876fb846a3eea5d5137171883a514", vin[1].getScriptSig().getHex());
		assertEquals(new Long("4294967295"), vin[1].getSequence());

		assertEquals("0239dd50c4f4b65dbfa0072e6abd3e904b7d2d9daf87cbc17c44f4f78836acf3", vin[2].getTxid());
		assertEquals(new Long(1), vin[2].getVout());
		assertEquals("3045022100ad5f296c4f5497336140c66243da445748e2299a4a0bf19f4a83ccc1d13ba14e02201565cc0a21211a57c85440fcd04b0aad12aab713acb770deb369ed3645c1558801 036c172e7c01419a063763653f1105284c350876fb846a3eea5d5137171883a514", vin[2].getScriptSig().getAsm());
		assertEquals("483045022100ad5f296c4f5497336140c66243da445748e2299a4a0bf19f4a83ccc1d13ba14e02201565cc0a21211a57c85440fcd04b0aad12aab713acb770deb369ed3645c155880121036c172e7c01419a063763653f1105284c350876fb846a3eea5d5137171883a514", vin[2].getScriptSig().getHex());
		assertEquals(new Long("4294967295"), vin[2].getSequence());

		Vout[] vout = result.getVout();
		assertEquals(new BigDecimal("999.99000000"), vout[0].getValue());
		assertEquals(new Long("0"), vout[0].getN());
		assertEquals("OP_DUP OP_HASH160 bc56f4975e095c6d8c5b649d47c8aadc068bb556 OP_EQUALVERIFY OP_CHECKSIG", vout[0].getScriptPubKey().getAsm());
		assertEquals("76a914bc56f4975e095c6d8c5b649d47c8aadc068bb55688ac", vout[0].getScriptPubKey().getHex());
		assertEquals(new Long("1"), vout[0].getScriptPubKey().getReqSigs());
		assertEquals("pubkeyhash", vout[0].getScriptPubKey().getType());
		assertEquals("KQNzWimL7CXoPGA1xRsjMpT8Tmaw637dZY", vout[0].getScriptPubKey().getAddresses()[0]);

		assertEquals(new BigDecimal("13000.00000000"), vout[1].getValue());
		assertEquals(new Long("1"), vout[1].getN());
		assertEquals("OP_DUP OP_HASH160 b516e35b64aee9805dcd3ff3a930ed94d3a633a7 OP_EQUALVERIFY OP_CHECKSIG", vout[1].getScriptPubKey().getAsm());
		assertEquals("76a914b516e35b64aee9805dcd3ff3a930ed94d3a633a788ac", vout[1].getScriptPubKey().getHex());
		assertEquals(new Long("1"), vout[1].getScriptPubKey().getReqSigs());
		assertEquals("pubkeyhash", vout[1].getScriptPubKey().getType());
		assertEquals("KPif2uJRjzUJVpNjHjUmDB2nrX474Njrj3", vout[1].getScriptPubKey().getAddresses()[0]);

		assertEquals(new BigDecimal("13000.00000000"), result.getAmount());
		assertNotNull(result.getConfirmations());
		assertEquals("000000001789312697a57d98a6ee0023311a48ac612c63032d84647e40b949db", result.getBlockhash());
		assertEquals(new Long(1), result.getBlockindex());
		assertEquals(new Long(1400064316), result.getBlocktime());
		assertEquals(new Long(1400064211), result.getTimereceived());
		assertEquals(null, result.getComment());
		assertEquals(null, result.getMessage());
		
		Details[] details = result.getDetails();
		assertEquals("develop", details[0].getAccount());
		assertEquals("KPif2uJRjzUJVpNjHjUmDB2nrX474Njrj3", details[0].getAddress());
		assertEquals("receive", details[0].getCategory());
		assertEquals(new BigDecimal("13000.00000000"), details[0].getAmount());
		
	}

	/**
	 * Test
	 * GetTransactionResult.call(JsonRPCClient jsonRPCClient, String transactionId)
	 * 
	 * 【テスト仕様】
	 * 送信の場合、コンソールでの実行結果とJSON-RPCでの実行結果を比較する。
	 * gettransaction b5ded1d59d1f0e5b294ab73ad2aa05cde7a09fec2d0001f3818a5598f0193acd
	 * 
	 * 【確認事項】
	 * 結果が一致すること。
	 * 
	 * @throws Exception
	 * @since 1.0
	 */
	@Test
	public void test2() throws Exception {
		GetTransaction obj = (GetTransaction)container.getComponent(GetTransaction.class);
		TestResources resources = (TestResources)container.getComponent(TestResources.class);
		if (resources.isTestGetTransaction_test2() == false) {
			System.out.println("test skip GetTransactionTestKuma.test2");
			return;
		}

		System.out.println("***** GetTransactionTestKuma.test2 *********************************");
		GetTransactionResult result = obj.call(jsonRPC, "b5ded1d59d1f0e5b294ab73ad2aa05cde7a09fec2d0001f3818a5598f0193acd");
		out(result);

		System.out.println("message = " + result.getMessage());
		
		assertEquals("b5ded1d59d1f0e5b294ab73ad2aa05cde7a09fec2d0001f3818a5598f0193acd", result.getTxid());
		assertEquals(new Long(1), result.getVersion());
		assertEquals(new Long(1400952262), result.getTime());
		assertEquals(new Long(0), result.getLocktime());
		
		Vin[] vin = result.getVin();
		assertEquals("e44a61516ba93934d759daa95dfd3b4c4235bfa52770df5f123fa94ca9de99a2", vin[0].getTxid());
		assertEquals(new Long(0), vin[0].getVout());
		assertEquals("3044022003fe8235cbccd6b9f12cca5fe62122d71c4979cba4932d671a60c22a1c693d6b02203b81fb3bea498b72c7147c4d0e10593925f8265fe7f4a73c4bd49c452d3316c301 02090d71adebc5df065feee9698a028a8a22780c66fedea6c97d0d72a5349f5626", vin[0].getScriptSig().getAsm());
		assertEquals("473044022003fe8235cbccd6b9f12cca5fe62122d71c4979cba4932d671a60c22a1c693d6b02203b81fb3bea498b72c7147c4d0e10593925f8265fe7f4a73c4bd49c452d3316c3012102090d71adebc5df065feee9698a028a8a22780c66fedea6c97d0d72a5349f5626", vin[0].getScriptSig().getHex());
		assertEquals(new Long("4294967295"), vin[0].getSequence());

		Vout[] vout = result.getVout();
		assertEquals(new BigDecimal("107.74543200"), vout[0].getValue());
		assertEquals(new Long("0"), vout[0].getN());
		assertEquals("OP_DUP OP_HASH160 2bc8612c92ae3ce54adad8e19f8acfa26761f7d2 OP_EQUALVERIFY OP_CHECKSIG", vout[0].getScriptPubKey().getAsm());
		assertEquals("76a9142bc8612c92ae3ce54adad8e19f8acfa26761f7d288ac", vout[0].getScriptPubKey().getHex());
		assertEquals(new Long("1"), vout[0].getScriptPubKey().getReqSigs());
		assertEquals("pubkeyhash", vout[0].getScriptPubKey().getType());
		assertEquals("KBCeQkfhfLfWwqVFtVMYRf9Qurn19E13gC", vout[0].getScriptPubKey().getAddresses()[0]);

		assertEquals(new BigDecimal("1.00000000"), vout[1].getValue());
		assertEquals(new Long("1"), vout[1].getN());
		assertEquals("OP_DUP OP_HASH160 a3df06c2dafc77471b9b0a6295ef68833a834b78 OP_EQUALVERIFY OP_CHECKSIG", vout[1].getScriptPubKey().getAsm());
		assertEquals("76a914a3df06c2dafc77471b9b0a6295ef68833a834b7888ac", vout[1].getScriptPubKey().getHex());
		assertEquals(new Long("1"), vout[1].getScriptPubKey().getReqSigs());
		assertEquals("pubkeyhash", vout[1].getScriptPubKey().getType());
		assertEquals("KN9cdJVAvivaE82XjLL3k3hDuQbiqriiCB", vout[1].getScriptPubKey().getAddresses()[0]);

		assertEquals(new BigDecimal("1.00000000"), vout[2].getValue());
		assertEquals(new Long("2"), vout[2].getN());
		assertEquals("OP_DUP OP_HASH160 b516e35b64aee9805dcd3ff3a930ed94d3a633a7 OP_EQUALVERIFY OP_CHECKSIG", vout[2].getScriptPubKey().getAsm());
		assertEquals("76a914b516e35b64aee9805dcd3ff3a930ed94d3a633a788ac", vout[2].getScriptPubKey().getHex());
		assertEquals(new Long("1"), vout[2].getScriptPubKey().getReqSigs());
		assertEquals("pubkeyhash", vout[2].getScriptPubKey().getType());
		assertEquals("KPif2uJRjzUJVpNjHjUmDB2nrX474Njrj3", vout[2].getScriptPubKey().getAddresses()[0]);

		assertEquals(new BigDecimal("-1.00000000"), result.getAmount());
		assertEquals(new BigDecimal("-0.01000000"), result.getFee());
		assertNotNull(result.getConfirmations());
		assertEquals("0000000003b8fbbf82c47eef134c1873cce0904bf899e8cfa8b9240a5a45c24e", result.getBlockhash());
		assertEquals(new Long(2), result.getBlockindex());
		assertEquals(new Long(1400952326), result.getBlocktime());
		assertEquals(new Long(1400952262), result.getTimereceived());
		assertEquals(null, result.getComment());
		assertEquals(null, result.getMessage());
		
		Details[] details = result.getDetails();
		assertEquals("develop", details[0].getAccount());
		assertEquals("KN9cdJVAvivaE82XjLL3k3hDuQbiqriiCB", details[0].getAddress());
		assertEquals("send", details[0].getCategory());
		assertEquals(new BigDecimal("-1.00000000"), details[0].getAmount());
		assertEquals(new BigDecimal("-0.01000000"), details[0].getFee());

		assertEquals("develop", details[1].getAccount());
		assertEquals("KPif2uJRjzUJVpNjHjUmDB2nrX474Njrj3", details[1].getAddress());
		assertEquals("send", details[1].getCategory());
		assertEquals(new BigDecimal("-1.00000000"), details[1].getAmount());
		assertEquals(new BigDecimal("-0.01000000"), details[1].getFee());

		assertEquals("develop", details[2].getAccount());
		assertEquals("KPif2uJRjzUJVpNjHjUmDB2nrX474Njrj3", details[2].getAddress());
		assertEquals("receive", details[2].getCategory());
		assertEquals(new BigDecimal("1.00000000"), details[2].getAmount());

	}

	/**
	 * Test
	 * GetTransactionResult.call(JsonRPCClient jsonRPCClient, String transactionId)
	 * 
	 * 【テスト仕様】
	 * 送信の場合、コンソールでの実行結果とJSON-RPCでの実行結果を比較する。
	 * gettransaction b5ded1d59d1f0e5b294ab73ad2aa05cde7a09fec2d0001f3818a5598f0193acd
	 * 
	 * 【確認事項】
	 * 結果が一致すること。
	 * 
	 * @throws Exception
	 * @since 1.0
	 */
	@Test
	public void test3() throws Exception {
		GetTransaction obj = (GetTransaction)container.getComponent(GetTransaction.class);
		TestResources resources = (TestResources)container.getComponent(TestResources.class);
		if (resources.isTestGetTransaction_test3() == false) {
			System.out.println("test skip GetTransactionTestKuma.test3");
			return;
		}

		System.out.println("***** GetTransactionTestKuma.test3 *********************************");
		GetTransactionResult result = obj.call(jsonRPC, "7e38c9023af7b17a5442dc818034f1e8616912b2b4a8c5e01c207dced8511fbf");
		out(result);

		assertEquals("7e38c9023af7b17a5442dc818034f1e8616912b2b4a8c5e01c207dced8511fbf", result.getTxid());
		assertEquals("commentA", result.getComment());
		assertEquals("", result.getMessage());
		assertEquals("commenttoA", result.getTo());
		
	}



	
	/**
	 * Test(Factory)
	 * GetTransactionResult.call(JsonRPCClient jsonRPCClient, String transactionId)
	 * 
	 * 【テスト仕様】
	 * 受信の場合、コンソールでの実行結果とJSON-RPCでの実行結果を比較する。
	 * gettransaction b5ded1d59d1f0e5b294ab73ad2aa05cde7a09fec2d0001f3818a5598f0193acd
	 * テストにはFactoryを利用する。
	 * 
	 * 【確認事項】
	 * 結果が一致すること。
	 * 
	 * @throws Exception
	 * @since 1.0
	 */
	@Test
	public void testFactory() throws Exception {
		// Factory
		GetTransaction obj = factory.getGetTransaction();
		TestResources resources = (TestResources)container.getComponent(TestResources.class);
		if (resources.isTestGetTransaction_testFactory() == false) {
			System.out.println("test skip GetTransactionTestKuma.testFactory");
			return;
		}

		System.out.println("***** GetTransactionTestKuma.test1 *********************************");
		GetTransactionResult result = obj.call(jsonRPC, "ce2e2691ff718cf0553759195096958760f11e778f6b90194575b904a23f298d");
		out(result);
		
		assertEquals("ce2e2691ff718cf0553759195096958760f11e778f6b90194575b904a23f298d", result.getTxid());
		assertEquals(new Long(1), result.getVersion());
		assertEquals(new Long(1400064211), result.getTime());
		assertEquals(new Long(0), result.getLocktime());
		
		Vin[] vin = result.getVin();
		assertEquals("93a239c7bad1177ab95ea5812463f77d5fd58c18c57a11319dd15bb6847f074f", vin[0].getTxid());
		assertEquals(new Long(1), vin[0].getVout());
		assertEquals("3045022100ae2f0cba383d6fe092419cbcbc01c7e4e2c4ede9119d77657be1347cdbe5a018022010b3ec64d344f0a23326c05135f00a75208ae7326a849efde7c5c5a6343c7f7101 036c172e7c01419a063763653f1105284c350876fb846a3eea5d5137171883a514", vin[0].getScriptSig().getAsm());
		assertEquals("483045022100ae2f0cba383d6fe092419cbcbc01c7e4e2c4ede9119d77657be1347cdbe5a018022010b3ec64d344f0a23326c05135f00a75208ae7326a849efde7c5c5a6343c7f710121036c172e7c01419a063763653f1105284c350876fb846a3eea5d5137171883a514", vin[0].getScriptSig().getHex());
		assertEquals(new Long("4294967295"), vin[0].getSequence());

		assertEquals("cfec848bbd300bcd37c0230a96bd20c5ae68ef5cfd2a8f0b5b1b203ec9fae0eb", vin[1].getTxid());
		assertEquals(new Long(1), vin[1].getVout());
		assertEquals("3046022100871098791b3fe08fc9f00c8eae99f6c2321cb08c1465ef4e1cf20fa5df045f67022100d32fe5f73fe926fa984135d1788d194b6668acc149d831472d972e533e4a41ce01 036c172e7c01419a063763653f1105284c350876fb846a3eea5d5137171883a514", vin[1].getScriptSig().getAsm());
		assertEquals("493046022100871098791b3fe08fc9f00c8eae99f6c2321cb08c1465ef4e1cf20fa5df045f67022100d32fe5f73fe926fa984135d1788d194b6668acc149d831472d972e533e4a41ce0121036c172e7c01419a063763653f1105284c350876fb846a3eea5d5137171883a514", vin[1].getScriptSig().getHex());
		assertEquals(new Long("4294967295"), vin[1].getSequence());

		assertEquals("0239dd50c4f4b65dbfa0072e6abd3e904b7d2d9daf87cbc17c44f4f78836acf3", vin[2].getTxid());
		assertEquals(new Long(1), vin[2].getVout());
		assertEquals("3045022100ad5f296c4f5497336140c66243da445748e2299a4a0bf19f4a83ccc1d13ba14e02201565cc0a21211a57c85440fcd04b0aad12aab713acb770deb369ed3645c1558801 036c172e7c01419a063763653f1105284c350876fb846a3eea5d5137171883a514", vin[2].getScriptSig().getAsm());
		assertEquals("483045022100ad5f296c4f5497336140c66243da445748e2299a4a0bf19f4a83ccc1d13ba14e02201565cc0a21211a57c85440fcd04b0aad12aab713acb770deb369ed3645c155880121036c172e7c01419a063763653f1105284c350876fb846a3eea5d5137171883a514", vin[2].getScriptSig().getHex());
		assertEquals(new Long("4294967295"), vin[2].getSequence());

		Vout[] vout = result.getVout();
		assertEquals(new BigDecimal("999.99000000"), vout[0].getValue());
		assertEquals(new Long("0"), vout[0].getN());
		assertEquals("OP_DUP OP_HASH160 bc56f4975e095c6d8c5b649d47c8aadc068bb556 OP_EQUALVERIFY OP_CHECKSIG", vout[0].getScriptPubKey().getAsm());
		assertEquals("76a914bc56f4975e095c6d8c5b649d47c8aadc068bb55688ac", vout[0].getScriptPubKey().getHex());
		assertEquals(new Long("1"), vout[0].getScriptPubKey().getReqSigs());
		assertEquals("pubkeyhash", vout[0].getScriptPubKey().getType());
		assertEquals("KQNzWimL7CXoPGA1xRsjMpT8Tmaw637dZY", vout[0].getScriptPubKey().getAddresses()[0]);

		assertEquals(new BigDecimal("13000.00000000"), vout[1].getValue());
		assertEquals(new Long("1"), vout[1].getN());
		assertEquals("OP_DUP OP_HASH160 b516e35b64aee9805dcd3ff3a930ed94d3a633a7 OP_EQUALVERIFY OP_CHECKSIG", vout[1].getScriptPubKey().getAsm());
		assertEquals("76a914b516e35b64aee9805dcd3ff3a930ed94d3a633a788ac", vout[1].getScriptPubKey().getHex());
		assertEquals(new Long("1"), vout[1].getScriptPubKey().getReqSigs());
		assertEquals("pubkeyhash", vout[1].getScriptPubKey().getType());
		assertEquals("KPif2uJRjzUJVpNjHjUmDB2nrX474Njrj3", vout[1].getScriptPubKey().getAddresses()[0]);

		assertEquals(new BigDecimal("13000.00000000"), result.getAmount());
		assertNotNull(result.getConfirmations());
		assertEquals("000000001789312697a57d98a6ee0023311a48ac612c63032d84647e40b949db", result.getBlockhash());
		assertEquals(new Long(1), result.getBlockindex());
		assertEquals(new Long(1400064316), result.getBlocktime());
		assertEquals(new Long(1400064211), result.getTimereceived());
		assertEquals(null, result.getComment());
		assertEquals(null, result.getMessage());
		
		Details[] details = result.getDetails();
		assertEquals("develop", details[0].getAccount());
		assertEquals("KPif2uJRjzUJVpNjHjUmDB2nrX474Njrj3", details[0].getAddress());
		assertEquals("receive", details[0].getCategory());
		assertEquals(new BigDecimal("13000.00000000"), details[0].getAmount());
		
	}
	
	private void out(GetTransactionResult result ) {
		System.out.println("json = " + result.getJsonString());
		System.out.println("txid = " + result.getTxid());
		System.out.println("version = " + result.getVersion());
		System.out.println("time = " + result.getTime());
		System.out.println("locktime = " + result.getLocktime());
		
		int i = 0;
		for (Vin vin : result.getVin()) {
			System.out.println("** vin " + i + " **");			
			System.out.println("txid = " + vin.getTxid());
			System.out.println("vout = " + vin.getVout());
			System.out.println("asm = " + vin.getScriptSig().getAsm());
			System.out.println("hex = " + vin.getScriptSig().getHex());
			System.out.println("sequence = " + vin.getSequence());
		}

		i = 0;
		for (Vout vout : result.getVout()) {
			System.out.println("** vout " + i + " **");			
			System.out.println("value = " + vout.getValue());
			System.out.println("n = " + vout.getN());
			System.out.println("asm = " + vout.getScriptPubKey().getAsm());
			System.out.println("hex = " + vout.getScriptPubKey().getHex());
			System.out.println("reqSigs = " + vout.getScriptPubKey().getReqSigs());
			System.out.println("type = " + vout.getScriptPubKey().getType());
			for (String address : vout.getScriptPubKey().getAddresses()) {
				System.out.println("address = " + address);
			}
		}
		
		System.out.println("amount = " + result.getAmount());
		System.out.println("confirmations = " + result.getConfirmations());
		System.out.println("blockhash = " + result.getBlockhash());
		System.out.println("blockindex = " + result.getBlockindex());
		System.out.println("blocktime = " + result.getBlocktime());
		System.out.println("timereceived = " + result.getTimereceived());
		System.out.println("comment = " + result.getComment());
		System.out.println("message = " + result.getMessage());
		System.out.println("to = " + result.getTo());

		i = 0;
		for (Details details : result.getDetails()) {
			System.out.println("** details " + i + " **");			
			System.out.println("account = " + details.getAccount());
			System.out.println("address = " + details.getAddress());
			System.out.println("category = " + details.getCategory());
			System.out.println("amount = " + details.getAmount());
			System.out.println("fee = " + details.getFee());
		}

	}

}
