package monalipse.views;

import java.io.UnsupportedEncodingException;
import java.net.MalformedURLException;
import java.net.URL;
import java.net.URLDecoder;
import java.net.URLEncoder;
import java.util.Arrays;
import java.util.StringTokenizer;
import java.util.logging.Logger;

import monalipse.MonalipsePlugin;
import monalipse.actions.BookmarkAction;
import monalipse.bookmark.BookmarkManager;
import monalipse.editors.ThreadEditorInput;
import monalipse.editors.ThreadViewerEditor;
import monalipse.server.BBSServerManager;
import monalipse.server.IBBSBoard;
import monalipse.server.IBBSReference;
import monalipse.server.IThreadContentProvider;
import monalipse.utils.CancelableRunner;
import monalipse.utils.CancelableRunner.ICancelableProgressMonitor;

import org.eclipse.core.resources.IResource;
import org.eclipse.core.resources.IResourceChangeEvent;
import org.eclipse.core.resources.IResourceChangeListener;
import org.eclipse.core.resources.ResourcesPlugin;
import org.eclipse.core.runtime.CoreException;
import org.eclipse.core.runtime.Platform;
import org.eclipse.core.runtime.QualifiedName;
import org.eclipse.jface.action.Action;
import org.eclipse.jface.action.ControlContribution;
import org.eclipse.jface.action.IAction;
import org.eclipse.jface.action.IMenuListener;
import org.eclipse.jface.action.IMenuManager;
import org.eclipse.jface.action.IStatusLineManager;
import org.eclipse.jface.action.IToolBarManager;
import org.eclipse.jface.action.MenuManager;
import org.eclipse.jface.action.Separator;
import org.eclipse.jface.resource.ImageDescriptor;
import org.eclipse.jface.text.ITextOperationTarget;
import org.eclipse.jface.viewers.DoubleClickEvent;
import org.eclipse.jface.viewers.IDoubleClickListener;
import org.eclipse.jface.viewers.ISelection;
import org.eclipse.jface.viewers.IStructuredContentProvider;
import org.eclipse.jface.viewers.IStructuredSelection;
import org.eclipse.jface.viewers.ITableLabelProvider;
import org.eclipse.jface.viewers.ITreeContentProvider;
import org.eclipse.jface.viewers.LabelProvider;
import org.eclipse.jface.viewers.TableTreeViewer;
import org.eclipse.jface.viewers.Viewer;
import org.eclipse.jface.viewers.ViewerSorter;
import org.eclipse.swt.SWT;
import org.eclipse.swt.custom.TableTree;
import org.eclipse.swt.events.FocusAdapter;
import org.eclipse.swt.events.FocusEvent;
import org.eclipse.swt.events.FocusListener;
import org.eclipse.swt.events.KeyAdapter;
import org.eclipse.swt.events.KeyEvent;
import org.eclipse.swt.events.SelectionEvent;
import org.eclipse.swt.events.SelectionListener;
import org.eclipse.swt.graphics.Image;
import org.eclipse.swt.widgets.Combo;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Control;
import org.eclipse.swt.widgets.Menu;
import org.eclipse.swt.widgets.TableColumn;
import org.eclipse.ui.IActionBars;
import org.eclipse.ui.IEditorPart;
import org.eclipse.ui.IMemento;
import org.eclipse.ui.ISelectionListener;
import org.eclipse.ui.IWorkbenchActionConstants;
import org.eclipse.ui.IWorkbenchPage;
import org.eclipse.ui.IWorkbenchPart;
import org.eclipse.ui.PartInitException;
import org.eclipse.ui.XMLMemento;
import org.eclipse.ui.internal.WorkbenchImages;
import org.eclipse.ui.internal.WorkbenchMessages;

public class ThreadListView extends AbstractTableTreeView implements ISelectionListener, IResourceChangeListener, SelectionListener
{
	public static final String ID_THREAD_LIST = ThreadListView.class.getName();

	private static final Logger logger = MonalipsePlugin.getLogger();

	private static final String TAG_INPUT = "input";
	private static final String TAG_URL = "url";
	private static final String TAG_TITLE = "title";

	private TableTreeViewer viewer;
	private ThreadListContentProvider provider;
	private CancelableRunner cancelable;
	private IAction reloadAction;
	private IAction abortAction;
	private IAction doubleClickAction;
	private IAction cutAction;
	private IAction copyAction;
	private IAction pasteAction;
	private IAction deleteAction;
	private IAction selectAllAction;
	private BookmarkAction bookmarkAction;
	private Combo searchCombo;
	private String constraint;
	private FocusListener focusListener;
	private IBBSBoard inputQueue;

	public ThreadListView()
	{
//logger.finest("enter");
		logger.finest("<init>");
	}
	
	protected ITreeContentProvider getContentProvider()
	{
		return provider;
	}
	
	protected ITableLabelProvider getLabelProvider()
	{
		return provider;
	}

	protected Logger getLogger()
	{
		return logger;
	}

	protected TableTreeViewer getViewer()
	{
		return viewer;
	}

	public void createPartControl(Composite parent)
	{
//logger.finest("enter");
		logger.finest("begin");
		viewer = new TableTreeViewer(parent, SWT.SINGLE | SWT.H_SCROLL | SWT.V_SCROLL | SWT.FULL_SELECTION);
		provider = new ThreadListContentProvider();
		viewer.setContentProvider(provider);
		viewer.setLabelProvider(provider);
//		viewer.setSorter(new NameSorter());
		viewer.setInput(ResourcesPlugin.getWorkspace());
		makeColumns();
		makeActions();
		hookContextMenu();
		hookDoubleClickAction();
		contributeToActionBars();
		initializeTextActions();
		updateSelectionDependentActions();
		getSite().setSelectionProvider(viewer);
		ResourcesPlugin.getWorkspace().addResourceChangeListener(this);
		getViewSite().getPage().addSelectionListener(this);
		viewer.getTableTree().addSelectionListener(this);
		hookDragAndDrop();
		partCreated();
		logger.finest("done");

		focusListener = new FocusAdapter()
			{
				public void focusGained(FocusEvent e)
				{
					logger.finest("begin");
					updateSelectionDependentActions();
					ensureSearchComboInit();
					logger.finest("done");
				}
			};
		viewer.getControl().addFocusListener(focusListener);

		viewer.getTableTree().getTable().addKeyListener(new KeyAdapter()
			{
				public void keyPressed(KeyEvent e)
				{
					if(e.keyCode == SWT.F5)
						reloadAction.run();
				}
			});
	}
	
	public void dispose()
	{
//logger.finest("enter");
		ResourcesPlugin.getWorkspace().removeResourceChangeListener(this);
		getViewSite().getPage().removeSelectionListener(this);
		super.dispose();
	}

	public void resourceChanged(IResourceChangeEvent event)
	{
//logger.finest("enter");
		if(viewer.getInput() instanceof IBBSBoard)
		{
			logger.finest("begin");
			IBBSBoard board = (IBBSBoard)viewer.getInput();
			if(board.threadListChanged(event))
			{
				logger.finest("threadListChanged");
				getSite().getShell().getDisplay().asyncExec(new Runnable()
					{
						public void run()
						{
							refreshList();
							if(viewer.getVisibleExpandedElements().length == 0)
								expandFirstFragment();
						}
					});
			}

			final Object[] changed = board.getChangedItems(event);
			if(changed != null)
			{
				logger.finest("update");
				getSite().getShell().getDisplay().asyncExec(new Runnable()
					{
						public void run()
						{
							viewer.update(changed, null);
						}
					});
			}

			if(BookmarkManager.bookmarkChanged(event))
			{
				logger.finest("bookmarkChanged");
				getSite().getShell().getDisplay().asyncExec(new Runnable()
					{
						public void run()
						{
							setTitleImage(MonalipsePlugin.getDefault().getLabelImageOf((IBBSBoard)viewer.getInput()));
							viewer.refresh(true);
						}
					});
			}
			logger.finest("done");
		}
	}

	private void refreshList()
	{
//logger.finest("enter");
		if(provider != null)
		{
			logger.finest("begin");
			IMemento memento = XMLMemento.createWriteRoot(ThreadListView.class.getName());
			saveState(memento);
			logger.finest("reset input");
			viewer.setInput(viewer.getInput());
			restoreState(memento);
			logger.finest("done");
		}
	}
	
	public void selectionChanged(IWorkbenchPart part, ISelection selection)
	{
//logger.finest("enter");
		if(!viewer.getTableTree().isDisposed() && part != this)
		{
			if(selection instanceof IStructuredSelection)
			{
				Object first = ((IStructuredSelection)selection).getFirstElement();
				if(first instanceof IBBSReference)
				{
					IBBSBoard board = BBSServerManager.getBoardOf(((IBBSReference)first).getURL());
					if(board != null)
						selectionChanged(board);
				}
			}
		}
	}

	private void selectionChanged(IBBSBoard board)
	{
		if(inputQueue == null && viewer.getInput() == board)
			return;
		if(inputQueue != null && inputQueue == board)
			return;

		inputQueue = board;
		new Thread(new Runnable()
			{
				public void run()
				{
					getSite().getShell().getDisplay().syncExec(new Runnable()
						{
							public void run()
							{
							}
						});
					
					IEditorPart editor = getSite().getWorkbenchWindow().getActivePage().getActiveEditor();
					if(editor instanceof ThreadViewerEditor)
					{
						// for opening editor
						((ThreadViewerEditor)editor).getCancelable().runAndJoin(editor, new CancelableRunner.ICancelableRunnableWithProgress()
							{
								public void run(ICancelableProgressMonitor monitor)
								{
								}
							});

						// for updating cache
						((ThreadViewerEditor)editor).getCancelable().runAndJoin(editor, new CancelableRunner.ICancelableRunnableWithProgress()
							{
								public void run(ICancelableProgressMonitor monitor)
								{
								}
							});
					}
					
					IBBSBoard board = inputQueue;
					if(board != null)
						board.ensureCacheLoaded();

					getSite().getShell().getDisplay().asyncExec(new Runnable()
						{
							public void run()
							{
								if(inputQueue != null)
								{
									IBBSBoard board = inputQueue;
									inputQueue = null;
									setInput(board);
								}
							}
						});
				}
			}).start();
	}

	private void setInput(IBBSBoard board)
	{
//logger.finest("enter");
		logger.finest("begin");

		setTitle(board.getName());
		setTitleImage(MonalipsePlugin.getDefault().getLabelImageOf(board));

		if(viewer.getInput() == null || !viewer.getInput().equals(board))
		{
			logger.finest("folder changed");

			if(viewer.getInput() instanceof IBBSBoard)
				saveSearchWord(((IBBSBoard)viewer.getInput()).getLogFolder());

			constraint = "";
			viewer.setInput(board);
			restoreSearchWord(board.getLogFolder());

//			getSite().getShell().getDisplay().asyncExec(new Runnable()
//				{
//					public void run()
//					{
//						getSite().getPage().activate(ThreadListView.this);
//					}
//				});

			if(provider.getElements(viewer.getInput()).length == 0)
			{
				logger.finest("empty thread list");
				getSite().getShell().getDisplay().asyncExec(new Runnable()
					{
						public void run()
						{
							reloadAction.run();
						}
					});
			}
			else
			{
				expandFirstFragment();
			}
		}
		logger.finest("done");
	}

	private void expandFirstFragment()
	{
//logger.finest("enter");
		if(provider.hasChildren(viewer.getInput()))
		{
			logger.finest("begin");
			Object[] fragments = provider.getElements(viewer.getInput());
			if(0 < fragments.length && provider.hasChildren(fragments[0]))
			{
				logger.finest("set expansion");
				viewer.setExpandedElements(new Object[]{fragments[0]});
			}
			logger.finest("done");
		}
	}

	public void widgetSelected(SelectionEvent e)
	{
//logger.finest("enter");
		logger.finest("begin");
		IBBSReference ref = bookmarkAction.updateState(viewer.getSelection());
		if(ref instanceof IThreadContentProvider)
			ThreadViewerEditor.activateEditor(getSite(), (IThreadContentProvider)ref);
		logger.finest("done");
	}
	
	public void widgetDefaultSelected(SelectionEvent e)
	{
	}
	
	private void saveSearchWord(IResource res)
	{
//logger.finest("enter");
		if(searchCombo != null && !searchCombo.isDisposed())
		{
			try
			{
				logger.finest("begin");
				StringBuffer buf = new StringBuffer();
				String[] keys = searchCombo.getItems();
				for(int i = 0; i < keys.length && i < 10; i++)
					buf.append(URLEncoder.encode(keys[i], "UTF-8")).append(" ");
				logger.finest("save search word: " + buf.toString());
				res.setPersistentProperty(new QualifiedName(getClass().getName(), "search"), buf.toString());
			}
			catch (UnsupportedEncodingException e)
			{
			}
			catch (CoreException e)
			{
			}
		}
	}
	
	private void restoreSearchWord(IResource res)
	{
//logger.finest("enter");
		if(searchCombo != null && !searchCombo.isDisposed())
		{
			try
			{
				searchCombo.removeAll();
				searchCombo.add("");
				String keys = res.getPersistentProperty(new QualifiedName(getClass().getName(), "search"));
				if(keys != null)
				{
					logger.finest("restore search word: " + keys);
					StringTokenizer tk = new StringTokenizer(keys);
					while(tk.hasMoreTokens())
						searchCombo.add(URLDecoder.decode(tk.nextToken(), "UTF-8"));
				}
			}
			catch (UnsupportedEncodingException e)
			{
			}
			catch (CoreException e)
			{
			}
		}
	}

	public void saveState(IMemento memento)
	{
//logger.finest("enter");
		super.saveState(memento);
		getLogger().finest("begin");
		if(getViewer() != null)
			saveInputState(memento);
		getLogger().finest("done");
	}

	private void saveInputState(IMemento memento)
	{
//logger.finest("enter");
		IMemento elementMem = memento.createChild(TAG_INPUT);
		if(viewer.getInput() instanceof IBBSBoard)
			elementMem.putString(TAG_URL, ((IBBSBoard)viewer.getInput()).getURL().toExternalForm());
		elementMem.putString(TAG_TITLE, getTitle());
		if(viewer.getInput() instanceof IBBSBoard)
			saveSearchWord(((IBBSBoard)viewer.getInput()).getLogFolder());
	}

	protected void restoreState(IMemento memento)
	{
		restoreInputState(memento);
		super.restoreState(memento);
	}

	private void restoreInputState(IMemento memento)
	{
//logger.finest("enter");
		IMemento elementMem = memento.getChild(TAG_INPUT);
		if(elementMem != null)
		{
			String url = elementMem.getString(TAG_URL);
			logger.finest(url);
			if(url != null)
			{
				try
				{
					IBBSBoard board = BBSServerManager.getBoardOf(new URL(url));
					if(board != null)
						viewer.setInput(board);
				}
				catch (MalformedURLException e)
				{
				}
			}
			
			String title = elementMem.getString(TAG_TITLE);
			if(title != null)
				setTitle(title);
		}
	}

	private void hookContextMenu()
	{
		MenuManager menuMgr = new MenuManager("#PopupMenu");
		menuMgr.setRemoveAllWhenShown(true);
		menuMgr.addMenuListener(new IMenuListener()
			{
				public void menuAboutToShow(IMenuManager manager)
				{
					ThreadListView.this.fillContextMenu(manager);
				}
			});
		Menu menu = menuMgr.createContextMenu(viewer.getControl());
		viewer.getControl().setMenu(menu);
		getSite().registerContextMenu(menuMgr, viewer);
	}

	private void contributeToActionBars()
	{
		IActionBars bars = getViewSite().getActionBars();
//		fillLocalPullDown(bars.getMenuManager());
		fillLocalToolBar(bars.getToolBarManager());
		fillLocalStatuBar(bars.getStatusLineManager());
	}

	private void fillLocalStatuBar(IStatusLineManager manager)
	{
		cancelable = new CancelableRunner(manager, getSite().getShell().getDisplay(), abortAction);
	}

	private void fillLocalPullDown(IMenuManager manager)
	{
		manager.add(reloadAction);
		manager.add(new Separator());
		manager.add(abortAction);
	}

	private void fillContextMenu(IMenuManager manager)
	{
		bookmarkAction.updateState(viewer.getSelection());
		manager.add(reloadAction);
		manager.add(abortAction);
		manager.add(new Separator());
		manager.add(bookmarkAction);
		// Other plug-ins can contribute there actions here
		manager.add(new Separator("Additions"));
	}

	private void fillLocalToolBar(IToolBarManager manager)
	{
		manager.add(new ControlContribution(ThreadListView.class.getName() + ".constrain")
			{
				protected Control createControl(Composite parent)
				{
					ViewUtils.ComboInfo info = ViewUtils.createCombo(parent, SWT.DROP_DOWN | SWT.BORDER);
					searchCombo = info.combo;
					searchCombo.addFocusListener(focusListener);
					searchCombo.setToolTipText("Search");
					searchCombo.addSelectionListener(new SelectionListener()
						{
							public void widgetSelected(SelectionEvent e)
							{
								selected(e);
							}
	
							public void widgetDefaultSelected(SelectionEvent e)
							{
								selected(e);
							}
							
							private void selected(SelectionEvent e)
							{
								String text = searchCombo.getText();
								if(0 < text.length())
								{
									if(Arrays.asList(searchCombo.getItems()).contains(text))
										searchCombo.remove(text);
									searchCombo.add(text, 1);
								}
								constraint = text;
								refreshList();
								searchCombo.setText(text);
								if(viewer.getVisibleExpandedElements().length == 0)
									expandFirstFragment();
							}
						});
					return info.control;
				}
			});

		manager.add(reloadAction);
		manager.add(abortAction);
	}

	private void ensureSearchComboInit()
	{
//logger.finest("enter");
		if(searchCombo.getItemCount() == 0)
		{
			searchCombo.add("");
			if(viewer.getInput() instanceof IBBSBoard)
				restoreSearchWord(((IBBSBoard)viewer.getInput()).getLogFolder());
		}
	}

	private void makeColumns()
	{
		TableTree table = viewer.getTableTree();
		table.getTable().setHeaderVisible(true);
		table.getTable().setLinesVisible(true);
		TableColumn col;
		col = new TableColumn(table.getTable(), SWT.LEFT);
		col.setText("Rank");
		col.setWidth(80);
		col = new TableColumn(table.getTable(), SWT.LEFT);
		col.setText("Title");
		col.setWidth(400);
		col = new TableColumn(table.getTable(), SWT.LEFT);
		col.setText("Response");
		col.setWidth(50);
		col = new TableColumn(table.getTable(), SWT.LEFT);
		col.setText("Cache");
		col.setWidth(50);
		col = new TableColumn(table.getTable(), SWT.LEFT);
		col.setText("Read");
		col.setWidth(50);
	}

	private void makeActions()
	{
		String iconPath = "icons/"; //$NON-NLS-1$		
		URL installURL = Platform.getPlugin(MonalipsePlugin.PLUGIN_ID).getDescriptor().getInstallURL();

		reloadAction = new Action()
			{
				public void run()
				{
					cancelable.run(cancelable, new CancelableRunner.ICancelableRunnableWithProgress()
						{
							public void run(CancelableRunner.ICancelableProgressMonitor monitor)
							{
								if(viewer.getInput() instanceof IBBSBoard)
								{
									logger.finest("refresh");
									((IBBSBoard)viewer.getInput()).updateThreadList();
								}
							}
						});
				}
			};
		reloadAction.setText("Reload");
		reloadAction.setToolTipText("Reload thread list");
		try
		{
			reloadAction.setImageDescriptor(ImageDescriptor.createFromURL(new URL(installURL, iconPath + "refresh_nav.gif"))); //$NON-NLS-1$
		}
		catch (MalformedURLException e)
		{
		}

		abortAction = new Action()
			{
				public void run()
				{
					cancelable.cancel();
				}
			};
		abortAction.setText("Abort");
		abortAction.setToolTipText("Abort downloading board list");
		abortAction.setEnabled(false);
		try
		{
			abortAction.setImageDescriptor(ImageDescriptor.createFromURL(new URL(installURL, iconPath + "stop_nav.gif"))); //$NON-NLS-1$
		}
		catch (MalformedURLException e)
		{
		}

		doubleClickAction = new Action()
			{
				public void run()
				{
					ISelection selection = viewer.getSelection();
					if(selection instanceof IStructuredSelection)
					{
						Object first = ((IStructuredSelection)selection).getFirstElement();
						if(first instanceof IThreadContentProvider)
						{
							logger.finest("begin");
							final IThreadContentProvider thread = (IThreadContentProvider)first;
							IWorkbenchPage page = getSite().getPage().getWorkbenchWindow().getActivePage();
							try
							{
								logger.finest("openEditor");
								ThreadEditorInput tei = new ThreadEditorInput(thread);
								tei.setPrefetch(true);
								IEditorPart part = page.openEditor(tei, ThreadViewerEditor.class.getName());
								if(part instanceof ThreadViewerEditor)
								{
									logger.finest("updateThread");
									((ThreadViewerEditor)part).updateThread(-1);
								}
								tei.setPrefetch(false);
							}
							catch(PartInitException ex)
							{
								ex.printStackTrace();
							}
							logger.finest("done");
						}
						else
						{
							viewer.setExpandedState(first, !viewer.getExpandedState(first));
						}
					}
				}
			};
		
		bookmarkAction = new BookmarkAction();
	}
	
	private IAction createAction(final String id, final int action, String image)
	{
		Action act = new Action(WorkbenchMessages.getString("Workbench." + id))
			{
				public void run()
				{
					ITextOperationTarget target = getTextOperationTarget();
					if(target != null)
						target.doOperation(action);
				}
			};
		if(image != null)
		{
			act.setImageDescriptor(WorkbenchImages.getImageDescriptor(image));
			act.setDisabledImageDescriptor(WorkbenchImages.getImageDescriptor(image + "_DISABLED"));
			act.setHoverImageDescriptor(WorkbenchImages.getImageDescriptor(image + "_HOVER"));
		}
		getViewSite().getActionBars().setGlobalActionHandler(id, act);
		return act;
	}

	private void initializeTextActions()
	{
		cutAction = createAction(IWorkbenchActionConstants.CUT, ITextOperationTarget.CUT, "IMG_CTOOL_CUT_EDIT");
		copyAction = createAction(IWorkbenchActionConstants.COPY, ITextOperationTarget.COPY, "IMG_CTOOL_COPY_EDIT");
		pasteAction = createAction(IWorkbenchActionConstants.PASTE, ITextOperationTarget.PASTE, "IMG_CTOOL_PASTE_EDIT");
		deleteAction = createAction(IWorkbenchActionConstants.DELETE, ITextOperationTarget.DELETE, "IMG_CTOOL_DELETE_EDIT");
		selectAllAction = createAction(IWorkbenchActionConstants.SELECT_ALL, ITextOperationTarget.SELECT_ALL, null);

		MenuManager manager = new MenuManager(null, null);
		manager.setRemoveAllWhenShown(true);
		manager.addMenuListener(new IMenuListener()
			{
				public void menuAboutToShow(IMenuManager mgr)
				{
					fillContextMenu(mgr);
				}
			});
	}
	
	private ITextOperationTarget getTextOperationTarget()
	{
		ITextOperationTarget target = null;
		if(searchCombo != null && searchCombo.isFocusControl())
			target = new ComboTextOperationTarget(searchCombo);
		return target;
	}

	protected void updateSelectionDependentActions()
	{
		ITextOperationTarget target = getTextOperationTarget();
		cutAction.setEnabled(target != null && target.canDoOperation(ITextOperationTarget.CUT));
		copyAction.setEnabled(target != null && target.canDoOperation(ITextOperationTarget.COPY));
		pasteAction.setEnabled(target != null && target.canDoOperation(ITextOperationTarget.PASTE));
		deleteAction.setEnabled(target != null && target.canDoOperation(ITextOperationTarget.DELETE));
		selectAllAction.setEnabled(target != null && target.canDoOperation(ITextOperationTarget.SELECT_ALL));
	}

	private void hookDoubleClickAction()
	{
		viewer.addDoubleClickListener(new IDoubleClickListener()
			{
				public void doubleClick(DoubleClickEvent event)
				{
					doubleClickAction.run();
				}
			});
	}
	
	private void hookDragAndDrop()
	{
//		viewer.addDragSupport(DND.DROP_MOVE, new Transfer[]{TextTransfer.getInstance()}, new DragSourceListener()
//			{
//				public void dragStart(DragSourceEvent event)
//				{
//System.err.println(event);
//					if(viewer.getSelection() instanceof IStructuredSelection)
//						event.doit = 0 < ((IStructuredSelection)viewer.getSelection()).size();
//					else
//						event.doit = false;
//				}
//	
//				public void dragSetData(DragSourceEvent event)
//				{
//					if(viewer.getSelection() instanceof IStructuredSelection)
//					{
//						IStructuredSelection sel = (IStructuredSelection)viewer.getSelection();
//						if(sel.getFirstElement() instanceof IBBSReference)
//							event.data = ((IBBSReference)sel.getFirstElement()).getURL();
//					}
//
//					if(event.data == null)
//						event.data = "";
//				}
//	
//				public void dragFinished(DragSourceEvent event)
//				{
//				}
//			});
	}

	public void setFocus()
	{
		viewer.getControl().setFocus();
	}

	class NameSorter extends ViewerSorter
	{
	}
	
	private class ThreadListContentProvider extends LabelProvider implements IStructuredContentProvider, ITreeContentProvider, ITableLabelProvider
	{
		private static final int THREAD_LIST_DIVIDE_SIZE = 50;

		public Object[] getElements(Object inputElement)
		{
//logger.finest("enter");
			if(inputElement instanceof IBBSBoard)
			{
				IThreadContentProvider[] cps = ((IBBSBoard)inputElement).getThreadList(constraint);
				Fragment[] fragments = new Fragment[(cps.length + THREAD_LIST_DIVIDE_SIZE - 1) / THREAD_LIST_DIVIDE_SIZE];
				for(int i = 0; i < fragments.length; i++)
					fragments[i] = new Fragment(cps, THREAD_LIST_DIVIDE_SIZE * i);
				return fragments;
			}
			else
			{
				return new Object[0];
			}
		}

		public Object[] getChildren(Object parentElement)
		{
//logger.finest("enter");
			if(parentElement instanceof Fragment)
				return ((Fragment)parentElement).getChildren();
			else
				return null;
		}

		public Object getParent(Object element)
		{
//logger.finest("enter");
			if(element instanceof Fragment)
			{
				return viewer.getInput();
			}
			else if(element instanceof IThreadContentProvider)
			{
				int index = ((IThreadContentProvider)element).getIndex() / THREAD_LIST_DIVIDE_SIZE;
				Object[] list = getElements(viewer.getInput());
				if(0 <= index && index < list.length)
					return list[index];
			}

			return null;
		}

		public boolean hasChildren(Object element)
		{
//logger.finest("enter");
			return element instanceof Fragment || element instanceof IBBSBoard;
		}

		public void inputChanged(Viewer viewer, Object oldInput, Object newInput)
		{
//logger.finest("enter");
		}

		public String getText(Object element)
		{
//logger.finest("enter");
			if(element instanceof Fragment)
				return ((Fragment)element).getText();
			else if(element instanceof IThreadContentProvider)
				return ((IThreadContentProvider)element).getName();
			else
				return "";
		}

		public Image getColumnImage(Object element, int columnIndex)
		{
//logger.finest("enter");
			if(element instanceof IThreadContentProvider && columnIndex == 1 && element instanceof IBBSReference)
				return MonalipsePlugin.getDefault().getLabelImageOf((IBBSReference)element);
			return null;
		}

		public String getColumnText(Object element, int columnIndex)
		{
//logger.finest("enter");
			if(element instanceof IThreadContentProvider)
			{
				IThreadContentProvider thread = (IThreadContentProvider)element;
				switch(columnIndex)
				{
				case 0:
					return String.valueOf(thread.getIndex() + 1);
				
				case 1:
					return thread.getName();
				
				case 2:
					return String.valueOf(thread.getResponseCountHint());
				
				case 3:
					return String.valueOf(thread.getCachedCount());
				
				case 4:
					return getPersistentCountProperty(thread, "read");
				}
			}
			else if(element instanceof Fragment)
			{
				if(columnIndex == 0)
					return ((Fragment)element).getText();
			}
			return "";
		}
		
		private String getPersistentCountProperty(IThreadContentProvider thread, String type)
		{
			try
			{
				String count = thread.getLogFile().getPersistentProperty(new QualifiedName(ThreadViewerEditor.class.getName(), type));
				if(count == null)
					count = "0";
				return count;
			}
			catch (CoreException e)
			{
			}
			return "0";
		}
		
		private class Fragment
		{
			private IThreadContentProvider[] children;
			private int start;
			
			public Fragment(IThreadContentProvider[]  children, int start)
			{
//logger.finest("enter");
				this.children = children;
				this.start = start;
			}
			
			public IThreadContentProvider[] getChildren()
			{
//logger.finest("enter");
				IThreadContentProvider[] ch = new IThreadContentProvider[Math.min(children.length - start, THREAD_LIST_DIVIDE_SIZE)];
				for(int i = 0; i < ch.length; i++)
					ch[i] = children[start + i];
				return ch;
			}
			
			public String getText()
			{
//logger.finest("enter");
				return (start + 1) + "..." + (start + THREAD_LIST_DIVIDE_SIZE);
			}
			
			public boolean equals(Object obj)
			{
				if(obj instanceof Fragment)
					return ((Fragment)obj).start == start;
				else
					return false;
			}
			
			public int hashCode()
			{
				return start;
			}
		}

	}
}