package monalipse.views;

import java.util.ArrayList;
import java.util.List;
import java.util.logging.Logger;

import org.eclipse.jface.viewers.IStructuredSelection;
import org.eclipse.jface.viewers.ITableLabelProvider;
import org.eclipse.jface.viewers.ITreeContentProvider;
import org.eclipse.jface.viewers.StructuredSelection;
import org.eclipse.jface.viewers.TableTreeViewer;
import org.eclipse.ui.IMemento;
import org.eclipse.ui.IViewSite;
import org.eclipse.ui.PartInitException;
import org.eclipse.ui.part.ViewPart;

public abstract class AbstractTableTreeView extends ViewPart
{
	private static final String TAG_COLUMNWIDTH = "columnWidth";
	private static final String TAG_COLUMN = "column";
	private static final String TAG_WIDTH = "width";
	private static final String TAG_SELECTION = "selection";
	private static final String TAG_ELEMENT = "element";
	private static final String TAG_EXPANDED = "expanded";
	private static final String TAG_LABEL = "label";

	private IMemento memento;

	protected abstract Logger getLogger();
	protected abstract TableTreeViewer getViewer();
	protected abstract ITreeContentProvider getContentProvider();
	protected abstract ITableLabelProvider getLabelProvider();

	public void init(IViewSite site, IMemento memento) throws PartInitException
	{
		super.init(site, memento);
		this.memento = memento;
	}
	
	protected void partCreated()
	{
		if(memento != null)
			restoreState(memento);
		memento = null;
	}

	public void saveState(IMemento memento)
	{
//logger.finest("enter");
		getLogger().finest("begin");
		if(getViewer() == null)
		{
			if(this.memento != null)
				memento.putMemento(this.memento);
		}
		else
		{
			saveColumnState(memento);
			saveExpansionState(memento);
			saveSelectionState(memento);
		}
		getLogger().finest("done");
	}

	private void saveColumnState(IMemento memento)
	{
//logger.finest("enter");
		IMemento widthMem = memento.createChild(TAG_COLUMNWIDTH);
		for(int i = 0; i < getViewer().getTableTree().getTable().getColumnCount(); i++)
		{
			IMemento columnMem = widthMem.createChild(TAG_COLUMN);
			columnMem.putInteger(TAG_WIDTH, getViewer().getTableTree().getTable().getColumn(i).getWidth());
		}
	}

	private void saveSelectionState(IMemento memento)
	{
//logger.finest("enter");
		setElements(memento.createChild(TAG_SELECTION), ((IStructuredSelection)getViewer().getSelection()).toArray());
	}

	private void saveExpansionState(IMemento memento)
	{
//logger.finest("enter");
		setElements(memento.createChild(TAG_EXPANDED), getViewer().getVisibleExpandedElements());
	}
	
	private void setElements(IMemento memento, Object[] elements)
	{
//logger.finest("enter");
		if(getContentProvider() == null)
			return;

		if(elements.length > 0)
		{
			for (int i = 0; i < elements.length; i++)
			{
				Object o = elements[i];
				IMemento elementMem = memento.createChild(TAG_ELEMENT);
				while(getContentProvider().getParent(o) != null)
				{
					IMemento labelMem = elementMem.createChild(TAG_LABEL);
					labelMem.putString(TAG_LABEL, getLabelProvider().getColumnText(o, 0));
					o = getContentProvider().getParent(o);
				}
			}
		}
	}

	protected void restoreState(IMemento memento)
	{
//logger.finest("enter");
		getLogger().finest("begin");
		restoreColumnState(memento);
		restoreExpansionState(memento);
		restoreSelectionState(memento);
		getLogger().finest("done");
	}

	private void restoreColumnState(IMemento memento)
	{
//logger.finest("enter");
		IMemento widthMem = memento.getChild(TAG_COLUMNWIDTH);
		if(widthMem != null)
		{
			IMemento[] columnMem = widthMem.getChildren(TAG_COLUMN);
			if(columnMem != null)
			{
				for(int i = 0; i < getViewer().getTableTree().getTable().getColumnCount() && i < columnMem.length; i++)
					getViewer().getTableTree().getTable().getColumn(i).setWidth(columnMem[i].getInteger(TAG_WIDTH).intValue());
			}
		}
	}

	private void restoreSelectionState(IMemento memento)
	{
//logger.finest("enter");
		IMemento childMem;
		childMem = memento.getChild(TAG_SELECTION);
		if (childMem != null)
			getViewer().setSelection(new StructuredSelection(getElements(childMem)));
	}
	
	private void restoreExpansionState(IMemento memento)
	{
//logger.finest("enter");
		IMemento childMem = memento.getChild(TAG_EXPANDED);
		if (childMem != null)
			getViewer().setExpandedElements(getElements(childMem).toArray());
	}
	
	private List getElements(IMemento memento)
	{
//logger.finest("enter");
		ArrayList list = new ArrayList();

		IMemento[] elementMem = memento.getChildren(TAG_ELEMENT);
		for (int i = 0; i < elementMem.length; i++)
		{
			IMemento[] labelMem = elementMem[i].getChildren(TAG_LABEL);
			Object[] children = getContentProvider().getElements(getViewer().getInput());
			Object e = null;
			for(int j = labelMem.length - 1; 0 <= j; j--)
			{
				e = findElement(children, labelMem[j].getString(TAG_LABEL));
				if(e == null)
					break;
				else
					children = getContentProvider().getChildren(e);
			}
			if(e != null)
				list.add(e);
		}
		
		return list;
	}
	
	private Object findElement(Object[] list, String label)
	{
//logger.finest("enter");
		if(list == null)
			return null;
		for(int i = 0; i < list.length; i++)
		{
			if(getLabelProvider().getColumnText(list[i], 0).equals(label))
				return list[i];
		}
		return null;
	}

}
