package monalipse.server.giko;

import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.lang.reflect.InvocationTargetException;
import java.net.MalformedURLException;
import java.net.URL;
import java.util.ArrayList;
import java.util.List;
import java.util.Properties;

import javax.xml.transform.Transformer;
import javax.xml.transform.TransformerException;
import javax.xml.transform.TransformerFactory;
import javax.xml.transform.dom.DOMResult;
import javax.xml.transform.dom.DOMSource;
import javax.xml.transform.stream.StreamResult;
import javax.xml.transform.stream.StreamSource;

import monalipse.MonalipsePlugin;
import monalipse.server.AbstractBBSServer;
import monalipse.server.IBoardListContentProvider;
import org.eclipse.core.resources.IFile;
import org.eclipse.core.resources.IFolder;
import org.eclipse.core.resources.IProject;
import org.eclipse.core.resources.IResourceChangeEvent;
import org.eclipse.core.resources.IResourceDelta;
import org.eclipse.core.runtime.CoreException;
import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.jface.viewers.LabelProvider;
import org.eclipse.jface.viewers.Viewer;
import org.eclipse.swt.graphics.Image;
import org.eclipse.ui.ISharedImages;
import org.eclipse.ui.IWorkbenchWindow;
import org.eclipse.ui.PlatformUI;
import org.eclipse.ui.actions.WorkspaceModifyOperation;
import org.w3c.dom.Document;
import org.w3c.dom.Element;
import org.w3c.dom.NodeList;
import org.xml.sax.SAXException;

import com.meterware.httpunit.GetMethodWebRequest;
import com.meterware.httpunit.WebRequest;
import com.meterware.httpunit.WebResponse;

class BoardListContentProvider extends LabelProvider implements IBoardListContentProvider
{
	private static String BBSMENU_URL = "http://www.ff.iij4u.or.jp/~ch2/bbsmenu.html";
	public static String[] BBS_DOMAINS = new String[]{"2ch.net", "machibbs.com", "bbspink.com"};

	private static Transformer transformer;

	private IWorkbenchWindow workbenchWindow;
	private IProject input;
	private Category root;
	private Properties logMap;

	public BoardListContentProvider(IWorkbenchWindow workbenchWindow)
	{
		this.workbenchWindow = workbenchWindow;
	}
	
	public static IFolder getLogFolderOf(IProject input, String baseURL)
	{
		Properties logMap = new Properties();
		IFile mapFile = input.getFile(".logmap");

		try
		{
			MonalipsePlugin.ensureSynchronized(mapFile);
			if(mapFile.exists())
			{
				InputStream in = mapFile.getContents();
				try
				{
					logMap.load(in);
				}
				finally
				{
					in.close();
				}
				String log = logMap.getProperty(baseURL);
				if(log != null)
					return input.getFolder(log);
			}
		}
		catch (IOException e)
		{
			e.printStackTrace();
		}
		catch (CoreException e)
		{
			e.printStackTrace();
		}
		
		return null;
	}
	
	public boolean resourceChanged(IResourceChangeEvent event)
	{
		return input != null && AbstractBBSServer.resourceModified(IResourceDelta.ADDED | IResourceDelta.REMOVED | IResourceDelta.CHANGED, event.getDelta(), input.getFile(".bbsmenu"));
	}
	
	public String getText(Object element)
	{
		if(element instanceof Category)
			return ((Category)element).getName();
		else if(element instanceof Board)
			return ((Board)element).getName();
		else
			return null;
	}
	
	public Image getImage(Object element)
	{
		String imageKey = ISharedImages.IMG_OBJ_ELEMENT;
		if (element instanceof Category)
			imageKey = ISharedImages.IMG_OBJ_FOLDER;
		return PlatformUI.getWorkbench().getSharedImages().getImage(imageKey);
	}

	public boolean hasChildren(Object element)
	{
		if(element == input)
			return root.hasChildren();
		else if(element instanceof Category)
			return ((Category)element).hasChildren();
		else
			return false;
	}

	public Object[] getChildren(Object parentElement)
	{
		if(parentElement == input)
			return root.getChildren();
		else if(parentElement instanceof Category)
			return ((Category)parentElement).getChildren();
		else
			return new Object[0];
	}
	
	public Object[] getElements(Object inputElement)
	{
		return getChildren(inputElement);
	}
	
	public Object getParent(Object element)
	{
		if(element == input)
			return root;
		else if(element instanceof Category)
			return ((Category)element).getCategory();
		else if(element instanceof Board)
			return ((Board)element).getCategory();
		else
			return root;
	}
	
	public void inputChanged(Viewer viewer, Object oldInput, Object newInput)
	{
		input = null;

		if(newInput instanceof IProject)
			input = (IProject)newInput;
		
		logMap = new Properties();
		if(input != null)
		{
			IFile mapFile = input.getFile(".logmap");
			try
			{
				MonalipsePlugin.ensureSynchronized(mapFile);
				if(mapFile.exists())
				{
					InputStream in = mapFile.getContents();
					try
					{
						logMap.load(in);
					}
					finally
					{
						in.close();
					}
				}
			}
			catch (IOException e)
			{
				e.printStackTrace();
			}
			catch (CoreException e)
			{
				e.printStackTrace();
			}
		}

		Category oldRoot = root;
		root = new Category(null, "<root>");
		Document doc = getCachedMenu();
		if(doc != null)
		{
			parseBBSMenu(doc.getDocumentElement(), root, null);
			
			final List urlList = new ArrayList();

			Object[] categories = root.getChildren();
			for(int i = 0; i < categories.length; i++)
			{
				Category category = (Category)categories[i];
				Object[] boards = category.getChildren();
				for(int j = 0; j < boards.length; j++)
				{
					Board board = (Board)boards[j];
					final String url = board.getURL().toExternalForm();
					String log = logMap.getProperty(url);
					if(log == null && oldRoot != null)
					{
						Category oldCategory = (Category)oldRoot.getChild(category.getName());
						if(oldCategory != null)
						{
							Board oldBoard = (Board)oldCategory.getChild(board.getName());
							if(oldBoard != null)
								log = logMap.getProperty(oldBoard.getURL().toExternalForm());
						}
					}
					
					if(log == null)
						log = "log" + logMap.size();
					
					if(!logMap.getProperty(log, "").equals(url))
						urlList.add(url);
					
					logMap.setProperty(url, log);
					logMap.setProperty(log, url);
					board.setLogFolder(input.getFolder(log));
				}
			}

			if(0 < urlList.size())
			{
				GikoServer.asyncExec(workbenchWindow, new WorkspaceModifyOperation()
					{
						protected void execute(IProgressMonitor monitor) throws InvocationTargetException
						{
							monitor.beginTask("", urlList.size());
	
							for(int i = 0; i < urlList.size(); i++)
							{
								String url = (String)urlList.get(i);
								IFolder logFolder = input.getFolder((String)logMap.get(url));
								try
								{
									if(!logFolder.exists())
										logFolder.create(false, true, null);
									IFile locationFile = logFolder.getFile(".location");
									MonalipsePlugin.ensureSynchronized(locationFile);
									if(locationFile.exists())
										locationFile.setContents(MonalipsePlugin.getUTFInputStream(url), false, false, null);
									else
										locationFile.create(MonalipsePlugin.getUTFInputStream(url), false, null);
								}
								catch (CoreException e)
								{
									e.printStackTrace();
								}
								catch (IOException e)
								{
									e.printStackTrace();
								}
								monitor.worked(1);
							}
							
							monitor.done();
						}
					});
			}
		}

		if(input != null)
		{
			final IFile mapFile = input.getFile(".logmap");
			GikoServer.asyncExec(workbenchWindow, new WorkspaceModifyOperation()
				{
					protected void execute(IProgressMonitor monitor) throws InvocationTargetException
					{
						try
						{
							MonalipsePlugin.ensureSynchronized(mapFile);
							ByteArrayOutputStream bout = new ByteArrayOutputStream();
							logMap.store(bout, "");
							bout.close();
							if(mapFile.exists())
								mapFile.setContents(new ByteArrayInputStream(bout.toByteArray()), false, false, null);
							else
								mapFile.create(new ByteArrayInputStream(bout.toByteArray()), false, null);
						}
						catch (IOException e)
						{
							e.printStackTrace();
						}
						catch (CoreException e)
						{
							e.printStackTrace();
						}
					}
				});
		}
	}
	
	private Document getCachedMenu()
	{
		if(input != null)
		{
			IFile file = input.getFile(".bbsmenu");
			try
			{
				MonalipsePlugin.ensureSynchronized(file);
				if(!file.exists())
					return null;
				InputStream in = file.getContents();
				try
				{
					DOMResult res = new DOMResult();
					getTransformer().transform(new StreamSource(in), res);
					return (Document)res.getNode();
				}
				finally
				{
					in.close();
				}
			}
			catch (CoreException e)
			{
				e.printStackTrace();
			}
			catch (TransformerException e)
			{
				e.printStackTrace();
			}
			catch (IOException e)
			{
				e.printStackTrace();
			}
		}
		
		return null;
	}
	
	private void setCachedMenu(Document doc) throws TransformerException, IOException
	{
		ByteArrayOutputStream bout = new ByteArrayOutputStream();
		getTransformer().transform(new DOMSource(doc), new StreamResult(bout));
		bout.close();
		final byte[] bytes = bout.toByteArray();
		GikoServer.asyncExec(workbenchWindow, new WorkspaceModifyOperation()
			{
				protected void execute(IProgressMonitor monitor) throws InvocationTargetException
				{
					try
					{
						IFile cache = input.getFile(".bbsmenu");
						MonalipsePlugin.ensureSynchronized(cache);
						if(cache.exists())
							cache.setContents(new ByteArrayInputStream(bytes), false, false, monitor);
						else
							cache.create(new ByteArrayInputStream(bytes), false, monitor);
					}
					catch (CoreException e)
					{
						throw new InvocationTargetException(e);
					}
				}
			});
	}

	private static void parseBBSMenu(Element element, Category root, Category category)
	{
		NodeList children = element.getChildNodes();
		for(int i = 0; i < children.getLength(); i++)
		{
			if(children.item(i) instanceof Element)
			{
				Element child = (Element)children.item(i);
				if(child.getTagName().equals("a") && category != null)
				{
					try
					{
						String url = child.getAttribute("href");
						URL href = new URL(url);
						String name = child.getFirstChild().getNodeValue();
						if(isBBS(href))
						{
							if(!category.hasChildren())
								root.addChild(category);
							category.addChild(new Board(category, name, href));
						}
					}
					catch(MalformedURLException e)
					{
					}
				}
				else if(child.getTagName().equals("b"))
				{
					String s = child.getFirstChild().getNodeValue();
					if(0 < s.length())
						category = new Category(root, s);
				}
				else
				{
					parseBBSMenu(child, root, category);
				}
			}
		}
	}

	private static boolean isBBS(URL href)
	{
		for(int i = 0; i < BBS_DOMAINS.length; i++)
		{
			if(href.getHost().endsWith(BBS_DOMAINS[i]) && 1 < href.getFile().length() && href.getFile().endsWith("/"))
				return true;
		}
		return false;
	}

	public void updateBoardList()
	{
		if(input == null)
			return;

		try
		{
			WebRequest req = new GetMethodWebRequest(BBSMENU_URL);
//			req.setHeaderField("User-Agent", "Monazilla/1.00 (monalipse/0.01)");
			WebResponse wr = GikoServer.getWebConversation().getResponse(req);
			
			if(wr.getResponseCode() == 200)
				setCachedMenu(wr.getDOM());
			
			wr.getInputStream().close();
		}
		catch (TransformerException e)
		{
			e.printStackTrace();
		}
		catch (MalformedURLException e)
		{
			e.printStackTrace();
		}
		catch (IOException e)
		{
			e.printStackTrace();
		}
		catch (SAXException e)
		{
			e.printStackTrace();
		}
	}

	private static Transformer getTransformer() throws TransformerException
	{
		if(transformer == null)
		{
			TransformerFactory tf = TransformerFactory.newInstance();
			transformer = tf.newTransformer();
			transformer.setOutputProperty("method", "xml");
			transformer.setOutputProperty("standalone", "yes");
			transformer.setOutputProperty("indent", "yes");
		}
		return transformer;
	}

}
