package monalipse.views;

import java.io.UnsupportedEncodingException;
import java.lang.reflect.InvocationTargetException;
import java.net.MalformedURLException;
import java.net.URL;
import java.net.URLDecoder;
import java.net.URLEncoder;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;
import java.util.StringTokenizer;

import monalipse.MonalipsePlugin;
import monalipse.editors.IThreadViewerEditor;
import monalipse.editors.ThreadEditorInput;
import monalipse.editors.win32.ThreadViewerEditorWin32;
import monalipse.part.CancelableRunner;
import monalipse.server.BBSServerManager;
import monalipse.server.IBBSServer;
import monalipse.server.IThreadContentProvider;
import monalipse.server.IThreadListContentProvider;
import org.eclipse.core.resources.IFolder;
import org.eclipse.core.resources.IResource;
import org.eclipse.core.resources.IResourceChangeEvent;
import org.eclipse.core.resources.IResourceChangeListener;
import org.eclipse.core.resources.ResourcesPlugin;
import org.eclipse.core.runtime.CoreException;
import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.core.runtime.Path;
import org.eclipse.core.runtime.Platform;
import org.eclipse.core.runtime.QualifiedName;
import org.eclipse.jface.action.Action;
import org.eclipse.jface.action.ControlContribution;
import org.eclipse.jface.action.IAction;
import org.eclipse.jface.action.IMenuListener;
import org.eclipse.jface.action.IMenuManager;
import org.eclipse.jface.action.IStatusLineManager;
import org.eclipse.jface.action.IToolBarManager;
import org.eclipse.jface.action.MenuManager;
import org.eclipse.jface.action.Separator;
import org.eclipse.jface.operation.IRunnableWithProgress;
import org.eclipse.jface.resource.ImageDescriptor;
import org.eclipse.jface.viewers.DoubleClickEvent;
import org.eclipse.jface.viewers.IDoubleClickListener;
import org.eclipse.jface.viewers.ISelection;
import org.eclipse.jface.viewers.IStructuredSelection;
import org.eclipse.jface.viewers.LabelProvider;
import org.eclipse.jface.viewers.StructuredSelection;
import org.eclipse.jface.viewers.TableTreeViewer;
import org.eclipse.jface.viewers.ViewerSorter;
import org.eclipse.swt.SWT;
import org.eclipse.swt.custom.TableTree;
import org.eclipse.swt.events.SelectionEvent;
import org.eclipse.swt.events.SelectionListener;
import org.eclipse.swt.widgets.Combo;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Control;
import org.eclipse.swt.widgets.Menu;
import org.eclipse.swt.widgets.TableColumn;
import org.eclipse.ui.IActionBars;
import org.eclipse.ui.IEditorPart;
import org.eclipse.ui.IMemento;
import org.eclipse.ui.ISelectionListener;
import org.eclipse.ui.IViewSite;
import org.eclipse.ui.IWorkbenchPage;
import org.eclipse.ui.IWorkbenchPart;
import org.eclipse.ui.PartInitException;
import org.eclipse.ui.XMLMemento;
import org.eclipse.ui.part.ViewPart;

public class ThreadListView extends ViewPart implements ISelectionListener, IResourceChangeListener, SelectionListener
{
	public static final String ID_THREAD_LIST = ThreadListView.class.getName();

	private static final String TAG_INPUT = "input";
	private static final String TAG_FOLDER = "folder";
	private static final String TAG_COLUMNWIDTH = "columnWidth";
	private static final String TAG_COLUMN = "column";
	private static final String TAG_WIDTH = "width";
	private static final String TAG_SELECTION = "selection";
	private static final String TAG_ELEMENT = "element";
	private static final String TAG_EXPANDED = "expanded";
	private static final String TAG_LABEL = "label";
	private static final String TAG_TITLE = "title";

	private TableTreeViewer viewer;
	private IBBSServer server;
	private IThreadListContentProvider provider;
	private CancelableRunner cancelable;
	private IAction reloadAction;
	private IAction abortAction;
	private IAction doubleClickAction;
	private IMemento memento;
	private Combo searchCombo;

	public ThreadListView()
	{
	}
	
	public void init(IViewSite site, IMemento memento) throws PartInitException
	{
		super.init(site, memento);
		this.memento = memento;
	}

	public void createPartControl(Composite parent)
	{
		viewer = new TableTreeViewer(parent, SWT.MULTI | SWT.H_SCROLL | SWT.V_SCROLL | SWT.FULL_SELECTION);
		viewer.setContentProvider(new NullTreeContentProvider());
		viewer.setLabelProvider(new LabelProvider());
//		viewer.setSorter(new NameSorter());
		viewer.setInput(ResourcesPlugin.getWorkspace());
		makeColumns();
		makeActions();
		hookContextMenu();
		hookDoubleClickAction();
		contributeToActionBars();
		ResourcesPlugin.getWorkspace().addResourceChangeListener(this);
		getViewSite().getPage().addSelectionListener(this);
		viewer.getTableTree().addSelectionListener(this);
		if(memento != null)
			restoreState(memento);
		memento = null;
	}

	public void resourceChanged(IResourceChangeEvent event)
	{
		if(provider != null)
		{
			if(provider.resourceChanged(event))
			{
				refreshList();
				if(viewer.getVisibleExpandedElements().length == 0)
					expandFirstFragment();
			}

			Object[] changed = provider.getChangedItems(event);
			if(changed != null)
				viewer.update(changed, null);
		}
	}

	private void refreshList()
	{
		if(provider != null)
		{
			IMemento memento = XMLMemento.createWriteRoot(ThreadListView.class.getName());
			saveState(memento);
			viewer.setInput(viewer.getInput());
			restoreState(memento);
		}
	}
	
	public void selectionChanged(IWorkbenchPart part, ISelection selection)
	{
		if(!viewer.getTableTree().isDisposed() && selection instanceof IStructuredSelection && part instanceof BoardListView)
		{
			Object first = ((IStructuredSelection)selection).getFirstElement();
			
			String title = ((BoardListView)part).getBoardNameOf(first);
			if(title != null)
				setTitle(title);

			IBBSServer server = ((BoardListView)part).getServer();
			if(this.server != server || viewer.getContentProvider() == null)
			{
				this.server = server;
				provider = server.getThreadListContentProvider();
				viewer.setContentProvider(provider);
				viewer.setLabelProvider(provider);
			}
			
			IFolder logFolder = server.getLogFolderOf(first);
			if(viewer.getInput() == null || (logFolder != null && !viewer.getInput().equals(logFolder)))
			{
				if(viewer.getInput() instanceof IFolder)
					saveSearchWord((IFolder)viewer.getInput());
				if(provider != null)
					provider.setConstraint("");
				viewer.setInput(logFolder);
				restoreSearchWord(logFolder);
				if(provider != null)
				{
					getSite().getShell().getDisplay().asyncExec(new Runnable()
						{
							public void run()
							{
								getSite().getPage().activate(ThreadListView.this);
							}
						});

					if(provider.getElements(viewer.getInput()).length == 0)
					{
						getSite().getShell().getDisplay().asyncExec(new Runnable()
							{
								public void run()
								{
									reloadAction.run();
								}
							});
					}
					else
					{
						expandFirstFragment();
					}
				}
			}
		}
	}

	private void expandFirstFragment()
	{
		Object[] fragments = provider.getChildren(viewer.getInput());
		if(0 < fragments.length && provider.hasChildren(fragments[0]))
			viewer.setExpandedElements(new Object[]{fragments[0]});
	}

	public void widgetDefaultSelected(SelectionEvent e)
	{
	}

	public void widgetSelected(SelectionEvent e)
	{
		ISelection selection = viewer.getSelection();
		if(selection instanceof IStructuredSelection)
		{
			Object first = ((IStructuredSelection)selection).getFirstElement();
			if(first instanceof IThreadContentProvider)
			{
				final IThreadContentProvider thread = (IThreadContentProvider)first;
				IWorkbenchPage page = getSite().getPage().getWorkbenchWindow().getActivePage();
				if(page.findEditor(new ThreadEditorInput(thread)) != null)
				{
					try
					{
						page.openEditor(new ThreadEditorInput(thread), "monalipse.editors.ThreadViewerEditor");
					}
					catch(PartInitException ex)
					{
						ex.printStackTrace();
					}
				}
			}
		}
	}
	
	private void saveSearchWord(IResource res)
	{
		if(searchCombo != null && !searchCombo.isDisposed())
		{
			try
			{
				StringBuffer buf = new StringBuffer();
				String[] keys = searchCombo.getItems();
				for(int i = 0; i < keys.length && i < 10; i++)
					buf.append(URLEncoder.encode(keys[i], "UTF-8")).append(" ");
				res.setPersistentProperty(new QualifiedName(getClass().getName(), "search"), buf.toString());
			}
			catch (UnsupportedEncodingException e)
			{
			}
			catch (CoreException e)
			{
			}
		}
	}
	
	private void restoreSearchWord(IResource res)
	{
		if(searchCombo != null && !searchCombo.isDisposed())
		{
			try
			{
				searchCombo.removeAll();
				searchCombo.add("");
				String keys = res.getPersistentProperty(new QualifiedName(getClass().getName(), "search"));
				if(keys != null)
				{
					StringTokenizer tk = new StringTokenizer(keys);
					while(tk.hasMoreTokens())
						searchCombo.add(URLDecoder.decode(tk.nextToken(), "UTF-8"));
				}
			}
			catch (UnsupportedEncodingException e)
			{
			}
			catch (CoreException e)
			{
			}
		}
	}

	public void saveState(IMemento memento)
	{
		if(viewer == null)
		{
			if(this.memento != null)
				memento.putMemento(this.memento);
			return;
		}
		saveColumnState(memento);
		saveInputState(memento);
		saveExpansionState(memento);
		saveSelectionState(memento);
	}

	private void saveColumnState(IMemento memento)
	{
		IMemento widthMem = memento.createChild(TAG_COLUMNWIDTH);
		for(int i = 0; i < viewer.getTableTree().getTable().getColumnCount(); i++)
		{
			IMemento columnMem = widthMem.createChild(TAG_COLUMN);
			columnMem.putInteger(TAG_WIDTH, viewer.getTableTree().getTable().getColumn(i).getWidth());
		}
	}

	private void saveInputState(IMemento memento)
	{
		if(server != null)
		{
			IMemento elementMem = memento.createChild(TAG_INPUT);
			elementMem.putString(TAG_FOLDER, ((IFolder)viewer.getInput()).getFullPath().toString());
			elementMem.putString(TAG_TITLE, getTitle());
			saveSearchWord((IFolder)viewer.getInput());
		}
	}

	private void saveSelectionState(IMemento memento)
	{
		setElements(memento.createChild(TAG_SELECTION), ((IStructuredSelection)viewer.getSelection()).toArray());
	}

	private void saveExpansionState(IMemento memento)
	{
		setElements(memento.createChild(TAG_EXPANDED), viewer.getVisibleExpandedElements());
	}
	
	private void setElements(IMemento memento, Object[] elements)
	{
		if(provider == null)
			return;

		if(elements.length > 0)
		{
			for (int i = 0; i < elements.length; i++)
			{
				Object o = elements[i];
				IMemento elementMem = memento.createChild(TAG_ELEMENT);
				while(provider.getParent(o) != null)
				{
					IMemento labelMem = elementMem.createChild(TAG_LABEL);
					labelMem.putString(TAG_LABEL, provider.getColumnText(o, 0));
					o = provider.getParent(o);
				}
			}
		}
	}

	private void restoreState(IMemento memento)
	{
		restoreColumnState(memento);
		restoreInputState(memento);
		restoreExpansionState(memento);
		restoreSelectionState(memento);
	}

	private void restoreColumnState(IMemento memento)
	{
		IMemento widthMem = memento.getChild(TAG_COLUMNWIDTH);
		if(widthMem != null)
		{
			IMemento[] columnMem = widthMem.getChildren(TAG_COLUMN);
			if(columnMem != null)
			{
				for(int i = 0; i < viewer.getTableTree().getTable().getColumnCount() && i < columnMem.length; i++)
					viewer.getTableTree().getTable().getColumn(i).setWidth(columnMem[i].getInteger(TAG_WIDTH).intValue());
			}
		}
	}

	private void restoreInputState(IMemento memento)
	{
		IMemento elementMem = memento.getChild(TAG_INPUT);
		if(elementMem != null)
		{
			String folder = elementMem.getString(TAG_FOLDER);
			if(folder != null)
			{
				IFolder fld = ResourcesPlugin.getWorkspace().getRoot().getFolder(new Path(folder));

				server = BBSServerManager.getInstanceOf(fld.getProject(), getSite().getWorkbenchWindow());
				if(server != null)
				{
					provider = server.getThreadListContentProvider();
					viewer.setContentProvider(provider);
					viewer.setLabelProvider(provider);
					viewer.setInput(fld);
					restoreSearchWord(fld);
				}
			}
			
			String title = elementMem.getString(TAG_TITLE);
			if(title != null)
				setTitle(title);
		}
	}

	private void restoreSelectionState(IMemento memento)
	{
		IMemento childMem;
		childMem = memento.getChild(TAG_SELECTION);
		if (childMem != null)
			viewer.setSelection(new StructuredSelection(getElements(childMem)));
	}
	
	private void restoreExpansionState(IMemento memento)
	{
		IMemento childMem = memento.getChild(TAG_EXPANDED);
		if (childMem != null)
			viewer.setExpandedElements(getElements(childMem).toArray());
	}
	
	private List getElements(IMemento memento)
	{
		ArrayList list = new ArrayList();

		IMemento[] elementMem = memento.getChildren(TAG_ELEMENT);
		for (int i = 0; i < elementMem.length; i++)
		{
			IMemento[] labelMem = elementMem[i].getChildren(TAG_LABEL);
			Object[] children = provider.getElements(viewer.getInput());
			Object e = null;
			for(int j = labelMem.length - 1; 0 <= j; j--)
			{
				e = findElement(children, labelMem[j].getString(TAG_LABEL));
				if(e == null)
					break;
				else
					children = provider.getChildren(e);
			}
			if(e != null)
				list.add(e);
		}
		
		return list;
	}
	
	private Object findElement(Object[] list, String label)
	{
		if(list == null)
			return null;
		for(int i = 0; i < list.length; i++)
		{
			if(provider.getColumnText(list[i], 0).equals(label))
				return list[i];
		}
		return null;
	}

	private void hookContextMenu()
	{
		MenuManager menuMgr = new MenuManager("#PopupMenu");
		menuMgr.setRemoveAllWhenShown(true);
		menuMgr.addMenuListener(new IMenuListener()
			{
				public void menuAboutToShow(IMenuManager manager)
				{
					ThreadListView.this.fillContextMenu(manager);
				}
			});
		Menu menu = menuMgr.createContextMenu(viewer.getControl());
		viewer.getControl().setMenu(menu);
		getSite().registerContextMenu(menuMgr, viewer);
	}

	private void contributeToActionBars()
	{
		IActionBars bars = getViewSite().getActionBars();
//		fillLocalPullDown(bars.getMenuManager());
		fillLocalToolBar(bars.getToolBarManager());
		fillLocalStatuBar(bars.getStatusLineManager());
	}

	private void fillLocalStatuBar(IStatusLineManager manager)
	{
		cancelable = new CancelableRunner(manager, getSite().getShell().getDisplay(), abortAction);
	}

	private void fillLocalPullDown(IMenuManager manager)
	{
		manager.add(reloadAction);
		manager.add(new Separator());
		manager.add(abortAction);
	}

	private void fillContextMenu(IMenuManager manager)
	{
		manager.add(reloadAction);
		manager.add(abortAction);
		// Other plug-ins can contribute there actions here
		manager.add(new Separator("Additions"));
	}

	private void fillLocalToolBar(IToolBarManager manager)
	{
		manager.add(new ControlContribution(ThreadListView.class.getName() + ".constrain")
			{
				protected Control createControl(Composite parent)
				{
					searchCombo = new Combo(parent, SWT.DROP_DOWN);
					searchCombo.setToolTipText("Search");
					searchCombo.add("");
					if(viewer.getInput() instanceof IFolder)
						restoreSearchWord((IFolder)viewer.getInput());
					searchCombo.addSelectionListener(new SelectionListener()
						{
							public void widgetSelected(SelectionEvent e)
							{
								selected(e);
							}
	
							public void widgetDefaultSelected(SelectionEvent e)
							{
								selected(e);
							}
							
							private void selected(SelectionEvent e)
							{
								String text = searchCombo.getText();
								if(0 < text.length())
								{
									if(Arrays.asList(searchCombo.getItems()).contains(text))
										searchCombo.remove(text);
									searchCombo.add(text, 1);
								}
								provider.setConstraint(text);
								refreshList();
								searchCombo.setText(text);
								if(viewer.getVisibleExpandedElements().length == 0)
									expandFirstFragment();
							}
						});
					return searchCombo;
				}
			});

		manager.add(reloadAction);
		manager.add(abortAction);
	}

	private void makeColumns()
	{
		TableTree table = viewer.getTableTree();
		table.getTable().setHeaderVisible(true);
		table.getTable().setLinesVisible(true);
		TableColumn col;
		col = new TableColumn(table.getTable(), SWT.LEFT);
		col.setText("Rank");
		col.setWidth(80);
		col = new TableColumn(table.getTable(), SWT.LEFT);
		col.setText("Title");
		col.setWidth(400);
		col = new TableColumn(table.getTable(), SWT.LEFT);
		col.setText("Response");
		col.setWidth(50);
		col = new TableColumn(table.getTable(), SWT.LEFT);
		col.setText("Cache");
		col.setWidth(50);
		col = new TableColumn(table.getTable(), SWT.LEFT);
		col.setText("Read");
		col.setWidth(50);
	}

	private void makeActions()
	{
		String iconPath = "icons/"; //$NON-NLS-1$		
		URL installURL = Platform.getPlugin(MonalipsePlugin.PLUGIN_ID).getDescriptor().getInstallURL();

		reloadAction = new Action()
			{
				public void run()
				{
					cancelable.run(cancelable, new IRunnableWithProgress()
						{
							public void run(IProgressMonitor monitor) throws InvocationTargetException, InterruptedException
							{
								if(provider != null)
									provider.updateThreadList();
							}
						});
				}
			};
		reloadAction.setText("Reload");
		reloadAction.setToolTipText("Reload thread list");
		try
		{
			reloadAction.setImageDescriptor(ImageDescriptor.createFromURL(new URL(installURL, iconPath + "refresh_nav.gif"))); //$NON-NLS-1$
		}
		catch (MalformedURLException e)
		{
		}

		abortAction = new Action()
			{
				public void run()
				{
					cancelable.cancel();
				}
			};
		abortAction.setText("Abort");
		abortAction.setToolTipText("Abort downloading board list");
		abortAction.setEnabled(false);
		try
		{
			abortAction.setImageDescriptor(ImageDescriptor.createFromURL(new URL(installURL, iconPath + "stop_nav.gif"))); //$NON-NLS-1$
		}
		catch (MalformedURLException e)
		{
		}

		doubleClickAction = new Action()
			{
				public void run()
				{
					ISelection selection = viewer.getSelection();
					if(selection instanceof IStructuredSelection)
					{
						Object first = ((IStructuredSelection)selection).getFirstElement();
						if(first instanceof IThreadContentProvider)
						{
							final IThreadContentProvider thread = (IThreadContentProvider)first;
							IWorkbenchPage page = getSite().getPage().getWorkbenchWindow().getActivePage();
							try
							{
								IEditorPart part = page.openEditor(new ThreadEditorInput(thread), ThreadViewerEditorWin32.class.getName());
								if(part instanceof IThreadViewerEditor)
									((IThreadViewerEditor)part).updateThread();
							}
							catch(PartInitException ex)
							{
								ex.printStackTrace();
							}
						}
					}
				}
			};
	}

	private void hookDoubleClickAction()
	{
		viewer.addDoubleClickListener(new IDoubleClickListener()
		{
			public void doubleClick(DoubleClickEvent event)
			{
				doubleClickAction.run();
			}
		});
	}

	public void setFocus()
	{
		viewer.getControl().setFocus();
	}

	class NameSorter extends ViewerSorter
	{
	}

}