package monalipse.wizards;

/*
 * (c) Copyright IBM Corp. 2000, 2001.
 * All Rights Reserved.
 */
import java.io.ByteArrayInputStream;
import java.io.IOException;
import java.lang.reflect.InvocationTargetException;
import java.net.MalformedURLException;
import java.net.URL;

import monalipse.MonalipsePlugin;
import org.eclipse.core.resources.IFile;
import org.eclipse.core.resources.IProject;
import org.eclipse.core.resources.IProjectDescription;
import org.eclipse.core.resources.IResourceStatus;
import org.eclipse.core.resources.IWorkspace;
import org.eclipse.core.resources.ResourcesPlugin;
import org.eclipse.core.runtime.CoreException;
import org.eclipse.core.runtime.IConfigurationElement;
import org.eclipse.core.runtime.IExecutableExtension;
import org.eclipse.core.runtime.IPath;
import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.core.runtime.OperationCanceledException;
import org.eclipse.core.runtime.Platform;
import org.eclipse.core.runtime.Status;
import org.eclipse.core.runtime.SubProgressMonitor;
import org.eclipse.jface.dialogs.ErrorDialog;
import org.eclipse.jface.dialogs.IDialogSettings;
import org.eclipse.jface.dialogs.MessageDialog;
import org.eclipse.jface.resource.ImageDescriptor;
import org.eclipse.jface.viewers.IStructuredSelection;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Control;
import org.eclipse.swt.widgets.Text;
import org.eclipse.ui.IWorkbench;
import org.eclipse.ui.PlatformUI;
import org.eclipse.ui.actions.WorkspaceModifyOperation;
import org.eclipse.ui.dialogs.WizardNewProjectCreationPage;
import org.eclipse.ui.plugin.AbstractUIPlugin;
import org.eclipse.ui.wizards.newresource.BasicNewProjectResourceWizard;
import org.eclipse.ui.wizards.newresource.BasicNewResourceWizard;

public abstract class NewMonalipseProjectCreationWizard extends BasicNewResourceWizard implements IExecutableExtension
{
	private WizardNewProjectCreationPage mainPage;
	private IProject newProject;
	private IConfigurationElement configElement;
	private static String PAGE_PROBLEMS_TITLE = MonalipsePlugin.getResourceString("NewProject.errorOpeningPage"); //$NON-NLS-1$
	private static String WINDOW_PROBLEMS_TITLE = MonalipsePlugin.getResourceString("NewProject.errorOpeningWindow"); //$NON-NLS-1$

	public NewMonalipseProjectCreationWizard()
	{
		AbstractUIPlugin plugin = (AbstractUIPlugin)Platform.getPlugin(PlatformUI.PLUGIN_ID);
		IDialogSettings workbenchSettings = plugin.getDialogSettings();
		IDialogSettings section = workbenchSettings.getSection(getClass().getName());
		if(section == null)
			section = workbenchSettings.addNewSection(getClass().getName());
		setDialogSettings(section);
	}

	protected abstract Class getBBSServerImplClass();
	protected abstract String getInitialProjectName();

	public void addPages()
	{
		super.addPages();
		mainPage = new WizardNewProjectCreationPage("monalipseNewProjectPage") //$NON-NLS-1$
			{
				public void createControl(Composite parent)
				{
					super.createControl(parent);
					validate(parent);
				}
				
				private void validate(Control control)
				{
					if(control instanceof Composite)
					{
						Control[] children = ((Composite)control).getChildren();
						for(int i = 0; i < children.length; i++)
							validate(children[i]);
					}
					else if(control instanceof Text)
					{
						final Text text = (Text)control;
						if(text.getText().equals(getInitialProjectName()))
						{
							NewMonalipseProjectCreationWizard.this.getShell().getDisplay().asyncExec(new Runnable()
								{
									public void run()
									{
										text.setText(getInitialProjectName());
									}
								});
						}
					}
				}
			};
		mainPage.setTitle(MonalipsePlugin.getResourceString("NewProject.title")); //$NON-NLS-1$
		mainPage.setDescription(MonalipsePlugin.getResourceString("NewProject.description")); //$NON-NLS-1$
		mainPage.setInitialProjectName(getInitialProjectName());
		this.addPage(mainPage);
	}

	private IProject createNewProject()
	{
		if (newProject != null)
			return newProject;

		final IProject newProjectHandle = mainPage.getProjectHandle();

		IPath defaultPath = Platform.getLocation();
		IPath newPath = mainPage.getLocationPath();
		if(defaultPath.equals(newPath))
			newPath = null;
		IWorkspace workspace = ResourcesPlugin.getWorkspace();
		final IProjectDescription description = workspace.newProjectDescription(newProjectHandle.getName());
		description.setLocation(newPath);

		WorkspaceModifyOperation op = new WorkspaceModifyOperation()
			{
				protected void execute(IProgressMonitor monitor) throws CoreException
				{
					createProject(description, newProjectHandle, monitor);
				}
			};

		try
		{
			getContainer().run(true, true, op);
		}
		catch (InterruptedException e)
		{
			return null;
		}
		catch (InvocationTargetException e)
		{
			Throwable t = e.getTargetException();
			if(t instanceof CoreException)
			{
				if(((CoreException)t).getStatus().getCode() == IResourceStatus.CASE_VARIANT_EXISTS)
				{
					MessageDialog.openError(getShell(), MonalipsePlugin.getResourceString("NewProject.errorMessage"), //$NON-NLS-1$
											MonalipsePlugin.formatResourceString("NewProject.caseVariantExistsError", new String[] { newProjectHandle.getName()})); //$NON-NLS-1$,
				}
				else
				{
					ErrorDialog.openError(getShell(), MonalipsePlugin.getResourceString("NewProject.errorMessage"), //$NON-NLS-1$
											null, ((CoreException) t).getStatus());
				}
			}
			else
			{
				Platform.getPlugin(PlatformUI.PLUGIN_ID).getLog().log(new Status(Status.ERROR, PlatformUI.PLUGIN_ID, 0, t.toString(), t));
				MessageDialog.openError(getShell(), MonalipsePlugin.getResourceString("NewProject.errorMessage"), //$NON-NLS-1$
										MonalipsePlugin.formatResourceString("NewProject.internalError", new Object[] { t.getMessage()})); //$NON-NLS-1$
			}
			return null;
		}

		newProject = newProjectHandle;

		return newProject;
	}

	private static void addMonalipseNature(IProject project, IProgressMonitor monitor) throws CoreException
	{
		monitor.beginTask("", 1); //$NON-NLS-1$
		if(!project.hasNature(MonalipsePlugin.NATURE_ID))
		{
			IProjectDescription description = project.getDescription();
			String[] prevNatures = description.getNatureIds();
			String[] newNatures = new String[prevNatures.length + 1];
			System.arraycopy(prevNatures, 0, newNatures, 0, prevNatures.length);
			newNatures[prevNatures.length] = MonalipsePlugin.NATURE_ID;
			description.setNatureIds(newNatures);
			project.setDescription(description, monitor);
		}
		monitor.worked(1);
		monitor.done();
	}
	
	private void configureProject(IProject project, IProgressMonitor monitor) throws CoreException
	{
		IFile file = project.getFile(".classname"); //$NON-NLS-1$
		try
		{
			file.create(MonalipsePlugin.getUTFInputStream(getBBSServerImplClass().getName()), true, monitor);
		}
		catch (IOException e)
		{
			throw new RuntimeException(e);
		}
	}
	
	private void createProject(IProjectDescription description, IProject projectHandle, IProgressMonitor monitor) throws CoreException, OperationCanceledException
	{
		try
		{
			monitor.beginTask("", 4000); //$NON-NLS-1$

			projectHandle.create(description, new SubProgressMonitor(monitor, 1000));

			if (monitor.isCanceled())
				throw new OperationCanceledException();

			projectHandle.open(new SubProgressMonitor(monitor, 1000));

			if (monitor.isCanceled())
				throw new OperationCanceledException();

			addMonalipseNature(projectHandle, new SubProgressMonitor(monitor, 1000));

			if (monitor.isCanceled())
				throw new OperationCanceledException();

			configureProject(projectHandle, new SubProgressMonitor(monitor, 1000));
		}
		finally
		{
			monitor.done();
		}
	}

	public IProject getNewProject()
	{
		return newProject;
	}

	public void init(IWorkbench workbench, IStructuredSelection currentSelection)
	{
		super.init(workbench, currentSelection);
		setNeedsProgressMonitor(true);
		setWindowTitle(MonalipsePlugin.getResourceString("NewProject.windowTitle")); //$NON-NLS-1$
	}

	protected void initializeDefaultPageImageDescriptor()
	{
		String iconPath = "icons/full/"; //$NON-NLS-1$
		try
		{
			URL installURL = Platform.getPlugin(PlatformUI.PLUGIN_ID).getDescriptor().getInstallURL();
			URL url = new URL(installURL, iconPath + "wizban/newprj_wiz.gif"); //$NON-NLS-1$
			ImageDescriptor desc = ImageDescriptor.createFromURL(url);
			setDefaultPageImageDescriptor(desc);
		}
		catch (MalformedURLException e)
		{
		}
	}

	public boolean performFinish()
	{
		createNewProject();

		if (newProject == null)
			return false;

		BasicNewProjectResourceWizard.updatePerspective(configElement);
		selectAndReveal(newProject);

		return true;
	}

	public void setInitializationData(IConfigurationElement cfig, String propertyName, Object data)
	{
		configElement = cfig;
	}
}
