package monalipse;

import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import java.io.DataOutputStream;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.io.InputStream;
import java.lang.reflect.InvocationTargetException;
import java.net.MalformedURLException;
import java.net.URL;
import java.text.DateFormat;
import java.text.MessageFormat;
import java.text.SimpleDateFormat;
import java.util.Calendar;
import java.util.Locale;
import java.util.MissingResourceException;
import java.util.ResourceBundle;
import java.util.logging.ConsoleHandler;
import java.util.logging.Formatter;
import java.util.logging.Handler;
import java.util.logging.LogManager;
import java.util.logging.LogRecord;
import java.util.logging.Logger;

import monalipse.bookmark.BookmarkManager;
import monalipse.server.BBSServerManager;
import monalipse.server.IBBSBoard;
import monalipse.server.IThreadContentProvider;
import monalipse.views.IBBSReference;
import org.eclipse.core.resources.IFile;
import org.eclipse.core.resources.IProject;
import org.eclipse.core.resources.IResource;
import org.eclipse.core.resources.IResourceDelta;
import org.eclipse.core.resources.IWorkspace;
import org.eclipse.core.resources.IWorkspaceRoot;
import org.eclipse.core.resources.ResourcesPlugin;
import org.eclipse.core.runtime.CoreException;
import org.eclipse.core.runtime.IPluginDescriptor;
import org.eclipse.core.runtime.NullProgressMonitor;
import org.eclipse.core.runtime.Platform;
import org.eclipse.jface.operation.IRunnableWithProgress;
import org.eclipse.jface.resource.CompositeImageDescriptor;
import org.eclipse.jface.resource.ImageDescriptor;
import org.eclipse.jface.resource.ImageRegistry;
import org.eclipse.swt.graphics.Image;
import org.eclipse.swt.graphics.ImageData;
import org.eclipse.swt.graphics.Point;
import org.eclipse.ui.IWorkbenchWindow;
import org.eclipse.ui.plugin.AbstractUIPlugin;

public class MonalipsePlugin extends AbstractUIPlugin
{
	public static final String PLUGIN_ID = "monalipse" ; //$NON-NLS-1$
	public static final String NATURE_ID = PLUGIN_ID + ".monalipsenature" ; //$NON-NLS-1$
	
	public static final String IMG_OBJ_BOARD = MonalipsePlugin.class.getName() + ".obj.board";
	public static final String IMG_OBJ_THREAD = MonalipsePlugin.class.getName() + ".obj.thread";
	public static final String IMG_OBJ_FOLDER = MonalipsePlugin.class.getName() + ".obj.folder";
	public static final String IMG_OVR_BOOKMARK = MonalipsePlugin.class.getName() + ".ovr.bookmark";
	public static final String IMG_OVR_SUCCESS = MonalipsePlugin.class.getName() + ".ovr.success";

	private static MonalipsePlugin plugin;

	private ResourceBundle resourceBundle;

	static
	{
		try
		{
			FileInputStream inputStream = new FileInputStream(new File(ResourcesPlugin.getWorkspace().getRoot().getLocation().toFile(), "logging.properties"));
			LogManager.getLogManager().readConfiguration(inputStream);
			inputStream.close();
			System.err.println("logging.properties read");
		}
		catch (FileNotFoundException ex)
		{
		}
		catch (IOException ex)
		{
		}
	}

	public MonalipsePlugin(IPluginDescriptor descriptor)
	{
		super(descriptor);
		plugin = this;
		try
		{
			resourceBundle = ResourceBundle.getBundle("monalipse.MonalipsePluginResources");
		}
		catch (MissingResourceException x)
		{
			resourceBundle = null;
		}
	}
	
	protected void initializeImageRegistry(ImageRegistry reg)
	{
		String iconPath = "icons/"; //$NON-NLS-1$		
		URL installURL = Platform.getPlugin(MonalipsePlugin.PLUGIN_ID).getDescriptor().getInstallURL();
		try
		{
			reg.put(IMG_OBJ_BOARD, ImageDescriptor.createFromURL(new URL(installURL, iconPath + "category_obj.gif"))); //$NON-NLS-1$
			reg.put(IMG_OBJ_THREAD, ImageDescriptor.createFromURL(new URL(installURL, iconPath + "file_obj.gif"))); //$NON-NLS-1$
			reg.put(IMG_OBJ_FOLDER, ImageDescriptor.createFromURL(new URL(installURL, iconPath + "fldr_obj.gif"))); //$NON-NLS-1$
			reg.put(IMG_OVR_BOOKMARK, ImageDescriptor.createFromURL(new URL(installURL, iconPath + "bkmrk_ovr.gif"))); //$NON-NLS-1$
			reg.put(IMG_OVR_SUCCESS, ImageDescriptor.createFromURL(new URL(installURL, iconPath + "success_ovr.gif"))); //$NON-NLS-1$
		}
		catch (MalformedURLException e)
		{
		}
	}
	
	public Image getOverlayImage(String base, boolean bookmarked, boolean updated)
	{
		ImageRegistry reg = getImageRegistry();
		String key = base + bookmarked + updated;
		Image image = reg.get(key);
		if(image == null)
		{
			OverlayImageDescriptor ovr = new OverlayImageDescriptor(reg.get(base).getImageData());
			if(bookmarked)
				ovr.setTopRightImage(reg.get(IMG_OVR_BOOKMARK).getImageData());
			if(updated)
				ovr.setBottomLeftImage(reg.get(IMG_OVR_SUCCESS).getImageData());
			reg.put(key, ovr);
			image = reg.get(key);
		}
		return image;
	}
	
	public Image getLabelImageOf(IBBSReference element)
	{
		boolean bookmarked = false;
		if(element instanceof IBBSReference)
			bookmarked = BookmarkManager.getBookmarks().findItem(element.getURL()) != null;

		boolean updated = false;

		IThreadContentProvider thread = null;
		
		if(element instanceof IThreadContentProvider)
			thread = (IThreadContentProvider)element;
		else if(!(element instanceof IBBSBoard))
			thread = BBSServerManager.getThreadContentProviderOf(element.getURL());
		
		if(thread != null)
			updated = thread.hasNewResponses();

		if(thread == null)
			return MonalipsePlugin.getDefault().getOverlayImage(MonalipsePlugin.IMG_OBJ_BOARD, bookmarked, updated);
		else
			return MonalipsePlugin.getDefault().getOverlayImage(MonalipsePlugin.IMG_OBJ_THREAD, bookmarked, updated);
	}

	public static MonalipsePlugin getDefault()
	{
		return plugin;
	}

	public static IWorkspace getWorkspace()
	{
		return ResourcesPlugin.getWorkspace();
	}

	public static String getResourceString(String key)
	{
		ResourceBundle bundle = MonalipsePlugin.getDefault().getResourceBundle();
		try
		{
			return bundle.getString(key);
		}
		catch(MissingResourceException e)
		{
			return key;
		}
	}

	public static String formatResourceString(String key, Object[] args)
	{
		return MessageFormat.format(getResourceString(key), args);
	}

	public ResourceBundle getResourceBundle()
	{
		return resourceBundle;
	}

	public static InputStream getUTFInputStream(String str) throws IOException
	{
		ByteArrayOutputStream bout = new ByteArrayOutputStream();
		DataOutputStream dout = new DataOutputStream(bout);
		dout.writeUTF(str);
		return new ByteArrayInputStream(bout.toByteArray());
	}

	public static void ensureSynchronized(IFile file) throws CoreException
	{
		if(!file.isSynchronized(IResource.DEPTH_ZERO))
			file.refreshLocal(IResource.DEPTH_ZERO, new NullProgressMonitor());
	}

	public static Logger getLogger(Class clazz)
	{
		Logger logger = Logger.getLogger(clazz.getName());
		Handler handler = new ConsoleHandler();
		handler.setFormatter(new Formatter()
			{
				private DateFormat dateFormat = new SimpleDateFormat("HH:mm:ss", Locale.US);

				public String format(LogRecord record)
				{
					StringBuffer buf = new StringBuffer();
					Calendar cal = Calendar.getInstance();
					cal.setTimeInMillis(record.getMillis());
					buf.append(dateFormat.format(cal.getTime()));
					buf.append(' ');
					String clazz = record.getSourceClassName();
					buf.append(clazz.substring(clazz.lastIndexOf('.') + 1, clazz.length()));
					buf.append("#");
					buf.append(record.getSourceMethodName());
					int sp = Math.max(50 - buf.length(), 1);
					for(int i = 0; i < sp; i++)
						buf.append(' ');
					buf.append(record.getMessage());
					buf.append('\n');
					return buf.toString();
				}

				private void append2DigitsNum(StringBuffer buf, int n)
				{
					if(n < 10)
						buf.append('0');
					buf.append(n);
				}
			});
		logger.addHandler(handler);
		logger.setUseParentHandlers(false);
		return logger;
	}
	
	public static IProject getProject()
	{
		IWorkspace ws = ResourcesPlugin.getWorkspace();
		IWorkspaceRoot root = ws.getRoot();
		IProject[] proj = root.getProjects();
		for(int i = 0; i < proj.length; i++)
		{
			try
			{
				if(proj[i].getNature(MonalipsePlugin.NATURE_ID) != null)
					return proj[i];
			}
			catch (CoreException e)
			{
			}
		}
		return null;
	}
	
	public static boolean projectModified(IResourceDelta delta)
	{
		IProject project = getProject();
		return resourceModified(IResourceDelta.ADDED | IResourceDelta.REMOVED, delta, project);
	}

	public static boolean resourceModified(int type, IResourceDelta delta, IResource target)
	{
		boolean res = false;

		if(delta != null)
		{
			IResourceDelta[] affectedChildren = delta.getAffectedChildren(type);
			for (int i = 0; i < affectedChildren.length; i++)
			{
				if(affectedChildren[i].getResource().equals(target))
					return true;
				res |= resourceModified(type, affectedChildren[i], target);
			}
		}

		return res;
	}

	public static void asyncExec(final IWorkbenchWindow workbenchWindow, final IRunnableWithProgress run)
	{
		workbenchWindow.getShell().getDisplay().asyncExec(new Runnable()
			{
				public void run()
				{
					try
					{
						if(!workbenchWindow.getShell().isDisposed())
							workbenchWindow.run(false, false, run);
					}
					catch(InterruptedException e)
					{
						e.printStackTrace();
					}
					catch(InvocationTargetException e)
					{
						e.printStackTrace();
					}
				}
			});
	}

	public static void syncExec(final IWorkbenchWindow workbenchWindow, final IRunnableWithProgress run)
	{
		workbenchWindow.getShell().getDisplay().syncExec(new Runnable()
			{
				public void run()
				{
					try
					{
						if(!workbenchWindow.getShell().isDisposed())
							workbenchWindow.run(false, false, run);
					}
					catch(InterruptedException e)
					{
						e.printStackTrace();
					}
					catch(InvocationTargetException e)
					{
						e.printStackTrace();
					}
				}
			});
	}

	public class OverlayImageDescriptor extends CompositeImageDescriptor
	{
		private ImageData baseImage;
		private ImageData topLeftImage;
		private ImageData bottomLeftImage;
		private ImageData bottomRightImage;
		private ImageData topRightImage;

		public OverlayImageDescriptor(ImageData baseImage)
		{
			this.baseImage = baseImage;
		}

		protected Point getSize()
		{
			return new Point(baseImage.width, baseImage.height);
		}

		protected void drawCompositeImage(int width, int height)
		{
			drawImage(baseImage, 0, 0);

			if(topLeftImage != null)
				drawImage(topLeftImage, 0, 0);
			if(bottomLeftImage != null)
				drawImage(bottomLeftImage, 0, height - bottomLeftImage.height);
			if(bottomRightImage != null)
				drawImage(bottomRightImage, width - bottomRightImage.width, height - bottomRightImage.height);
			if(topRightImage != null)
				drawImage(topRightImage, width - topRightImage.width, 0);
		}
		
		public void setTopLeftImage(ImageData topLeftImage)
		{
			this.topLeftImage = topLeftImage;
		}
		
		public void setBottomLeftImage(ImageData bottomLeftImage)
		{
			this.bottomLeftImage = bottomLeftImage;
		}
		
		public void setBottomRightImage(ImageData bottomRightImage)
		{
			this.bottomRightImage = bottomRightImage;
		}
		
		public void setTopRightImage(ImageData topRightImage)
		{
			this.topRightImage = topRightImage;
		}
	}
}
