package monalipse.views;

import java.util.ArrayList;
import java.util.List;
import java.util.logging.Logger;

import org.eclipse.core.resources.IResourceChangeListener;
import org.eclipse.jface.viewers.ILabelProvider;
import org.eclipse.jface.viewers.IStructuredSelection;
import org.eclipse.jface.viewers.ITreeContentProvider;
import org.eclipse.jface.viewers.StructuredSelection;
import org.eclipse.jface.viewers.TreeViewer;
import org.eclipse.ui.IMemento;
import org.eclipse.ui.IViewSite;
import org.eclipse.ui.PartInitException;
import org.eclipse.ui.part.ViewPart;

public abstract class AbstractTreeView extends ViewPart implements IResourceChangeListener
{
	private static final String TAG_SELECTION = "selection";
	private static final String TAG_ELEMENT = "element";
	private static final String TAG_EXPANDED = "expanded";
	private static final String TAG_LABEL = "label";

	private static final String TAG_VERTICAL_POSITION = "verticalPosition";
	private static final String TAG_HORIZONTAL_POSITION = "horizontalPosition";
	private static final String TAG_FILTERS = "filters";
	private static final String TAG_FILTER = "filter";
	
	private IMemento memento;

	protected abstract Logger getLogger();
	protected abstract TreeViewer getViewer();
	protected abstract ITreeContentProvider getContentProvider();
	protected abstract ILabelProvider getLabelProvider();

	public void init(IViewSite site, IMemento memento) throws PartInitException
	{
		super.init(site, memento);
		this.memento = memento;
	}
	
	protected void partCreated()
	{
		if(memento != null)
			restoreState(memento);
		memento = null;
	}

	public void saveState(IMemento memento)
	{
		getLogger().finest("begin");
		if(getViewer() == null)
		{
			if(this.memento != null)
				memento.putMemento(this.memento);
		}
		else
		{
			saveExpansionState(memento);
			saveSelectionState(memento);
		}
		getLogger().finest("done");
	}

	private void saveSelectionState(IMemento memento)
	{
		setElements(memento.createChild(TAG_SELECTION), ((IStructuredSelection)getViewer().getSelection()).toArray());
	}

	private void saveExpansionState(IMemento memento)
	{
		setElements(memento.createChild(TAG_EXPANDED), getViewer().getVisibleExpandedElements());
	}
	
	private void setElements(IMemento memento, Object[] elements)
	{
		if(elements.length > 0)
		{
			for (int i = 0; i < elements.length; i++)
			{
				Object o = elements[i];
				IMemento elementMem = memento.createChild(TAG_ELEMENT);
				while(getContentProvider().getParent(o) != null)
				{
					IMemento labelMem = elementMem.createChild(TAG_LABEL);
					labelMem.putString(TAG_LABEL, getLabelProvider().getText(o));
					o = getContentProvider().getParent(o);
				}
			}
		}
	}

	protected void restoreState(IMemento memento)
	{
		getLogger().finest("begin");
		restoreExpansionState(memento);
		restoreSelectionState(memento);
		getLogger().finest("done");
	}

	private void restoreSelectionState(IMemento memento)
	{
		IMemento childMem;
		childMem = memento.getChild(TAG_SELECTION);
		if (childMem != null)
			getViewer().setSelection(new StructuredSelection(getElements(childMem)));
	}
	
	private void restoreExpansionState(IMemento memento)
	{
		IMemento childMem = memento.getChild(TAG_EXPANDED);
		if (childMem != null)
			getViewer().setExpandedElements(getElements(childMem).toArray());
	}
	
	private List getElements(IMemento memento)
	{
		ArrayList list = new ArrayList();

		IMemento[] elementMem = memento.getChildren(TAG_ELEMENT);
		for (int i = 0; i < elementMem.length; i++)
		{
			IMemento[] labelMem = elementMem[i].getChildren(TAG_LABEL);
			Object[] children = getContentProvider().getElements(getViewer().getInput());
			Object e = null;
			for(int j = labelMem.length - 1; 0 <= j; j--)
			{
				e = findElement(children, labelMem[j].getString(TAG_LABEL));
				if(e == null)
					break;
				else
					children = getContentProvider().getChildren(e);
			}
			if(e != null)
				list.add(e);
		}
		
		return list;
	}
	
	private Object findElement(Object[] list, String label)
	{
		if(list == null)
			return null;
		for(int i = 0; i < list.length; i++)
		{
			if(getLabelProvider().getText(list[i]).equals(label))
				return list[i];
		}
		return null;
	}


}
