package monalipse.views;

import java.io.UnsupportedEncodingException;
import java.net.MalformedURLException;
import java.net.URL;
import java.net.URLDecoder;
import java.net.URLEncoder;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;
import java.util.StringTokenizer;
import java.util.logging.Logger;

import monalipse.MonalipsePlugin;
import monalipse.actions.BookmarkAction;
import monalipse.bookmark.BookmarkManager;
import monalipse.editors.ThreadEditorInput;
import monalipse.editors.ThreadViewerEditor;
import monalipse.server.BBSServerManager;
import monalipse.server.IBBSBoard;
import monalipse.server.IThreadContentProvider;
import monalipse.utils.CancelableRunner;
import org.eclipse.core.resources.IResource;
import org.eclipse.core.resources.IResourceChangeEvent;
import org.eclipse.core.resources.IResourceChangeListener;
import org.eclipse.core.resources.ResourcesPlugin;
import org.eclipse.core.runtime.CoreException;
import org.eclipse.core.runtime.Platform;
import org.eclipse.core.runtime.QualifiedName;
import org.eclipse.jface.action.Action;
import org.eclipse.jface.action.ControlContribution;
import org.eclipse.jface.action.IAction;
import org.eclipse.jface.action.IMenuListener;
import org.eclipse.jface.action.IMenuManager;
import org.eclipse.jface.action.IStatusLineManager;
import org.eclipse.jface.action.IToolBarManager;
import org.eclipse.jface.action.MenuManager;
import org.eclipse.jface.action.Separator;
import org.eclipse.jface.resource.ImageDescriptor;
import org.eclipse.jface.viewers.DoubleClickEvent;
import org.eclipse.jface.viewers.IDoubleClickListener;
import org.eclipse.jface.viewers.ISelection;
import org.eclipse.jface.viewers.IStructuredContentProvider;
import org.eclipse.jface.viewers.IStructuredSelection;
import org.eclipse.jface.viewers.ITableLabelProvider;
import org.eclipse.jface.viewers.ITreeContentProvider;
import org.eclipse.jface.viewers.LabelProvider;
import org.eclipse.jface.viewers.StructuredSelection;
import org.eclipse.jface.viewers.TableTreeViewer;
import org.eclipse.jface.viewers.Viewer;
import org.eclipse.jface.viewers.ViewerSorter;
import org.eclipse.swt.SWT;
import org.eclipse.swt.custom.TableTree;
import org.eclipse.swt.events.KeyAdapter;
import org.eclipse.swt.events.KeyEvent;
import org.eclipse.swt.events.SelectionEvent;
import org.eclipse.swt.events.SelectionListener;
import org.eclipse.swt.graphics.Image;
import org.eclipse.swt.widgets.Combo;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Control;
import org.eclipse.swt.widgets.Menu;
import org.eclipse.swt.widgets.TableColumn;
import org.eclipse.ui.IActionBars;
import org.eclipse.ui.IEditorPart;
import org.eclipse.ui.IMemento;
import org.eclipse.ui.ISelectionListener;
import org.eclipse.ui.IViewSite;
import org.eclipse.ui.IWorkbenchPage;
import org.eclipse.ui.IWorkbenchPart;
import org.eclipse.ui.PartInitException;
import org.eclipse.ui.XMLMemento;
import org.eclipse.ui.part.ViewPart;

public class ThreadListView extends ViewPart implements ISelectionListener, IResourceChangeListener, SelectionListener
{
	public static final String ID_THREAD_LIST = ThreadListView.class.getName();

	private static final Logger logger = MonalipsePlugin.getLogger(ThreadListView.class);

	private static final String TAG_INPUT = "input";
	private static final String TAG_URL = "url";
	private static final String TAG_COLUMNWIDTH = "columnWidth";
	private static final String TAG_COLUMN = "column";
	private static final String TAG_WIDTH = "width";
	private static final String TAG_SELECTION = "selection";
	private static final String TAG_ELEMENT = "element";
	private static final String TAG_EXPANDED = "expanded";
	private static final String TAG_LABEL = "label";
	private static final String TAG_TITLE = "title";

	private TableTreeViewer viewer;
	private ThreadListContentProvider provider;
	private CancelableRunner cancelable;
	private IAction reloadAction;
	private IAction abortAction;
	private IAction doubleClickAction;
	private BookmarkAction bookmarkAction;
	private IMemento memento;
	private Combo searchCombo;
	private String constraint;

	public ThreadListView()
	{
		logger.finest("<init>");
	}
	
	public void init(IViewSite site, IMemento memento) throws PartInitException
	{
		super.init(site, memento);
		this.memento = memento;
	}

	public void createPartControl(Composite parent)
	{
		logger.finest("begin");
		viewer = new TableTreeViewer(parent, SWT.SINGLE | SWT.H_SCROLL | SWT.V_SCROLL | SWT.FULL_SELECTION);
		provider = new ThreadListContentProvider();
		viewer.setContentProvider(provider);
		viewer.setLabelProvider(provider);
//		viewer.setSorter(new NameSorter());
		viewer.setInput(ResourcesPlugin.getWorkspace());
		makeColumns();
		makeActions();
		hookContextMenu();
		hookDoubleClickAction();
		contributeToActionBars();
		ResourcesPlugin.getWorkspace().addResourceChangeListener(this);
		getViewSite().getPage().addSelectionListener(this);
		viewer.getTableTree().addSelectionListener(this);
		if(memento != null)
			restoreState(memento);
		memento = null;
		hookDragAndDrop();
		logger.finest("done");

		viewer.getTableTree().getTable().addKeyListener(new KeyAdapter()
			{
				public void keyPressed(KeyEvent e)
				{
					if(e.keyCode == SWT.F5)
						reloadAction.run();
				}
			});
	}

	public void resourceChanged(IResourceChangeEvent event)
	{
		if(viewer.getInput() instanceof IBBSBoard)
		{
			logger.finest("begin");
			IBBSBoard board = (IBBSBoard)viewer.getInput();
			if(board.threadListChanged(event))
			{
				logger.finest("threadListChanged");
				refreshList();
				if(viewer.getVisibleExpandedElements().length == 0)
					expandFirstFragment();
			}

			Object[] changed = board.getChangedItems(event);
			if(changed != null)
			{
				logger.finest("update");
				viewer.update(changed, null);
			}

			if(BookmarkManager.bookmarkChanged(event))
			{
				setTitleImage(MonalipsePlugin.getDefault().getLabelImageOf((IBBSBoard)viewer.getInput()));
				viewer.refresh(true);
			}
			logger.finest("done");
		}
	}

	private void refreshList()
	{
		if(provider != null)
		{
			logger.finest("begin");
			IMemento memento = XMLMemento.createWriteRoot(ThreadListView.class.getName());
			saveState(memento);
			logger.finest("reset input");
			viewer.setInput(viewer.getInput());
			restoreState(memento);
			logger.finest("done");
		}
	}
	
	public void selectionChanged(IWorkbenchPart part, ISelection selection)
	{
		if(!viewer.getTableTree().isDisposed())
		{
			if(selection instanceof IStructuredSelection)
			{
				Object first = ((IStructuredSelection)selection).getFirstElement();
				if(first instanceof IBBSReference)
				{
					IBBSBoard board = BBSServerManager.getBoardOf(((IBBSReference)first).getURL());
					if(board != null)
						selectionChanged(board);
				}
			}
		}
	}

	private void selectionChanged(IBBSBoard board)
	{
		logger.finest("begin");

		setTitle(board.getName());
		setTitleImage(MonalipsePlugin.getDefault().getLabelImageOf(board));

		if(viewer.getInput() == null || !viewer.getInput().equals(board))
		{
			logger.finest("folder changed");

			if(viewer.getInput() instanceof IBBSBoard)
				saveSearchWord(((IBBSBoard)viewer.getInput()).getLogFolder());

			constraint = "";
			viewer.setInput(board);
			restoreSearchWord(board.getLogFolder());

//			getSite().getShell().getDisplay().asyncExec(new Runnable()
//				{
//					public void run()
//					{
//						getSite().getPage().activate(ThreadListView.this);
//					}
//				});

			if(provider.getElements(viewer.getInput()).length == 0)
			{
				logger.finest("empty thread list");
				getSite().getShell().getDisplay().asyncExec(new Runnable()
					{
						public void run()
						{
							reloadAction.run();
						}
					});
			}
			else
			{
				expandFirstFragment();
			}
		}
		logger.finest("done");
	}

	private void expandFirstFragment()
	{
		if(provider.hasChildren(viewer.getInput()))
		{
			logger.finest("begin");
			Object[] fragments = provider.getElements(viewer.getInput());
			if(0 < fragments.length && provider.hasChildren(fragments[0]))
			{
				logger.finest("set expansion");
				viewer.setExpandedElements(new Object[]{fragments[0]});
			}
			logger.finest("done");
		}
	}

	public void widgetSelected(SelectionEvent e)
	{
		logger.finest("begin");
		IBBSReference ref = bookmarkAction.updateState(getSite().getWorkbenchWindow(), viewer.getSelection());
		if(ref instanceof IThreadContentProvider)
			ThreadViewerEditor.activateEditor(getSite(), (IThreadContentProvider)ref);
		logger.finest("done");
	}
	
	public void widgetDefaultSelected(SelectionEvent e)
	{
	}
	
	private void saveSearchWord(IResource res)
	{
		if(searchCombo != null && !searchCombo.isDisposed())
		{
			try
			{
				logger.finest("begin");
				StringBuffer buf = new StringBuffer();
				String[] keys = searchCombo.getItems();
				for(int i = 0; i < keys.length && i < 10; i++)
					buf.append(URLEncoder.encode(keys[i], "UTF-8")).append(" ");
				logger.finest("save search word: " + buf.toString());
				res.setPersistentProperty(new QualifiedName(getClass().getName(), "search"), buf.toString());
			}
			catch (UnsupportedEncodingException e)
			{
			}
			catch (CoreException e)
			{
			}
		}
	}
	
	private void restoreSearchWord(IResource res)
	{
		if(searchCombo != null && !searchCombo.isDisposed())
		{
			try
			{
				searchCombo.removeAll();
				searchCombo.add("");
				String keys = res.getPersistentProperty(new QualifiedName(getClass().getName(), "search"));
				if(keys != null)
				{
					logger.finest("restore search word: " + keys);
					StringTokenizer tk = new StringTokenizer(keys);
					while(tk.hasMoreTokens())
						searchCombo.add(URLDecoder.decode(tk.nextToken(), "UTF-8"));
				}
			}
			catch (UnsupportedEncodingException e)
			{
			}
			catch (CoreException e)
			{
			}
		}
	}

	public void saveState(IMemento memento)
	{
		logger.finest("begin");
		if(viewer == null)
		{
			if(this.memento != null)
				memento.putMemento(this.memento);
		}
		else
		{
			saveColumnState(memento);
			saveInputState(memento);
			saveExpansionState(memento);
			saveSelectionState(memento);
		}
		logger.finest("done");
	}

	private void saveColumnState(IMemento memento)
	{
		IMemento widthMem = memento.createChild(TAG_COLUMNWIDTH);
		for(int i = 0; i < viewer.getTableTree().getTable().getColumnCount(); i++)
		{
			IMemento columnMem = widthMem.createChild(TAG_COLUMN);
			columnMem.putInteger(TAG_WIDTH, viewer.getTableTree().getTable().getColumn(i).getWidth());
		}
	}

	private void saveInputState(IMemento memento)
	{
		IMemento elementMem = memento.createChild(TAG_INPUT);
		if(viewer.getInput() instanceof IBBSBoard)
			elementMem.putString(TAG_URL, ((IBBSBoard)viewer.getInput()).getURL().toExternalForm());
		elementMem.putString(TAG_TITLE, getTitle());
		if(viewer.getInput() instanceof IBBSBoard)
			saveSearchWord(((IBBSBoard)viewer.getInput()).getLogFolder());
	}

	private void saveSelectionState(IMemento memento)
	{
		setElements(memento.createChild(TAG_SELECTION), ((IStructuredSelection)viewer.getSelection()).toArray());
	}

	private void saveExpansionState(IMemento memento)
	{
		setElements(memento.createChild(TAG_EXPANDED), viewer.getVisibleExpandedElements());
	}
	
	private void setElements(IMemento memento, Object[] elements)
	{
		if(provider == null)
			return;

		if(elements.length > 0)
		{
			for (int i = 0; i < elements.length; i++)
			{
				Object o = elements[i];
				IMemento elementMem = memento.createChild(TAG_ELEMENT);
				while(provider.getParent(o) != null)
				{
					IMemento labelMem = elementMem.createChild(TAG_LABEL);
					labelMem.putString(TAG_LABEL, provider.getColumnText(o, 0));
					o = provider.getParent(o);
				}
			}
		}
	}

	private void restoreState(IMemento memento)
	{
		logger.finest("begin");
		restoreColumnState(memento);
		restoreInputState(memento);
		restoreExpansionState(memento);
		restoreSelectionState(memento);
		logger.finest("done");
	}

	private void restoreColumnState(IMemento memento)
	{
		IMemento widthMem = memento.getChild(TAG_COLUMNWIDTH);
		if(widthMem != null)
		{
			IMemento[] columnMem = widthMem.getChildren(TAG_COLUMN);
			if(columnMem != null)
			{
				for(int i = 0; i < viewer.getTableTree().getTable().getColumnCount() && i < columnMem.length; i++)
					viewer.getTableTree().getTable().getColumn(i).setWidth(columnMem[i].getInteger(TAG_WIDTH).intValue());
			}
		}
	}

	private void restoreInputState(IMemento memento)
	{
		IMemento elementMem = memento.getChild(TAG_INPUT);
		if(elementMem != null)
		{
			String url = elementMem.getString(TAG_URL);
			logger.finest(url);
			if(url != null)
			{
				try
				{
					IBBSBoard board = BBSServerManager.getBoardOf(new URL(url));
					if(board != null)
						viewer.setInput(board);
				}
				catch (MalformedURLException e)
				{
				}
			}
			
			String title = elementMem.getString(TAG_TITLE);
			if(title != null)
				setTitle(title);
		}
	}

	private void restoreSelectionState(IMemento memento)
	{
		IMemento childMem;
		childMem = memento.getChild(TAG_SELECTION);
		if (childMem != null)
			viewer.setSelection(new StructuredSelection(getElements(childMem)));
	}
	
	private void restoreExpansionState(IMemento memento)
	{
		IMemento childMem = memento.getChild(TAG_EXPANDED);
		if (childMem != null)
			viewer.setExpandedElements(getElements(childMem).toArray());
	}
	
	private List getElements(IMemento memento)
	{
		ArrayList list = new ArrayList();

		IMemento[] elementMem = memento.getChildren(TAG_ELEMENT);
		for (int i = 0; i < elementMem.length; i++)
		{
			IMemento[] labelMem = elementMem[i].getChildren(TAG_LABEL);
			Object[] children = provider.getElements(viewer.getInput());
			Object e = null;
			for(int j = labelMem.length - 1; 0 <= j; j--)
			{
				e = findElement(children, labelMem[j].getString(TAG_LABEL));
				if(e == null)
					break;
				else
					children = provider.getChildren(e);
			}
			if(e != null)
				list.add(e);
		}
		
		return list;
	}
	
	private Object findElement(Object[] list, String label)
	{
		if(list == null)
			return null;
		for(int i = 0; i < list.length; i++)
		{
			if(provider.getColumnText(list[i], 0).equals(label))
				return list[i];
		}
		return null;
	}

	private void hookContextMenu()
	{
		MenuManager menuMgr = new MenuManager("#PopupMenu");
		menuMgr.setRemoveAllWhenShown(true);
		menuMgr.addMenuListener(new IMenuListener()
			{
				public void menuAboutToShow(IMenuManager manager)
				{
					ThreadListView.this.fillContextMenu(manager);
				}
			});
		Menu menu = menuMgr.createContextMenu(viewer.getControl());
		viewer.getControl().setMenu(menu);
		getSite().registerContextMenu(menuMgr, viewer);
	}

	private void contributeToActionBars()
	{
		IActionBars bars = getViewSite().getActionBars();
//		fillLocalPullDown(bars.getMenuManager());
		fillLocalToolBar(bars.getToolBarManager());
		fillLocalStatuBar(bars.getStatusLineManager());
	}

	private void fillLocalStatuBar(IStatusLineManager manager)
	{
		cancelable = new CancelableRunner(manager, getSite().getShell().getDisplay(), abortAction);
	}

	private void fillLocalPullDown(IMenuManager manager)
	{
		manager.add(reloadAction);
		manager.add(new Separator());
		manager.add(abortAction);
	}

	private void fillContextMenu(IMenuManager manager)
	{
		bookmarkAction.updateState(getSite().getWorkbenchWindow(), viewer.getSelection());
		manager.add(reloadAction);
		manager.add(abortAction);
		manager.add(new Separator());
		manager.add(bookmarkAction);
		// Other plug-ins can contribute there actions here
		manager.add(new Separator("Additions"));
	}

	private void fillLocalToolBar(IToolBarManager manager)
	{
		manager.add(new ControlContribution(ThreadListView.class.getName() + ".constrain")
			{
				protected Control createControl(Composite parent)
				{
					searchCombo = new Combo(parent, SWT.DROP_DOWN);
					searchCombo.setToolTipText("Search");
					searchCombo.add("");
					if(viewer.getInput() instanceof IBBSBoard)
						restoreSearchWord(((IBBSBoard)viewer.getInput()).getLogFolder());
					searchCombo.addSelectionListener(new SelectionListener()
						{
							public void widgetSelected(SelectionEvent e)
							{
								selected(e);
							}
	
							public void widgetDefaultSelected(SelectionEvent e)
							{
								selected(e);
							}
							
							private void selected(SelectionEvent e)
							{
								String text = searchCombo.getText();
								if(0 < text.length())
								{
									if(Arrays.asList(searchCombo.getItems()).contains(text))
										searchCombo.remove(text);
									searchCombo.add(text, 1);
								}
								constraint = text;
								refreshList();
								searchCombo.setText(text);
								if(viewer.getVisibleExpandedElements().length == 0)
									expandFirstFragment();
							}
						});
					return searchCombo;
				}
			});

		manager.add(reloadAction);
		manager.add(abortAction);
	}

	private void makeColumns()
	{
		TableTree table = viewer.getTableTree();
		table.getTable().setHeaderVisible(true);
		table.getTable().setLinesVisible(true);
		TableColumn col;
		col = new TableColumn(table.getTable(), SWT.LEFT);
		col.setText("Rank");
		col.setWidth(80);
		col = new TableColumn(table.getTable(), SWT.LEFT);
		col.setText("Title");
		col.setWidth(400);
		col = new TableColumn(table.getTable(), SWT.LEFT);
		col.setText("Response");
		col.setWidth(50);
		col = new TableColumn(table.getTable(), SWT.LEFT);
		col.setText("Cache");
		col.setWidth(50);
		col = new TableColumn(table.getTable(), SWT.LEFT);
		col.setText("Read");
		col.setWidth(50);
	}

	private void makeActions()
	{
		String iconPath = "icons/"; //$NON-NLS-1$		
		URL installURL = Platform.getPlugin(MonalipsePlugin.PLUGIN_ID).getDescriptor().getInstallURL();

		reloadAction = new Action()
			{
				public void run()
				{
					cancelable.run(cancelable, new CancelableRunner.ICancelableRunnableWithProgress()
						{
							public void run(CancelableRunner.ICancelableProgressMonitor monitor)
							{
								if(viewer.getInput() instanceof IBBSBoard)
								{
									logger.finest("refresh");
									((IBBSBoard)viewer.getInput()).updateThreadList(getSite().getWorkbenchWindow());
								}
							}
						});
				}
			};
		reloadAction.setText("Reload");
		reloadAction.setToolTipText("Reload thread list");
		try
		{
			reloadAction.setImageDescriptor(ImageDescriptor.createFromURL(new URL(installURL, iconPath + "refresh_nav.gif"))); //$NON-NLS-1$
		}
		catch (MalformedURLException e)
		{
		}

		abortAction = new Action()
			{
				public void run()
				{
					cancelable.cancel();
				}
			};
		abortAction.setText("Abort");
		abortAction.setToolTipText("Abort downloading board list");
		abortAction.setEnabled(false);
		try
		{
			abortAction.setImageDescriptor(ImageDescriptor.createFromURL(new URL(installURL, iconPath + "stop_nav.gif"))); //$NON-NLS-1$
		}
		catch (MalformedURLException e)
		{
		}

		doubleClickAction = new Action()
			{
				public void run()
				{
					ISelection selection = viewer.getSelection();
					if(selection instanceof IStructuredSelection)
					{
						Object first = ((IStructuredSelection)selection).getFirstElement();
						if(first instanceof IThreadContentProvider)
						{
							logger.finest("begin");
							final IThreadContentProvider thread = (IThreadContentProvider)first;
							IWorkbenchPage page = getSite().getPage().getWorkbenchWindow().getActivePage();
							try
							{
								logger.finest("openEditor");
								IEditorPart part = page.openEditor(new ThreadEditorInput(thread), ThreadViewerEditor.class.getName());
								if(part instanceof ThreadViewerEditor)
								{
									logger.finest("updateThread");
									((ThreadViewerEditor)part).updateThread(-1);
								}
							}
							catch(PartInitException ex)
							{
								ex.printStackTrace();
							}
							logger.finest("done");
						}
					}
				}
			};
		
		bookmarkAction = new BookmarkAction();
	}

	private void hookDoubleClickAction()
	{
		viewer.addDoubleClickListener(new IDoubleClickListener()
		{
			public void doubleClick(DoubleClickEvent event)
			{
				doubleClickAction.run();
			}
		});
	}
	
	private void hookDragAndDrop()
	{
//		viewer.addDragSupport(DND.DROP_MOVE, new Transfer[]{TextTransfer.getInstance()}, new DragSourceListener()
//			{
//				public void dragStart(DragSourceEvent event)
//				{
//System.err.println(event);
//					if(viewer.getSelection() instanceof IStructuredSelection)
//						event.doit = 0 < ((IStructuredSelection)viewer.getSelection()).size();
//					else
//						event.doit = false;
//				}
//	
//				public void dragSetData(DragSourceEvent event)
//				{
//					if(viewer.getSelection() instanceof IStructuredSelection)
//					{
//						IStructuredSelection sel = (IStructuredSelection)viewer.getSelection();
//						if(sel.getFirstElement() instanceof IBBSReference)
//							event.data = ((IBBSReference)sel.getFirstElement()).getURL();
//					}
//
//					if(event.data == null)
//						event.data = "";
//				}
//	
//				public void dragFinished(DragSourceEvent event)
//				{
//				}
//			});
	}

	public void setFocus()
	{
		viewer.getControl().setFocus();
	}

	class NameSorter extends ViewerSorter
	{
	}
	
	private class ThreadListContentProvider extends LabelProvider implements IStructuredContentProvider, ITreeContentProvider, ITableLabelProvider
	{
		private static final int THREAD_LIST_DIVIDE_SIZE = 50;

		public Object[] getElements(Object inputElement)
		{
			if(inputElement instanceof IBBSBoard)
			{
				IThreadContentProvider[] cps = ((IBBSBoard)inputElement).getThreadList(constraint);
				Fragment[] fragments = new Fragment[(cps.length + THREAD_LIST_DIVIDE_SIZE - 1) / THREAD_LIST_DIVIDE_SIZE];
				for(int i = 0; i < fragments.length; i++)
					fragments[i] = new Fragment(cps, THREAD_LIST_DIVIDE_SIZE * i);
				return fragments;
			}
			else
			{
				return new Object[0];
			}
		}

		public Object[] getChildren(Object parentElement)
		{
			if(parentElement instanceof Fragment)
				return ((Fragment)parentElement).getChildren();
			else
				return null;
		}

		public Object getParent(Object element)
		{
			if(element instanceof Fragment)
			{
				return viewer.getInput();
			}
			else if(element instanceof IThreadContentProvider)
			{
				int index = ((IThreadContentProvider)element).getIndex() / THREAD_LIST_DIVIDE_SIZE;
				Object[] list = getElements(viewer.getInput());
				if(0 <= index && index < list.length)
					return list[index];
			}

			return null;
		}

		public boolean hasChildren(Object element)
		{
			return element instanceof Fragment || element instanceof IBBSBoard;
		}

		public void inputChanged(Viewer viewer, Object oldInput, Object newInput)
		{
		}

		public String getText(Object element)
		{
			if(element instanceof Fragment)
				return ((Fragment)element).getText();
			else if(element instanceof IThreadContentProvider)
				return ((IThreadContentProvider)element).getName();
			else
				return "";
		}

		public Image getColumnImage(Object element, int columnIndex)
		{
			if(element instanceof IThreadContentProvider && columnIndex == 1 && element instanceof IBBSReference)
				return MonalipsePlugin.getDefault().getLabelImageOf((IBBSReference)element);
			return null;
		}

		public String getColumnText(Object element, int columnIndex)
		{
			if(element instanceof IThreadContentProvider)
			{
				IThreadContentProvider thread = (IThreadContentProvider)element;
				switch(columnIndex)
				{
				case 0:
					return String.valueOf(thread.getIndex() + 1);
				
				case 1:
					return thread.getName();
				
				case 2:
					return String.valueOf(thread.getResponseCountHint());
				
				case 3:
					return String.valueOf(thread.getCachedCount());
				
				case 4:
					return getPersistentCountProperty(thread, ".read");
				}
			}
			else if(element instanceof Fragment)
			{
				if(columnIndex == 0)
					return ((Fragment)element).getText();
			}
			return "";
		}
		
		private String getPersistentCountProperty(IThreadContentProvider thread, String type)
		{
			try
			{
				if(viewer.getInput() instanceof IBBSBoard)
				{
					String count = ((IBBSBoard)viewer.getInput()).getLogFolder().getPersistentProperty(new QualifiedName(ThreadViewerEditor.class.getName(), thread.getID() + type));
					if(count == null)
						count = "0";
					return count;
				}
			}
			catch (CoreException e)
			{
			}
			return "0";
		}
		
		private class Fragment
		{
			private IThreadContentProvider[] children;
			private int start;
			
			public Fragment(IThreadContentProvider[]  children, int start)
			{
				this.children = children;
				this.start = start;
			}
			
			public IThreadContentProvider[] getChildren()
			{
				IThreadContentProvider[] ch = new IThreadContentProvider[Math.min(children.length - start, THREAD_LIST_DIVIDE_SIZE)];
				for(int i = 0; i < ch.length; i++)
					ch[i] = children[start + i];
				return ch;
			}
			
			public String getText()
			{
				return (start + 1) + "..." + (start + THREAD_LIST_DIVIDE_SIZE);
			}
			
			public boolean equals(Object obj)
			{
				if(obj instanceof Fragment)
					return ((Fragment)obj).start == start;
				else
					return false;
			}
			
			public int hashCode()
			{
				return start;
			}
		}

	}
}