package monalipse.server;

import java.awt.Graphics2D;
import java.awt.image.BufferedImage;
import java.awt.image.DataBuffer;
import java.awt.image.DataBufferByte;
import java.awt.image.IndexColorModel;
import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.lang.ref.SoftReference;
import java.lang.reflect.InvocationTargetException;
import java.net.ConnectException;
import java.net.SocketTimeoutException;
import java.net.URL;
import java.net.UnknownHostException;
import java.util.ArrayList;
import java.util.Date;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.logging.Logger;

import javax.imageio.ImageIO;
import javax.imageio.ImageReadParam;
import javax.imageio.ImageReader;
import javax.imageio.stream.ImageInputStream;

import monalipse.MonalipsePlugin;
import org.apache.commons.httpclient.HttpClient;
import org.apache.commons.httpclient.methods.GetMethod;
import org.eclipse.core.resources.IFile;
import org.eclipse.core.resources.IFolder;
import org.eclipse.core.resources.IResource;
import org.eclipse.core.resources.IResourceChangeEvent;
import org.eclipse.core.resources.IResourceDelta;
import org.eclipse.core.runtime.CoreException;
import org.eclipse.core.runtime.IAdaptable;
import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.core.runtime.QualifiedName;
import org.eclipse.swt.graphics.Device;
import org.eclipse.swt.graphics.Image;
import org.eclipse.swt.graphics.ImageData;
import org.eclipse.swt.graphics.PaletteData;
import org.eclipse.swt.graphics.RGB;
import org.eclipse.swt.program.Program;
import org.eclipse.ui.IWorkbenchWindow;
import org.eclipse.ui.actions.WorkspaceModifyOperation;

public class LinkedObject implements IAdaptable
{
	private static final Logger logger = MonalipsePlugin.getLogger();

	private static final ImageData LINK_IMAGE_DATA;
	private static final ImageData WAIT_IMAGE_DATA;
	private static final ImageData ERROR_IMAGE_DATA;
	private static final IndexColorModel INDEX_COLOR_MODEL;
	private static final PaletteData INDEX_PALETTE_DATA;
	
	public static int IMAGE_SIZE = 64;
	
	static
	{
		LINK_IMAGE_DATA = getFittedImage(getScaledImage(MonalipsePlugin.getDefault().getImageRegistry().get(MonalipsePlugin.IMG_OBJ_LINK).getImageData(), 32), IMAGE_SIZE);
		WAIT_IMAGE_DATA = getFittedImage(getScaledImage(MonalipsePlugin.getDefault().getImageRegistry().get(MonalipsePlugin.IMG_OBJ_CLOCK).getImageData(), 32), IMAGE_SIZE);
		ERROR_IMAGE_DATA = getFittedImage(getScaledImage(MonalipsePlugin.getDefault().getImageRegistry().get(MonalipsePlugin.IMG_OBJ_DISCONNECT).getImageData(), 32), IMAGE_SIZE);

		int[] colors = new int[256];
		RGB[] rgbs = new RGB[256];
		for(int r = 0; r < 6; r++)
		{
			for(int g = 0; g < 6; g++)
			{
				for(int b = 0; b < 6; b++)
				{
					int idx = r * 36 + g * 6 + b;
					colors[idx] = ((r * 51) << 16) | ((g * 51) << 8) | (b * 51);
					rgbs[idx] = new RGB(r * 51, g * 51, b * 51);
				}
			}
		}
		
		int g = 256 / (256 - 6 * 6 * 6);
		int gray = g * 3;
		for(int i = 6 * 6 * 6; i < 256; i++)
		{
			colors[i] = (gray << 16) | (gray << 8) | gray;
			rgbs[i] = new RGB(gray, gray, gray);
			gray += g;
		}
		
		INDEX_COLOR_MODEL = new IndexColorModel(8, 256, colors, 0, false, -1, DataBuffer.TYPE_BYTE);
		INDEX_PALETTE_DATA = new PaletteData(rgbs);
	}

	private IThreadContentProvider thread;
	private int responseNumber;
	private URL url;
	private String urlText;
	private IFile file;
	private SoftReference imageDataReference;
	private ImageData imageData;
	private Image image;
	private boolean isImage;
	private QualifiedName statusKey;

	public LinkedObject(IThreadContentProvider thread, int responseNumber, URL url)
	{
		this.thread = thread;
		this.responseNumber = responseNumber;
		this.url = url;
		urlText = url.toExternalForm();
		String dir = url.toExternalForm().toLowerCase();
		file = thread.getLogFolder().getFolder(thread.getID() + ".images").getFile(dir.substring(dir.lastIndexOf('/') + 1, dir.length()));
		isImage = isImage(file.getName().toLowerCase());
		statusKey = new QualifiedName(getClass().getName(), url + ".status");
	}
	
	public boolean equals(Object obj)
	{
		if(obj instanceof LinkedObject)
		{
			LinkedObject linked = (LinkedObject)obj;
			return linked.thread.equals(thread) && linked.responseNumber == responseNumber && linked.urlText.equals(urlText);
		}
		return false;
	}
	
	public int hashCode()
	{
		return urlText.hashCode();
	}
	
	public boolean isImage()
	{
		return isImage;
	}

	public IThreadContentProvider getThread()
	{
		return thread;
	}

	public int getResponseNumber()
	{
		return responseNumber;
	}
	
	public synchronized boolean fetchObject(final IWorkbenchWindow workbenchWindow, boolean force) throws SocketTimeoutException, ConnectException, UnknownHostException
	{
		if(!force)
		{
			try
			{
				String status = thread.getLogFile().getPersistentProperty(statusKey);
				if(status != null && status.equals("error"))
				{
					dispose();
					return false;
				}
			}
			catch (CoreException e)
			{
			}
		}

		try
		{
			MonalipsePlugin.ensureSynchronized(file);
			if(!file.exists())
			{
				logger.finest("begin " + url);

				HttpClient hc = new HttpClient();
				hc.setTimeout(10000);
				
				GetMethod get = new GetMethod();
				get.setRequestHeader("User-Agent", "Mozilla/4.0 (compatible; MSIE 5.5; Windows NT 5.0)");
				get.setRequestHeader("Referer", urlText);
				get.setRequestHeader("Accept", "image/png, image/jpeg, image/gif, */*");
				get.setRequestHeader("Accept-Language", "ja");
				get.setRequestHeader("Accept-Encoding", "gzip");
				get.setRequestHeader("Accept-Charset", "Shift_JIS, *");
				get.setPath(url.getPath());
				get.setHttp11(false);

//				hc.startSession(url.getHost(), url.getPort(), "127.0.0.1", 8080);
				hc.startSession(url);
				try
				{
					hc.executeMethod(get);
					MonalipsePlugin.log(logger, get);

					if(get.getStatusLine().getStatusCode() == 200)
					{
						logger.finest("download");
						InputStream in = get.getResponseBodyAsStream();
						ByteArrayOutputStream out = new ByteArrayOutputStream();
						byte[] b = new byte[32768];
						int header = 16;
						for(int i = 0; i < header; i++)
							b[i] = (byte)in.read();
						logger.finest("image:" + isImage(b) + " archive:" + isArchive(b));
						if(isImage(b) || isArchive(b))
						{
							out.write(b, 0, header);
							while(true)
							{
								int r = in.read(b);
								if(r == -1)
									break;
								out.write(b, 0, r);
							}
							out.close();
							logger.finest("complete");
							long mod = 0;
							try
							{
								mod = Date.parse(get.getResponseHeader("Last-Modified").getValue());
							}
							catch (RuntimeException e)
							{
							}
							final long lastModified = mod;
							final byte[] bytes = out.toByteArray();
							MonalipsePlugin.asyncExec(workbenchWindow, new WorkspaceModifyOperation()
								{
									protected void execute(IProgressMonitor monitor) throws InvocationTargetException
									{
										try
										{
											createLogFolder(workbenchWindow, monitor);
											MonalipsePlugin.ensureSynchronized(file);
											if(file.exists())
												file.setContents(new ByteArrayInputStream(bytes), false, false, monitor);
											else
												file.create(new ByteArrayInputStream(bytes), false, monitor);
											imageData = null;
											imageDataReference = null;
											if(lastModified != 0)
											{
												file.getLocation().toFile().setLastModified(lastModified);
												MonalipsePlugin.ensureSynchronized(file);
											}
										}
										catch (CoreException e)
										{
											throw new InvocationTargetException(e);
										}
									}
							
									private void createLogFolder(IWorkbenchWindow workbenchWindow, IProgressMonitor monitor)
									{
										thread.getBoard().getServer().createServerFolder(workbenchWindow, monitor);
										thread.getBoard().createLogFolder(workbenchWindow, monitor);
										try
										{
											if(!file.getParent().exists())
												((IFolder)file.getParent()).create(true, true, monitor);
										}
										catch (CoreException e)
										{
											e.printStackTrace();
										}
									}
								});
							return true;
						}
						{
							thread.getLogFile().setPersistentProperty(statusKey, "error");
							dispose();
						}
					}
					else if(get.getStatusLine().getStatusCode() == 404 ||
							get.getStatusLine().getStatusCode() == 403 ||
							get.getStatusLine().getStatusCode() == 302)
					{
						thread.getLogFile().setPersistentProperty(statusKey, "error");
						dispose();
					}
				}
				finally
				{
					hc.endSession();
					logger.finest("done");
				}
			}
		}
		catch (CoreException e)
		{
			e.printStackTrace();
		}
		catch (SocketTimeoutException e)
		{
			throw e;
		}
		catch (ConnectException e)
		{
			throw e;
		}
		catch (UnknownHostException e)
		{
			throw e;
		}
		catch (IOException e)
		{
			e.printStackTrace();
		}
		
		return false;
	}

	public boolean loadImage()
	{
		if(imageData != null || (imageDataReference != null && imageDataReference.get() != null))
			return false;

		try
		{
			MonalipsePlugin.ensureSynchronized(file);
			if(file.exists())
			{
				if(isImage)
				{
					try
					{
						int size = IMAGE_SIZE;

						String name = file.getName();
						Iterator readers = ImageIO.getImageReadersBySuffix(name.substring(name.lastIndexOf('.') + 1, name.length()));
						
						if(readers.hasNext())
						{
							ImageReader reader = (ImageReader)readers.next();
							
							ImageInputStream in = null;
							try
							{
								in = ImageIO.createImageInputStream(file.getLocation().toFile());
								reader.setInput(in);
								ImageReadParam param = reader.getDefaultReadParam();
								int skip = (int)Math.ceil(Math.max(reader.getWidth(0), reader.getHeight(0)) / (double)size);
								param.setSourceSubsampling(skip, skip, 0, 0);
								BufferedImage image = reader.read(0, param);

								BufferedImage indexed = new BufferedImage(size, size, BufferedImage.TYPE_BYTE_INDEXED, INDEX_COLOR_MODEL);
								Graphics2D g = indexed.createGraphics();
								g.setColor(java.awt.Color.white);
								g.fillRect(0, 0, size, size);
								g.drawImage(image, (size - image.getWidth()) / 2, (size - image.getHeight()) / 2, null);
								g.dispose();
								
								ImageData data = new ImageData(size, size, 8, INDEX_PALETTE_DATA, 1, ((DataBufferByte)indexed.getRaster().getDataBuffer()).getData());
								imageDataReference = new SoftReference(data);

//								ImageData data = new ImageData(size, size, 24, new PaletteData(0xff0000, 0x00ff00, 0x0000ff));
//								imageDataReference = new SoftReference(data);
//								int[] pixels = new int[size * size];
//								for(int i = 0; i < pixels.length; i++)
//									pixels[i] = 0x00ffffff;
//								int w = Math.min(size, image.getWidth());
//								int h = Math.min(size, image.getHeight());
//								image.getRGB(0, 0, w, h, pixels, (size - w) / 2 + (size - h) / 2 * size, size);
//								for(int i = 0; i < size; i++)
//									data.setPixels(0, i, size, pixels, size * i);
								
								return true;
							}
							catch (IOException ex)
							{
								ex.printStackTrace();
							}
							finally
							{
								reader.dispose();
								if(in != null)
									in.close();
							}
						}

//						BufferedImage image = ImageIO.read(file.getLocation().toFile());
//						double scale = (double)size / Math.max(image.getWidth(), image.getHeight());
//						int w = (int)(image.getWidth() * scale);
//						int h = (int)(image.getHeight() * scale);
//
//						BufferedImage scaled = new BufferedImage(size, size, BufferedImage.TYPE_INT_RGB);
//						Graphics2D g = scaled.createGraphics();
//						g.setColor(java.awt.Color.white);
//						g.fillRect(0, 0, size, size);
//						g.setRenderingHint(RenderingHints.KEY_RENDERING, RenderingHints.VALUE_RENDER_QUALITY);
//						g.drawImage(image, (size - w) / 2, (size - h) / 2, w, h, null);
//						g.dispose();
//
//						imageData = new ImageData(size, size, 24, new PaletteData(0xff0000, 0x00ff00, 0x0000ff));
//						int[] pixels = new int[size * size];
//						scaled.getRGB(0, 0, size, size, pixels, 0, size);
//						for(int i = 0; i < size; i++)
//							imageData.setPixels(0, i, size, pixels, size * i);
//						return true;
					}
					catch(Throwable e)
					{
						e.printStackTrace();
					}
				}
				
				String name = file.getName();
				imageDataReference = new SoftReference(getImageDataOf(name.substring(name.lastIndexOf('.') + 1, name.length())));
			}
			else
			{
				String status = thread.getLogFile().getPersistentProperty(statusKey);
				if(status != null && status.equals("error"))
					imageData = ERROR_IMAGE_DATA;
				else
					imageData = LINK_IMAGE_DATA;
			}
		}
		catch (Exception e)
		{
			e.printStackTrace();
		}
		
		return true;
	}

	public Image getImage(Device device)
	{
		ImageData data = imageData == null ? (imageDataReference == null ? null : (ImageData)imageDataReference.get()) : imageData;
		if(image == null && data != null)
			image = new Image(device, data);
		return image;
	}

	public String getName()
	{
		return file.getName();
	}

	public URL getURL()
	{
		return url;
	}

	public String getURLText()
	{
		return urlText;
	}
	
	public IFile getCachedFile()
	{
		return file;
	}
	
	public void dispose()
	{
		if(image != null)
			image.dispose();
		image = null;
	}
	
	public Object getAdapter(Class adapter)
	{
		if(adapter == IBBSReference.class)
		{
			return new IBBSReference()
				{
					public String getName()
					{
						return thread.getName();
					}
	
					public URL getURL()
					{
						return thread.getURL();
					}
	
					public int getResponseNumber()
					{
						return LinkedObject.this.getResponseNumber();
					}
				};
		}
		return null;
	}
	
	private static ImageData getImageDataOf(String extension)
	{
		Program p;
		p = Program.findProgram(extension);
		if(p != null)
			return getFittedImage(getScaledImage(p.getImageData(), 32), IMAGE_SIZE);
		else
			return null;
	}

	private static ImageData getScaledImage(ImageData data, int size)
	{
		if(data.width == size && data.height == size)
			return data;

		double scale = (double)size / Math.max(data.width, data.height);
		data = data.scaledTo((int)(data.width * scale), (int)(data.height * scale));
		
		return data;
	}
	
	private static ImageData getFittedImage(ImageData data, int size)
	{
		if(data.width == size && data.height == size)
			return data;
		
		int white;
		boolean convertColor = false;
		
		try
		{
			white = data.palette.getPixel(new RGB(0xff, 0xff, 0xff));
		}
		catch(IllegalArgumentException e)
		{
			white = 0x00ffffff;
			convertColor = true;
		}

		int[] whites = new int[size];
		int[] pixels = new int[size];
		for(int i = 0; i < whites.length; i++)
			whites[i] = white;
		int xoff = (size - data.width) / 2;
		int yoff = (size - data.height) / 2;

		ImageData fit = new ImageData(size, size, convertColor ? 24 : data.depth, convertColor ? new PaletteData(0xff0000, 0x00ff00, 0x0000ff) : data.palette);
		for(int i = 0; i < yoff; i++)
			fit.setPixels(0, i, size, whites, 0);
		for(int y = 0; y < data.height; y++)
		{
			System.arraycopy(whites, 0, pixels, 0, size);
			if(convertColor && !data.palette.isDirect)
			{
				for(int i = 0; i < data.width; i++)
				{
					RGB rgb = data.palette.colors[data.getPixel(i, y)];
					pixels[xoff + i] = (rgb.red << 16) | (rgb.green << 8) | rgb.blue;
				}
			}
			else
			{
				data.getPixels(0, y, data.width, pixels, xoff);
			}
			fit.setPixels(0, yoff + y, size, pixels, 0);
		}
		for(int i = yoff + data.height; i < size; i++)
			fit.setPixels(0, i, size, whites, 0);
		return fit;
	}
	
	public static boolean matches(String fileName)
	{
		String lc = fileName.toLowerCase();
		return isImage(lc) || isArchive(lc);
	}

	private static boolean isImage(String lc)
	{
		return lc.endsWith(".jpg") || lc.endsWith(".jpeg") || lc.endsWith(".gif") || lc.endsWith(".png") || lc.endsWith(".bmp");
	}
	
	private static boolean isArchive(String lc)
	{
		return lc.endsWith(".zip") || lc.endsWith(".lzh") || lc.endsWith(".rar") || lc.endsWith(".gca");
	}

	private static boolean isImage(byte[] b)
	{
		return (b[0] == (byte)0xff && b[1] == (byte)0xd8) ||
				(b[0] == (byte)'G'&& b[1] == (byte)'I' && b[2] == (byte)'F') ||
				(b[0] == (byte)0x89 && b[1] == (byte)0x50 && b[2] == (byte)0x4e && b[3] == (byte)0x47) ||
				(b[0] == (byte)'B'&& b[1] == (byte)'M');
	}

	private static boolean isArchive(byte[] b)
	{
		return (b[0] == (byte)0x50 && b[1] == (byte)0x4b) ||
				(b[2] == (byte)'-'&& b[3] == (byte)'L' && b[4] == (byte)'H') ||
				(b[0] == (byte)'R' && b[1] == (byte)'a' && b[2] == (byte)'r') ||
				(b[0] == (byte)'G'&& b[1] == (byte)'C' && b[2] == (byte)'A');
	}

	public static LinkedObject[] getChangedNodes(LinkedObject[] objects, IResourceChangeEvent event)
	{
		Map cachedResources = new HashMap();
		for(int i = 0; i < objects.length; i++)
			addNodeMap(cachedResources, objects[i].file, objects[i]);

		List list = new ArrayList();
		nodeModified(IResourceDelta.ADDED | IResourceDelta.REMOVED | IResourceDelta.CHANGED, event.getDelta(), cachedResources, list);
		if(list.size() == 0)
			return null;

		objects = new LinkedObject[list.size()];
		list.toArray(objects);
		return objects;
	}
	
	private static void addNodeMap(Map map, Object key, LinkedObject node)
	{
		List list;
		if(map.containsKey(key))
		{
			list = (List)map.get(key);
		}
		else
		{
			list = new ArrayList();
			map.put(key, list);
		}

		list.add(node);
	}

	private static void nodeModified(int type, IResourceDelta delta, Map cachedResources, List resultNodes)
	{
		if(delta != null)
		{
			IResourceDelta[] affectedChildren = delta.getAffectedChildren(type);
			for (int i = 0; i < affectedChildren.length; i++)
			{
				IResource res = affectedChildren[i].getResource();
				if(res instanceof IFile)
				{
					List nodeList = (List)cachedResources.get(res);
					if(nodeList != null)
					{
						for(int j = 0; j < nodeList.size(); j++)
							((LinkedObject)nodeList.get(j)).dispose();
						resultNodes.addAll(nodeList);
					}
				}
				nodeModified(type, affectedChildren[i], cachedResources, resultNodes);
			}
		}
	}
}
