package monalipse.server.giko;

import java.io.BufferedReader;
import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import java.io.DataInputStream;
import java.io.DataOutputStream;
import java.io.IOException;
import java.io.InputStreamReader;
import java.io.UnsupportedEncodingException;
import java.lang.reflect.InvocationTargetException;
import java.net.MalformedURLException;
import java.net.URL;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.logging.Logger;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

import javax.xml.transform.TransformerException;

import monalipse.MonalipsePlugin;
import monalipse.server.IBBSBoard;
import monalipse.server.IBBSServer;
import monalipse.server.IBoardTreeNode;
import monalipse.server.IThreadContentProvider;
import org.apache.commons.httpclient.Header;
import org.apache.commons.httpclient.HttpClient;
import org.apache.commons.httpclient.methods.GetMethod;
import org.eclipse.core.resources.IFile;
import org.eclipse.core.resources.IFolder;
import org.eclipse.core.resources.IResource;
import org.eclipse.core.resources.IResourceChangeEvent;
import org.eclipse.core.resources.IResourceDelta;
import org.eclipse.core.runtime.CoreException;
import org.eclipse.core.runtime.IAdaptable;
import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.core.runtime.QualifiedName;
import org.eclipse.ui.IWorkbenchWindow;
import org.eclipse.ui.actions.WorkspaceModifyOperation;

class Board implements IAdaptable, IBBSBoard, IBoardTreeNode
{
	private static final Logger logger = MonalipsePlugin.getLogger();

	private IBoardTreeNode parent;
	private String name;
	private String primaryURLString;
	private URL primaryURL;
	private List obsoleteURLs;
	private GikoServer server;
	private IFolder logFolder;
	private List threadList;
	private Map cacheMap = new HashMap();

	public Board(GikoServer server, IBoardTreeNode parent, IFolder logFolder, String name, URL primaryURL, String[] obsoleteURLs)
	{
		this.server = server;
		this.parent = parent;
		this.logFolder = logFolder;
		this.name = name;
		this.primaryURL = primaryURL;
		this.primaryURLString = primaryURL.toExternalForm();
		this.obsoleteURLs = new ArrayList(Arrays.asList(obsoleteURLs));
	}
	
	public boolean equals(Object obj)
	{
		if(obj instanceof Board)
			return ((Board)obj).primaryURLString.equals(primaryURLString);
		else
			return false;
	}
	
	public int hashCode()
	{
		return primaryURLString.hashCode();
	}
	
	public IBBSServer getServer()
	{
		return server;
	}
	
	public IFolder getLogFolder()
	{
		return logFolder;
	}

	public String getName()
	{
		return name;
	}
	
	public URL getURL()
	{
		return primaryURL;
	}
	
	public int getResponseNumber()
	{
		return -1;
	}
	
	public String[] getObsoleteURLs()
	{
		String[] obs = new String[obsoleteURLs.size()];
		obsoleteURLs.toArray(obs);
		return obs;
	}
	
	public boolean matches(String url)
	{
		if(primaryURLString.equals(url))
			return true;
		for(int i = 0; i < obsoleteURLs.size(); i++)
		{
			if(obsoleteURLs.get(i).equals(url))
				return true;
		}
		return false;
	}

	public Object getAdapter(Class adapter)
	{
		return null;
	}
	
	public IBoardTreeNode[] getChildren()
	{
		return new IBoardTreeNode[0];
	}

	public IBoardTreeNode getParent()
	{
		return parent;
	}

	public boolean hasChildren()
	{
		return false;
	}

	public IThreadContentProvider[] getThreadList(String constraint)
	{
		ensureCacheLoaded();
		
		List list;
		if(constraint != null && 0 < constraint.length())
		{
			constraint = toRegularForm(constraint);
			list = new ArrayList();
			for(int i = 0; i < threadList.size(); i++)
			{
				ThreadContentProvider cp = (ThreadContentProvider)threadList.get(i);
				if(isVisible(cp.getName(), constraint))
					list.add(cp);
			}
		}
		else
		{
			list = threadList;
		}

		IThreadContentProvider[] res = new IThreadContentProvider[list.size()];
		list.toArray(res);
		return res;
	}
	
	boolean containsID(String id)
	{
		return cacheMap.isEmpty() || cacheMap.containsKey(id);
	}

	public IThreadContentProvider getThreadOf(URL url)
	{
		ensureCacheLoaded();
		ThreadReference ref = ThreadReference.of(url);
		ThreadContentProvider cp = (ThreadContentProvider)cacheMap.get(ref.getID());
		if(cp != null)
			return cp;
		else if(ref.getReferenceType() == ThreadReference.REFERENCE_THREAD)
			return new ThreadContentProvider(this, ref.getBoardURL(), ref.getID(), 0, null, 0);
		else
			return null;
	}

	public URL getURLOf(IFile file)
	{
		ThreadContentProvider cp = (ThreadContentProvider)cacheMap.get(file.getName());
		if(cp != null)
			return cp.getURL();
		else
			return getURL();
	}

	public boolean threadListChanged(IResourceChangeEvent event)
	{
		return MonalipsePlugin.resourceModified(IResourceDelta.ADDED | IResourceDelta.REMOVED | IResourceDelta.CHANGED, event.getDelta(), logFolder.getFile(".subject"));
	}

	public synchronized void updateThreadList(IWorkbenchWindow workbenchWindow)
	{
		try
		{
			HttpClient hc = GikoServer.createHttpClient();
			GetMethod get = GikoServer.createGetMethod();

			URL url = new URL(getURL(), "subject.txt");
			get.setRequestHeader("Accept-Encoding", "gzip");
			get.setPath(url.getFile());
			try
			{
				String lastModified = getLogFolder().getPersistentProperty(new QualifiedName(Board.class.getName(), "Last-Modified"));
				if(lastModified != null)
					get.setRequestHeader("If-Modified-Since", lastModified);
			}
			catch (CoreException e)
			{
			}

			hc.startSession(url);
			hc.executeMethod(get);
			MonalipsePlugin.log(logger, get);
			
			if(get.getStatusLine().getStatusCode() == 302)
			{
				hc.endSession();
				hc.startSession(getURL());
				get = GikoServer.createGetMethod();
				get.setPath(getURL().getFile());
				hc.executeMethod(get);
				if(get.getStatusLine().getStatusCode() == 200)
				{
					BufferedReader r = new BufferedReader(new InputStreamReader(get.getResponseBodyAsStream(), "Windows-31J"));
					Pattern p = Pattern.compile("http.*" + getURL().getFile());
					while(true)
					{
						String line = r.readLine();
						if(line == null)
							break;
						Matcher m = p.matcher(line);
						if(m.find())
						{
							setURL(new URL(m.group()));
							server.saveCachedMenu(workbenchWindow);
							hc.endSession();
							get = GikoServer.createGetMethod();
							get.setPath(getURL().getFile());
							get.setRequestHeader("Accept-Encoding", "gzip");
							hc.startSession(new URL(getURL(), "subject.txt"));
							hc.executeMethod(get);
						}
					}
				}
			}


			if(get.getStatusLine().getStatusCode() == 200)
			{
				BufferedReader r = new BufferedReader(new InputStreamReader(GikoServer.getResponseBodyAsStream(get), "Windows-31J"));
				try
				{
					threadList = new ArrayList();
					cacheMap = new HashMap();
					String delim = null;
					String br = null;
					while(true)
					{
						String line = r.readLine();
						if(line == null)
							break;
						line = line.trim();
						if(0 < line.length())
						{
							if(delim == null)
							{
								int c = line.indexOf(",");
								int b = line.indexOf("<>");
								if(c == -1)
								{
									if(b == -1)
										break;
									else
										delim = "<>";
								}
								else
								{
									if(b == -1)
										delim = ",";
									else if(c < b)
										delim = "b";
									else if(b < c)
										delim = "<>";
								}
								
								if(line.endsWith(")"))
									br = "(";
								else if(line.endsWith(">"))
									br = "<";
								else if(line.endsWith("\uff09")) // FULLWIDTH RIGHT PARENTHESIS
									br = "\uff08"; // FULLWIDTH LEFT PARENTHESIS
								else
									break;
							}
							
							String id = line.substring(0, line.indexOf(delim)).trim();
							String name = line.substring(id.length() + delim.length(), line.lastIndexOf(br)).trim();
							int count = Integer.parseInt(line.substring(line.lastIndexOf(br) + 1, line.length() - 1).trim());
							
							ThreadContentProvider thread = new ThreadContentProvider(this, getURL(), id, threadList.size(), name, count);
							threadList.add(thread);
							cacheMap.put(thread.getID(), thread);
						}
					}
					
					Header h = get.getResponseHeader("Last-Modified");
					saveCachedSubject(workbenchWindow, h == null ? null : h.getValue());
				}
				finally
				{
					r.close();
					hc.endSession();
				}
			}
		}
		catch (MalformedURLException e)
		{
			e.printStackTrace();
		}
		catch (UnsupportedEncodingException e)
		{
			e.printStackTrace();
		}
		catch (IOException e)
		{
			e.printStackTrace();
		}
		catch (TransformerException e)
		{
			e.printStackTrace();
		}
	}

	public void setURL(URL url)
	{
		if(!primaryURLString.equals(url.toExternalForm()))
		{
			addObsoleteURL(primaryURLString);
			primaryURL = url;
			primaryURLString = url.toExternalForm();
		}
	}

	public void addObsoleteURL(String url)
	{
		if(!obsoleteURLs.contains(url))
			obsoleteURLs.add(url);
	}

	public Object[] getChangedItems(IResourceChangeEvent event)
	{
		List list = new ArrayList();
		getModifiedThread(event.getDelta(), list);
		if(0 < list.size())
			return list.toArray();
		else
			return null;
	}

	private void getModifiedThread(IResourceDelta delta, List list)
	{
		if(delta != null)
		{
			IResourceDelta[] affectedChildren = delta.getAffectedChildren(IResourceDelta.ADDED | IResourceDelta.REMOVED | IResourceDelta.CHANGED);
			for (int i = 0; i < affectedChildren.length; i++)
			{
				IResource res = affectedChildren[i].getResource();
				if(cacheMap.containsKey(res.getName()))
					list.add(cacheMap.get(res.getName()));
				getModifiedThread(affectedChildren[i], list);
			}
		}
	}
	
	public void createLogFolder(IWorkbenchWindow workbenchWindow, IProgressMonitor monitor)
	{
		MonalipsePlugin.syncExec(workbenchWindow, new WorkspaceModifyOperation()
			{
				protected void execute(IProgressMonitor monitor) throws InvocationTargetException
				{
					try
					{
						if(!getLogFolder().exists())
							getLogFolder().create(true, true, monitor);
					}
					catch (CoreException e)
					{
						throw new InvocationTargetException(e);
					}
				}
			});
	}

	private void saveCachedSubject(final IWorkbenchWindow workbenchWindow, final String lastModified) throws IOException
	{
		ByteArrayOutputStream bout = new ByteArrayOutputStream();
		DataOutputStream dout = new DataOutputStream(bout);
		SubjectCacheFile.writeCache(dout, threadList);
		dout.close();
		final byte[] bytes = bout.toByteArray();
		MonalipsePlugin.asyncExec(workbenchWindow, new WorkspaceModifyOperation()
			{
				protected void execute(IProgressMonitor monitor) throws InvocationTargetException
				{
					try
					{
						server.createServerFolder(workbenchWindow, monitor);
						createLogFolder(workbenchWindow, monitor);
						IFile cache = logFolder.getFile(".subject");
						MonalipsePlugin.ensureSynchronized(cache);
						if(cache.exists())
							cache.setContents(new ByteArrayInputStream(bytes), false, false, monitor);
						else
							cache.create(new ByteArrayInputStream(bytes), false, monitor);
						if(lastModified != null)
							getLogFolder().setPersistentProperty(new QualifiedName(Board.class.getName(), "Last-Modified"), lastModified);
					}
					catch (CoreException e)
					{
						throw new InvocationTargetException(e);
					}
				}
			});
	}

	private synchronized void ensureCacheLoaded()
	{
		if(threadList == null)
		{
			threadList = new ArrayList();
			cacheMap = new HashMap();
			IFile subject = logFolder.getFile(".subject");
			try
			{
				MonalipsePlugin.ensureSynchronized(subject);
				if(subject.exists())
				{
					SubjectCacheFile subj = null;
					try
					{
						subj = SubjectCacheFile.of(new DataInputStream(subject.getContents()));
						if(subj != null)
						{
							URL baseURL = getURL();
							for(int i = 0; i < subj.count; i++)
							{
								ThreadContentProvider thread = subj.readSubject(this, i);
								threadList.add(thread);
								cacheMap.put(thread.getID(), thread);
							}
						}
					}
					finally
					{
						if(subj != null)
							subj.close();
					}
				}
			}
			catch (MalformedURLException e)
			{
				e.printStackTrace();
			}
			catch (CoreException e)
			{
				e.printStackTrace();
			}
			catch (IOException e)
			{
				e.printStackTrace();
			}
		}
	}
	
	private boolean isVisible(String title, String constraint)
	{
		if(constraint == null || constraint.length() == 0)
			return true;
		return toRegularForm(title).indexOf(constraint) != -1;
	}
	
	private String toRegularForm(String str)
	{
		StringBuffer buf = new StringBuffer();
		for(int i = 0; i < str.length(); i++)
		{
			char ch = str.charAt(i);
			if('\uff41' <= ch && ch <= '\uff5a')
				ch = (char)('a' + (ch - '\uff41'));
			else if('\uff21' <= ch && ch <= '\uff3a')
				ch = (char)('a' + (ch - '\uff21'));
			else if('A' <= ch && ch <= 'Z')
				ch = (char)('a' + (ch - 'A'));
			else if('\uff10' <= ch && ch <= '\uff19')
				ch = (char)('0' + (ch - '\uff10'));
			buf.append((char)ch);
		}
		return buf.toString();
	}
	
	private static class SubjectCacheFile
	{
		private static final int SUBJECT_CACHE_VERSION = 0x01;
		private DataInputStream din;
		private int count;
		
		public static SubjectCacheFile of(DataInputStream din) throws IOException
		{
			SubjectCacheFile res = null;
			try
			{
				int version = din.readInt();
				if(version == SUBJECT_CACHE_VERSION)
					res = new SubjectCacheFile(din);
			}
			finally
			{
				if(res == null)
					din.close();
			}
			return res;
		}

		private SubjectCacheFile(DataInputStream din) throws IOException
		{
			this.din = din;
			count = din.readInt();
		}
		
		public ThreadContentProvider readSubject(Board board, int index) throws IOException
		{
			String id = din.readUTF();
			String name = din.readUTF();
			int responseCount = din.readInt();
			return new ThreadContentProvider(board, board.getURL(), id, index, name, responseCount);
		}
		
		public void close()
		{
			try
			{
				if(din != null)
					din.close();
				din = null;
			}
			catch (IOException e)
			{
			}
		}
		
		public static void writeCache(DataOutputStream dout, List list) throws IOException
		{
			dout.writeInt(SUBJECT_CACHE_VERSION);
			dout.writeInt(list.size());
			for(int i = 0; i < list.size(); i++)
			{
				ThreadContentProvider thread = (ThreadContentProvider)list.get(i);
				dout.writeUTF(thread.getID());
				dout.writeUTF(thread.getName());
				dout.writeInt(thread.getResponseCountHint());
			}
		}
	}

}
