package monalipse.server.giko;

import java.io.BufferedReader;
import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import java.io.DataInputStream;
import java.io.DataOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.io.OutputStream;
import java.lang.reflect.InvocationTargetException;
import java.net.MalformedURLException;
import java.net.URL;
import java.net.URLConnection;
import java.util.ArrayList;
import java.util.List;
import java.util.logging.Logger;
import java.util.zip.GZIPInputStream;

import javax.xml.transform.TransformerException;

import monalipse.MonalipsePlugin;
import monalipse.server.AbstractBBSServer;
import monalipse.server.IBBSBoard;
import monalipse.server.IBBSServer;
import monalipse.server.IBoardTreeNode;
import org.apache.commons.httpclient.Header;
import org.apache.commons.httpclient.HttpClient;
import org.apache.commons.httpclient.HttpMethod;
import org.apache.commons.httpclient.methods.GetMethod;
import org.apache.commons.httpclient.methods.PostMethod;
import org.eclipse.core.resources.IFile;
import org.eclipse.core.resources.IFolder;
import org.eclipse.core.resources.IResource;
import org.eclipse.core.resources.IResourceChangeEvent;
import org.eclipse.core.resources.IResourceDelta;
import org.eclipse.core.runtime.CoreException;
import org.eclipse.core.runtime.IAdaptable;
import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.core.runtime.Platform;
import org.eclipse.core.runtime.PluginVersionIdentifier;
import org.eclipse.jface.preference.IPreferenceStore;
import org.eclipse.jface.util.IPropertyChangeListener;
import org.eclipse.jface.util.PropertyChangeEvent;
import org.eclipse.ui.IWorkbenchWindow;
import org.eclipse.ui.actions.WorkspaceModifyOperation;
import org.w3c.dom.Document;
import org.w3c.dom.Element;
import org.w3c.dom.NodeList;
import org.w3c.tidy.Tidy;

public class GikoServer extends AbstractBBSServer
{
	public static final String PREF_BBSMENU_URL = "bbsmenu.url";
	private static final String DEFAULT_BBSMENU_URL = "http://www.ff.iij4u.or.jp/~ch2/bbsmenu.html";
	public static final String PREF_OYSTER_ID = "oyster.id";
	private static final String DEFAULT_OYSTER_ID = "";
	public static final String PREF_OYSTER_PASSWORD = "oyster.password";
	private static final String DEFAULT_OYSTER_PASSWORD = "";

	private static final long LOGIN_INTERVAL = 60 * 60 * 1000;

	private static final String USER_AGENT;
	private static String userAgentPrefix = "Monazilla/1.00";
	
	private static HttpClient httpClient = new HttpClient();

	private static String sessionID;
	private static long loginTime;
	
	private static final Logger logger = MonalipsePlugin.getLogger();

	static
	{
		PluginVersionIdentifier versionID = Platform.getPluginRegistry().getPluginDescriptor(MonalipsePlugin.PLUGIN_ID).getVersionIdentifier();
		USER_AGENT = "monalipse/" + versionID;

		IPreferenceStore store = MonalipsePlugin.getDefault().getPreferenceStore();
		store.setDefault(PREF_BBSMENU_URL, DEFAULT_BBSMENU_URL);
		store.setDefault(PREF_OYSTER_ID, DEFAULT_OYSTER_ID);
		store.setDefault(PREF_OYSTER_PASSWORD, DEFAULT_OYSTER_PASSWORD);
		
		MonalipsePlugin.getDefault().getPreferenceStore().addPropertyChangeListener(new IPropertyChangeListener()
			{
				public void propertyChange(PropertyChangeEvent event)
				{
					if(event.getProperty().equals(PREF_OYSTER_ID) ||
						event.getProperty().equals(PREF_OYSTER_PASSWORD))
					{
						loginTime = 0;
					}
				}
			});
	}
	
	public static HttpClient createHttpClient()
	{
		HttpClient client = new HttpClient();
		client.setTimeout(10000);
		return client;
	}
	
	public static GetMethod createGetMethod()
	{
		return (GetMethod)setMethodOptions(new GetMethod()
			{
				public String getRequestCharSet()
				{
					return "Windows-31J";
				}
				
				public String getResponseCharSet()
				{
					return "Windows-31J";
				}
			});
	}

	public static PostMethod createPostMethod()
	{
		return (PostMethod)setMethodOptions(new PostMethod()
			{
				public String getRequestCharSet()
				{
					return "Windows-31J";
				}
				
				public String getResponseCharSet()
				{
					return "Windows-31J";
				}
			});
	}
	
	private static HttpMethod setMethodOptions(HttpMethod method)
	{
		method.setFollowRedirects(false);
		method.setRequestHeader("User-Agent", userAgentPrefix + " (" + USER_AGENT + ")");
		return method;
	}
	
	public static InputStream getResponseBodyAsStream(HttpMethod get) throws IOException
	{
		InputStream in = get.getResponseBodyAsStream();
		Header h = get.getResponseHeader("Content-Encoding");
		if(h != null && h.getValue().equals("gzip"))
			in = new GZIPInputStream(in);
		return in;
	}

	public synchronized static String getOysterSessionID(IProgressMonitor monitor)
	{
		monitor.beginTask("Oyster", 10);
		IPreferenceStore store = MonalipsePlugin.getDefault().getPreferenceStore();
		String id = store.getString(GikoServer.PREF_OYSTER_ID);
		String password = store.getString(GikoServer.PREF_OYSTER_PASSWORD);
		try
		{
			if(id.length() != 0 && password.length() != 0)
			{
				if(System.currentTimeMillis() - loginTime < LOGIN_INTERVAL)
					return sessionID;
				
				try
				{
					monitor.subTask("Loggin in...");

					String body = "ID=" + id + "&PW=" + password;		
					URLConnection conn = new URL("https://tiger2.he.net/~tora3n2c/futen.cgi").openConnection();
					conn.setRequestProperty("User-Agent", "DOLIB/1.00");
					conn.setRequestProperty("X-2ch-UA", USER_AGENT);
					conn.setRequestProperty("Content-Length", String.valueOf(body.length()));
					conn.setDoInput(true);
					conn.setDoOutput(true);
					OutputStream out = conn.getOutputStream();
					out.write(body.getBytes());
					
					monitor.worked(5);
	
					BufferedReader r = new BufferedReader(new InputStreamReader(conn.getInputStream()));
					try
					{
						String res = r.readLine();
						res = res.substring("SESSION-ID=".length(), res.length());
						String ua = res.substring(0, res.indexOf(':'));
						
						if(!ua.equals("ERROR"))
						{
							monitor.worked(5);
							userAgentPrefix = ua;
							sessionID = res;
							loginTime = System.currentTimeMillis();
							return sessionID;
						}
					}
					finally
					{
						r.close();
					}
				}
				catch (MalformedURLException e)
				{
				}
				catch (IOException e)
				{
				}
			}
		}
		finally
		{
			monitor.done();
		}
		return null;
	}

	private Category boardTree;
	private int nextLogNumber;

	public void initialize(IFolder folder)
	{
		super.initialize(folder);
		IPreferenceStore pref = MonalipsePlugin.getDefault().getPreferenceStore();
		pref.setDefault(PREF_BBSMENU_URL, DEFAULT_BBSMENU_URL);
		pref.setDefault(PREF_OYSTER_ID, DEFAULT_OYSTER_ID);
		pref.setDefault(PREF_OYSTER_PASSWORD, DEFAULT_OYSTER_PASSWORD);
	}
	
	public boolean boardListChanged(IResourceChangeEvent event)
	{
		return MonalipsePlugin.resourceModified(IResourceDelta.ADDED | IResourceDelta.REMOVED | IResourceDelta.CHANGED, event.getDelta(), getServerFolder().getFile(".bbsmenu"));
	}

	public IBoardTreeNode getBoardTree()
	{
		ensureBoardListLoaded();
		return boardTree;
	}

	private synchronized void ensureBoardListLoaded()
	{
		if(boardTree == null)
		{
			boardTree = new Category(this, null, "<root>");
			IFile bbsmenu = getServerFolder().getFile(".bbsmenu");
			try
			{
				MonalipsePlugin.ensureSynchronized(bbsmenu);
				if(bbsmenu.exists())
				{
					BoardListCacheFile cache = null;
					try
					{
						cache = BoardListCacheFile.of(new DataInputStream(bbsmenu.getContents()));
						if(cache != null)
						{
							cache.read(this, boardTree);
							nextLogNumber = cache.nextLogNumber;
						}
					}
					finally
					{
						if(cache != null)
							cache.close();
					}
				}
			}
			catch (MalformedURLException e)
			{
				e.printStackTrace();
			}
			catch (CoreException e)
			{
				e.printStackTrace();
			}
			catch (IOException e)
			{
				e.printStackTrace();
			}
		}
	}

	public synchronized void updateBoardList(IWorkbenchWindow workbenchWindow)
	{
		HttpClient hc = createHttpClient();
		try
		{
			URL url = new URL(MonalipsePlugin.getDefault().getPreferenceStore().getString(GikoServer.PREF_BBSMENU_URL));

			GetMethod get = createGetMethod();
			get.setPath(url.getFile());

			hc.startSession(url);
			hc.executeMethod(get);
			MonalipsePlugin.log(logger, get);
			
			if(get.getStatusLine().getStatusCode() == 200)
			{
				Category root = new Category(this, null, "<root>");
				Tidy tidy = new Tidy();
				tidy.setCharEncoding(org.w3c.tidy.Configuration.UTF8);
				tidy.setQuiet(true);
				tidy.setShowWarnings(true);
				Document document = tidy.parseDOM(new ByteArrayInputStream(get.getResponseBodyAsString().getBytes("UTF-8")), null);
				parseBBSMenu(document.getDocumentElement(), root, null);
				boardTree = root;
				saveCachedMenu(workbenchWindow);
			}
		}
		catch (TransformerException e)
		{
			e.printStackTrace();
		}
		catch (MalformedURLException e)
		{
			e.printStackTrace();
		}
		catch (IOException e)
		{
			e.printStackTrace();
		}
		finally
		{
			try
			{
				hc.endSession();
			}
			catch (IOException e)
			{
			}
		}
	}

	public String getTargetName()
	{
		return GikoServer.class.getName();
	}
	
	public IBBSBoard getBoardOf(URL url)
	{
		ensureBoardListLoaded();
		Board board = getBoardOf(url, boardTree);
		if(board != null)
			return board;
		
		ThreadReference ref = ThreadReference.of(url);
		if(ref != null)
		{
			String dir = ref.getBoardURL().getFile();
			IBoardTreeNode[] categories = boardTree.getChildren();
			for(int i = 0; i < categories.length; i++)
			{
				IBoardTreeNode[] boards = categories[i].getChildren();
				for(int j = 0; j < boards.length; j++)
				{
					board = (Board)boards[j];
					if(board.getURL().toExternalForm().indexOf(dir) != -1)
					{
						board.addObsoleteURL(ref.getBoardURL().toExternalForm());
						return board;
					}
				}
			}
		}
		
		return null;
	}
	
	private static Board getBoardOf(URL url, IBoardTreeNode root)
	{
		ThreadReference ref = ThreadReference.of(url);
		if(ref != null)
		{
			String boardURL = ref.getBoardURL().toExternalForm();
			IBoardTreeNode[] categories = root.getChildren();
			for(int i = 0; i < categories.length; i++)
			{
				IBoardTreeNode[] boards = categories[i].getChildren();
				for(int j = 0; j < boards.length; j++)
				{
					Board board = (Board)boards[j];
					if(board.matches(boardURL))
						return board;
				}
			}
		}
		return null;
	}
	
	public URL getURLOf(IFile file)
	{
		IResource log = file.getParent();
		IResource srv = log.getParent();
		while(srv != null && !srv.equals(getServerFolder()))
		{
			log = srv;
			srv = srv.getParent();
		}
		
		if(srv != null)
		{
			String dir = log.getName();
			IBoardTreeNode[] categories = boardTree.getChildren();
			for(int i = 0; i < categories.length; i++)
			{
				IBoardTreeNode[] boards = categories[i].getChildren();
				for(int j = 0; j < boards.length; j++)
				{
					Board board = (Board)boards[j];
					if(board.getLogFolder().getName().equals(dir))
						return board.getURLOf(file);
				}
			}
		}

		return null;
	}

	public void createServerFolder(IWorkbenchWindow workbenchWindow, IProgressMonitor monitor)
	{
		MonalipsePlugin.syncExec(workbenchWindow, new WorkspaceModifyOperation()
			{
				protected void execute(IProgressMonitor monitor) throws InvocationTargetException
				{
					try
					{
						if(!getServerFolder().exists())
							getServerFolder().create(true, true, monitor);
					}
					catch (CoreException e)
					{
						throw new InvocationTargetException(e);
					}
				}
			});
	}

	public void saveCachedMenu(final IWorkbenchWindow workbenchWindow) throws TransformerException, IOException
	{
		ByteArrayOutputStream bout = new ByteArrayOutputStream();
		DataOutputStream dout = new DataOutputStream(bout);
		BoardListCacheFile.writeCache(dout, boardTree, nextLogNumber);
		dout.close();
		final byte[] bytes = bout.toByteArray();
		MonalipsePlugin.asyncExec(workbenchWindow, new WorkspaceModifyOperation()
			{
				protected void execute(IProgressMonitor monitor) throws InvocationTargetException
				{
					try
					{
						createServerFolder(workbenchWindow, monitor);
						IFile cache = getServerFolder().getFile(".bbsmenu");
						MonalipsePlugin.ensureSynchronized(cache);
						if(cache.exists())
							cache.setContents(new ByteArrayInputStream(bytes), false, false, monitor);
						else
							cache.create(new ByteArrayInputStream(bytes), false, monitor);
					}
					catch (CoreException e)
					{
						throw new InvocationTargetException(e);
					}
				}
			});
	}

	private void parseBBSMenu(Element element, Category root, Category category)
	{
		NodeList children = element.getChildNodes();
		for(int i = 0; i < children.getLength(); i++)
		{
			if(children.item(i) instanceof Element)
			{
				Element child = (Element)children.item(i);
				if(child.getTagName().equals("a") && category != null)
				{
					try
					{
						String url = child.getAttribute("href");
						String name = child.getFirstChild().getNodeValue();
						URL href = new URL(url);
						Board board = getBoardOf(href, root);
						
						if(board == null)
						{
							Board exists = getBoardOf(href, boardTree);
							if(exists != null)
								board = new Board(this, category, exists.getLogFolder(), exists.getName(), exists.getURL(), exists.getObsoleteURLs());
						}
						
						if(board == null)
						{
							Category cat = (Category)boardTree.getChild(category.getName());
							if(cat != null)
							{
								board = (Board)cat.getChild(name);
								if(board != null)
									board.setURL(href);
							}
						}
						
						if(board == null)
						{
							ThreadReference ref = ThreadReference.of(href);
							if(ref != null && ref.getReferenceType() == ThreadReference.REFERENCE_BOARD)
							{
								board = new Board(this, category, getServerFolder().getFolder("log" + nextLogNumber), name, href, new String[0]);
								nextLogNumber++;
							}
						}

						if(board != null)
						{
							if(!category.hasChildren())
								root.addChild(category);
							category.addChild(board);
						}
					}
					catch(MalformedURLException e)
					{
					}
				}
				else if(child.getTagName().equals("b"))
				{
					String s = child.getFirstChild().getNodeValue();
					if(0 < s.length())
						category = new Category(this, root, s);
				}
				else
				{
					parseBBSMenu(child, root, category);
				}
			}
		}
	}

	// int version
	// int nextLogNumber
	// int <categoryCount>
	// {
	//   UTF name
	//   int <boardCount>
	//   {
	//     UTF logFolder
	//     UTF name
	//     UTF primaryURL
	//     int <obsoleteURLCount>
	//     {
	//       UTF obsoleteURL
	//     }*
	//   }*
	// }*
	private static class BoardListCacheFile
	{
		private static final int BOARD_LIST_CACHE_VERSION = 0x03;
		private DataInputStream din;
		public int nextLogNumber;
		
		public static BoardListCacheFile of(DataInputStream din) throws IOException
		{
			BoardListCacheFile res = null;
			try
			{
				int version = din.readInt();
				if(version == BOARD_LIST_CACHE_VERSION)
					res = new BoardListCacheFile(din);
			}
			finally
			{
				if(res == null)
					din.close();
			}
			return res;
		}

		private BoardListCacheFile(DataInputStream din) throws IOException
		{
			this.din = din;
			nextLogNumber = din.readInt();
		}
		
		public void read(GikoServer server, Category root) throws IOException
		{
			int categoryCount = din.readInt();
			for(int i = 0; i < categoryCount; i++)
			{
				Category cat = new Category(server, root, din.readUTF());
				root.addChild(cat);
				int boardCount = din.readInt();
				for(int j = 0; j < boardCount; j++)
				{
					String logFolder = din.readUTF();
					String name = din.readUTF();
					URL url = new URL(din.readUTF());
					String[] obs = new String[din.readInt()];
					for(int k = 0; k < obs.length; k++)
						obs[k] = din.readUTF();
					cat.addChild(new Board(server, cat, server.getServerFolder().getFolder(logFolder), name, url, obs));
				}
			}
		}
		
		public void close()
		{
			try
			{
				if(din != null)
					din.close();
				din = null;
			}
			catch (IOException e)
			{
			}
		}
		
		public static void writeCache(DataOutputStream dout, Category root, int nextLogNumber) throws IOException
		{
			dout.writeInt(BOARD_LIST_CACHE_VERSION);
			dout.writeInt(nextLogNumber);
			IBoardTreeNode[] categories = root.getChildren();
			dout.writeInt(categories.length);
			for(int i = 0; i < categories.length; i++)
			{
				IBoardTreeNode category = categories[i];
				dout.writeUTF(category.getName());
				IBoardTreeNode[] boards = category.getChildren();
				dout.writeInt(boards.length);
				for(int j = 0; j < boards.length; j++)
				{
					Board board = (Board)boards[j];
					dout.writeUTF(board.getLogFolder().getName());
					dout.writeUTF(board.getName());
					dout.writeUTF(board.getURL().toExternalForm());
					String[] obs = board.getObsoleteURLs();
					dout.writeInt(obs.length);
					for(int k = 0; k < obs.length; k++)
						dout.writeUTF(obs[k]);
				}
			}
		}
	}

	private static class Category implements IAdaptable, IBoardTreeNode
	{
		private GikoServer server;
		private IBoardTreeNode parent;
		private String name;
		private List children = new ArrayList();
		private IBoardTreeNode[] childrenArray;
	
		public Category(GikoServer server, IBoardTreeNode parent, String name)
		{
			this.parent = parent;
			this.name = name;
		}
		
		public boolean equals(Object obj)
		{
			if(obj instanceof Category)
				return ((Category)obj).getName().equals(getName());
			else
				return false;
		}
		
		public int hashCode()
		{
			return getName().hashCode();
		}

		public IBBSServer getServer()
		{
			return server;
		}
		
		public String getName()
		{
			return name;
		}
	
		public boolean hasChildren()
		{
			return !children.isEmpty();
		}
		
		public IBoardTreeNode[] getChildren()
		{
			if(childrenArray == null)
			{
				childrenArray = new IBoardTreeNode[children.size()];
				children.toArray(childrenArray);
			}
			return childrenArray;
		}
	
		public IBoardTreeNode getParent()
		{
			return parent;
		}
		
		public Object getAdapter(Class adapter)
		{
			return null;
		}
		
		public void addChild(IBoardTreeNode child)
		{
			children.add(child);
			childrenArray = null;
		}
		
		public IBoardTreeNode getChild(String name)
		{
			for(int i = 0; i < children.size(); i++)
			{
				IBoardTreeNode e = (IBoardTreeNode)children.get(i);
				if(e.getName().equals(name))
					return e;
			}
			return null;
		}
	}
}
