package monalipse.views;

import java.net.ConnectException;
import java.net.MalformedURLException;
import java.net.SocketTimeoutException;
import java.net.URL;
import java.net.UnknownHostException;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.logging.Logger;

import monalipse.MonalipsePlugin;
import monalipse.editors.ThreadViewerEditor;
import monalipse.server.BBSServerManager;
import monalipse.server.IBBSReference;
import monalipse.server.IThreadContentProvider;
import monalipse.server.LinkedObject;
import monalipse.utils.CancelableRunner;
import org.apache.commons.collections.SequencedHashMap;
import org.apache.commons.threadpool.DefaultThreadPool;
import org.apache.commons.threadpool.ThreadPool;
import org.eclipse.core.resources.IFile;
import org.eclipse.core.resources.IResourceChangeEvent;
import org.eclipse.core.resources.IResourceChangeListener;
import org.eclipse.core.resources.ResourcesPlugin;
import org.eclipse.core.runtime.CoreException;
import org.eclipse.core.runtime.Platform;
import org.eclipse.jface.action.Action;
import org.eclipse.jface.action.IAction;
import org.eclipse.jface.action.IMenuListener;
import org.eclipse.jface.action.IMenuManager;
import org.eclipse.jface.action.IStatusLineManager;
import org.eclipse.jface.action.IToolBarManager;
import org.eclipse.jface.action.MenuManager;
import org.eclipse.jface.action.Separator;
import org.eclipse.jface.resource.ImageDescriptor;
import org.eclipse.jface.viewers.DoubleClickEvent;
import org.eclipse.jface.viewers.IDoubleClickListener;
import org.eclipse.jface.viewers.ISelection;
import org.eclipse.jface.viewers.IStructuredContentProvider;
import org.eclipse.jface.viewers.IStructuredSelection;
import org.eclipse.jface.viewers.ITableLabelProvider;
import org.eclipse.jface.viewers.LabelProvider;
import org.eclipse.jface.viewers.TableViewer;
import org.eclipse.jface.viewers.Viewer;
import org.eclipse.jface.viewers.ViewerSorter;
import org.eclipse.swt.SWT;
import org.eclipse.swt.custom.TreeEditor;
import org.eclipse.swt.dnd.DND;
import org.eclipse.swt.dnd.DragSource;
import org.eclipse.swt.dnd.DragSourceEvent;
import org.eclipse.swt.dnd.DragSourceListener;
import org.eclipse.swt.dnd.FileTransfer;
import org.eclipse.swt.dnd.Transfer;
import org.eclipse.swt.events.KeyAdapter;
import org.eclipse.swt.events.KeyEvent;
import org.eclipse.swt.events.PaintEvent;
import org.eclipse.swt.events.PaintListener;
import org.eclipse.swt.events.SelectionEvent;
import org.eclipse.swt.events.SelectionListener;
import org.eclipse.swt.graphics.Device;
import org.eclipse.swt.graphics.Font;
import org.eclipse.swt.graphics.FontData;
import org.eclipse.swt.graphics.Image;
import org.eclipse.swt.program.Program;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Menu;
import org.eclipse.swt.widgets.Table;
import org.eclipse.swt.widgets.TableColumn;
import org.eclipse.ui.IActionBars;
import org.eclipse.ui.IPartListener;
import org.eclipse.ui.ISelectionListener;
import org.eclipse.ui.IWorkbenchPart;
import org.eclipse.ui.part.ViewPart;

public class LinkedImagesView extends ViewPart implements SelectionListener, ISelectionListener, IResourceChangeListener, IPartListener
{
	public static final String ID_LINKED_IMAGES = LinkedImagesView.class.getName();

	private static final Logger logger = MonalipsePlugin.getLogger();

	private TableViewer viewer;
	private IAction downloadAction;
	private IAction abortAction;
	private IAction doubleClickAction;
	private LinkedImagesContentProvider provider;
	private IThreadContentProvider thread;
	private CancelableRunner cancelable;
	private TreeEditor treeEditor;
	private boolean loading;
	private boolean active;
	private ArrayList loadQueue = new ArrayList();
	private ThreadPool downloadThreadPool = new DefaultThreadPool(4);
	private ThreadViewerEditor editor;
	private int textOffset;

	public LinkedImagesView()
	{
//logger.finest("enter");
	}

	public void createPartControl(Composite parent)
	{
//logger.finest("enter");
		logger.finest("begin");
		viewer = new TableViewer(parent, SWT.MULTI | SWT.H_SCROLL | SWT.V_SCROLL | SWT.FULL_SELECTION);
		provider = new LinkedImagesContentProvider();
		viewer.setContentProvider(provider);
		viewer.setLabelProvider(provider);
//		viewer.setSorter(new NameSorter());
		viewer.setInput(this);
		FontData font = viewer.getTable().getFont().getFontData()[0];
		viewer.getTable().setFont(new Font(getSite().getShell().getDisplay(), font.getName(), 47, font.getStyle()));
		makeColumns();
		makeActions();
		hookContextMenu();
		hookDoubleClickAction();
		contributeToActionBars();
		getSite().setSelectionProvider(viewer);
		viewer.getTable().addSelectionListener(this);
		ResourcesPlugin.getWorkspace().addResourceChangeListener(this);
		getViewSite().getPage().addSelectionListener(this);
		getSite().getWorkbenchWindow().getPartService().addPartListener(this);
		hookDragAndDrop();
		
		viewer.getTable().addPaintListener(new PaintListener()
			{
				public void paintControl(PaintEvent e)
				{
					if(viewer.getInput() instanceof LinkedObject[])
					{
						int itemHeight = viewer.getTable().getItemHeight();
						int ox = 0;
						if(viewer.getTable().getHorizontalBar() != null)
							ox = viewer.getTable().getHorizontalBar().getSelection();
						int oy = 0;
						if(viewer.getTable().getVerticalBar() != null)
							oy = viewer.getTable().getVerticalBar().getSelection() * itemHeight;
						e.gc.setFont(getSite().getShell().getDisplay().getSystemFont());
						LinkedObject[] nodes = (LinkedObject[])viewer.getInput();
						int yoff = (itemHeight - e.gc.textExtent("").y) / 2;
						drawNodes(e, nodes, ox, oy, 0, yoff, -1);
					}
				}
				
				private int drawNodes(PaintEvent e, LinkedObject[] nodes, int ox, int oy, int y, int yoff, int responseNumber)
				{
					for(int i = 0; i < nodes.length; i++)
					{
						int itemHeight = viewer.getTable().getItemHeight();

						if(responseNumber != nodes[i].getResponseNumber())
						{
							responseNumber = nodes[i].getResponseNumber();
							e.gc.drawString(">>" + responseNumber, 1 - ox, 1 + y - oy, false);
						}

						e.gc.drawString(nodes[i].getName(), textOffset + 2 - ox, y + yoff - oy, true);
						y += itemHeight;
					}
					return y;
				}
			});

		viewer.getTable().addKeyListener(new KeyAdapter()
			{
				public void keyPressed(KeyEvent e)
				{
					if(e.keyCode == SWT.F5)
						downloadAction.run();
				}
			});

		loading = true;
		Thread imageLoader = new Thread(new Runnable()
			{
				public void run()
				{
					while(true)
					{
						final LinkedObject obj;

						synchronized(loadQueue)
						{
							while(loading && loadQueue.isEmpty())
							{
								loadQueue.trimToSize();

								try
								{
									loadQueue.wait();
								}
								catch (InterruptedException e)
								{
								}
							}
							
							if(loadQueue.isEmpty())
								return;
							
							obj = (LinkedObject)loadQueue.remove(0);
						}
						
						try
						{
							if(obj.loadImage())
							{
								getSite().getShell().getDisplay().asyncExec(new Runnable()
									{
										public void run()
										{
											update(obj);
											packColumns();
										}
									});
							}
						}
						catch (RuntimeException e)
						{
							e.printStackTrace();
						}
					}
				}
			});
		imageLoader.setPriority(Thread.MIN_PRIORITY);
		imageLoader.start();
		logger.finest("done");
	}
	
	public void partActivated(IWorkbenchPart part)
	{
//logger.finest("enter");
		update(part);
	}

	public void partBroughtToTop(IWorkbenchPart part)
	{
//logger.finest("enter");
		update(part);
	}

	public void partClosed(IWorkbenchPart part)
	{
		if(part == LinkedImagesView.this)
		{
			synchronized(loadQueue)
			{
				loading = false;
				loadQueue.notify();
			}
		}
	}

	public void partDeactivated(IWorkbenchPart part)
	{
		if(part == LinkedImagesView.this)
			active = false;
	}

	public void partOpened(IWorkbenchPart part)
	{
//logger.finest("enter");
		update(part);
	}
	
	private void update(IWorkbenchPart part)
	{
		if(part instanceof ThreadViewerEditor)
		{
			IThreadContentProvider cp = ((ThreadViewerEditor)part).getContentProvider();
			if(thread == null || cp == null || !thread.equals(cp))
			{
				LinkedObject[] linked = ((ThreadViewerEditor)part).getLinkedObjects();
				if(linked != null)
				{
					selectionChanged(linked);
					thread = cp;
					editor = (ThreadViewerEditor)part;
				}
			}
		}

		if(part == LinkedImagesView.this)
		{
			synchronized(loadQueue)
			{
				active = true;
				loadQueue.notify();
			}
		}
	}

	private void update(LinkedObject obj)
	{
		logger.finest("begin");
		viewer.update(obj, null);
		List list = Arrays.asList((LinkedObject[])viewer.getInput());
		int i = list.indexOf(obj);
		if(i != -1)
		{
			int oy = 0;
			if(viewer.getTable().getVerticalBar() != null)
				oy = viewer.getTable().getVerticalBar().getSelection();
			int itemHeight = viewer.getTable().getItemHeight();
			viewer.getTable().redraw(0, (i - oy) * itemHeight, viewer.getTable().getSize().x, itemHeight, true);
		}
		logger.finest("done");
	}
	
	public void dispose()
	{
//logger.finest("enter");
		ResourcesPlugin.getWorkspace().removeResourceChangeListener(this);
		getViewSite().getPage().removeSelectionListener(this);
		getSite().getWorkbenchWindow().getPartService().removePartListener(this);
		super.dispose();
	}
	
	public void resourceChanged(IResourceChangeEvent event)
	{
//logger.finest("enter");
		logger.finest("begin");
		if(thread != null)
		{
			if(thread.threadChanged(event) && editor != null)
			{
				int scroll = 0;
				if(viewer.getTable().getVerticalBar() != null)
					scroll = viewer.getTable().getVerticalBar().getSelection();
				ISelection sel = viewer.getSelection();
				selectionChanged(editor.getLinkedObjects());
				viewer.setSelection(sel);
				if(viewer.getTable().getVerticalBar() != null)
					viewer.getTable().getVerticalBar().setSelection(scroll);
				textOffset = 0;
				packColumns();
			}
			else
			{
				LinkedObject[] nodes = LinkedObject.getChangedNodes((LinkedObject[])viewer.getInput(), event);
				if(nodes != null)
				{
					synchronized(loadQueue)
					{
						loadQueue.addAll(0, Arrays.asList(nodes));
						loadQueue.notify();
					}
				}
			}
		}
		logger.finest("done");
	}
	
	public void selectionChanged(IWorkbenchPart part, ISelection selection)
	{
//logger.finest("enter");
		logger.finest("begin");
		if(!viewer.getTable().isDisposed())
		{
			if(selection instanceof IBBSReference && ((IBBSReference)selection).getResponseNumber() != -1)
			{
				int itemHeight = viewer.getTable().getItemHeight();
				if(LinkedObject.IMAGE_SIZE < itemHeight)
				{
					IThreadContentProvider thread = BBSServerManager.getThreadContentProviderOf(((IBBSReference)selection).getURL());
					if(thread != null && this.thread != null && this.thread.equals(thread))
					{
						int sel = ((IBBSReference)selection).getResponseNumber();
						Object[] objs = provider.getElements(viewer.getInput());
						if(0 < objs.length)
						{
							int row = objs.length - 1;
							for(int i = 0; i < objs.length; i++)
							{
								if(sel <= ((LinkedObject)objs[i]).getResponseNumber())
								{
									row = i;
									break;
								}
							}
							if(0 <= row && row < objs.length)
								viewer.reveal(objs[row]);
						}
					}
				}
			}
//			if(selection instanceof IStructuredSelection)
//			{
//				Object first = ((IStructuredSelection)selection).getFirstElement();
//				if(first instanceof IBBSReference)
//				{
//					IThreadContentProvider thread = BBSServerManager.getThreadContentProviderOf(((IBBSReference)first).getURL());
//					if(thread != null)
//					{
//						if(this.thread == null || !this.thread.equals(thread))
//						{
//							selectionChanged(thread.getLinkedObjects());
//							this.thread = thread;
//							editor = null;
//						}
//					}
//				}
//			}
		}
		logger.finest("done");
	}
	
	private void selectionChanged(LinkedObject[] linked)
	{
//logger.finest("enter");
		logger.finest("begin");
		viewer.setInput(linked);
		textOffset = 0;
		packColumns();
		logger.finest("done");
	}

	public void widgetSelected(SelectionEvent e)
	{
		logger.finest("begin");
		ISelection selection = viewer.getSelection();
		if(selection instanceof IStructuredSelection)
		{
			Object first = ((IStructuredSelection)selection).getFirstElement();
			if(first instanceof LinkedObject)
			{
				LinkedObject node = (LinkedObject)first;
				if(editor != null)
					editor.scrollTo(node.getResponseNumber());
			}
		}
		logger.finest("done");
	}

	public void widgetDefaultSelected(SelectionEvent e)
	{
	}

	private void hookContextMenu()
	{
		MenuManager menuMgr = new MenuManager("#PopupMenu");
		menuMgr.setRemoveAllWhenShown(true);
		menuMgr.addMenuListener(new IMenuListener()
			{
				public void menuAboutToShow(IMenuManager manager)
				{
					LinkedImagesView.this.fillContextMenu(manager);
				}
			});
		Menu menu = menuMgr.createContextMenu(viewer.getControl());
		viewer.getControl().setMenu(menu);
		getSite().registerContextMenu(menuMgr, viewer);
	}

	private void contributeToActionBars()
	{
		IActionBars bars = getViewSite().getActionBars();
//		fillLocalPullDown(bars.getMenuManager());
		fillLocalToolBar(bars.getToolBarManager());
		fillLocalStatusBar(bars.getStatusLineManager());
	}

	private void fillLocalStatusBar(IStatusLineManager manager)
	{
		cancelable = new CancelableRunner(manager, getSite().getShell().getDisplay(), abortAction);
	}

	private void fillLocalPullDown(IMenuManager manager)
	{
		manager.add(downloadAction);
		manager.add(abortAction);
	}

	private void fillContextMenu(IMenuManager manager)
	{
		manager.add(downloadAction);
		manager.add(abortAction);
		manager.add(new Separator("Additions"));
	}

	private void fillLocalToolBar(IToolBarManager manager)
	{
		manager.add(downloadAction);
		manager.add(abortAction);
//		manager.add(new Separator());
//		drillDownAdapter.addNavigationActions(manager);
	}
	
	private void packColumns()
	{
		if(textOffset != viewer.getTable().getItemHeight())
		{
			textOffset = viewer.getTable().getItemHeight();
			TableColumn[] columns = viewer.getTable().getColumns();
			columns[0].setWidth(textOffset);
			viewer.getTable().redraw();
		}
	}

	private void makeColumns()
	{
		Table table = viewer.getTable();
		table.setHeaderVisible(false);
		table.setLinesVisible(false);
		TableColumn col;
		col = new TableColumn(table, SWT.LEFT);
		col = new TableColumn(table, SWT.LEFT);
		col.setWidth(200);
	}

	private void makeActions()
	{
		String iconPath = "icons/"; //$NON-NLS-1$		
		URL installURL = Platform.getPlugin(MonalipsePlugin.PLUGIN_ID).getDescriptor().getInstallURL();

		downloadAction = new Action()
			{
				public void run()
				{
					cancelable.run(null, new CancelableRunner.ICancelableRunnableWithProgress()
						{
							public void run(final CancelableRunner.ICancelableProgressMonitor monitor)
							{
								if(viewer.getInput() instanceof LinkedObject[])
								{
									LinkedObject[] objects = (LinkedObject[])viewer.getInput();
									monitor.beginTask("", objects.length);
									monitor.subTask("downloading...");

									SequencedHashMap map = new SequencedHashMap();
									for(int i = 0; i < objects.length; i++)
									{
										if(objects[i].isImage())
											addWork(map, objects[i]);
									}
									for(int i = 0; i < objects.length; i++)
									{
										if(!objects[i].isImage())
											addWork(map, objects[i]);
									}
									
									final int[] remain = new int[]{objects.length};
									
									List lists = map.sequence();
									for(int i = 0; i < lists.size(); i++)
									{
										final List list = (List)map.get(lists.get(i));
										downloadThreadPool.invokeLater(new Runnable()
											{
												public void run()
												{
													try
													{
														for(int i = 0; i < list.size() && !monitor.isCanceled(); i++)
														{
															LinkedObject obj = (LinkedObject)list.get(i);
															fetchImage(obj, false);
															monitor.worked(1);
														}
													}
													finally
													{
														synchronized(remain)
														{
															remain[0] -= list.size();
															remain.notify();
														}
													}
												}
											});
									}
									
									synchronized(remain)
									{
										try
										{
											while(0 < remain[0] && !monitor.isCanceled())
												remain.wait();
										}
										catch (InterruptedException e)
										{
										}
									}

									monitor.done();
								}
							}
							
							public void addWork(Map map, LinkedObject node)
							{
								List list = (List)map.get(node.getURL().getHost());
								if(list == null)
								{
									list = new ArrayList();
									map.put(node.getURL().getHost(), list);
								}
								list.add(node);
							}
						});
				}
			};
		downloadAction.setText("Download Images");
		downloadAction.setToolTipText("Download Images");
		try
		{
			downloadAction.setImageDescriptor(ImageDescriptor.createFromURL(new URL(installURL, iconPath + "refresh_nav.gif"))); //$NON-NLS-1$
		}
		catch (MalformedURLException e)
		{
		}

		abortAction = new Action()
			{
				public void run()
				{
					cancelable.cancel();
				}
			};
		abortAction.setText("Abort");
		abortAction.setToolTipText("Abort downloading images");
		abortAction.setEnabled(false);
		try
		{
			abortAction.setImageDescriptor(ImageDescriptor.createFromURL(new URL(installURL, iconPath + "stop_nav.gif"))); //$NON-NLS-1$
		}
		catch (MalformedURLException e)
		{
		}

		doubleClickAction = new Action()
			{
				public void run()
				{
					ISelection selection = viewer.getSelection();
					if(selection instanceof IStructuredSelection)
					{
						Object first = ((IStructuredSelection)selection).getFirstElement();
						if(first instanceof LinkedObject)
						{
							final LinkedObject obj = (LinkedObject)first;
							IFile file = obj.getCachedFile();
							try
							{
								MonalipsePlugin.ensureSynchronized(file);
								if(file.exists())
								{
									Program.launch(file.getLocation().toOSString());
								}
								else
								{
									cancelable.run(null, new CancelableRunner.ICancelableRunnableWithProgress()
										{
											public void run(final CancelableRunner.ICancelableProgressMonitor monitor)
											{
												monitor.beginTask("", 2);
												monitor.subTask("downloading...");
												monitor.worked(1);
												fetchImage(obj, true);
												monitor.worked(1);
												monitor.done();
											}
										});
								}
							}
							catch (CoreException e)
							{
							}
						}
					}
				}
			};
	}
	
	private void fetchImage(final LinkedObject obj, boolean force)
	{
		boolean success = false;
		try
		{
			success = obj.fetchObject(getSite().getWorkbenchWindow(), force);
		}
		catch (SocketTimeoutException e)
		{
			return;
		}
		catch (ConnectException e)
		{
			return;
		}
		catch (UnknownHostException e)
		{
			return;
		}
		catch(Exception e)
		{
		}
		finally
		{
			if(!success)
			{
				if(obj.loadImage())
				{
					getSite().getShell().getDisplay().asyncExec(new Runnable()
						{
							public void run()
							{
								update(obj);
								packColumns();
							}
						});
				}
			}
		}
	}

	private void hookDoubleClickAction()
	{
		viewer.addDoubleClickListener(new IDoubleClickListener()
			{
				public void doubleClick(DoubleClickEvent event)
				{
					doubleClickAction.run();
				}
			});
	}
	
	private void hookDragAndDrop()
	{
		DragSource dragSource = new DragSource(viewer.getTable(), DND.DROP_COPY);
		dragSource.setTransfer(new Transfer[]{FileTransfer.getInstance()});
		dragSource.addDragListener(new DragSourceListener()
			{
				public void dragStart(DragSourceEvent event)
				{
					ISelection selection = viewer.getSelection();
					if(selection instanceof IStructuredSelection)
					{
						Object first = ((IStructuredSelection)selection).getFirstElement();
						if(first instanceof LinkedObject)
						{
							Object[] nodes = ((IStructuredSelection)selection).toArray();
							for(int i = 0; i < nodes.length; i++)
							{
								IFile file = ((LinkedObject)nodes[i]).getCachedFile();
								try
								{
									MonalipsePlugin.ensureSynchronized(file);
									if(file.exists())
									{
										event.doit = true;
										return;
									}
								}
								catch (CoreException e)
								{
								}
							}
						}
					}
					event.doit = false;
				}
	
				public void dragSetData(DragSourceEvent event)
				{
					ISelection selection = viewer.getSelection();
					if(selection instanceof IStructuredSelection)
					{
						Object first = ((IStructuredSelection)selection).getFirstElement();
						if(first instanceof LinkedObject)
						{
							Object[] nodes = ((IStructuredSelection)selection).toArray();
							List list = new ArrayList();
							for(int i = 0; i < nodes.length; i++)
							{
								IFile file = ((LinkedObject)nodes[i]).getCachedFile();
								try
								{
									MonalipsePlugin.ensureSynchronized(file);
									if(file.exists())
										list.add(file.getLocation().toOSString());
								}
								catch (CoreException e)
								{
								}
							}
							String[] files = new String[list.size()];
							list.toArray(files);
							event.data = files;
						}
					}
				}
	
				public void dragFinished(DragSourceEvent event)
				{
				}
			});
	}

	public void setFocus()
	{
		viewer.getControl().setFocus();
	}

	class NameSorter extends ViewerSorter
	{
	}

	private class LinkedImagesContentProvider extends LabelProvider implements IStructuredContentProvider, ITableLabelProvider
	{
		private Map imageMap = new HashMap();
		private Device device = getSite().getShell().getDisplay();

		public Object[] getElements(Object inputElement)
		{
//logger.finest("enter");
			if(inputElement instanceof LinkedObject[])
				return (LinkedObject[])inputElement;
			return new Object[0];
		}

		public Image getImage(Object element)
		{
//logger.finest("enter");
			if(element instanceof LinkedObject)
				return ((LinkedObject)element).getImage(device);
			return null;
		}

		public String getText(Object element)
		{
//logger.finest("enter");
			if(element instanceof LinkedObject)
				return ((LinkedObject)element).getName();
			return element.toString();
		}

		public void inputChanged(Viewer viewer, Object oldInput, Object newInput)
		{
//logger.finest("enter");
			int reuse = 0;
			if(oldInput instanceof LinkedObject[] && newInput instanceof LinkedObject[])
			{
				LinkedObject[] oi = (LinkedObject[])oldInput;
				LinkedObject[] ni = (LinkedObject[])newInput;
				for(int i = 0; i < oi.length && i < ni.length; i++)
				{
					if(oi[i].equals(ni[i]))
					{
						reuse++;
						ni[i] = oi[i];
					}
					else
					{
						break;
					}
				}
			}

			if(oldInput instanceof LinkedObject[])
			{
				LinkedObject[] objs = (LinkedObject[])oldInput;
				for(int i = reuse; i < objs.length; i++)
					objs[i].dispose();
			}
			
			synchronized(loadQueue)
			{
				loadQueue.clear();
				if(newInput instanceof LinkedObject[])
					loadQueue.addAll(Arrays.asList((LinkedObject[])newInput));
				loadQueue.notify();
			}
		}
		
		public Image getColumnImage(Object element, int columnIndex)
		{
			if(columnIndex == 0)
				return getImage(element);
			else
				return null;
		}
		
		public String getColumnText(Object element, int columnIndex)
		{
//			if(columnIndex == 2)
//				return getText(element);
//			else
				return "";
		}
	}

}