package monalipse.editors;

import java.net.MalformedURLException;
import java.net.URL;
import java.util.ArrayList;
import java.util.List;
import java.util.logging.Logger;

import monalipse.MonalipsePlugin;
import monalipse.bookmark.BookmarkManager;
import monalipse.server.BBSServerManager;
import monalipse.server.IBBSBoard;
import monalipse.server.IBBSReference;
import monalipse.server.ILinkedLineFragment;
import monalipse.server.IResponseEnumeration;
import monalipse.server.IThreadContentProvider;
import monalipse.server.LinkedObject;
import monalipse.server.Response;
import monalipse.utils.CancelableRunner;
import monalipse.widgets.ColoredText;
import monalipse.widgets.ColoredText.Attributes;
import monalipse.widgets.ColoredText.LinkTarget;
import monalipse.widgets.ColoredText.TextEvent;
import monalipse.widgets.ColoredText.ToolTipTarget;
import org.eclipse.core.resources.IFile;
import org.eclipse.core.resources.IMarker;
import org.eclipse.core.resources.IResourceChangeEvent;
import org.eclipse.core.resources.IResourceChangeListener;
import org.eclipse.core.resources.ResourcesPlugin;
import org.eclipse.core.runtime.CoreException;
import org.eclipse.core.runtime.IAdaptable;
import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.core.runtime.NullProgressMonitor;
import org.eclipse.core.runtime.QualifiedName;
import org.eclipse.jface.action.IMenuListener;
import org.eclipse.jface.action.IMenuManager;
import org.eclipse.jface.action.MenuManager;
import org.eclipse.jface.preference.PreferenceConverter;
import org.eclipse.jface.text.IFindReplaceTarget;
import org.eclipse.jface.text.ITextOperationTarget;
import org.eclipse.jface.util.IPropertyChangeListener;
import org.eclipse.jface.util.PropertyChangeEvent;
import org.eclipse.jface.viewers.ISelection;
import org.eclipse.jface.viewers.ISelectionChangedListener;
import org.eclipse.jface.viewers.ISelectionProvider;
import org.eclipse.jface.viewers.IStructuredSelection;
import org.eclipse.jface.viewers.SelectionChangedEvent;
import org.eclipse.jface.viewers.StructuredSelection;
import org.eclipse.swt.SWT;
import org.eclipse.swt.dnd.DND;
import org.eclipse.swt.dnd.DragSource;
import org.eclipse.swt.dnd.DragSourceEvent;
import org.eclipse.swt.dnd.DragSourceListener;
import org.eclipse.swt.dnd.FileTransfer;
import org.eclipse.swt.dnd.Transfer;
import org.eclipse.swt.events.ArmEvent;
import org.eclipse.swt.events.ArmListener;
import org.eclipse.swt.events.DisposeEvent;
import org.eclipse.swt.events.DisposeListener;
import org.eclipse.swt.events.KeyEvent;
import org.eclipse.swt.events.KeyListener;
import org.eclipse.swt.events.MenuEvent;
import org.eclipse.swt.events.MenuListener;
import org.eclipse.swt.events.MouseAdapter;
import org.eclipse.swt.events.MouseEvent;
import org.eclipse.swt.graphics.Color;
import org.eclipse.swt.graphics.Device;
import org.eclipse.swt.graphics.Font;
import org.eclipse.swt.graphics.FontData;
import org.eclipse.swt.graphics.Image;
import org.eclipse.swt.graphics.Point;
import org.eclipse.swt.graphics.RGB;
import org.eclipse.swt.program.Program;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Display;
import org.eclipse.swt.widgets.Label;
import org.eclipse.swt.widgets.Menu;
import org.eclipse.swt.widgets.MenuItem;
import org.eclipse.swt.widgets.ScrollBar;
import org.eclipse.ui.IEditorInput;
import org.eclipse.ui.IEditorPart;
import org.eclipse.ui.IEditorSite;
import org.eclipse.ui.IPartListener;
import org.eclipse.ui.ISelectionListener;
import org.eclipse.ui.IWorkbenchPage;
import org.eclipse.ui.IWorkbenchPart;
import org.eclipse.ui.IWorkbenchPartSite;
import org.eclipse.ui.PartInitException;
import org.eclipse.ui.part.EditorPart;

public class ThreadViewerEditor extends EditorPart implements ColoredText.ToolTipProvider, ColoredText.LinkProvider, IFindReplaceTarget, IPartListener, IResourceChangeListener, ISelectionListener
{
	private static String PREF_PREFIX = "threadViewer.";
	public static String PREF_BODY_COLOR = PREF_PREFIX + "body.color";
	private static RGB DEFAULT_BODY_COLOR = new RGB(0x00, 0x00, 0x00);
	public static String PREF_LINK_COLOR = PREF_PREFIX + "link.color";
	private static RGB DEFAULT_LINK_COLOR = new RGB(0x00, 0x00, 0xff);
	public static String PREF_LINK_CLICKED_COLOR = PREF_PREFIX + "link.clicked.color";
	private static RGB DEFAULT_LINK_CLICKED_COLOR = new RGB(0x66, 0x00, 0x99);
	public static String PREF_MARKING_COLOR = PREF_PREFIX + "marking.color";
	private static RGB DEFAULT_MARKING_COLOR = new RGB(0xff, 0x00, 0x00);
	public static String PREF_NAME_COLOR = PREF_PREFIX + "name.color";
	private static RGB DEFAULT_NAME_COLOR = new RGB(0x22, 0x8b, 0x22);
	public static String PREF_MAIL_COLOR = PREF_PREFIX + "mail.color";
	private static RGB DEFAULT_MAIL_COLOR = new RGB(0x00, 0x00, 0xff);
	public static String PREF_BACKGROUND_COLOR = PREF_PREFIX + "background.color";
	private static RGB DEFAULT_BACKGROUND_COLOR = new RGB(0xef, 0xef, 0xef);
	public static String PREF_NORMAL_FONT = PREF_PREFIX + "normal.font";
	private static FontData DEFAULT_NORMAL_FONT = new FontData("MS P Gothic", 12, 0);
	public static String PREF_LINE_HEIGHT = PREF_PREFIX + "line.height";
	private static int DEFAULT_LINE_HEIGHT = 18;

	static
	{
		PreferenceConverter.setDefault(MonalipsePlugin.getDefault().getPreferenceStore(), PREF_BODY_COLOR, DEFAULT_BODY_COLOR);
		PreferenceConverter.setDefault(MonalipsePlugin.getDefault().getPreferenceStore(), PREF_LINK_COLOR, DEFAULT_LINK_COLOR);
		PreferenceConverter.setDefault(MonalipsePlugin.getDefault().getPreferenceStore(), PREF_LINK_CLICKED_COLOR, DEFAULT_LINK_CLICKED_COLOR);
		PreferenceConverter.setDefault(MonalipsePlugin.getDefault().getPreferenceStore(), PREF_MARKING_COLOR, DEFAULT_MARKING_COLOR);
		PreferenceConverter.setDefault(MonalipsePlugin.getDefault().getPreferenceStore(), PREF_NAME_COLOR, DEFAULT_NAME_COLOR);
		PreferenceConverter.setDefault(MonalipsePlugin.getDefault().getPreferenceStore(), PREF_MAIL_COLOR, DEFAULT_MAIL_COLOR);
		PreferenceConverter.setDefault(MonalipsePlugin.getDefault().getPreferenceStore(), PREF_BACKGROUND_COLOR, DEFAULT_BACKGROUND_COLOR);
		PreferenceConverter.setDefault(MonalipsePlugin.getDefault().getPreferenceStore(), PREF_NORMAL_FONT, DEFAULT_NORMAL_FONT);
		MonalipsePlugin.getDefault().getPreferenceStore().setDefault(PREF_LINE_HEIGHT, DEFAULT_LINE_HEIGHT);
	}

	private static final Logger logger = MonalipsePlugin.getLogger();

	private ThreadEditorInput input;
	private Display display;
	private ColoredText text;
	private ThreadViewerAttributes attributes;
	private int sequence;
	private int responseCount;
	private boolean writable;
	private boolean toolTipActive;
	private SelectionProvider selectionProvider;
	private boolean active = true;
	private boolean toBeUpdated;
	private LinkedObject[] linkedObjects;
	private boolean linkedObjectsValid;

	public ThreadViewerEditor()
	{
	}

	public void updateThread(final int scrollTo)
	{
		CancelableRunner cancelable = ((ThreadViewerEditorActionBarContributor)getEditorSite().getActionBarContributor()).getCancelable();
		cancelable.run(this, new CancelableRunner.ICancelableRunnableWithProgress()
			{
				public void run(CancelableRunner.ICancelableProgressMonitor monitor)
				{
					updateThread(monitor, true, scrollTo);
				}
			});
	}
	
	private void updateThread(CancelableRunner.ICancelableProgressMonitor monitor, final boolean download, int scroll)
	{
		monitor.beginTask("getting thread...", 130);

		try
		{
			IThreadContentProvider thread = getContentProvider();
			if(thread == null)
				return;
			
			display.syncExec(new Runnable()
				{
					public void run()
					{
						setTitleImage(MonalipsePlugin.getDefault().getLabelImageOf(getContentProvider()));
						for(int i = 0; i < text.getLineCount(); i++)
						{
							if(text.getLineAt(i) instanceof Response.HeaderLine)
								((Response.HeaderLine)text.getLineAt(i)).getResponse().unmarkNewResponse();
						}
						text.redraw();
					}
				});

			monitor.worked(10);
			IResponseEnumeration e;
			if(download)
				e = thread.updateResponses(monitor.getRunner().getSubProgressMonitor(monitor, 100), getSite().getWorkbenchWindow(), sequence, responseCount);
			else
				e = thread.getResponses(monitor.getRunner().getSubProgressMonitor(monitor, 100), sequence, responseCount);
			writable = e.isWritable();
			monitor.worked(10);
	
			if(e == null)
				return;

			try
			{
				if(!e.isPartialContent())
				{
					responseCount = 0;
					display.syncExec(new Runnable()
						{
							public void run()
							{
								text.clear();
							}
						});
				}

				monitor.worked(10);
				
				final ColoredText.TextPosition[] latestResponseLocation = new ColoredText.TextPosition[1];
				final ColoredText.TextPosition[] scrollTo = new ColoredText.TextPosition[1];
				if(download)
				{
					display.syncExec(new Runnable()
						{
							public void run()
							{
								if(responseCount != 0 && saveScrollPosition() == responseCount)
								{
									for(int i = 0; i < text.getLineCount(); i++)
									{
										ColoredText.Line line = text.getLineAt(i);
										if(line instanceof Response.HeaderLine &&
											((Response.HeaderLine)line).getReponseNumber() == responseCount)
										{
											latestResponseLocation[0] = new ColoredText.TextPosition(i, 0);
										}
									}
								}
							}
						});
				}
				
				if(scroll <= responseCount)
					scrollTo[0] = latestResponseLocation[0];

				final List responses = new ArrayList();
				long bulkWrite = System.currentTimeMillis();
				int writePeriod = responseCount + 10;
				
				Runnable applyLines = new Runnable()
					{
						public void run()
						{
							if(!download)
							{
								for(int i = 0; i < responses.size(); i++)
									((Response)responses.get(i)).unmarkNewResponse();
							}

							List lines = new ArrayList();
							for(int i = 0; i < responses.size(); i++)
							{
								((Response)responses.get(i)).getLines(lines);
								lines.add(new ColoredText.Line(0));
							}
							text.addLines(lines);
							responses.clear();
							
							if(scrollTo[0] != null)
							{
								Point pt = text.getPointFromIndex(scrollTo[0]);
								ScrollBar vert = text.getVerticalBar();
								if(vert.getSelection() + pt.y < vert.getMaximum() - vert.getThumb())
								{
									text.scrollTo(scrollTo[0], SWT.TOP);
									scrollTo[0] = null;
								}
							}
						}
					};

				try
				{
					while(!monitor.isCanceled() && e.hasNextResponse())
					{
						Response response = e.getNextResponse();
						if(response != null)
						{
							responses.add(response);

							if(responseCount == 0)
							{
								input.setTitle(e.getTitle());
								display.syncExec(new Runnable()
									{
										public void run()
										{
											setTitle(input.getTitle());
											((ThreadViewerEditorActionBarContributor)getEditorSite().getActionBarContributor()).updateBookmarkDependentActions();
											ColoredText.Line line = new ColoredText.Line(0);
											line.addLineFragment(new ColoredText.LineFragment(input.getTitle(), ThreadViewerAttributes.COLOR_MARKING, ThreadViewerAttributes.FONT_NORMAL, false));
											text.addLine(line);
											text.addLine(new ColoredText.Line(0));
										}
									});
							}

							responseCount++;
							
							if(responseCount < 5 || responseCount == writePeriod || 300 < System.currentTimeMillis() - bulkWrite || scroll == responseCount)
							{
								bulkWrite = System.currentTimeMillis();
								display.syncExec(applyLines);
								
								if(scroll == responseCount)
								{
									display.syncExec(new Runnable()
										{
											public void run()
											{
												for(int i = text.getLineCount() - 1; 0 <= i; i--)
												{
													ColoredText.Line line = text.getLineAt(i);
													if(line instanceof Response.HeaderLine)
													{
														scrollTo[0] = new ColoredText.TextPosition(i, 0);
														break;
													}
												}
											}
										});
								}
							}
							
							if(responseCount == 1)
							{
								String tooltipText = input.getTitle() + "\n" + input.getURL();
								input.setToolTipText(tooltipText);
								display.syncExec(new Runnable()
									{
										public void run()
										{
											setTitleToolTip(input.getToolTipText());
											setTitle(input.getTitle());
										}
									});
							}
						}
					}
				}
				catch (InterruptedException ex)
				{
				}
				finally
				{
					if(0 < responses.size())
						display.syncExec(applyLines);
				}

				sequence = e.getSequenceNumber();
				
				if(responseCount < scroll)
					scrollTo[0] = latestResponseLocation[0];

				if(scrollTo[0] != null)
				{
					display.syncExec(new Runnable()
						{
							public void run()
							{
								text.scrollTo(scrollTo[0], SWT.TOP);
							}
						});
				}
			}
			finally
			{
				if(linkedObjects != null)
					linkedObjectsValid = false;
				e.close();
			}
		}
		finally
		{
			monitor.done();
		}
	}
	
	public LinkedObject[] getLinkedObjects()
	{
		if(linkedObjects == null)
			linkedObjects = new LinkedObject[0];
		if(!linkedObjectsValid)
			updateLinkedObjects();
		return linkedObjects;
	}
	
	public void updateLinkedObjects()
	{
//logger.finest("enter");
		linkedObjectsValid = true;

		IThreadContentProvider thread = getContentProvider();
		if(thread == null)
			return;

		logger.finest("begin");
		List list = new ArrayList();
		int responseNumber = 0;
		for(int i = 0; i < text.getLineCount(); i++)
		{
			ColoredText.Line line = text.getLineAt(i);
			if(line instanceof Response.HeaderLine)
				responseNumber = ((Response.HeaderLine)line).getReponseNumber();
			for(int j = 0; j < line.getLineFragmentCount(); j++)
			{
				ColoredText.LineFragment f = line.getLineFragmentAt(j);
				if(f instanceof ILinkedLineFragment)
				{
					String link = ((ILinkedLineFragment)f).getURL();
					try
					{
						if(LinkedObject.matches(link))
						{
							if(list.size() < linkedObjects.length &&
								linkedObjects[list.size()].getResponseNumber() == responseNumber &&
								linkedObjects[list.size()].getURLText().equals(link))
							{
								list.add(linkedObjects[list.size()]);
							}
							else
							{
								list.add(new LinkedObject(thread, responseNumber, new URL(link)));
							}
						}
					}
					catch (MalformedURLException e)
					{
					}
				}
			}
		}
		
		linkedObjects = new LinkedObject[list.size()];
		list.toArray(linkedObjects);

		logger.finest("done");
	}
	
	public void disposeCache()
	{
		try
		{
			getContentProvider().getLogFile().delete(true, false, new NullProgressMonitor());
		}
		catch (CoreException e)
		{
		}
	}
	
	public void lockToolTip()
	{
		text.lockToolTip();
	}
	
	public ITextOperationTarget getTextOperationTarget()
	{
		return text;
	}

	public boolean scrollTo(final int responseNumber)
	{
//logger.finest("enter");
		final boolean[] res = new boolean[1];
		display.asyncExec(new Runnable()
			{
				public void run()
				{
					for(int i = 0; text != null && i < text.getLineCount(); i++)
					{
						ColoredText.Line line = text.getLineAt(i);
						if(line instanceof Response.HeaderLine)
						{
							if(((Response.HeaderLine)line).getReponseNumber() == responseNumber)
							{
								ColoredText.TextPosition read = new ColoredText.TextPosition(i, 0);
								text.scrollTo(read, SWT.TOP);
								res[0] = true;
								break;
							}
						}
					}
				}
			});
		return res[0];
	}
	
	public int getResponseCount()
	{
		return responseCount;
	}

	private int saveScrollPosition()
	{
//logger.finest("enter");
		if(text.getVerticalBar() == null || 0 < text.getVerticalBar().getSelection())
		{
			ColoredText.TextPosition bottomLine = text.getIndexFromPoint(new Point(0, text.getSize().y - 1), false);
			if(bottomLine.isValid())
			{
				for(int i = bottomLine.row; 0 <= i; i--)
				{
					ColoredText.Line line = text.getLineAt(i);
					if(line instanceof Response.HeaderLine)
					{
						int read = ((Response.HeaderLine)line).getReponseNumber();
						try
						{
							IThreadContentProvider cp = getContentProvider();
							cp.getLogFile().setPersistentProperty(new QualifiedName(ThreadViewerEditor.class.getName(), "read"), String.valueOf(read));
							cp.getLogFile().setPersistentProperty(new QualifiedName(ThreadViewerEditor.class.getName(), "readLine"), String.valueOf(bottomLine.row));
						}
						catch (CoreException e)
						{
						}
						return read;
					}
				}
			}
		}
		return 0;
	}
	
	private void loadScrollPosition()
	{
//logger.finest("enter");
		try
		{
			IThreadContentProvider cp = getContentProvider();
			ColoredText.TextPosition read = new ColoredText.TextPosition(Integer.parseInt(cp.getLogFile().getPersistentProperty(new QualifiedName(ThreadViewerEditor.class.getName(), "readLine"))), 0);
			text.scrollTo(read, SWT.BOTTOM);
		}
		catch (CoreException e)
		{
		}
		catch (NumberFormatException e)
		{
		}
	}

	public IThreadContentProvider getContentProvider()
	{
//logger.finest("enter");
		try
		{
			return BBSServerManager.getThreadContentProviderOf(new URL(input.getURL()));
		}
		catch (MalformedURLException e)
		{
			return null;
		}
	}

	public boolean isWritable()
	{
		return writable;
	}

	public void doSave(IProgressMonitor monitor)
	{
	}

	public void doSaveAs()
	{
	}

	public void gotoMarker(IMarker marker)
	{
	}

	public void init(IEditorSite site, IEditorInput input) throws PartInitException
	{
		setSite(site);
		setInput(input);
		
		if(input instanceof ThreadEditorInput)
			this.input = (ThreadEditorInput)input;
	}

	public boolean isDirty()
	{
		return false;
	}

	public boolean isSaveAsAllowed()
	{
		return false;
	}
	
	public static void activateEditor(IWorkbenchPartSite site, IThreadContentProvider thread)
	{
//logger.finest("enter");
		logger.finest("begin");
		IWorkbenchPage page = site.getPage().getWorkbenchWindow().getActivePage();
		IEditorPart part = page.findEditor(new ThreadEditorInput(thread));
		if(part != null)
		{
			try
			{
				logger.finest("activate");
				page.openEditor(new ThreadEditorInput(thread), ThreadViewerEditor.class.getName(), false);
			}
			catch(PartInitException ex)
			{
				ex.printStackTrace();
			}
		}
		logger.finest("done");
	}

	public void createPartControl(Composite parent)
	{
//logger.finest("enter");
		display = parent.getShell().getDisplay();
		
		attributes = new ThreadViewerAttributes(parent.getDisplay());

		MonalipsePlugin.getDefault().getPreferenceStore().addPropertyChangeListener(new IPropertyChangeListener()
			{
				public void propertyChange(PropertyChangeEvent event)
				{
					if(event.getProperty().startsWith(PREF_PREFIX))
					{
						attributes.dispose();
						attributes = new ThreadViewerAttributes(getSite().getShell().getDisplay());
						text.setAttributes(attributes);
					}
				}
			});
		
		text = new ColoredText(parent, SWT.SMOOTH);
		text.setAttributes(attributes);
		text.setBackgroundAttribute(ThreadViewerAttributes.COLOR_BACKGROUND);
		text.setToolTipProvider(this);
		text.setLinkProvider(this);
		text.addTextTrackListener(new ColoredText.TextTrackListener()
			{
				public void textEnter(TextEvent e)
				{
					if(e.fragment instanceof ColoredText.ToolTipTarget)
						getContentProvider().prefetchToolTip((ColoredText.ToolTipTarget)e.fragment);
				}
	
				public void textExit(TextEvent e)
				{
				}
	
				public void textHover(TextEvent e)
				{
				}
				
				public void textSelectionEnter(TextEvent e)
				{
				}
				
				public void textSelectionExit(TextEvent e)
				{
				}
				
				public void textSelectionHover(TextEvent e)
				{
				}

			});
		text.addSelectionChangedListener(new ISelectionChangedListener()
			{
				public void selectionChanged(SelectionChangedEvent event)
				{
					((ThreadViewerEditorActionBarContributor)getEditorSite().getActionBarContributor()).updateSelectionDependentActions();
					
					int sel = getSelectedResponseNumber();
					if(sel != -1)
						selectionProvider.fireSelectionChangedEvent(sel);
				}
			});
		text.addKeyListener(new KeyListener()
			{
				public void keyPressed(KeyEvent e)
				{
					if(e.keyCode == SWT.ESC)
						text.setSelection(ColoredText.TextPosition.INVALID, ColoredText.TextPosition.INVALID, true);
					else if(e.keyCode == SWT.F5)
						updateThread(-1);
				}
	
				public void keyReleased(KeyEvent e)
				{
				}
			});

		hookContextMenu();
		
		ResourcesPlugin.getWorkspace().addResourceChangeListener(this);
		getSite().getPage().addSelectionListener(this);
		getSite().getPage().addPartListener(this);
		selectionProvider = new SelectionProvider();
		getSite().setSelectionProvider(selectionProvider);

		CancelableRunner cancelable = ((ThreadViewerEditorActionBarContributor)getEditorSite().getActionBarContributor()).getCancelable();
		cancelable.setDisplay(display);
		cancelable.runDirect(new CancelableRunner.ICancelableRunnableWithProgress()
			{
				public void run(CancelableRunner.ICancelableProgressMonitor monitor)
				{
					updateThread(monitor, false, -1);
				}
			});

		Thread thread = new Thread(new Runnable()
			{
				public void run()
				{
					display.asyncExec(new Runnable()
						{
							public void run()
							{
								loadScrollPosition();
							}
						});
				}
			});
		thread.start();
		try
		{
			thread.join();
		}
		catch (InterruptedException e)
		{
		}
	}
	
	private int getSelectedResponseNumber()
	{
		if(text.getSelection() instanceof ColoredText.TextSelection)
		{
			ColoredText.TextPosition pos = ((ColoredText.TextSelection)text.getSelection()).from;
			if(pos.isValid())
			{
				for(int i = pos.row; 0 <= i; i--)
				{
					ColoredText.Line line = text.getLineAt(i);
					if(line instanceof Response.HeaderLine)
					{
						return ((Response.HeaderLine)line).getReponseNumber();
					}
				}
			}
		}

		return -1;
	}
	
	public void selectionChanged(IWorkbenchPart part, ISelection selection)
	{
		if(selection instanceof IStructuredSelection && getContentProvider() != null)
		{
			Object first = ((IStructuredSelection)selection).getFirstElement();
			IBBSReference ref = null;
			if(first instanceof IBBSReference)
				ref = (IBBSReference)first;
			else if(first instanceof IAdaptable)
				ref = (IBBSReference)((IAdaptable)first).getAdapter(IBBSReference.class);
			
			if(ref != null && ref.getResponseNumber() != -1 && ref.getURL().toExternalForm().equals(getContentProvider().getURL().toExternalForm()))
				scrollTo(ref.getResponseNumber());
		}
	}
	
	public void resourceChanged(IResourceChangeEvent event)
	{
//logger.finest("enter");
		logger.finest("begin");
		IThreadContentProvider thread = getContentProvider();
		if(thread != null)
		{
			boolean threadChanged = thread.threadChanged(event);
			boolean boardChanged = thread.getBoard().threadListChanged(event);
			boolean bookmarkChanged = BookmarkManager.bookmarkChanged(event);
			if(threadChanged || boardChanged || bookmarkChanged)
			{
				setTitleImage(MonalipsePlugin.getDefault().getLabelImageOf(thread));
//				if(!threadChanged && active)
//					updateThread(-1);
			}
		}
		logger.finest("done");
	}
	
	public void setToBeUpdated(boolean updated)
	{
		toBeUpdated = updated;
	}

	private void hookContextMenu()
	{
		MenuManager menuMgr = new MenuManager("#PopupMenu");
		menuMgr.setRemoveAllWhenShown(true);
		menuMgr.addMenuListener(new IMenuListener()
			{
				public void menuAboutToShow(IMenuManager manager)
				{
					ThreadViewerEditorActionBarContributor cont = (ThreadViewerEditorActionBarContributor)getEditorSite().getActionBarContributor();
					cont.contributeToContextMenu(manager, ThreadViewerEditor.this);
				}
			});
		final Menu menu = menuMgr.createContextMenu(text);
		menu.addMenuListener(new MenuListener()
			{
				private boolean showHover;
				private List listeners = new ArrayList();

				public void menuShown(MenuEvent e)
				{
					Display display = getSite().getShell().getDisplay();
					final Point cursotLocation = text.toControl(display.getCursorLocation());
					MenuItem[] items = menu.getItems();
					for(int i = 0; i < items.length; i++)
					{
						ArmListener listener = null;

						if(items[i].getText().startsWith(">>"))
						{
							try
							{
								String label = items[i].getText();
								final int res = Integer.parseInt(label.substring(2, label.length()));
								listener = new ArmListener()
									{
										public void widgetArmed(ArmEvent e)
										{
											if(!text.isToolTipVisible() || showHover)
											{
												text.showToolTip(cursotLocation.x, cursotLocation.y, new ResponseReferToolTipTarget(res, res), null, false, true);
												showHover = true;
											}
										}
									};
							}
							catch (NumberFormatException ex)
							{
							}
						}
						
						if(listener == null)
						{
							listener = new ArmListener()
								{
									public void widgetArmed(ArmEvent e)
									{
										if(showHover)
											text.disposeToolTip();
									}
								};
						}
						
						items[i].addArmListener(listener);
						listeners.add(listener);
					}
				}
				
				public void menuHidden(MenuEvent e)
				{
					showHover = false;
					MenuItem[] items = menu.getItems();
					for(int i = 0; i < items.length && i < listeners.size(); i++)
						items[i].removeArmListener((ArmListener)listeners.get(i));
					listeners.clear();
				}
			});
		text.setMenu(menu);
	}
	
	
	public void linkClicked(ColoredText text, LinkTarget target)
	{
		if(target instanceof ILinkedLineFragment)
		{
			String href = ((ILinkedLineFragment)target).getURL();
			try
			{
				IThreadContentProvider thread = BBSServerManager.getThreadContentProviderOf(new URL(href));
				if(thread != null)
				{
					IWorkbenchPage page = getSite().getWorkbenchWindow().getActivePage();
					try
					{
						boolean update = page.findEditor(new ThreadEditorInput(thread)) == null;
						IEditorPart part = page.openEditor(new ThreadEditorInput(thread), ThreadViewerEditor.class.getName());
						if(part instanceof ThreadViewerEditor)
						{
							int pos = ((ILinkedLineFragment)target).getTargetResponseNumber();
							if(0 < pos && ((ThreadViewerEditor)part).scrollTo(pos))
								pos = -1;
							if(update)
								((ThreadViewerEditor)part).updateThread(pos);
						}
					}
					catch(PartInitException ex)
					{
						ex.printStackTrace();
					}
					return;
				}

				IBBSBoard board = BBSServerManager.getBoardOf(new URL(href));
				if(board != null)
				{
					selectionProvider.fireSelectionChangedEvent(board);
					return;
				}

				if(href.startsWith("http"))
				{
					Program.launch(href);
					return;
				}
			}
			catch (MalformedURLException ex)
			{
				ex.printStackTrace();
			}
		}
	}
	
	public static Point createColoredTextToolTip(Composite parent, ColoredText sourceText, int maxWidth, String title, Response[] responses)
	{
		ColoredText text = new ColoredText(parent, SWT.SMOOTH);
		text.setAttributes(sourceText.getAttributes());
		text.setBackgroundAttribute(sourceText.getBackgroundAttribute());
		text.setToolTipProvider(sourceText.getToolTipProvider());
		text.setToolTipSource(sourceText);
		text.setLinkProvider(sourceText.getLinkProvider());

		if(title != null)
		{
			ColoredText.Line line = new ColoredText.Line(0);
			line.addLineFragment(new ColoredText.LineFragment(title, ThreadViewerEditor.ThreadViewerAttributes.COLOR_MARKING, ThreadViewerEditor.ThreadViewerAttributes.FONT_NORMAL, false));
			text.addLine(line);
			if(responses != null && 0 < responses.length)
				text.addLine(new ColoredText.Line(0));
		}

		if(responses != null && 0 < responses.length)
		{
			List lines = new ArrayList();

			for(int i = 0; i < responses.length; i++)
			{
				responses[i].getLines(lines);
				lines.add(new ColoredText.Line(0));
			}

			for(int i = lines.size() - 1; 0 <= i; i--)
			{
				if(((ColoredText.Line)lines.get(i)).getText().trim().length() == 0)
					lines.remove(i);
				else
					break;
			}

			text.addLines(lines);
		}

		Point size = text.computeSize(SWT.DEFAULT, SWT.DEFAULT, true);
		if(maxWidth < size.x)
			size = new Point(maxWidth, text.setWrapWidth(maxWidth));
		text.setSize(size);
		return size;
	}
	
	private Point createImageToolTip(Composite parent, URL url)
	{
		final LinkedObject obj = new LinkedObject(getContentProvider(), 0, url);
		obj.loadImage();
		Image image = obj.getImage(getSite().getShell().getDisplay());
		final IFile file = obj.getCachedFile();
		final Label label = new Label(parent, SWT.NONE);
		label.setImage(image);
		Point size = label.computeSize(SWT.DEFAULT, SWT.DEFAULT);
		
		label.addDisposeListener(new DisposeListener()
			{
				public void widgetDisposed(DisposeEvent e)
				{
					obj.dispose();
				}
			});

		label.addMouseListener(new MouseAdapter()
			{
				public void mouseDoubleClick(MouseEvent e)
				{
					try
					{
						MonalipsePlugin.ensureSynchronized(file);
						if(file.exists())
						{
							Program.launch(file.getLocation().toOSString());
							text.disposeToolTip();
						}
						else
						{
							new Thread(new Runnable()
								{
									public void run()
									{
										try
										{
											obj.fetchObject(getSite().getWorkbenchWindow(), true);
										}
										catch (Exception e)
										{
										}
										
										getSite().getShell().getDisplay().asyncExec(new Runnable()
											{
												public void run()
												{
													if(!label.isDisposed())
													{
														obj.dispose();
														obj.loadImage();
														label.setImage(obj.getImage(getSite().getShell().getDisplay()));
													}
												}
											});
									}
								}).start();
						}
					}
					catch (CoreException ex)
					{
					}
				}
			});

		DragSource dragSource = new DragSource(label, DND.DROP_COPY);
		dragSource.setTransfer(new Transfer[]{FileTransfer.getInstance()});
		dragSource.addDragListener(new DragSourceListener()
			{
				public void dragStart(DragSourceEvent event)
				{
					try
					{
						MonalipsePlugin.ensureSynchronized(file);
						if(file.exists())
						{
							event.doit = true;
							return;
						}
					}
					catch (CoreException e)
					{
					}
					event.doit = false;
				}
	
				public void dragSetData(DragSourceEvent event)
				{
					event.data = new String[0];
					try
					{
						MonalipsePlugin.ensureSynchronized(file);
						if(file.exists())
							event.data = new String[]{file.getLocation().toOSString()};
					}
					catch (CoreException e)
					{
					}
				}
	
				public void dragFinished(DragSourceEvent event)
				{
				}
			});
		
		return size;
	}

	public Point fillToolTip(Composite parent, ColoredText text, int maxWidth, ToolTipTarget target)
	{
		if(target instanceof ResponseReferToolTipTarget)
		{
			IThreadContentProvider thread = getContentProvider();
			if(thread == null)
				return new Point(0, 0);

			List responses = new ArrayList();
			thread.getResponses(responses, ((ResponseReferToolTipTarget)target).start, ((ResponseReferToolTipTarget)target).end);
			Response[] r = new Response[responses.size()];
			responses.toArray(r);
			return createColoredTextToolTip(parent, text, maxWidth, getTitle(), r);
		}
		else
		{
			Point size = getContentProvider().fillToolTip(parent, text, maxWidth, target, getTitle());
			
			if(size == null && target instanceof ILinkedLineFragment)
			{
				try
				{
					URL url = new URL(((ILinkedLineFragment)target).getURL());
					
					IBBSBoard board = BBSServerManager.getBoardOf(url);
					if(board != null)
					{
						Label label = new Label(parent, SWT.NONE);
						label.setText(board.getName());
						size = label.computeSize(SWT.DEFAULT, SWT.DEFAULT);
					}
					
					if(size == null && LinkedObject.matches(url.toExternalForm()))
						return createImageToolTip(parent, url);
					
					if(size == null)
					{
						Label label = new Label(parent, SWT.NONE);
						label.setText(url.toExternalForm());
						size = label.computeSize(SWT.DEFAULT, SWT.DEFAULT);
					}
				}
				catch (MalformedURLException e)
				{
				}
			}

			if(size == null)
			{
				Label label = new Label(parent, SWT.NONE);
				label.setText("(error)");
				size = label.computeSize(SWT.DEFAULT, SWT.DEFAULT);
			}
			
			return size;
		}
	}
	
	public Point fillToolTip(Composite parent, ColoredText text, int maxWidth, String selection)
	{
		return getContentProvider().fillToolTip(parent, text, maxWidth, selection, getTitle());
	}
	
	public void toolTipActivated(ColoredText text)
	{
		toolTipActive = true;
		((ThreadViewerEditorActionBarContributor)getEditorSite().getActionBarContributor()).updateToolTipDependentActions();
	}
	
	public void toolTipDeactivated(ColoredText text)
	{
		toolTipActive = false;
		((ThreadViewerEditorActionBarContributor)getEditorSite().getActionBarContributor()).updateToolTipDependentActions();
	}
	
	public boolean isToolTipActive()
	{
		return toolTipActive;
	}

	public void setFocus()
	{
		text.setFocus();
	}
	
	public void dispose()
	{
		ResourcesPlugin.getWorkspace().removeResourceChangeListener(this);
		getSite().getPage().removePartListener(this);
		if(text != null)
			text.dispose();
		text = null;
		if(attributes != null)
			attributes.dispose();
		attributes = null;
		super.dispose();
	}
	
	public boolean canPerformFind()
	{
		return true;
	}

	public int findAndSelect(int offset, String findString, boolean searchForward, boolean caseSensitive, boolean wholeWord)
	{
		ColoredText.TextPosition off = new ColoredText.TextPosition(text.getLineCount() - 1, text.getLineAt(text.getLineCount() - 1).getTextLength());
		for(int i = 0; i < text.getLineCount(); i++)
		{
			int len = text.getLineAt(i).getTextLength();
			if(offset < len)
			{
				off = new ColoredText.TextPosition(i, Math.max(offset, 0));
				break;
			}
			offset -= len;
		}
		return text.textPositionToIntPosition(text.findAndSelect(off, findString, searchForward, caseSensitive, wholeWord));
	}

	public Point getSelection()
	{
		ColoredText.TextSelection sel = text.getTextSelection();
		int start = text.textPositionToIntPosition(sel.from);
		int end = text.textPositionToIntPosition(sel.to);
		return new Point(start, end - start);
	}
	
	public String getSelectionText()
	{
		ColoredText.TextSelection sel = text.getTextSelection();
		return text.getText(sel.from, sel.to, false);
	}

	public boolean isEditable()
	{
		return false;
	}

	public void replaceSelection(String text)
	{
	}
	
	public Object getAdapter(Class adapter)
	{
		if(adapter == IFindReplaceTarget.class)
			return this;
		else
			return null;
	}
	
	public void partActivated(IWorkbenchPart part)
	{
		if(part == this)
		{
			active = true;
			
//			IThreadContentProvider thread = getContentProvider();
//			if(thread != null)
//			{
//				if(toBeUpdated  || thread.hasNewResponses())
//				{
//					updateThread(-1);
//					toBeUpdated = false;
//				}
//			}
		}
	}

	public void partBroughtToTop(IWorkbenchPart part)
	{
	}

	public void partClosed(IWorkbenchPart part)
	{
	}

	public void partDeactivated(IWorkbenchPart part)
	{
		if(text != null)
			text.disposeToolTip();

		if(part == this && active)
		{
			saveScrollPosition();
			active = false;
		}
	}

	public void partOpened(IWorkbenchPart part)
	{
	}
	
	private class ResponseSelection implements ISelection, IBBSReference
	{
		private int responseNumber;
		
		private ResponseSelection(int responseNumber)
		{
			this.responseNumber = responseNumber;
		}

		public boolean isEmpty()
		{
			return false;
		}

		public String getName()
		{
			return getTitle();
		}

		public int getResponseNumber()
		{
			return responseNumber;
		}

		public URL getURL()
		{
			IThreadContentProvider thread = getContentProvider();
			if(thread == null)
				return null;
			else
				return thread.getURL();
		}
	}
	
	private class ResponseReferToolTipTarget implements ColoredText.ToolTipTarget
	{
		private int start;
		private int end;

		public ResponseReferToolTipTarget(int start, int end)
		{
			this.start = start;
			this.end = end;
		}
	}

	private class SelectionProvider implements ISelectionProvider
	{
		private List selectionChangedListeners = new ArrayList();

		public void addSelectionChangedListener(ISelectionChangedListener listener)
		{
			selectionChangedListeners.add(listener);
		}
		
		public void removeSelectionChangedListener(ISelectionChangedListener listener)
		{
			selectionChangedListeners.remove(listener);
		}

		public ISelection getSelection()
		{
			int sel = getSelectedResponseNumber();
			if(sel != -1)
				return new ResponseSelection(sel);
			else
				return null;
		}

		public void setSelection(ISelection selection)
		{
			if(selection instanceof IBBSReference)
			{
				int sel = ((IBBSReference)selection).getResponseNumber();
				if(sel != -1)
					scrollTo(sel);
			}
		}

		private void fireSelectionChangedEvent(SelectionChangedEvent e)
		{
			for(int i = 0; i < selectionChangedListeners.size(); i++)
				((ISelectionChangedListener)selectionChangedListeners.get(i)).selectionChanged(e);
		}
	
		private void fireSelectionChangedEvent(IBBSBoard board)
		{
			fireSelectionChangedEvent(new SelectionChangedEvent(this, new StructuredSelection(board)));
		}
	
		private void fireSelectionChangedEvent(int responseNumber)
		{
			fireSelectionChangedEvent(new SelectionChangedEvent(this, new ResponseSelection(responseNumber)));
		}
	}

	public static class ThreadViewerAttributes implements Attributes
	{
		public static final int COLOR_BODY = 0;
		public static final int COLOR_LINK = 1;
		public static final int COLOR_LINK_CLICKED = 2;
		public static final int COLOR_MARKING = 3;
		public static final int COLOR_NAME = 4;
		public static final int COLOR_MAIL = 5;
		public static final int COLOR_BACKGROUND = 6;
		public static final int FONT_NORMAL = 0;
		public static final int FONT_BOLD = 1;
	
		private Color[] colors = new Color[7];
		private Font[] fonts = new Font[2];
		private int lineSkip;
		
		public ThreadViewerAttributes(Device device)
		{
			colors[COLOR_BODY] = new Color(device, PreferenceConverter.getColor(MonalipsePlugin.getDefault().getPreferenceStore(), PREF_BODY_COLOR));
			colors[COLOR_LINK] = new Color(device, PreferenceConverter.getColor(MonalipsePlugin.getDefault().getPreferenceStore(), PREF_LINK_COLOR));
			colors[COLOR_LINK_CLICKED] = new Color(device, PreferenceConverter.getColor(MonalipsePlugin.getDefault().getPreferenceStore(), PREF_LINK_CLICKED_COLOR));
			colors[COLOR_MARKING] = new Color(device, PreferenceConverter.getColor(MonalipsePlugin.getDefault().getPreferenceStore(), PREF_MARKING_COLOR));
			colors[COLOR_NAME] = new Color(device, PreferenceConverter.getColor(MonalipsePlugin.getDefault().getPreferenceStore(), PREF_NAME_COLOR));
			colors[COLOR_MAIL] = new Color(device, PreferenceConverter.getColor(MonalipsePlugin.getDefault().getPreferenceStore(), PREF_MAIL_COLOR));
			colors[COLOR_BACKGROUND] = new Color(device, PreferenceConverter.getColor(MonalipsePlugin.getDefault().getPreferenceStore(), PREF_BACKGROUND_COLOR));
			FontData fontData = PreferenceConverter.getFontData(MonalipsePlugin.getDefault().getPreferenceStore(), PREF_NORMAL_FONT);
			fonts[FONT_NORMAL] = new Font(device, fontData);
			fonts[FONT_BOLD] = new Font(device, new FontData(fontData.getName(), fontData.getHeight(), SWT.BOLD));
			lineSkip = MonalipsePlugin.getDefault().getPreferenceStore().getInt(PREF_LINE_HEIGHT);
		}
		
		public Color getColor(int key)
		{
			try
			{
				return colors[key];
			}
			catch(ArrayIndexOutOfBoundsException e)
			{
				return colors[0];
			}
		}
		
		public Font getFont(int key)
		{
			try
			{
				return fonts[key];
			}
			catch(ArrayIndexOutOfBoundsException e)
			{
				return fonts[0];
			}
		}
		
		public int getLineHeight()
		{
			return lineSkip;
		}
		
		public void dispose()
		{
			for(int i = 0; i < colors.length ; i++)
			{
				if(colors[i] != null)
					colors[i].dispose();
				colors[i] = null;
			}
			for(int i = 0; i < fonts.length ; i++)
			{
				if(fonts[i] != null)
					fonts[i].dispose();
				fonts[i] = null;
			}
		}
	}
}
